'use strict';

angular.module('xld.templates', ['xld.templates.environment', 'xld.templates.common']);

angular.module('NewEnvironment', ['ui.router', 'xl.components', 'xld.templates', 'ngLocalize', 'ngSanitize', 'xl.templates', 'ui.bootstrap', 'xl.online.commons']).config(["$stateProvider", "$urlRouterProvider", function ($stateProvider, $urlRouterProvider) {
    $urlRouterProvider.otherwise('/new-environment');
    $stateProvider.state('new-environment', {
        url: '/new-environment',
        templateUrl: 'environment/environments.html',
        controller: 'EnvironmentsManagerController',
        controllerAs: 'vm',
        bindToController: true
    });
}]).constant('xlDeployServicesRoot', 'deployit/').run(["$rootScope", "typeSystemService", "dipHandlers", function ($rootScope, typeSystemService, dipHandlers) {
    typeSystemService.loadTypesDescriptors();

    // Used in DIP Forms
    $rootScope.dipHandlers = dipHandlers;
}]);
'use strict';

angular.module('NewEnvironment').controller('SelectExistingContainerModalController', ["$scope", "$xlModalInstance", "modalData", "utils", "ciResource", "constantService", function ($scope, $xlModalInstance, modalData, utils, ciResource, constantService) {

    var vm = $scope;

    /**
     * Close dialog
     */
    var closeDialogModal = function closeDialogModal() {
        $xlModalInstance.dismiss();
    };

    vm.selectedContainers = [];

    vm.existingContainersHandlers = {
        itemSelectorHandler: function itemSelectorHandler() {
            return ciResource.findAllCisByType(modalData.type, modalData.parentId).then(function (response) {
                return _.map(response.data, function (item) {
                    item.id = utils.removePrefix(item.ref);
                    delete item.ref;
                    return item;
                });
            });
        }
    };

    vm.hasSelectedContainer = function () {
        return vm.selectedContainers.length > 0;
    };

    vm.cancel = function () {
        closeDialogModal();
    };

    vm.addContainer = function () {
        var selected = vm.selectedContainers[0];
        selected.id = constantService.ROOT.INFRASTRUCTURE + '/' + selected.id;
        $xlModalInstance.close(selected);
    };

    $xlModalInstance.handlers = {
        onEscModal: closeDialogModal,
        onCloseModal: closeDialogModal
    };
}]);
'use strict';

angular.module('NewEnvironment').controller('EditDictionaryModalController', ["$scope", "$timeout", "$xlModalInstance", "modalData", "typeSystemService", "confirmDialogService", function ($scope, $timeout, $xlModalInstance, modalData, typeSystemService, confirmDialogService) {

    var askConfirmation = function askConfirmation() {
        confirmDialogService.openConfirmDialog({
            message: 'Are you sure you want to leave? Any changes will be lost.',
            confirm: 'Yes', reject: 'Cancel', showIf: !$scope.formEditDictionary.form.$pristine
        }).then(function () {
            return $xlModalInstance.dismiss();
        });
    };

    $scope.formEditDictionary = { form: undefined };
    $scope.dictionary = modalData.dictionary;

    $xlModalInstance.handlers = {
        onEscModal: askConfirmation,
        onCloseModal: askConfirmation
    };

    $scope.isValidDictionary = function () {
        return $scope.formEditDictionary.form.$valid && !_.isEmpty($scope.dictionary.id);
    };

    $scope.cancel = function () {
        return askConfirmation();
    };

    $scope.create = function () {
        var input = $scope.dictionary.inputEntry;
        var encryptedInput = $scope.dictionary.encryptedInputEntry;
        var unsaved = !(input.key.trim().length || input.value.trim().length || encryptedInput.key.trim().length || encryptedInput.value.trim().length);

        confirmDialogService.openConfirmDialog({
            message: 'Are you sure you want to save the dictionary without added key and value?',
            confirm: 'Yes', reject: 'Cancel', showIf: !unsaved
        }).then(function () {
            return $xlModalInstance.close($scope.dictionary);
        });
    };
}]);
'use strict';

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

var ContainerService = function () {
    function ContainerService() {
        _classCallCheck(this, ContainerService);
    }

    _createClass(ContainerService, [{
        key: '$get',
        value: function $get($q, ciResource, directoryResource, constantService, ciService, typeSystemService, utils) {
            var saveCiContainer = function saveCiContainer(containerCiObj, ciContainer) {

                var promises = [];

                var metaType = typeSystemService.getTypeDescriptorOf(ciContainer.type);
                var allCis = [];
                if (metaType && metaType.root) {
                    allCis = ciService.createDirectoriesJsonCis(containerCiObj);
                }

                allCis.push(ciContainer);

                _.forEach(allCis, function (ci) {
                    var deferred = $q.defer();

                    ciResource.ciExists(ci.id).then(function (response) {
                        var exists = response.data.boolean;
                        if (!exists) {
                            // ci doesn't exist so we can create it
                            return deferred.resolve(ci);
                        } else {
                            return ciResource.get(ci.id).success(function (returnedCi) {
                                if (ci.type === constantService.TYPE.CORE_DIRECTORY && returnedCi.type !== constantService.TYPE.CORE_DIRECTORY) {
                                    return deferred.reject({
                                        isDirectoryError: true,
                                        message: 'A ci named ' + utils.removePrefix(ci.id) + ' already exists'
                                    });
                                } else if (ci.type !== constantService.TYPE.CORE_DIRECTORY) {
                                    return deferred.resolve(ci);
                                }
                                return deferred.resolve();
                            });
                        }
                    });
                    promises.push(deferred.promise);
                });

                return $q.all(promises).then(function (cis) {
                    var deferred = $q.defer();
                    ciResource.saveAllCis(_.without(cis, undefined)).then(deferred.resolve, function (error) {
                        return deferred.reject({
                            ciError: true,
                            message: error.data
                        });
                    });
                    return deferred.promise;
                });
            };

            var addCandidatesPathContainers = function addCandidatesPathContainers(meta, params) {
                return directoryResource.findAllDirectoriesByAncestor(constantService.ROOT.INFRASTRUCTURE, params.term).then(utils.getCiIdsWithoutPrefix);
            };

            return {
                saveCiContainer: saveCiContainer,
                addCandidatesPathContainers: addCandidatesPathContainers
            };
        }
    }]);

    return ContainerService;
}();

angular.module('NewEnvironment').provider('containerService', ContainerService);
'use strict';

angular.module('NewEnvironment').controller('NewContainerModalController', ["$scope", "$log", "$timeout", "$xlModalInstance", "$q", "ciResource", "modalData", "typeSystemService", "containerService", "constantService", "confirmDialogService", "ciService", function ($scope, $log, $timeout, $xlModalInstance, $q, ciResource, modalData, typeSystemService, containerService, constantService, confirmDialogService, ciService) {
    $scope.containerNameForm = {};
    $scope.typeDescriptor = {};

    $scope.containerTypeNames = typeSystemService.getContainerTypeNames();

    $scope.environmentId = modalData.environmentId;

    $scope.container = {
        name: undefined,
        path: ''
    };

    $scope.onTypeSelect = function (typeName) {
        $scope.typeDescriptor = typeSystemService.getTypeDescriptorOf(typeName);
        var metaParent = _.find($scope.typeDescriptor.properties, function (propertie) {
            return propertie.asContainment && propertie.kind === 'CI';
        });

        if (metaParent && modalData.parentContainer) {
            $scope.container[metaParent.name] = { id: modalData.parentContainer.id, type: modalData.parentContainer.type };
        }
    };

    $scope.containerTypeNames = _.map(modalData.containerTypes, 'type');
    if ($scope.containerTypeNames.length === 1) {
        $scope.container.type = $scope.containerTypeNames[0];
        $scope.onTypeSelect($scope.container.type);
    }

    $scope.create = function () {
        createContainer($scope.container).then(function () {
            $xlModalInstance.close($scope.container);
        }, function (error) {
            $log.error('Error', error);
        });
    };

    // Autocomplete container path
    $scope.containerPathAutocompleteHandlers = {
        addCandidates: containerService.addCandidatesPathContainers
    };

    var createContainer = function createContainer(container) {
        var path = modalData.parentContainer ? ciService.extractCiStructure(modalData.parentContainer.id).idWithoutRoot : $scope.container.path;
        var containerCiObj = ciService.normalizeCiId(constantService.ROOT.INFRASTRUCTURE, path, $scope.container.name);
        var containerCopy = _.cloneDeep(container);
        containerCopy.id = containerCiObj.id;
        $scope.container.id = containerCiObj.id;

        delete containerCopy.name;
        delete containerCopy.path;

        return containerService.saveCiContainer(containerCiObj, containerCopy);
    };

    var openConfirmDialogModal = function openConfirmDialogModal() {
        confirmDialogService.openConfirmDialog({
            message: 'Are you sure you want to leave? Any changes will be lost.',
            confirm: 'Yes', reject: 'Cancel', showIf: !$scope.containerNameForm.internal.$pristine
        }).then(function () {
            return $xlModalInstance.dismiss();
        });
    };

    $xlModalInstance.handlers = {};
    if (modalData.handleEsc) {
        $xlModalInstance.handlers.onEscModal = function () {
            return openConfirmDialogModal();
        };
    }

    if (modalData.hasClose) {
        $xlModalInstance.handlers.onCloseModal = function () {
            return openConfirmDialogModal();
        };
    }

    $scope.close = function () {
        return openConfirmDialogModal();
    };
}]);
'use strict';

angular.module('NewEnvironment').factory('environmentService', ["$q", "typeSystemService", "directoryResource", "versionResource", "environmentsResource", "ciResource", "constantService", "utils", "ciService", "dipHandlers", "$xlModal", function ($q, typeSystemService, directoryResource, versionResource, environmentsResource, ciResource, constantService, utils, ciService, dipHandlers, $xlModal) {

    function createJsonCis(environmentObj, extendedData) {
        var cis = ciService.createDirectoriesJsonCis(environmentObj);
        cis.push(ciService.createJsonCiWithData(environmentObj, constantService.TYPE.UDM_ENVIRONMENT, extendedData));

        return cis;
    }

    function createCiEnvironment(environmentObj, extendedData) {
        var promises = [];

        var allCis = createJsonCis(environmentObj, extendedData);
        _.forEach(allCis, function (ci) {
            var deferred = $q.defer();
            if (ci.type === constantService.TYPE.ENVIRONMENTS) {
                angular.extend(ci, extendedData);
            }
            ciResource.ciExists(ci.id).then(function (response) {
                var exists = response.data.boolean;
                if (!exists) {
                    // ci doesn't exist so we can create it
                    return deferred.resolve(ci);
                } else {
                    return ciResource.get(ci.id).success(function (returnedCi) {
                        if (ci.type === constantService.TYPE.CORE_DIRECTORY && returnedCi.type === constantService.TYPE.UDM_ENVIRONMENT) {
                            return deferred.reject({
                                isDirectoryError: true,
                                message: 'An environment named ' + utils.removePrefix(ci.id) + ' already exists'
                            });
                        } else if (ci.type === constantService.TYPE.UDM_ENVIRONMENT) {
                            return deferred.resolve(ci);
                        }
                        return deferred.resolve();
                    });
                }
            });
            promises.push(deferred.promise);
        });

        return $q.all(promises).then(function (cis) {
            var deferred = $q.defer();
            ciResource.saveAllCis(_.without(cis, undefined)).then(deferred.resolve, function (error) {
                return deferred.reject({
                    isEnvironmentError: true,
                    message: error.data
                });
            });
            return deferred.promise;
        });
    }

    var fetchContainerInstances = function fetchContainerInstances(containerTypes) {
        var promises = _.map(containerTypes, function (containerType) {
            return ciResource.findAllCis(containerType);
        });
        return $q.all(promises);
    };

    var fetchDeployableCis = function fetchDeployableCis(packageCi) {
        if (packageCi && packageCi.id) {
            return ciResource.get(packageCi.id).then(function (deployableCis) {
                return _.map(deployableCis.data.deployables, function (deployableRef) {
                    return { id: deployableRef };
                });
            });
        } else {
            var differed = $q.defer();
            differed.resolve([]);
            return differed.promise;
        }
    };

    var fetchContainerCis = function fetchContainerCis(deployables) {

        var arrPromises = _.map(deployables, function (deployable) {
            return ciResource.get(deployable.id);
        });

        return $q.all(arrPromises).then(function (deployableCis) {
            var allTypes = [];
            _.forEach(deployableCis, function (deployableCi) {
                var deployableType = deployableCi.data.type;
                var types = typeSystemService.getTypesByDeployableType(deployableType);
                allTypes = _.union(allTypes, types);
            });

            var containerTypes = _.map(allTypes, 'containerType');

            return fetchContainerInstances(containerTypes).then(function (result) {
                return _(result).map('data').flatten().uniqBy('ref').map(function (item) {
                    item.id = item.ref;
                    delete item.ref;
                    return item;
                }).value();
            });
        });
    };

    var addCandidatesPackages = function addCandidatesPackages(meta, params) {
        return dipHandlers.ciHandler(constantService.TYPE.UDM_VERSION, params.term);
    };

    var addCandidatesDeployables = function addCandidatesDeployables(deployables, params) {
        var differed = $q.defer();
        var candidates = _.filter(deployables, function (item) {
            return item.id.indexOf(params.term) !== -1;
        });
        differed.resolve(candidates);
        return differed.promise;
    };

    var addCandidatesContainers = function addCandidatesContainers(containers, params) {
        var differed = $q.defer();
        var candidates = _.filter(containers, function (item) {
            return item.id.indexOf(params.term) !== -1;
        });
        differed.resolve(candidates);
        return differed.promise;
    };

    var addCandidatesAllDictionaries = function addCandidatesAllDictionaries(containers, params) {
        return dipHandlers.ciHandler(constantService.TYPE.UDM_DICTIONARY, params.term);
    };

    var addCandidatesDirectories = function addCandidatesDirectories(meta, params) {
        return directoryResource.findAllDirectoriesByAncestor(constantService.ROOT.ENVIRONMENTS, params.term).then(utils.getCiIdsWithoutPrefix);
    };

    var addCandidatesAllContainers = function addCandidatesAllContainers(meta, params) {
        return dipHandlers.ciHandler(constantService.TYPE.UDM_CONTAINER, params.term);
    };

    var getContainerHierarchy = function getContainerHierarchy(typeName) {
        var containerHierarchy = computeContainerHierarchy(typeName);
        return enrichContainerHierarchy(containerHierarchy);
    };

    var enrichContainerHierarchy = function enrichContainerHierarchy(containerHierarchy) {
        var result = [];
        _.forEach(containerHierarchy, function (containerDescription) {
            var elem = {};
            elem.type = containerDescription.type;
            elem.subtypes = [];

            if (containerDescription.virtual) {
                elem.subtypes = typeSystemService.getTypeDescriptorOfSuperType(containerDescription.type);
            } else {
                elem.subtypes.push(containerDescription);
            }

            result.unshift(elem);
        });
        return result;
    };

    var computeContainerHierarchy = function computeContainerHierarchy(typeName, containers) {
        var containerHierarchy = containers ? containers : [];
        var typeDescriptorOf = typeSystemService.getTypeDescriptorOf(typeName);

        if (typeDescriptorOf) {
            containerHierarchy.push(typeDescriptorOf);
            var asContainmentProperties = _.find(typeDescriptorOf.properties, 'asContainment');

            if (asContainmentProperties) {
                computeContainerHierarchy(asContainmentProperties.referencedType, containerHierarchy);
            }
        } else {
            //Virtual
            containerHierarchy.push({ 'type': typeName, 'virtual': true });
        }
        return containerHierarchy;
    };

    var saveDictionary = function saveDictionary(newDictionary) {
        var ciObj = ciService.normalizeCiId(constantService.ROOT.ENVIRONMENTS, undefined, newDictionary.id);
        var ci = ciService.createJsonCiWithData(ciObj, constantService.TYPE.UDM_DICTIONARY, {
            entries: newDictionary.entries,
            encryptedEntries: newDictionary.encryptedEntries
        });

        return ciResource.saveCi(ci);
    };

    var editDictionary = function editDictionary(newDictionary) {
        var ciObj = ciService.extractCiStructure(newDictionary.id);
        var ci = ciService.createJsonCiWithData(ciObj, newDictionary.type, {
            entries: newDictionary.entries,
            encryptedEntries: newDictionary.encryptedEntries
        });
        return ciResource.editCi(ci.id, ci);
    };

    var openEditDictionaryModal = function openEditDictionaryModal(dictionary) {
        var openModal = function openModal() {
            var modalOptions = {
                templateUrl: 'environment/tabs/environment/edit-dictionary-modal/edit-dictionary-modal.html',
                windowClass: 'edit-dictionary-modal',
                controller: 'EditDictionaryModalController',
                resolve: {
                    modalData: function modalData() {
                        return {
                            dictionary: angular.copy(dictionary)
                        };
                    }
                }
            };

            return $xlModal.open(modalOptions).result;
        };

        if (!dictionary.entries) {
            return ciService.getCi(dictionary.id).then(function (response) {
                dictionary.entries = response.data.entries;
                dictionary.encryptedEntries = response.data.encryptedEntries;
                return openModal();
            });
        } else {
            return openModal();
        }
    };

    return {
        fetchDeployableCis: fetchDeployableCis,
        fetchContainerCis: fetchContainerCis,
        addCandidatesPackages: addCandidatesPackages,
        addCandidatesDeployables: addCandidatesDeployables,
        addCandidatesContainers: addCandidatesContainers,
        addCandidatesDirectories: addCandidatesDirectories,
        addCandidatesAllContainers: addCandidatesAllContainers,
        addCandidatesAllDictionaries: addCandidatesAllDictionaries,
        createCiEnvironment: createCiEnvironment,
        getContainerHierarchy: getContainerHierarchy,
        computeContainerHierarchy: computeContainerHierarchy,
        enrichContainerHierarchy: enrichContainerHierarchy,
        saveDictionary: saveDictionary,
        editDictionary: editDictionary,
        openEditDictionaryModal: openEditDictionaryModal
    };
}]);
'use strict';

angular.module('NewEnvironment').controller('EnvironmentController', ["$scope", "$interval", "$log", "$xlModal", "$q", "$window", "WizardHandler", "environmentService", "ciService", "typeSystemService", "utils", "confirmDialogService", "constantService", "ciTabsService", function ($scope, $interval, $log, $xlModal, $q, $window, WizardHandler, environmentService, ciService, typeSystemService, utils, confirmDialogService, constantService, ciTabsService) {
    var $tabData = $scope.$tabData;
    var parentVm = $scope.$parent.vm;

    /**
     * ContainerBundle
     */
    var ContainerBundle = function ContainerBundle(container) {
        this.type = container.type;
        this.subtypes = container.subtypes;
        this.container = undefined;
        this._isNewContainer = false;
    };

    ContainerBundle.prototype.activeButton = function () {
        var index = $tabData.containerHierarchyBundles.indexOf(this);
        if (index === 0) {
            return true;
        }
        return $tabData.containerHierarchyBundles[index - 1].isContainerSelected();
    };

    ContainerBundle.prototype.displayButton = function () {
        var index = $tabData.containerHierarchyBundles.indexOf(this);
        return !$tabData.containerHierarchyBundles[index].isContainerSelected();
    };

    ContainerBundle.prototype.getTitle = function () {
        return this.container ? {
            title: ciService.extractCiStructure(this.container.id).name,
            type: this.container.type
        } : { title: this.type };
    };

    ContainerBundle.prototype.isContainerSelected = function () {
        return !!this.container;
    };

    ContainerBundle.prototype.clearContainer = function () {
        this._isNewContainer = false;
        this.container = undefined;
    };

    ContainerBundle.prototype.isNewContainer = function () {
        return this._isNewContainer;
    };

    ContainerBundle.prototype.setNewContainer = function (value) {
        this._isNewContainer = value;
    };

    ContainerBundle.prototype.clearContainerAndChild = function () {
        var index = $tabData.containerHierarchyBundles.indexOf(this);
        for (var i = index; i < $tabData.containerHierarchyBundles.length; i++) {
            $tabData.containerHierarchyBundles[i].clearContainer();
        }
    };

    ContainerBundle.prototype.getContainer = function () {
        return this.container;
    };

    ContainerBundle.prototype.getParent = function () {
        var index = $tabData.containerHierarchyBundles.indexOf(this);
        if (index === 0 || index === -1) {
            return undefined;
        }
        var parent = $tabData.containerHierarchyBundles[index - 1].container;
        return parent ? { id: parent.id, type: parent.type } : undefined;
    };

    ContainerBundle.prototype.setParent = function () {
        var index = $tabData.containerHierarchyBundles.indexOf(this);
        if (index > 0) {
            var parentContainerBundle = $tabData.containerHierarchyBundles[index - 1];
            parentContainerBundle.updateFromId(this.extractParentId()).then(function () {
                parentContainerBundle.setParent();
            });
        }
    };

    ContainerBundle.prototype.extractParentId = function () {
        var objId = ciService.extractCiStructure(this.container.id);
        return objId.root + objId.path;
    };

    ContainerBundle.prototype.updateFromId = function (id) {
        var currentContainerBundle = this;

        return ciService.getCi(id).then(function (response) {
            currentContainerBundle.container = response.data;
        });
    };

    $tabData.isLoaded = false;
    if (!$tabData.initialized) {
        $scope.environmentDynamicFormModel = {};
        $scope.environmentForms = {};
        $tabData.environmentData = { members: [], dictionaries: [] };
        $tabData.envSetContainerHelp = false;
        $tabData.containerHelper = {
            deployableCandidates: [],
            containerCandidates: []
        };
        $tabData.containerChoice = 'create';
        $tabData.containerHierarchyBundles = [];
        $tabData.dictionaryChoice = 'create';
        $tabData.newDictionary = { id: '', entries: {}, inputEntry: {}, encryptedInputEntry: {} };
    }

    if ($tabData.mode !== constantService.MODES.CREATION && !$tabData.initialized) {

        ciService.getCi($tabData.environmentId).then(function (response) {
            var ci = response.data;
            var data = ciService.extractCiStructure(ci.id);
            $tabData.environmentId = data.idWithoutRoot;
            $tabData.environmentData = ci;
            $tabData.environmentData.dictionaries = _.map(ci.dictionaries, function (dictionary) {
                return { id: dictionary };
            });
            $tabData.environmentData.triggers = _.map(ci.triggers, function (trigger) {
                return { id: trigger, type: null };
            });
            $tabData.environmentData.members = _.map(ci.members, function (member) {
                return {
                    id: _.isString(member) ? utils.removePrefix(member) : utils.removePrefix(member.id),
                    type: _.isString(member) ? null : member.type
                };
            });

            $tabData.isLoaded = true;
        }, function (error) {
            $log.error(error);
        });
    } else {
        $tabData.isLoaded = true;
    }

    $scope.isLoaded = function () {
        return $tabData.isLoaded;
    };

    // Wizard Management
    $scope.verifyEnvNotExistAndGotoNextStep = function () {
        ciService.ciExists(constantService.ROOT.ENVIRONMENTS, $tabData.environmentId).then(function (response) {
            if (response.data.boolean) {
                $tabData.postEnvError = 'Environment already exists';
            } else {
                WizardHandler.wizard().next();
            }
        });
    };

    $scope.verifyContainerIsNotSelectAndGotoNextStep = function () {
        var condition = $tabData.environmentData.members && $tabData.environmentData.members.length > 0;
        confirmDialogService.openConfirmDialog({
            message: 'Containers is empty, are you sure you want to go to the next step ?',
            confirm: 'Yes', reject: 'Cancel', showIf: !condition
        }).then(function () {
            return WizardHandler.wizard().next();
        });
    };

    $scope.verifyDictonariesIsNotSelectAndGotoNextStep = function () {
        var condition = $tabData.environmentData.dictionaries && $tabData.environmentData.dictionaries.length > 0;
        confirmDialogService.openConfirmDialog({
            message: 'Dictionaries is empty, are you sure you want to go to the next step?',
            confirm: 'Yes', reject: 'Cancel', showIf: !condition
        }).then(function () {
            return WizardHandler.wizard().next();
        });
    };

    // STEP 1 : name/path
    // Autocomplete path
    $scope.envPathAutocompleteHandlers = {
        addCandidates: environmentService.addCandidatesDirectories
    };

    $scope.$watch('$tabData.environmentId', function () {
        $tabData.postEnvError = undefined;
    });

    // STEP 2 : containers
    // New Container Modal
    $scope.existingContainersHandlers = {
        itemSelectorHandler: function itemSelectorHandler() {
            return environmentService.addCandidatesAllContainers(undefined, '').then(function (data) {
                return _.map(data, function (container) {
                    return {
                        // Remove Infrastructure
                        id: utils.removePrefix(container.id),
                        type: container.type
                    };
                });
            });
        }
    };

    // Create New Containers
    $scope.containerTypeNames = typeSystemService.getContainerTypeNames();

    $scope.newContainerTypeSelected = function (type) {
        $tabData.containerHierarchyBundles.length = 0;
        var containerHierarchy = environmentService.getContainerHierarchy(type);
        $tabData.containerHierarchyBundles = _.map(containerHierarchy, function (container) {
            return new ContainerBundle(container);
        });
    };

    $scope.$watch('$tabData.newContainerType', function (type) {
        if (_.isUndefined(type)) {
            $tabData.containerHierarchyBundles.length = 0;
        }
    });

    var openNewContainerModal = function openNewContainerModal(containerBundle) {
        var xlModalInstance = $xlModal.open({
            templateUrl: 'environment/tabs/environment/container/container-modal.html',
            controller: 'NewContainerModalController',
            windowClass: 'xl-container-modal',
            resolve: {
                modalData: function modalData() {
                    return {
                        parentContainer: containerBundle.getParent(),
                        containerTypes: containerBundle.subtypes,
                        handleEsc: true,
                        hasClose: true,
                        environmentId: $tabData.environmentId
                    };
                }
            }
        });

        xlModalInstance.result.then(function (selectedItem) {
            var index = $tabData.containerHierarchyBundles.indexOf(containerBundle);

            var containerCopy = angular.copy(selectedItem);
            delete containerCopy.name;
            delete containerCopy.path;

            containerBundle.container = containerCopy;
            containerBundle.setNewContainer(true);

            for (var i = index + 1; i < $tabData.containerHierarchyBundles.length; i++) {
                $tabData.containerHierarchyBundles[i].clearContainer();
                $tabData.containerHierarchyBundles[i].setNewContainer(true);
            }
        });
    };

    $scope.createNewContainer = function (containerBundle) {
        openNewContainerModal(containerBundle);
    };

    $scope.selectExistingContainer = function (containerBundle) {
        var parentId = containerBundle.getParent() ? containerBundle.getParent().id : undefined;

        var xlModalInstance = $xlModal.open({
            templateUrl: 'environment/tabs/environment/select-existing-container/select-existing-container-modal.html',
            controller: 'SelectExistingContainerModalController',
            windowClass: 'xl-select-existing-container-modal',
            resolve: {
                modalData: function modalData() {
                    return {
                        type: containerBundle.type,
                        parentId: parentId
                    };
                }
            }
        });

        xlModalInstance.result.then(function (selectedItem) {
            containerBundle.container = selectedItem;
            containerBundle.setParent();
        });
    };

    $scope.addContainer = function () {
        var containerHierarchyBundles = $tabData.containerHierarchyBundles;
        if (!_.isUndefined(containerHierarchyBundles) && !_.isUndefined(containerHierarchyBundles[containerHierarchyBundles.length - 1])) {
            var container = containerHierarchyBundles[containerHierarchyBundles.length - 1].getContainer();
            $tabData.environmentData.members.push({
                id: utils.removePrefix(container.id),
                type: container.type
            });
            $tabData.newContainerType = '';
            containerHierarchyBundles.length = 0;
        }
    };

    $scope.isValidContainer = function () {
        var containerHierarchyBundles = $tabData.containerHierarchyBundles;
        return _.isUndefined(containerHierarchyBundles) || _.isUndefined(containerHierarchyBundles[containerHierarchyBundles.length - 1]) || !containerHierarchyBundles[containerHierarchyBundles.length - 1].isContainerSelected();
    };

    // Set-Of-Ci
    $scope.envContainersAutocompleteHandlers = {
        addCandidates: environmentService.addCandidatesAllContainers,
        onNewItem: openNewContainerModal
    };

    // Container Helper

    // 1 - $watch package
    $scope.$watch('$tabData.containerHelper.package', function (value, oldValue) {
        if (value === oldValue) {
            return;
        }
        $tabData.containerHelper.deployableCandidates.length = 0;
        $tabData.containerHelper.containerCandidates.length = 0;
        $tabData.containerHelper.deployable = undefined;
    });

    // 2 - $watch deployable
    $scope.$watch('$tabData.containerHelper.deployable', function (value, oldValue) {
        if (value === oldValue) {
            return;
        }
        $tabData.containerHelper.containerCandidates.length = 0;
    });

    // 1 - handler package
    $scope.envContainersHelperPackageHandlers = {
        addCandidates: environmentService.addCandidatesPackages,
        onSelect: function onSelect(packageCi) {
            environmentService.fetchDeployableCis(packageCi).then(function (deployables) {
                $tabData.containerHelper.deployableCandidates = deployables;
            });
        }
    };

    // 2 - handler deployable
    $scope.envContainersHelperDeployableHandlers = {
        addCandidates: function addCandidates(meta, params) {
            return environmentService.addCandidatesDeployables($tabData.containerHelper.deployableCandidates, params);
        },
        onSelect: function onSelect(deployableCi) {
            environmentService.fetchContainerCis([deployableCi]).then(function (data) {
                $tabData.containerHelper.containerCandidates = data;
            });
        }
    };

    // 3 - handler container
    $scope.envContainersHelperContainerHandlers = {
        addCandidates: function addCandidates(meta, params) {
            if ($tabData.containerHelper.containerCandidates.length === 0) {
                return environmentService.addCandidatesAllContainers(meta, params);
            } else {
                return environmentService.addCandidatesContainers($tabData.containerHelper.containerCandidates, params);
            }
        },
        onNewItem: _.noop
    };

    // STEP 3 : dictionaries
    // List-Of-Ci
    $scope.existingDictionariesHandlers = {
        itemSelectorHandler: function itemSelectorHandler() {
            return environmentService.addCandidatesAllDictionaries(undefined, '').then(function (data) {
                return data;
            });
        }
    };

    $scope.envDictionariesAutocompleteHandlers = {
        addCandidates: environmentService.addCandidatesAllDictionaries,
        onEdit: function onEdit(item) {
            environmentService.openEditDictionaryModal(item).then(function (dictionary) {
                environmentService.editDictionary(dictionary).then(function (response) {
                    var foundDictionary = _.find($tabData.environmentData.dictionaries, { id: response.data.id });
                    angular.copy(response.data, foundDictionary);
                });
            });
        },
        onDelete: function onDelete(item) {
            _.remove($tabData.environmentData.dictionaries, item);
        }
    };

    $scope.isValidDictionary = function () {
        return !_.isEmpty($tabData.newDictionary.id);
    };

    $scope.$watch('$tabData.newDictionary.id', function () {
        $scope.postError = undefined;
    });

    $scope.addDictionary = function () {
        var input = $tabData.newDictionary.inputEntry;
        var encryptedInput = $tabData.newDictionary.encryptedInputEntry;
        var unsaved = !(input.key.trim().length || input.value.trim().length || encryptedInput.key.trim().length || encryptedInput.value.trim().length);
        confirmDialogService.openConfirmDialog({
            message: 'Are you sure you want to save the dictionary without added key and value?',
            confirm: 'Yes', reject: 'Cancel', showIf: !unsaved
        }).then(function () {
            $scope.postError = undefined;
            environmentService.saveDictionary($tabData.newDictionary).then(function (response) {
                $tabData.environmentData.dictionaries.push(response.data);
                $tabData.newDictionary.id = '';
                $tabData.newDictionary.entries = {};
                $tabData.newDictionary.encryptedEntries = {};
            }, function (error) {
                $scope.postError = error.data && !error.xlPreventDefault ? error.data : undefined;
            });
        });
    };

    // STEP 4 : advanced properties
    $scope.environmentMetadata = typeSystemService.getTypeDescriptorOf(constantService.TYPE.UDM_ENVIRONMENT) || [];
    $scope.environmentMetadata.properties = _.filter($scope.environmentMetadata.properties, function (it) {
        return it.name !== 'members' && it.name !== 'dictionaries';
    });

    // SAVE ENVIRONMENT
    $scope.saveEnvironment = function () {
        var extendedData = _.cloneDeep($tabData.environmentData);
        _.forEach(extendedData.members, function (container) {
            container.id = constantService.ROOT.INFRASTRUCTURE + '/' + container.id;
        });

        // Prepare Json
        var id = utils.addLeadingSlashIfNotExists($tabData.environmentId);
        var environmentCIObj = ciService.extractCiStructure(constantService.ROOT.ENVIRONMENTS + id);

        environmentService.createCiEnvironment(environmentCIObj, extendedData).then(function () {
            parentVm.unfoldPathToNode(environmentCIObj.idWithoutRoot);
            ciTabsService.switchToViewEnvironment(environmentCIObj.idWithoutRoot);
            $scope.closeEnvironmentWindow();
        }, _.noop);
    };

    $scope.closeEnvironmentWindow = function () {
        return $interval(function () {
            return angular.element('#close-package-window').triggerHandler('click');
        }, 1, 1);
    };

    // Global
    $scope.$watch('environmentForms.envNameForm.$pristine', function (value) {
        if (value === false) {
            $scope.$setTabPristine();
        }
    });

    $tabData.initialized = true;

    var updateSectionHeight = function updateSectionHeight() {
        var headerHeight = 30;
        var sectionHeight = $window.innerHeight - headerHeight;
        angular.element('.xl-steps-content .steps section').css('height', sectionHeight);
    };

    $interval(updateSectionHeight, 1000, 1);

    angular.element($window).bind('resize', function () {
        updateSectionHeight();
    });
}]);
'use strict';

angular.module('NewEnvironment').factory('ciTabsService', ["utils", "$interval", "$uiTabs", "constantService", "confirmDialogService", function (utils, $interval, $uiTabs, constantService, confirmDialogService) {

    /**
     * Opens or replace a CI tab
     * @param targetState target tab type
     * @param options tab options
     * @param tabMode
     *  FORCE_NEW - always opens new tab,
     *  FORCE_SAME - switches the same tab if possible,
     *  AUTO - if current tab is in visual mode, switches the current tab, if not - opens new
     */
    var openOrReplaceTab = function openOrReplaceTab(targetState, options, tabMode) {
        var currentTab = $uiTabs.getActiveTab();
        if (!currentTab || !!currentTab.isWelcome || tabMode === 'FORCE_NEW') {
            $uiTabs.addTab(targetState, options);
        } else if (currentTab && (currentTab.$tabData.mode === 'visual' || tabMode === 'FORCE_SAME')) {
            $uiTabs.addTab(targetState, options, currentTab.$$id);
        }
    };

    // environment tabs
    var newEnvironment = function newEnvironment() {
        openOrReplaceTab('environments.env.create', {
            $tabData: {
                mode: constantService.MODES.CREATION
            }
        }, 'FORCE_NEW');
    };
    var viewEnvironment = function viewEnvironment(environmentId, forceNewTab) {
        openOrReplaceTab('environments.env.view', {
            title: utils.removePrefix(environmentId),
            $tabData: {
                environmentId: environmentId,
                mode: constantService.MODES.VIEW
            }
        }, forceNewTab);
    };

    var closePopupDialog = function closePopupDialog(tab, msg) {
        return confirmDialogService.openConfirmDialog({
            message: msg, confirm: 'Yes', reject: 'Cancel', showIf: !tab.pristine
        }).then(function () {
            return $interval(function () {
                return angular.element('#close-package-window').triggerHandler('click');
            }, 1, 1);
        });
    };

    var editEnvironment = function editEnvironment() {
        var currentTab = $uiTabs.getActiveTab();
        openOrReplaceTab('environments.env.edit', { title: currentTab.title, initialized: currentTab.initialized, data: currentTab.data }, 'FORCE_SAME');
    };

    var switchToViewEnvironment = function switchToViewEnvironment() {
        $uiTabs.getActiveTab().pristine = true;
        $uiTabs.removeTab($uiTabs.getActiveTab().$$id);
    };

    /**
     * Public API
     */
    return {
        newEnvironment: newEnvironment,
        viewEnvironment: viewEnvironment,
        editEnvironment: editEnvironment,
        closePopupDialog: closePopupDialog,
        switchToViewEnvironment: switchToViewEnvironment
    };
}]);
'use strict';

angular.module('NewEnvironment').config(["$uiTabsProvider", function ($uiTabsProvider) {

    $uiTabsProvider.tab('environments.env.create', {
        title: 'New environment',
        templateUrl: 'environment/tabs/environment/environment-ci.html',
        controller: 'EnvironmentController'
    }).tab('environments.env.view', {
        title: 'Visual environment',
        templateUrl: 'environment/tabs/environment/environment-ci.html',
        controller: 'EnvironmentController'
    }).welcome({
        templateUrl: 'environment/tabs/welcome/welcome.html'
    }).onClose(['ciTabsService', function (ciTabsService) {
        return function (tab) {
            return ciTabsService.closePopupDialog(tab, 'Are you sure you want to leave? Your changes will be lost.');
        };
    }]);
}]);
'use strict';

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

var EnvironmentsService = function () {
    function EnvironmentsService() {
        _classCallCheck(this, EnvironmentsService);
    }

    _createClass(EnvironmentsService, [{
        key: '$get',
        value: function $get($q, environmentsResource, constantService, directoryResource) {
            var enviromentTypeId = 'envClass';
            var directoryTypeId = 'dirClass';

            var addTypeToReturnedData = function addTypeToReturnedData(responseData, classType) {
                _.forEach(responseData, function (it) {
                    it.classType = classType;
                });
                return responseData;
            };

            var parseDataToSingleNode = function parseDataToSingleNode(data) {
                var path = data.ref;
                var classType = data.classType;
                var labels = path.split('/');
                var id = 'id_' + path;
                var idDom = id.replace(new RegExp('/', 'g'), '-');
                idDom = idDom.replace(new RegExp(' ', 'g'), '-');
                var node = { id: 'id_' + path, path: path, label: _.last(labels), classType: classType, idDom: idDom };

                if (classType !== enviromentTypeId) {
                    node.children = [];
                }

                return node;
            };

            var findEnvStructureByParent = function findEnvStructureByParent(parentPath, parentType) {
                var parent = parentPath ? parentPath : constantService.ROOT.ENVIRONMENTS;
                var promises = [];
                if (!parentType || parentType === directoryTypeId) {
                    // we look for root nodes or children of a directory.
                    var directoriesPromise = directoryResource.findAllDirectoriesByParent(parent).then(function (response) {
                        return addTypeToReturnedData(response.data, directoryTypeId);
                    });
                    var applicationsPromise = environmentsResource.findAllEnvironmentsByParent(parent).then(function (response) {
                        return addTypeToReturnedData(response.data, enviromentTypeId);
                    });
                    promises.push(directoriesPromise);
                    promises.push(applicationsPromise);
                }

                return $q.all(promises).then(function (allData) {
                    return _(allData).flatten().uniq().map(parseDataToSingleNode).value();
                });
            };

            return {
                environmentypeId: enviromentTypeId,
                directoryTypeId: directoryTypeId,
                findEnvStructureByParent: findEnvStructureByParent
            };
        }
    }]);

    return EnvironmentsService;
}();

angular.module('NewEnvironment').provider('environmentsService', EnvironmentsService);
'use strict';

angular.module('NewEnvironment').controller('EnvironmentsManagerController', ["$scope", "treeService", "environmentsService", "ciTabsService", function ($scope, treeService, environmentsService, ciTabsService) {

    var vm = this;

    vm.environments = [];
    vm.currentNode = {};

    vm.selectNode = function (node) {
        vm.currentNode.path = node.path;
    };

    vm.unfoldPathToNode = function (targetNodePath) {
        var nodeLabelsInPath = targetNodePath.split('/');
        if (targetNodePath[0] === '/') {
            nodeLabelsInPath.splice(0, 1);
        }
        var parentNode = { path: 'Environments', children: vm.environments, classType: 'dirClass' };
        treeService.unfoldNode(nodeLabelsInPath, 0, parentNode, vm.selectNode, environmentsService.findEnvStructureByParent);
    };

    vm.getRootEnvs = function () {
        environmentsService.findEnvStructureByParent().then(function (rootCIs) {
            vm.environments = rootCIs;
        });
    };

    vm.findNodeChildren = function (node) {
        return environmentsService.findEnvStructureByParent(node.path, node.classType);
    };

    vm.treeOptions = {};
    vm.getRootEnvs();

    vm.newFolder = ciTabsService.newFolder;
    vm.newEnvironment = ciTabsService.newEnvironment;
    vm.discoverEnvironment = ciTabsService.discoverEnvironment;
}]);
//# sourceMappingURL=environment.js.map
