'use strict';

angular.module('xl.widget', ['ui.select2', 'xl-tooltip', 'xl-wizard', 'xl.widget.date', 'xl.widget.time', 'xl-widget-modal', 'ngDragDrop', 'uuid4', 'vs-repeat']);
'use strict';

angular.module('xl-wizard', []);

angular.module('xl-wizard').directive('xlWizardStep', function () {
    return {
        restrict: 'A',
        replace: true,
        transclude: true,
        scope: {
            wzTitle: '@'
        },
        require: '^xlWidgetWizard',
        templateUrl: function templateUrl(element, attributes) {
            return attributes.template || 'xl-widget-wizard-step.html';
        },
        link: function link($scope, $element, $attrs, wizard) {
            $scope.state = { wzTitle: $scope.wzTitle };
            wizard.addStep($scope);
        }
    };
}).directive('xlWidgetWizard', function () {
    return {
        restrict: 'A',
        replace: true,
        transclude: true,
        scope: {
            currentStep: '=',
            onFinish: '&',
            hideIndicators: '=',
            editMode: '=',
            name: '@',
            navigable: '=',
            steps: '='
        },
        templateUrl: function templateUrl(element, attributes) {
            return attributes.template || 'src/xl-widget/xl-widget-wizard/xl-widget-wizard.html';
        },
        controller: ['$scope', '$element', 'WizardHandler', function ($scope, $element, WizardHandler) {

            WizardHandler.addWizard($scope.name || WizardHandler.defaultName, this);
            $scope.$on('$destroy', function () {
                WizardHandler.removeWizard($scope.name || WizardHandler.defaultName);
            });

            $scope.steps = $scope.steps || [];

            $scope.$watch('currentStep', function (step) {
                if (!step) return;
                var stepTitle = $scope.selectedStep.wzTitle;
                if ($scope.selectedStep && stepTitle !== $scope.currentStep) {
                    $scope.goTo(_.find($scope.steps, { wzTitle: $scope.currentStep }));
                }
            });

            $scope.$watch('[editMode, steps.length]', function () {
                var editMode = $scope.editMode;
                if (_.isNil(editMode) || _.isNil(editMode)) return;

                if (editMode) {
                    _.forEach($scope.steps, function (step) {
                        step.completed = true;
                    });
                }
            }, true);

            this.addStep = function (step) {
                var foundStep = _.find($scope.steps, { wzTitle: step.state.wzTitle });
                if (foundStep) {
                    step.state = foundStep;
                    if (foundStep.selected) {
                        $scope.selectedStep = foundStep;
                        if (!_.isUndefined($scope.currentStep)) {
                            $scope.currentStep = foundStep;
                        }
                    }
                } else {
                    $scope.steps.push(step.state);
                    if ($scope.steps.length === 1) {
                        $scope.goTo($scope.steps[0]);
                    }
                }
            };

            $scope.goTo = function (step) {
                unselectAll();
                $scope.selectedStep = step;
                if (!_.isUndefined($scope.currentStep)) {
                    $scope.currentStep = step.wzTitle;
                }
                step.selected = true;
                $scope.$emit('xlWidgetWizard:stepChanged', { step: step, index: _.indexOf($scope.steps, step) });
            };

            $scope.currentStepNumber = function () {
                return _.indexOf($scope.steps, $scope.selectedStep) + 1;
            };

            function unselectAll() {
                _.forEach($scope.steps, function (step) {
                    step.selected = false;
                });
                $scope.selectedStep = null;
            }

            this.next = function (draft) {
                var index = _.indexOf($scope.steps, $scope.selectedStep);
                if (!draft) {
                    $scope.selectedStep.completed = true;
                }
                if (index === $scope.steps.length - 1) {
                    this.finish();
                } else {
                    $scope.goTo($scope.steps[index + 1]);
                }
            };

            this.goTo = function (step) {
                var stepTo;
                if (_.isNumber(step)) {
                    stepTo = $scope.steps[step];
                } else {
                    stepTo = _.find($scope.steps, { wzTitle: step });
                }
                $scope.goTo(stepTo);
            };

            this.finish = function () {
                if ($scope.onFinish) {
                    $scope.onFinish();
                }
            };

            this.cancel = this.previous = function () {
                var index = _.indexOf($scope.steps, $scope.selectedStep);
                if (index === 0) {
                    throw new Error('Can\'t go back. It\'s already in step 0');
                } else {
                    $scope.goTo($scope.steps[index - 1]);
                }
            };
        }]
    };
}).run(['$templateCache', function ($templateCache) {
    $templateCache.put('xl-widget-wizard-step.html', '<section ng-show=\"state.selected\" ng-class=\"{current: state.selected, done: state.completed}\" class=\"step\" ng-transclude>\n' + '</section>');
}]).factory('WizardHandler', function () {
    var service = {};

    var wizards = {};

    service.defaultName = 'defaultWizard';

    service.addWizard = function (name, wizard) {
        wizards[name] = wizard;
    };

    service.removeWizard = function (name) {
        delete wizards[name];
    };

    service.wizard = function (name) {
        var nameToUse = name;
        if (!name) {
            nameToUse = service.defaultName;
        }

        return wizards[nameToUse];
    };

    return service;
});

function wizardButtonDirective(action) {
    angular.module('xl-wizard').directive(action, function () {
        return {
            restrict: 'A',
            replace: false,
            require: '^xlWidgetWizard',
            link: function link($scope, $element, $attrs, wizard) {

                $element.on('click', function (e) {
                    e.preventDefault();
                    $scope.$apply(function () {
                        $scope.$eval($attrs[action]);
                        wizard[action.replace('xlWizard', '').toLowerCase()]();
                    });
                });
            }
        };
    });
}

wizardButtonDirective('xlWizardNext');
wizardButtonDirective('xlWizardPrevious');
wizardButtonDirective('xlWizardFinish');
wizardButtonDirective('xlWizardCancel');
'use strict';

angular.module('xl.widget').factory('xlWidgetUtils', function () {

    var hasRequiredError = function hasRequiredError(ngModel, isRequired) {
        return !!(isRequired && (!ngModel || ngModel === '' || angular.isArray(ngModel) && ngModel.length === 0 || angular.isObject(ngModel) && !angular.isDate(ngModel) && _.keys(ngModel).length === 0));
    };

    return {
        hasRequiredError: hasRequiredError
    };
});

angular.module('xl.widget').directive('dynamicName', function ($compile, $parse, $timeout) {
    return {
        restrict: 'A',
        terminal: true,
        priority: 100000,
        link: function link(scope, elem) {
            $timeout(function () {
                var name = $parse(elem.attr('dynamic-name'))(scope);
                elem.removeAttr('dynamic-name');
                elem.attr('name', name);
                $compile(elem)(scope);
            });
        }
    };
});

angular.module('xl.widget').directive('removeFormControl', function ($timeout) {
    return {
        require: '?^form',
        restrict: 'A',
        link: function link(scope, element, attrs, form) {
            if (!form) {
                return;
            }

            $timeout(function () {
                var control = form[element.attr('name')];
                form.$removeControl(control);
            });
        }
    };
});

/**
 * Generates a randomId
 */
angular.module('xl.widget').directive('xlRandomIdGenerator', function () {
    return {
        restrict: 'A',
        link: {
            pre: function pre(scope, element, attrs, ctrl) {
                var metadata = scope.$eval(attrs.xlRandomIdGenerator);
                ctrl.randomId = (metadata ? metadata.name + '_' : '') + Math.random().toString().substr(2);
            }
        },
        controller: _.noop
    };
});

/**
 * Handles right click
 */
angular.module('xl.widget').directive('xlRightClick', function ($parse) {
    return {
        restrict: "A",
        link: function link(scope, elem, attrs) {
            var func = $parse(attrs.xlRightClick);
            elem.bind('contextmenu', function (event) {
                event.preventDefault();
                scope.$apply(func);
            });
        }
    };
});

/**
 * Has to have the xlRandomIdGenerator as a parent directive.
 * Takes randomId generated in xlRandomIdGenerator and sets it in the attribute xlRandomAttr
 */
angular.module('xl.widget').directive('xlRandomAttr', function () {
    return {
        restrict: 'A',
        require: '?^xlRandomIdGenerator',
        link: {
            pre: function pre(scope, element, attrs, randomIdGeneratorCtrl) {
                if (randomIdGeneratorCtrl) {
                    element.attr(attrs.xlRandomAttr, randomIdGeneratorCtrl.randomId + (attrs.xlRandomSuffix ? attrs.xlRandomSuffix : ''));
                }
            }
        }
    };
});

angular.module('xl.widget').directive('stopPristinePropagation', function () {
    return {
        restrict: 'A',
        require: '?ngModel',
        link: function link(scope, element, attrs, ngModel) {
            if (!ngModel) {
                return;
            }
            ngModel.$pristine = false;
        }
    };
});

angular.module('xl.widget').directive('xlPattern', function ($animate) {
    return {
        restrict: 'A',
        require: 'ngModel',
        link: function link(scope, elem, attrs, ngModelCtrl) {

            var PRISTINE_CLASS = 'ng-pristine',
                DIRTY_CLASS = 'ng-dirty';

            // pattern validator
            var pattern = attrs.xlPattern,
                patternValidator,
                match;

            if (pattern) {
                var validateRegex = function validateRegex(regexp, value) {
                    var validity = ngModelCtrl.$isEmpty(value) || regexp.test(value);
                    ngModelCtrl.$setValidity('pattern', validity);

                    if (!validity) {
                        // if not valid, just remove pristine (ng-pristine class) to show the red control
                        ngModelCtrl.$dirty = true;
                        ngModelCtrl.$pristine = false;
                        $animate.removeClass(elem, PRISTINE_CLASS);
                        $animate.addClass(elem, DIRTY_CLASS);
                    }

                    return value;
                };
                match = pattern.match(/^\/(.*)\/([gim]*)$/);
                if (match) {
                    pattern = new RegExp(match[1], match[2]);
                    patternValidator = function patternValidator(value) {
                        return validateRegex(pattern, value);
                    };
                } else {
                    patternValidator = function patternValidator(value) {
                        var patternObj = scope.$eval(pattern);

                        if (!patternObj || !patternObj.test) {
                            throw new Error('Not a regex: ' + pattern);
                        }
                        return validateRegex(patternObj, value);
                    };
                }

                ngModelCtrl.$formatters.push(patternValidator);
                ngModelCtrl.$parsers.push(patternValidator);
            }
        }
    };
});
'use strict';

angular.module('xl.widget').directive('xlWidgetTreeView', [function () {
    return {
        restrict: 'A',
        scope: {
            config: '=',
            options: '=',
            currentNode: '=',
            nodes: '=',
            datasource: '=',
            draggable: '=',
            expand: '=',
            broadcastEventName: '=',
            classStyle: '='
        },
        transclude: true,
        templateUrl: 'src/xl-widget/xl-widget-tree/xl-widget-tree.html',
        controller: ['$transclude', function ($transclude) {
            this.transclude = $transclude;
        }]
    };
}]).directive('treeNodes', ['$compile', function ($compile) {
    return {
        restrict: 'A',
        scope: {
            config: '=',
            options: '=',
            currentNode: '=',
            treeNodes: '=',
            datasource: '=',
            draggable: '=',
            expand: '=',
            broadcastEventName: '=',
            classStyle: '='
        },
        templateUrl: 'src/xl-widget/xl-widget-tree/tree-nodes.html',
        require: '^xlWidgetTreeView',
        compile: function compile(el) {
            var contents = el.contents().remove();
            var compiled = void 0;

            var appendContextMenu = function appendContextMenu(scope, contents) {
                if (scope.config && scope.config.contextMenu) {
                    var line = contents.find('.treenode-line');
                    line.attr('xl-widget-context-menu', '');
                    line.attr('context-menu-provider', 'config.contextMenuProvider');
                    line.attr('context-menu-source-data', 'node');
                }
            };

            var expandNodes = function expandNodes(nodes) {
                _.forEach(nodes, function (node) {
                    var hasChildren = !_.isEmpty(_.result(node, 'children'));
                    node.open = hasChildren;
                    if (hasChildren) {
                        expandNodes(node.children);
                    }
                });
            };

            return function link(scope, el, attr, ctrl) {

                var selectNode = function selectNode(selectedNode) {
                    if (scope.broadcastEventName) {
                        scope.$emit(scope.broadcastEventName, selectedNode);
                    }

                    if (scope.currentNode) {
                        scope.currentNode.path = selectedNode.path;
                    }
                };

                scope.selectNodeHead = function (selectedNode, $event) {
                    $event.stopPropagation();
                    //Collapse or Expand
                    selectedNode.open = !selectedNode.open;
                    if (scope.datasource && _.isEmpty(selectedNode.children)) {
                        scope.datasource(selectedNode).then(function (childrenNodes) {
                            var newNodes = childrenNodes;
                            if (childrenNodes && _.isEmpty(childrenNodes)) {
                                newNodes = undefined;
                            }
                            selectedNode.children = newNodes;
                        });
                    }
                };
                scope.selectNodeLabel = function (selectedNode) {
                    //remove highlight from previous node
                    removeCurrentSelection();
                    //remove highlight from previous node
                    if (scope.currentNode && scope.currentNode.selected) {
                        delete scope.currentNode.selected;
                    }

                    //set highlight to selected node
                    selectNode(selectedNode);
                    if (scope.options && scope.options.clickAction) {
                        scope.options.clickAction(selectedNode);
                    }
                };

                scope.dblClickAction = function (selectedNode) {
                    selectNode(selectedNode);
                    if (scope.options && scope.options.dblClickAction) {
                        scope.options.dblClickAction(selectedNode);
                    }
                };

                var removeCurrentSelection = function removeCurrentSelection() {
                    if (scope.currentNode && scope.currentNode.selected) {
                        delete scope.currentNode.selected;
                    }
                };

                scope.isNodeSelected = function (node) {
                    return _.result(scope, 'currentNode.path') === node.path;
                };

                scope.wholeLineClass = function (node, hoverOver) {
                    var result = { selected: scope.isNodeSelected(node), hoverOver: hoverOver };
                    if (node.classNode) {
                        result[node.classNode] = true;
                    }
                    return result;
                };

                ctrl.transclude(function (elem) {
                    var labelElement = contents.find('.tree-label');
                    if (!_.isEmpty(elem)) {
                        labelElement.empty();
                        labelElement.append(elem);
                    }
                });

                if (!compiled) {
                    appendContextMenu(scope, contents);
                    compiled = $compile(contents);
                }

                compiled(scope, function (clone) {
                    return el.append(clone);
                });
                scope.$watch('treeNodes', function () {
                    if (scope.expand) {
                        expandNodes(scope.treeNodes);
                    }
                });

                if (scope.expand) {
                    expandNodes(scope.treeNodes);
                }

                scope.hasChildren = function (node) {
                    return !_.isUndefined(node.children);
                };
            };
        }
    };
}]);
'use strict';

angular.module('xl-tooltip-position', []).factory('$xlposition', ['$document', '$window', function ($document, $window) {

    function getStyle(el, cssprop) {
        if (el.currentStyle) {
            //IE
            return el.currentStyle[cssprop];
        } else if ($window.getComputedStyle) {
            return $window.getComputedStyle(el)[cssprop];
        }
        // finally try and get inline style
        return el.style[cssprop];
    }

    function isStaticPositioned(element) {
        return (getStyle(element, 'position') || 'static') === 'static';
    }

    var parentOffsetEl = function parentOffsetEl(element) {
        var docDomEl = $document[0];
        var offsetParent = element.offsetParent || docDomEl;
        while (offsetParent && offsetParent !== docDomEl && isStaticPositioned(offsetParent)) {
            offsetParent = offsetParent.offsetParent;
        }
        return offsetParent || docDomEl;
    };

    return {
        position: function position(element) {
            var elBCR = this.offset(element);
            var offsetParentBCR = { top: 0, left: 0 };
            var offsetParentEl = parentOffsetEl(element[0]);
            if (offsetParentEl !== $document[0]) {
                offsetParentBCR = this.offset(angular.element(offsetParentEl));
                offsetParentBCR.top += offsetParentEl.clientTop - offsetParentEl.scrollTop;
                offsetParentBCR.left += offsetParentEl.clientLeft - offsetParentEl.scrollLeft;
            }

            var boundingClientRect = element[0].getBoundingClientRect();
            return {
                width: boundingClientRect.width || element.prop('offsetWidth'),
                height: boundingClientRect.height || element.prop('offsetHeight'),
                top: elBCR.top - offsetParentBCR.top,
                left: elBCR.left - offsetParentBCR.left
            };
        },

        offset: function offset(element) {
            var boundingClientRect = element[0].getBoundingClientRect();
            return {
                width: boundingClientRect.width || element.prop('offsetWidth'),
                height: boundingClientRect.height || element.prop('offsetHeight'),
                top: boundingClientRect.top + ($window.pageYOffset || $document[0].body.scrollTop || $document[0].documentElement.scrollTop),
                left: boundingClientRect.left + ($window.pageXOffset || $document[0].body.scrollLeft || $document[0].documentElement.scrollLeft)
            };
        }
    };
}]);

angular.module('xl-tooltip', ['xl-tooltip-position']).provider('$xltooltip', function () {

    var defaultOptions = {
        placement: 'bottom',
        animation: true,
        popupDelay: 300
    };

    // Default hide triggers for each show trigger
    var triggerMap = {
        'mouseenter': 'mouseleave',
        'click': 'click',
        'focus': 'blur'
    };

    // The options specified to the provider globally.
    var globalOptions = {};

    this.options = function (value) {
        angular.extend(globalOptions, value);
    };

    this.setTriggers = function setTriggers(triggers) {
        angular.extend(triggerMap, triggers);
    };

    function snake_case(name) {
        var regexp = /[A-Z]/g;
        var separator = '-';
        return name.replace(regexp, function (letter, pos) {
            return (pos ? separator : '') + letter.toLowerCase();
        });
    }

    this.$get = ['$window', '$compile', '$timeout', '$parse', '$document', '$xlposition', '$interpolate', function ($window, $compile, $timeout, $parse, $document, $xlposition, $interpolate) {
        return function $xltooltip(type, prefix, defaultTriggerShow) {
            var options = angular.extend({}, defaultOptions, globalOptions);

            function getTriggers(trigger) {
                var show = trigger || options.trigger || defaultTriggerShow;
                var hide = triggerMap[show] || show;
                return {
                    show: show,
                    hide: hide
                };
            }

            var directiveName = snake_case(type);

            var startSym = $interpolate.startSymbol();
            var endSym = $interpolate.endSymbol();
            var template = '<div ' + directiveName + '-popup ' + 'title="' + startSym + 'tt_title' + endSym + '" ' + 'content="' + startSym + 'tt_content' + endSym + '" ' + 'placement="' + startSym + 'tt_placement' + endSym + '" ' + 'animation="tt_animation" ' + 'is-open="tt_isOpen"' + '>' + '</div>';

            return {
                restrict: 'EA',
                scope: true,
                compile: function compile(tElem, tAttrs) {
                    var tooltipLinker = $compile(template);

                    return function link(scope, element, attrs) {
                        var tooltip;
                        var transitionTimeout;
                        var popupTimeout;
                        var appendToBody = angular.isDefined(options.appendToBody) ? options.appendToBody : false;
                        var triggers = getTriggers(undefined);
                        var hasRegisteredTriggers = false;
                        var hasEnableExp = angular.isDefined(attrs[prefix + 'Enable']);

                        function observeAttribute(name, defaultVal, fn) {
                            if (_.isUndefined(attrs[name])) {
                                attrs[name] = defaultVal;
                            }
                            attrs.$observe(name, fn);
                        }

                        var positionTooltip = function positionTooltip() {
                            var position, ttWidth, ttHeight, ttPosition;
                            // Get the position of the directive element.
                            position = appendToBody ? $xlposition.offset(element) : $xlposition.position(element);

                            // Get the height and width of the tooltip so we can center it.
                            ttWidth = tooltip.prop('offsetWidth');
                            ttHeight = tooltip.prop('offsetHeight');

                            // Calculate the tooltip's top and left coordinates to center it with
                            // this directive.
                            switch (scope.tt_placement) {
                                case 'right':
                                    ttPosition = {
                                        top: position.top + position.height / 2 - ttHeight / 2,
                                        left: position.left + position.width
                                    };
                                    break;
                                case 'bottom':
                                    ttPosition = {
                                        top: position.top + position.height,
                                        left: position.left + position.width / 2
                                    };
                                    break;
                                case 'left':
                                    ttPosition = {
                                        top: position.top + position.height / 2 - ttHeight / 2,
                                        left: position.left - ttWidth
                                    };
                                    break;
                                default:
                                    ttPosition = {
                                        top: position.top - ttHeight,
                                        left: position.left + position.width / 2
                                    };
                                    break;
                            }

                            ttPosition.top += 'px';
                            ttPosition.left += 'px';

                            tooltip.css(ttPosition);
                        };

                        scope.tt_isOpen = false;

                        function toggleTooltipBind() {
                            if (!scope.tt_isOpen) {
                                showTooltipBind();
                            } else {
                                hideTooltipBind();
                            }
                        }

                        function showTooltipBind() {
                            if (hasEnableExp && !scope.$eval(attrs[prefix + 'Enable'])) {
                                return;
                            }
                            if (scope.tt_popupDelay) {
                                popupTimeout = $timeout(show, scope.tt_popupDelay, false);
                                popupTimeout.then(function (reposition) {
                                    reposition();
                                });
                            } else {
                                show()();
                            }
                        }

                        function hideTooltipBind() {
                            scope.$apply(function () {
                                hide();
                            });
                        }

                        function show() {
                            // Don't show empty tooltips.
                            if (!scope.tt_content) {
                                return angular.noop;
                            }

                            createTooltip();

                            if (transitionTimeout) {
                                $timeout.cancel(transitionTimeout);
                            }

                            // Set the initial positioning.
                            tooltip.css({ top: 0, left: 0, display: 'block' });

                            // Now we add it to the DOM because need some info about it. But it's not
                            // visible yet anyway.
                            if (appendToBody) {
                                $document.find('body').append(tooltip);
                            } else {
                                element.after(tooltip);
                            }

                            positionTooltip();

                            // And show the tooltip.
                            scope.tt_isOpen = true;
                            scope.$digest(); // digest required as $apply is not called

                            // Return positioning function as promise callback for correct
                            // positioning after draw.
                            return positionTooltip;
                        }

                        // Hide the tooltip popup element.
                        function hide() {
                            // First things first: we don't show it anymore.
                            scope.tt_isOpen = false;

                            //if tooltip is going to be shown after delay, we must cancel this
                            $timeout.cancel(popupTimeout);

                            if (scope.tt_animation) {
                                transitionTimeout = $timeout(removeTooltip, 500);
                            } else {
                                removeTooltip();
                            }
                        }

                        function createTooltip() {
                            // There can only be one tooltip element per directive shown at once.
                            if (tooltip) {
                                removeTooltip();
                            }
                            tooltip = tooltipLinker(scope, _.noop);

                            // Get contents rendered into the tooltip
                            scope.$digest();
                        }

                        function removeTooltip() {
                            if (tooltip) {
                                tooltip.remove();
                                tooltip = null;
                            }
                        }

                        observeAttribute(type, undefined, function (val) {
                            scope.tt_content = val;

                            if (!val && scope.tt_isOpen) {
                                hide();
                            }
                        });

                        observeAttribute(prefix + 'Title', undefined, function (val) {
                            scope.tt_title = val;
                        });

                        observeAttribute(prefix + 'Placement', options.placement, function (val) {
                            scope.tt_placement = val;
                        });

                        observeAttribute(prefix + 'PopupDelay', options.popupDelay, function (val) {
                            var delay = parseInt(val, 10);
                            scope.tt_popupDelay = !isNaN(delay) ? delay : options.popupDelay;
                        });

                        var unregisterTriggers = function unregisterTriggers() {
                            if (hasRegisteredTriggers) {
                                element.unbind(triggers.show, showTooltipBind);
                                element.unbind(triggers.hide, hideTooltipBind);
                            }
                        };
                        observeAttribute(prefix + 'Trigger', defaultTriggerShow, function (val) {
                            unregisterTriggers();

                            triggers = getTriggers(val);

                            if (triggers.show === triggers.hide) {
                                element.bind(triggers.show, toggleTooltipBind);
                            } else {
                                element.bind(triggers.show, showTooltipBind);
                                element.bind(triggers.hide, hideTooltipBind);
                            }

                            hasRegisteredTriggers = true;
                        });

                        var animation = scope.$eval(attrs[prefix + 'Animation']);
                        scope.tt_animation = angular.isDefined(animation) ? !!animation : options.animation;

                        observeAttribute(prefix + 'AppendToBody', appendToBody, function (val) {
                            appendToBody = angular.isDefined(val) ? $parse(val)(scope) : appendToBody;
                        });

                        // if a tooltip is attached to <body> we need to remove it on
                        // location change as its parent scope will probably not be destroyed
                        // by the change.
                        if (appendToBody) {
                            scope.$on('$locationChangeSuccess', function closeTooltipOnLocationChangeSuccess() {
                                if (scope.tt_isOpen) {
                                    hide();
                                }
                            });
                        }

                        // Make sure tooltip is destroyed and removed.
                        scope.$on('$destroy', function onDestroyTooltip() {
                            $timeout.cancel(transitionTimeout);
                            $timeout.cancel(popupTimeout);
                            unregisterTriggers();
                            removeTooltip();
                        });
                    };
                }
            };
        };
    }];
}).directive('xlTooltipPopup', function () {
    return {
        restrict: 'EA',
        replace: true,
        scope: { content: '@', placement: '@', animation: '&', isOpen: '&' },
        templateUrl: 'template/tooltip/xl-tooltip-popup.html'
    };
}).directive('xlTooltip', ['$xltooltip', function ($xltooltip) {
    return $xltooltip('xlTooltip', 'xlTooltip', 'mouseenter');
}]).run(['$templateCache', function ($templateCache) {
    $templateCache.put('template/tooltip/xl-tooltip-popup.html', '<div class=\"xl-tooltip {{placement}}\" ng-class=\"{ in: isOpen(), fade: animation() }\">\n' + '  <div class=\"xl-tooltip-arrow\"></div>\n' + '  <div class=\"xl-tooltip-arrow-border\"></div>\n' + '  <div class=\"xl-tooltip-inner\" ng-bind=\"content\"></div>\n' + '</div>\n');
}]);
'use strict';

angular.module('xl.widget').directive('xlWidgetTime', ['xlWidgetUtils', function (xlWidgetUtils) {
    return {
        restrict: 'EA',
        replace: true,
        templateUrl: 'src/xl-widget/xl-widget-time/xl-widget-time.html',
        scope: {
            ngModel: '=',
            options: '=',
            displayMode: '='
        },
        controller: function controller($scope) {
            $scope.$watch('options', function () {
                if ($scope.options) {
                    $scope.hasRequiredError = xlWidgetUtils.hasRequiredError;
                }
                if (_.isUndefined($scope.ngModel) && !_.isUndefined($scope.options) && !_.isUndefined($scope.options['default']) && !_.isNull($scope.options['default'])) {
                    $scope.ngModel = $scope.options['default'];
                }
            });
        }
    };
}]);
'use strict';
/* jshint ignore:start */

angular.module('xl.widget.time', ['xl.widget.date.dateParser', 'xl.widget.date.tooltip']).provider('$timepicker', function () {

    var defaults = this.defaults = {
        animation: 'am-fade',
        prefixClass: 'timepicker',
        placement: 'bottom-left',
        template: 'src/xl-widget/xl-widget-time/timepicker/timepicker.html',
        trigger: 'focus',
        container: false,
        keyboard: true,
        html: false,
        delay: 0,
        // lang: $locale.id,
        useNative: true,
        timeType: 'date',
        timeFormat: 'shortTime',
        modelTimeFormat: null,
        autoclose: false,
        minTime: -Infinity,
        maxTime: +Infinity,
        length: 5,
        hourStep: 1,
        minuteStep: 5,
        iconUp: 'glyphicon glyphicon-chevron-up',
        iconDown: 'glyphicon glyphicon-chevron-down',
        arrowBehavior: 'pager'
    };

    this.$get = function ($window, $document, $rootScope, $interval, $sce, $locale, dateFilter, $tooltipLabs) {

        var isNative = /(ip(a|o)d|iphone|android)/ig.test($window.navigator.userAgent);
        var isTouch = 'createTouch' in $window.document && isNative;
        if (!defaults.lang) defaults.lang = $locale.id;

        function timepickerFactory(element, controller, config) {

            var $timepicker = $tooltipLabs(element, angular.extend({}, defaults, config));
            var parentScope = config.scope;
            var options = $timepicker.$options;
            var scope = $timepicker.$scope;

            // View vars

            var selectedIndex = 0;
            var startDate = controller.$dateValue || new Date();
            var viewDate = {
                hour: startDate.getHours(),
                meridian: startDate.getHours() < 12,
                minute: startDate.getMinutes(),
                second: startDate.getSeconds(),
                millisecond: startDate.getMilliseconds()
            };

            var format = $locale.DATETIME_FORMATS[options.timeFormat] || options.timeFormat;
            var formats = /(h+)([:\.])?(m+)[ ]?(a?)/i.exec(format).slice(1);
            scope.$iconUp = options.iconUp;
            scope.$iconDown = options.iconDown;

            // Scope methods

            scope.$select = function (date, index) {
                $timepicker.select(date, index);
                element[0].blur();
            };
            scope.$moveIndex = function (value, index) {
                $timepicker.$moveIndex(value, index);
            };
            scope.$switchMeridian = function (date) {
                $timepicker.switchMeridian(date);
            };

            // Public methods

            $timepicker.update = function (date) {
                // console.warn('$timepicker.update() newValue=%o', date);
                if (angular.isDate(date) && !isNaN(date.getTime())) {
                    $timepicker.$date = date;
                    angular.extend(viewDate, {
                        hour: date.getHours(),
                        minute: date.getMinutes(),
                        second: date.getSeconds(),
                        millisecond: date.getMilliseconds()
                    });
                    $timepicker.$build();
                } else if (!$timepicker.$isBuilt) {
                    $timepicker.$build();
                }
            };

            $timepicker.select = function (date, index, keep) {
                // console.warn('$timepicker.select', date, scope.$mode);
                if (!controller.$dateValue || isNaN(controller.$dateValue.getTime())) controller.$dateValue = new Date(1970, 0, 1);
                if (!angular.isDate(date)) date = new Date(date);
                if (index === 0) controller.$dateValue.setHours(date.getHours());else if (index === 1) controller.$dateValue.setMinutes(date.getMinutes());
                controller.$setViewValue(controller.$dateValue);
                controller.$render();
                if (options.autoclose && !keep) {
                    $timepicker.hide(true);
                }
            };

            $timepicker.switchMeridian = function (date) {
                var hours = (date || controller.$dateValue).getHours();
                controller.$dateValue.setHours(hours < 12 ? hours + 12 : hours - 12);
                controller.$setViewValue(controller.$dateValue);
                controller.$render();
            };

            // Protected methods

            $timepicker.$build = function () {
                // console.warn('$timepicker.$build() viewDate=%o', viewDate);
                var i,
                    midIndex = scope.midIndex = parseInt(options.length / 2, 10);
                var hours = [],
                    hour;
                for (i = 0; i < options.length; i++) {
                    hour = new Date(1970, 0, 1, viewDate.hour - (midIndex - i) * options.hourStep);
                    hours.push({
                        date: hour,
                        label: dateFilter(hour, formats[0]),
                        selected: $timepicker.$date && $timepicker.$isSelected(hour, 0),
                        disabled: $timepicker.$isDisabled(hour, 0)
                    });
                }
                var minutes = [],
                    minute;
                for (i = 0; i < options.length; i++) {
                    minute = new Date(1970, 0, 1, 0, viewDate.minute - (midIndex - i) * options.minuteStep);
                    minutes.push({
                        date: minute,
                        label: dateFilter(minute, formats[2]),
                        selected: $timepicker.$date && $timepicker.$isSelected(minute, 1),
                        disabled: $timepicker.$isDisabled(minute, 1)
                    });
                }

                var rows = [];
                for (i = 0; i < options.length; i++) {
                    rows.push([hours[i], minutes[i]]);
                }
                scope.rows = rows;
                scope.showAM = !!formats[3];
                scope.isAM = ($timepicker.$date || hours[midIndex].date).getHours() < 12;
                scope.timeSeparator = formats[1];
                $timepicker.$isBuilt = true;
            };

            $timepicker.$isSelected = function (date, index) {
                if (!$timepicker.$date) return false;else if (index === 0) {
                    return date.getHours() === $timepicker.$date.getHours();
                } else if (index === 1) {
                    return date.getMinutes() === $timepicker.$date.getMinutes();
                }
            };

            $timepicker.$isDisabled = function (date, index) {
                var selectedTime;
                if (index === 0) {
                    selectedTime = date.getTime() + viewDate.minute * 6e4;
                } else if (index === 1) {
                    selectedTime = date.getTime() + viewDate.hour * 36e5;
                }
                return selectedTime < options.minTime * 1 || selectedTime > options.maxTime * 1;
            };

            scope.$arrowAction = function (value, index) {
                if (options.arrowBehavior === 'picker') {
                    $timepicker.$setTimeByStep(value, index);
                } else {
                    $timepicker.$moveIndex(value, index);
                }
            };

            $timepicker.$setTimeByStep = function (value, index) {
                var newDate = new Date($timepicker.$date);
                var hours = newDate.getHours(),
                    hoursLength = dateFilter(newDate, 'h').length;
                var minutes = newDate.getMinutes(),
                    minutesLength = dateFilter(newDate, 'mm').length;
                if (index === 0) {
                    newDate.setHours(hours - parseInt(options.hourStep, 10) * value);
                } else {
                    newDate.setMinutes(minutes - parseInt(options.minuteStep, 10) * value);
                }
                $timepicker.select(newDate, index, true);
                parentScope.$digest();
            };

            $timepicker.$moveIndex = function (value, index) {
                var targetDate;
                if (index === 0) {
                    targetDate = new Date(1970, 0, 1, viewDate.hour + value * options.length, viewDate.minute);
                    angular.extend(viewDate, { hour: targetDate.getHours() });
                } else if (index === 1) {
                    targetDate = new Date(1970, 0, 1, viewDate.hour, viewDate.minute + value * options.length * options.minuteStep);
                    angular.extend(viewDate, { minute: targetDate.getMinutes() });
                }
                $timepicker.$build();
            };

            $timepicker.$onMouseDown = function (evt) {
                // Prevent blur on mousedown on .dropdown-menu
                if (evt.target.nodeName.toLowerCase() !== 'input') evt.preventDefault();
                evt.stopPropagation();
                // Emulate click for mobile devices
                if (isTouch) {
                    var targetEl = angular.element(evt.target);
                    if (targetEl[0].nodeName.toLowerCase() !== 'button') {
                        targetEl = targetEl.parent();
                    }
                    targetEl.triggerHandler('click');
                }
            };

            $timepicker.$onKeyDown = function (evt) {
                if (!/(38|37|39|40|13)/.test(evt.keyCode) || evt.shiftKey || evt.altKey) return;
                evt.preventDefault();
                evt.stopPropagation();

                // Close on enter
                if (evt.keyCode === 13) return $timepicker.hide(true);

                // Navigate with keyboard
                var newDate = new Date($timepicker.$date);
                var hours = newDate.getHours(),
                    hoursLength = dateFilter(newDate, 'h').length;
                var minutes = newDate.getMinutes(),
                    minutesLength = dateFilter(newDate, 'mm').length;
                var lateralMove = /(37|39)/.test(evt.keyCode);
                var count = 2 + !!formats[3] * 1;

                // Navigate indexes (left, right)
                if (lateralMove) {
                    if (evt.keyCode === 37) selectedIndex = selectedIndex < 1 ? count - 1 : selectedIndex - 1;else if (evt.keyCode === 39) selectedIndex = selectedIndex < count - 1 ? selectedIndex + 1 : 0;
                }

                // Update values (up, down)
                var selectRange = [0, hoursLength];
                if (selectedIndex === 0) {
                    if (evt.keyCode === 38) newDate.setHours(hours - parseInt(options.hourStep, 10));else if (evt.keyCode === 40) newDate.setHours(hours + parseInt(options.hourStep, 10));
                    selectRange = [0, hoursLength];
                } else if (selectedIndex === 1) {
                    if (evt.keyCode === 38) newDate.setMinutes(minutes - parseInt(options.minuteStep, 10));else if (evt.keyCode === 40) newDate.setMinutes(minutes + parseInt(options.minuteStep, 10));
                    selectRange = [hoursLength + 1, hoursLength + 1 + minutesLength];
                } else if (selectedIndex === 2) {
                    if (!lateralMove) $timepicker.switchMeridian();
                    selectRange = [hoursLength + 1 + minutesLength + 1, hoursLength + 1 + minutesLength + 3];
                }
                $timepicker.select(newDate, selectedIndex, true);
                createSelection(selectRange[0], selectRange[1]);
                parentScope.$digest();
            };

            // Private

            function createSelection(start, end) {
                if (element[0].createTextRange) {
                    var selRange = element[0].createTextRange();
                    selRange.collapse(true);
                    selRange.moveStart('character', start);
                    selRange.moveEnd('character', end);
                    selRange.select();
                } else if (element[0].setSelectionRange) {
                    element[0].setSelectionRange(start, end);
                } else if (angular.isUndefined(element[0].selectionStart)) {
                    element[0].selectionStart = start;
                    element[0].selectionEnd = end;
                }
            }

            function focusElement() {
                element[0].focus();
            }

            // Overrides

            var _init = $timepicker.init;
            $timepicker.init = function () {
                if (isNative && options.useNative) {
                    element.prop('type', 'time');
                    element.css('-webkit-appearance', 'textfield');
                    return;
                } else if (isTouch) {
                    element.prop('type', 'text');
                    element.attr('readonly', 'true');
                    element.on('click', focusElement);
                }
                _init();
            };

            var _destroy = $timepicker.destroy;
            $timepicker.destroy = function () {
                if (isNative && options.useNative) {
                    element.off('click', focusElement);
                }
                _destroy();
            };

            var _show = $timepicker.show;
            $timepicker.show = function () {
                _show();
                $interval(function () {
                    $timepicker.$element.on(isTouch ? 'touchstart' : 'mousedown', $timepicker.$onMouseDown);
                    if (options.keyboard) {
                        element.on('keydown', $timepicker.$onKeyDown);
                    }
                }, 1, 1);
            };

            var _hide = $timepicker.hide;
            $timepicker.hide = function (blur) {
                $timepicker.$element.off(isTouch ? 'touchstart' : 'mousedown', $timepicker.$onMouseDown);
                if (options.keyboard) {
                    element.off('keydown', $timepicker.$onKeyDown);
                }
                _hide(blur);
            };

            return $timepicker;
        }

        timepickerFactory.defaults = defaults;
        return timepickerFactory;
    };
}).directive('bsTimepickerLabs', function ($window, $parse, $q, $locale, dateFilter, $timepicker, $dateParserLabs, $timeout) {

    var defaults = $timepicker.defaults;
    var isNative = /(ip(a|o)d|iphone|android)/ig.test($window.navigator.userAgent);

    return {
        restrict: 'EAC',
        require: 'ngModel',
        link: function postLink(scope, element, attr, controller) {

            // Directive options
            var options = { scope: scope, controller: controller };
            angular.forEach(['placement', 'container', 'delay', 'trigger', 'keyboard', 'html', 'animation', 'template', 'autoclose', 'timeType', 'timeFormat', 'modelTimeFormat', 'useNative', 'hourStep', 'minuteStep', 'length', 'arrowBehavior'], function (key) {
                if (angular.isDefined(attr[key])) options[key] = attr[key];
            });

            // Visibility binding support
            attr.bsShow && scope.$watch(attr.bsShow, function (newValue, oldValue) {
                if (!timepicker || angular.isUndefined(newValue)) return;
                if (angular.isString(newValue)) newValue = newValue.match(',?(timepicker),?');
                newValue === true ? timepicker.show() : timepicker.hide();
            });

            // Initialize timepicker
            if (isNative && (options.useNative || defaults.useNative)) options.timeFormat = 'HH:mm';
            var timepicker = $timepicker(element, controller, options);
            options = timepicker.$options;

            // Initialize parser
            var dateParser = $dateParserLabs({ format: options.timeFormat, lang: options.lang });

            // Observe attributes for changes
            angular.forEach(['minTime', 'maxTime'], function (key) {
                // console.warn('attr.$observe(%s)', key, attr[key]);
                angular.isDefined(attr[key]) && attr.$observe(key, function (newValue) {
                    if (newValue === 'now') {
                        timepicker.$options[key] = new Date().setFullYear(1970, 0, 1);
                    } else if (angular.isString(newValue) && newValue.match(/^".+"$/)) {
                        timepicker.$options[key] = +new Date(newValue.substr(1, newValue.length - 2));
                    } else {
                        timepicker.$options[key] = dateParser.parse(newValue, new Date(1970, 0, 1, 0));
                    }
                    !isNaN(timepicker.$options[key]) && timepicker.$build();
                });
            });

            // Watch model for changes
            scope.$watch(attr.ngModel, function (newValue, oldValue) {
                // console.warn('scope.$watch(%s)', attr.ngModel, newValue, oldValue, controller.$dateValue);
                timepicker.update(controller.$dateValue);
            }, true);

            // viewValue -> $parsers -> modelValue
            controller.$parsers.unshift(function (viewValue) {
                // console.warn('$parser("%s"): viewValue=%o', element.attr('ng-model'), viewValue);
                // Null values should correctly reset the model value & validity
                if (!viewValue) {
                    controller.$setValidity('date', true);
                    return viewValue;
                }
                var parsedTime = dateParser.parse(viewValue, controller.$dateValue);
                if (!parsedTime || isNaN(parsedTime.getTime())) {
                    controller.$setValidity('date', false);
                } else {
                    var isValid = parsedTime.getTime() >= options.minTime && parsedTime.getTime() <= options.maxTime;
                    controller.$setValidity('date', isValid);
                    // Only update the model when we have a valid date
                    if (isValid) controller.$dateValue = parsedTime;
                }
                if (options.timeType === 'string') {
                    return dateFilter(parsedTime, options.modelTimeFormat || options.timeFormat);
                } else if (options.timeType === 'number') {
                    return controller.$dateValue.getTime();
                } else if (options.timeType === 'iso') {
                    return controller.$dateValue.toISOString();
                } else {
                    return new Date(controller.$dateValue);
                }
            });

            // modelValue -> $formatters -> viewValue
            controller.$formatters.push(function (modelValue) {
                // console.warn('$formatter("%s"): modelValue=%o (%o)', element.attr('ng-model'), modelValue, typeof modelValue);
                var date;
                if (angular.isUndefined(modelValue) || modelValue === null) {
                    date = NaN;
                } else if (angular.isDate(modelValue)) {
                    date = modelValue;
                } else if (options.timeType === 'string') {
                    date = dateParser.parse(modelValue, null, options.modelTimeFormat);
                } else {
                    date = new Date(modelValue);
                }
                // Setup default value?
                // if(isNaN(date.getTime())) date = new Date(new Date().setMinutes(0) + 36e5);
                controller.$dateValue = date;
                return controller.$dateValue;
            });

            // viewValue -> element
            controller.$render = function () {
                // console.warn('$render("%s"): viewValue=%o', element.attr('ng-model'), controller.$viewValue);
                element.val(!controller.$dateValue || isNaN(controller.$dateValue.getTime()) ? '' : dateFilter(controller.$dateValue, options.timeFormat));
            };

            // Garbage collection
            scope.$on('$destroy', function () {
                timepicker.destroy();
                options = null;
                timepicker = null;
            });

            element.on('blur keyup change', function () {
                scope.$evalAsync(function () {
                    return controller.$setViewValue(element.val());
                });
            });
        }
    };
});
/* jshint ignore:end */
'use strict';

angular.module('xl.widget').directive('xlWidgetTags', function ($interval) {
    return {
        restrict: 'A',
        scope: {
            ngModel: '=',
            options: '=',
            selectOptions: '=',
            displayMode: '='
        },
        replace: true,
        templateUrl: 'src/xl-widget/xl-widget-tags/xl-widget-tags.html',
        link: function link(scope, element) {
            var clearSelection = function clearSelection() {
                var selectedTags = element.find('li');
                _.forEach(selectedTags, function (el) {
                    angular.element(el).removeClass('select2-search-choice-selected');
                });
            };

            var inputSelector = '[ng-class=\"options.label ? \'xl-components-input\' : \'xl-components-input-full\'\"]';
            element.find(inputSelector).append('<button class="xl-select2-button-add ng-hide">Add</button>');
            var addButton = element.find(inputSelector + ' button.xl-select2-button-add');

            $interval(function () {
                _.forEach(element.find('li'), function (el) {
                    if (el.getAttribute('class') === 'select2-search-choice') {
                        var firstDiv = el.getElementsByTagName('div')[0];
                        if (firstDiv && !firstDiv.textContent) {
                            el.parentElement.removeChild(el);
                        }
                    }
                });

                var input = element.find('.select2-container input');
                var select2 = element.find('.select2-input');
                select2.on('select2-close', function () {
                    return addButton.addClass('ng-hide');
                });

                input.keyup(function () {
                    return addButton.toggleClass('ng-hide', input.val() === '');
                });
                element.find(inputSelector + ' button.xl-select2-button-add').on('mousedown', function () {
                    $interval(function () {
                        return angular.element('#select2-drop-mask').triggerHandler('click');
                    }, 1, 1);
                });
            }, 1, 1);

            scope.$on('itemMatched', function (event, args) {
                clearSelection();
                var selectedTags = element.find('li:contains(' + args + ')');
                selectedTags.each(function (i, el) {
                    var element = angular.element(el);
                    if (element.text().trim() === args) {
                        element.addClass('select2-search-choice-selected');
                    }
                });
            });

            scope.$on('itemUnMatched', function () {
                return clearSelection();
            });
        },
        controller: function controller($scope) {
            var itemMatching = false;

            var createSearchChoice = function createSearchChoice(term) {
                if (_.includes($scope.ngModel, term)) {
                    itemMatching = true;
                    $scope.$apply(function () {
                        return $scope.$emit('itemMatched', term);
                    });
                } else if (itemMatching && !_.includes($scope.ngModel, term)) {
                    itemMatching = false;
                    $scope.$apply(function () {
                        return $scope.$emit('itemUnMatched', term);
                    });
                }
                return { id: term, text: term };
            };

            if (angular.isDefined($scope.selectOptions)) {
                if (angular.isUndefined($scope.selectOptions.width)) {
                    $scope.selectOptions.width = 'resolve';
                }
                if (angular.isUndefined($scope.selectOptions.dropdownCssClass)) {
                    $scope.selectOptions.dropdownCssClass = _.constant('xl-components xl-widget-select-div');
                }

                $scope.selectOptions.multiple = true;
                $scope.selectOptions.selectOnBlur = true;
                $scope.selectOptions.width = 'off';
                $scope.selectOptions.simple_tags = true;
                $scope.selectOptions.tags = [];
                $scope.selectOptions.createSearchChoice = createSearchChoice;
                $scope.selectOptions.initSelection = true;
            }
        }
    };
});
'use strict';

var _slicedToArray = function () { function sliceIterator(arr, i) { var _arr = []; var _n = true; var _d = false; var _e = undefined; try { for (var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true) { _arr.push(_s.value); if (i && _arr.length === i) break; } } catch (err) { _d = true; _e = err; } finally { try { if (!_n && _i["return"]) _i["return"](); } finally { if (_d) throw _e; } } return _arr; } return function (arr, i) { if (Array.isArray(arr)) { return arr; } else if (Symbol.iterator in Object(arr)) { return sliceIterator(arr, i); } else { throw new TypeError("Invalid attempt to destructure non-iterable instance"); } }; }();

angular.module('xl.widget').directive('xlWidgetString', [function () {
    return {
        restrict: 'A',
        scope: {
            ngModel: '=',
            options: '=',
            isDisabled: '=',
            displayMode: '=',
            keyDownHandler: '='
        },
        replace: true,
        templateUrl: 'src/xl-widget/xl-widget-string/xl-widget-string.html',
        require: ['?^form', 'ngModel'],
        link: function link(scope, elem, attrs, ctrls) {
            var _ctrls = _slicedToArray(ctrls, 2);

            var form = _ctrls[0];
            var ngModelCtrl = _ctrls[1];


            var setValidity = function setValidity(state) {
                var controller = _.get(form, scope.options.name);
                if (controller) {
                    controller.$setValidity(scope.options.name, state);
                    form.$setValidity(scope.options.name, state);
                }
            };

            scope.$watch('options', function () {
                if (scope.options) {
                    (function () {
                        scope.isTextArea = scope.options.size === 'LARGE';

                        var validators = scope.options.validators;
                        if (validators) {
                            _.forEach(_.keys(validators), function (validatorName) {
                                ngModelCtrl.$validators[validatorName] = function (model, view) {
                                    var result = validators[validatorName](model, view);
                                    setValidity(result);
                                    return result;
                                };
                            });
                        }
                    })();
                }
                if (_.isUndefined(scope.ngModel) && !_.isUndefined(scope.options) && !_.isUndefined(scope.options['default']) && !_.isNull(scope.options['default'])) {
                    scope.ngModel = scope.options['default'];
                }
            });
            scope.erase = function () {
                ngModelCtrl.$setViewValue('');
                ngModelCtrl.$validate();
            };
        },
        controller: function controller($scope) {
            // trying to prevent safari/ie bug which opens up a pop DEPL-7854
            $scope.setValue = function ($event) {
                if ($event && $event.isDefaultPrevented()) return;

                if (!$event || $event.keyCode === 13 || $event.keyCode === 0 || $event.which === 13 || $event.which === 0) {
                    $event.preventDefault();
                }
                if ($scope.keyDownHandler) {
                    $scope.keyDownHandler($event);
                }
            };

            $scope.$watch('ngModel', function (newValue) {
                if (newValue === '') {
                    $scope.erase();
                }
            });
        }
    };
}]);
'use strict';

angular.module('xl.widget').directive('xlWidgetSpinner', [function () {
    return {
        restrict: 'A',
        replace: true,
        templateUrl: 'src/xl-widget/xl-widget-spinner/xl-widget-spinner.html'
    };
}]);
'use strict';

angular.module('xl.widget').directive('xlWidgetSelect', function (xlWidgetUtils) {
    return {
        restrict: 'A',
        require: 'ngModel',
        scope: {
            ngModel: '=',
            options: '=',
            selectOptions: '=',
            selectData: '=',
            displayMode: '='
        },
        replace: true,
        templateUrl: 'src/xl-widget/xl-widget-select/xl-widget-select.html',
        link: function link(scope, elem, attrs, ngModelCtrl) {
            scope.$watch('ngModel', function () {
                ngModelCtrl.$setValidity('required', !xlWidgetUtils.hasRequiredError(scope.ngModel, scope.options.required));
            });
        },
        controller: function controller($scope) {
            if (!_.isUndefined($scope.selectOptions)) {
                $scope.selectOptions.width = 'off';
            } else {
                $scope.selectOptions = { width: 'off' };
            }

            if (angular.isDefined($scope.selectOptions) && angular.isUndefined($scope.selectOptions.dropdownCssClass)) {
                $scope.selectOptions.dropdownCssClass = _.constant('xl-components xl-widget-select-div');
            }
        }
    };
});
'use strict';

angular.module('xl.widget').directive('xlWidgetProgressbar', function ($log, xlWidgetProgressbarService) {
    return {
        restrict: 'A',
        scope: {
            currentProgress: '=',
            options: '=',
            status: '='
        },
        templateUrl: 'src/xl-widget/xl-widget-progressbar/xl-widget-progressbar.html',
        link: function link(scope) {
            scope.min = scope.options.minValue || 0;
            scope.max = scope.options.maxValue || 100;
            scope.currentStatus = 'inProgress';
            scope.innerBarStyle = {};
            scope.innerBarClass = undefined;

            var STATUSES = {
                'inProgress': undefined,
                'success': 'xl-progressbar-success',
                'error': 'xl-progressbar-error'
            };

            scope.cancelProgress = function () {
                if (scope.options.cancel) {
                    scope.options.cancel();
                }
            };

            scope.applySuccessProgress = function () {
                if (scope.options.close) {
                    scope.options.close();
                }
            };

            scope.isCancelable = function () {
                return !!scope.options.cancel;
            };

            scope.displayButtonByStatus = function (expectedStatus) {
                return scope.isCancelable() && scope.currentStatus === expectedStatus;
            };

            scope.updateStatus = function () {
                if (scope.status === 'error') {
                    scope.currentStatus = 'error';
                } else if (scope.status === 'success') {
                    scope.currentStatus = 'success';
                } else {
                    scope.currentStatus = 'inProgress';
                }
            };

            scope.updateInnerDisplay = function (newRatio) {
                var updatedRatio = newRatio;
                scope.innerBarClass = STATUSES[scope.currentStatus];
                if (scope.currentStatus === 'success' || scope.currentStatus === 'error') {
                    updatedRatio = '100%';
                }
                scope.innerBarStyle.width = updatedRatio;
            };

            scope.calculateRatio = function () {
                var result = 0;
                try {
                    result = xlWidgetProgressbarService.calculateRatio(scope.min, scope.max, scope.currentProgress);
                } catch (e) {
                    if (e.message.search('greater than the maximal') > 0) {
                        result = 100;
                    } else if (e.message.search('lesser than the minimum ') > 0) {
                        result = 0;
                    }
                    if (scope.options.strictError) {
                        $log.error(e.message);
                    }
                }
                return result;
            };

            scope.$watch(function () {
                return scope.status + scope.currentProgress;
            }, function () {
                var newRatio = scope.calculateRatio();
                scope.updateStatus();
                scope.updateInnerDisplay(newRatio + '%');
            });
        }
    };
});
'use strict';

angular.module('xl.widget').factory('xlWidgetProgressbarService', function () {
    function calculateRatio(minValue, maxValue, currentValue) {
        if (currentValue < minValue) {
            throw new Error('invalid value for calculating ratio: current value is lesser than the minimum value');
        }
        if (currentValue > maxValue) {
            throw new Error('invalid value for calculating ratio: current value is greater than the maximal value');
        }

        var floatResult = 100 * (currentValue - minValue) / (maxValue - minValue);
        return Math.round(floatResult);
    }

    return {
        calculateRatio: calculateRatio
    };
});
'use strict';

angular.module('xl.widget').directive('xlOverlay', function ($timeout) {
    return {
        link: function link(scope, element, attr) {
            var overlayElement = angular.element('<div class="xl-screen-overlay"></div>');
            var mainElement = angular.element(element[0]);
            angular.element(overlayElement).appendTo(element[0]);

            var timerPromise;
            scope.$watch(function () {
                return scope.$eval(attr.xlOverlay);
            }, function (value, oldValue) {
                if (_.isUndefined(oldValue)) {
                    return;
                }

                if (!_.isUndefined(timerPromise)) {
                    $timeout.cancel(timerPromise);
                    timerPromise = undefined;
                }

                if (value === true || _.isUndefined(value)) {
                    var xlOverlayDelay = scope.$eval(attr.xlOverlayDelay) || 0;
                    if (xlOverlayDelay) {
                        timerPromise = $timeout(showOverlay, xlOverlayDelay);
                    } else {
                        showOverlay();
                    }
                } else {
                    hideOverlay();
                }
            });

            function hideOverlay() {
                overlayElement.addClass('xl-screen-overlay-hide');
                mainElement.removeClass('xl-screen-overlay-blur');
            }

            function showOverlay() {
                overlayElement.removeClass('xl-screen-overlay-hide');
                mainElement.addClass('xl-screen-overlay-blur');
            }
        }
    };
});
'use strict';

angular.module('xl.widget').factory('xlNotify', function ($interval) {
    var notifications = [];

    var typesClasses = {
        'INFO': 'xl-widget-notification-info',
        'SUCCESS': 'xl-widget-notification-success',
        'WARN': 'xl-widget-notification-warn',
        'ERROR': 'xl-widget-notification-error'
    };

    function Notification(message, type, options) {
        options = options || {};
        this.type = type || 'INFO';

        if (!angular.isObject(message)) {
            message = { message: message || 'No message', title: this.type };
        }

        this.body = message;
        this.duration = options.duration || 5000;
        this.typeClass = typesClasses[this.type] || typesClasses.INFO;

        var self = this;
        $interval(function () {
            notifications.splice(notifications.indexOf(self), 1);
        }, this.duration, 1);
    }

    Notification.prototype.close = function () {
        var index = notifications.indexOf(this);
        notifications.splice(index, 1);
    };

    var notify = function notify(message, type, options) {
        notifications.push(new Notification(message, type, options));
    };

    var getNotifications = function getNotifications() {
        return notifications;
    };

    // Public API
    return {
        notify: notify,
        getNotifications: getNotifications
    };
});

angular.module('xl.widget').directive('xlNotifyView', function (xlNotify) {
    return {
        restrict: 'EA',
        scope: {},
        replace: false,
        templateUrl: 'src/xl-widget/xl-widget-notify/xl-widget-notify.html',
        link: function link(scope) {
            scope.notifications = xlNotify.getNotifications();

            scope.dismiss = function (notification) {
                notification.close();
            };
        }
    };
});
'use strict';

angular.module('xl-widget-modal', ['xl-widget-modal-transition']).factory('$$stackedMap', function () {
    return {
        createNew: function createNew() {
            var stack = [];

            return {
                add: function add(key, value) {
                    stack.push({
                        key: key,
                        value: value
                    });
                },
                get: function get(key) {
                    for (var i = 0; i < stack.length; i++) {
                        if (key === stack[i].key) {
                            return stack[i];
                        }
                    }
                },
                keys: function keys() {
                    var keys = [];
                    for (var i = 0; i < stack.length; i++) {
                        keys.push(stack[i].key);
                    }
                    return keys;
                },
                top: function top() {
                    return stack[stack.length - 1];
                },
                remove: function remove(key) {
                    var idx = -1;
                    for (var i = 0; i < stack.length; i++) {
                        if (key === stack[i].key) {
                            idx = i;
                            break;
                        }
                    }
                    return stack.splice(idx, 1)[0];
                },
                removeTop: function removeTop() {
                    return stack.splice(stack.length - 1, 1)[0];
                },
                length: function length() {
                    return stack.length;
                }
            };
        }
    };
}).directive('xlModalBackdrop', ['$timeout', function ($timeout) {
    return {
        restrict: 'EA',
        replace: true,
        templateUrl: 'backdrop.html',
        link: function link(scope) {

            scope.animate = false;

            $timeout(function () {
                scope.animate = true;
            });
        }
    };
}]).directive('xlModalWindow', ['$xlModalStack', '$timeout', function ($xlModalStack, $timeout) {
    return {
        restrict: 'EA',
        scope: {
            index: '@',
            animate: '=',
            hasCloseButton: '='
        },
        replace: true,
        transclude: true,
        templateUrl: 'window.html',
        link: function link(scope, element, attrs) {
            scope.windowClass = attrs.windowClass || '';

            $timeout(function () {
                // trigger CSS transitions
                scope.animate = true;
                // focus a freshly-opened modal
                element[0].focus();
            });

            scope.close = function (evt) {
                var modal = $xlModalStack.getTop();
                if (modal && modal.value.backdrop && modal.value.backdrop !== 'static' && evt.target === evt.currentTarget) {
                    evt.preventDefault();
                    evt.stopPropagation();

                    if (modal.key.handlers && modal.key.handlers.onCloseModal) {
                        modal.key.handlers.onCloseModal();
                    }
                }
            };
        }
    };
}]).factory('$xlModalStack', ['$xlTransition', '$timeout', '$document', '$compile', '$rootScope', '$$stackedMap', function ($xlTransition, $timeout, $document, $compile, $rootScope, $$stackedMap) {

    var OPENED_MODAL_CLASS = 'xl-modal-open';

    var backdropDomEl, backdropScope;
    var openedWindows = $$stackedMap.createNew();
    var $xlModalStack = {};

    function backdropIndex() {
        var topBackdropIndex = -1;
        var opened = openedWindows.keys();
        for (var i = 0; i < opened.length; i++) {
            if (openedWindows.get(opened[i]).value.backdrop) {
                topBackdropIndex = i;
            }
        }
        return topBackdropIndex;
    }

    var deregistrationCallback = $rootScope.$watch(backdropIndex, function (newBackdropIndex) {
        if (backdropScope) {
            backdropScope.index = newBackdropIndex;
        }
    });

    $rootScope.$on('$destroy', deregistrationCallback);

    function removeModalWindow(modalInstance) {

        var body = $document.find('body').eq(0);
        var modalWindow = openedWindows.get(modalInstance).value;

        //clean up the stack
        openedWindows.remove(modalInstance);

        //remove window DOM element
        removeAfterAnimate(modalWindow.modalDomEl, modalWindow.modalScope, 300, checkRemoveBackdrop);
        body.toggleClass(OPENED_MODAL_CLASS, openedWindows.length() > 0);
    }

    function checkRemoveBackdrop() {
        //remove backdrop if no longer needed
        if (backdropDomEl && backdropIndex() === -1) {
            var backdropScopeRef = backdropScope;
            removeAfterAnimate(backdropDomEl, backdropScope, 150, function () {
                backdropScopeRef.$destroy();
                backdropScopeRef = null;
            });
            backdropDomEl = undefined;
            backdropScope = undefined;
        }
    }

    function removeAfterAnimate(domEl, scope, emulateTime, done) {

        function afterAnimating() {
            if (afterAnimating.done) {
                return;
            }
            afterAnimating.done = true;

            domEl.remove();
            if (done) {
                done();
            }
        }

        // Closing animation
        scope.animate = false;

        var transitionEndEventName = $xlTransition.transitionEndEventName;
        if (transitionEndEventName) {
            // transition out
            var timeout = $timeout(afterAnimating, emulateTime);

            domEl.bind(transitionEndEventName, function () {
                $timeout.cancel(timeout);
                afterAnimating();
                scope.$apply();
            });
        } else {
            // Ensure this call is async
            $timeout(afterAnimating, 0);
        }
    }

    $document.bind('keydown', function (evt) {
        var modal;

        if (evt.which === 27) {
            modal = openedWindows.top();
            if (modal && modal.value.keyboard && modal.key.handlers && modal.key.handlers.onEscModal) {
                modal.key.handlers.onEscModal();
            }
        }
    });

    $xlModalStack.open = function (modalInstance, modal) {

        openedWindows.add(modalInstance, {
            deferred: modal.deferred,
            modalScope: modal.scope,
            backdrop: modal.backdrop,
            keyboard: modal.keyboard
        });

        var body = $document.find('body').eq(0),
            currBackdropIndex = backdropIndex();

        if (currBackdropIndex >= 0 && !backdropDomEl) {
            backdropScope = $rootScope.$new(true);
            backdropScope.index = currBackdropIndex;
            backdropDomEl = $compile('<div xl-modal-backdrop></div>')(backdropScope);
            body.append(backdropDomEl);
        }

        var angularDomEl = angular.element('<div xl-modal-window></div>');
        angularDomEl.attr('window-class', modal.windowClass);
        angularDomEl.attr('has-close-button', !!modalInstance.handlers && !!modalInstance.handlers.onCloseModal);
        angularDomEl.attr('index', openedWindows.length() - 1);
        angularDomEl.attr('animate', 'animate');
        angularDomEl.html(modal.content);

        var modalDomEl = $compile(angularDomEl)(modal.scope);
        openedWindows.top().value.modalDomEl = modalDomEl;
        body.append(modalDomEl);
        body.addClass(OPENED_MODAL_CLASS);
    };

    $xlModalStack.close = function (modalInstance, result) {
        var modalWindow = openedWindows.get(modalInstance).value;
        if (modalWindow) {
            modalWindow.deferred.resolve(result);
            removeModalWindow(modalInstance);
        }
    };

    $xlModalStack.dismiss = function (modalInstance, reason) {
        var modalWindow = openedWindows.get(modalInstance).value;
        if (modalWindow) {
            modalWindow.deferred.reject(reason);
            removeModalWindow(modalInstance);
        }
    };

    $xlModalStack.dismissAll = function (reason) {
        var topModal = this.getTop();
        while (topModal) {
            this.dismiss(topModal.key, reason);
            topModal = this.getTop();
        }
    };

    $xlModalStack.getTop = function () {
        return openedWindows.top();
    };

    return $xlModalStack;
}]).provider('$xlModal', function () {

    var $xlModalProvider = {
        options: {
            backdrop: true, //can be also false or 'static'
            keyboard: true
        },
        $get: ['$injector', '$rootScope', '$q', '$http', '$templateCache', '$controller', '$xlModalStack', function ($injector, $rootScope, $q, $http, $templateCache, $controller, $xlModalStack) {

            var $xlModal = {};

            function getTemplatePromise(options) {
                return options.template ? $q.when(options.template) : $http.get(options.templateUrl, { cache: $templateCache }).then(function (result) {
                    return result.data;
                });
            }

            function getResolvePromises(resolves) {
                var promisesArr = [];
                angular.forEach(resolves, function (value, key) {
                    if (angular.isFunction(value) || angular.isArray(value)) {
                        promisesArr.push($q.when($injector.invoke(value)));
                    }
                });
                return promisesArr;
            }

            $xlModal.open = function (modalOptions) {

                var modalResultDeferred = $q.defer();
                var modalOpenedDeferred = $q.defer();

                //prepare an instance of a modal to be injected into controllers and returned to a caller
                var modalInstance = {
                    result: modalResultDeferred.promise,
                    opened: modalOpenedDeferred.promise,
                    close: function close(result) {
                        $xlModalStack.close(modalInstance, result);
                    },
                    dismiss: function dismiss(reason) {
                        $xlModalStack.dismiss(modalInstance, reason);
                    }
                };

                //merge and clean up options
                modalOptions = angular.extend({}, $xlModalProvider.options, modalOptions);
                modalOptions.resolve = modalOptions.resolve || {};

                //verify options
                if (!modalOptions.template && !modalOptions.templateUrl) {
                    throw new Error('One of template or templateUrl options is required.');
                }

                var templateAndResolvePromise = $q.all([getTemplatePromise(modalOptions)].concat(getResolvePromises(modalOptions.resolve)));

                templateAndResolvePromise.then(function resolveSuccess(tplAndVars) {

                    var modalScope = (modalOptions.scope || $rootScope).$new();
                    modalScope.$close = modalInstance.close;
                    modalScope.$dismiss = modalInstance.dismiss;

                    var ctrlLocals = {};
                    var resolveIter = 1;

                    //controllers
                    if (modalOptions.controller) {
                        ctrlLocals.$scope = modalScope;
                        ctrlLocals.$xlModalInstance = modalInstance;
                        angular.forEach(modalOptions.resolve, function (value, key) {
                            ctrlLocals[key] = tplAndVars[resolveIter++];
                        });
                        $controller(modalOptions.controller, ctrlLocals, modalOptions.bindToController, modalOptions.controllerAs);
                    }

                    var stackModal = {
                        backdrop: modalOptions.backdrop,
                        bindToController: modalOptions.bindToController,
                        content: tplAndVars[0],
                        deferred: modalResultDeferred,
                        keyboard: modalOptions.keyboard,
                        scope: modalScope,
                        windowClass: modalOptions.windowClass
                    };

                    $xlModalStack.open(modalInstance, stackModal);
                }, function resolveError(reason) {
                    modalResultDeferred.reject(reason);
                });

                templateAndResolvePromise.then(function () {
                    modalOpenedDeferred.resolve(true);
                }, function () {
                    modalOpenedDeferred.reject(false);
                });

                return modalInstance;
            };

            return $xlModal;
        }]
    };

    return $xlModalProvider;
}).run(['$templateCache', function ($templateCache) {
    $templateCache.put('backdrop.html', '<div class=\"xl-modal-backdrop fade\" ng-class=\"{in: animate}\" ng-style=\"{\'z-index\': 1040 + index*10}\"></div>');

    $templateCache.put('window.html', '<div tabindex="-1" class="xl-modal fade {{ windowClass }}" ng-class="{in: animate}" ng-style="{\'z-index\': 1050 + index*10, display: \'block\'}">\n' + '    <div class="xl-modal-dialog" ng-cloak><span ng-class="{\'xl-modal-close\': hasCloseButton}" data-ng-click="close($event)"></span><div class="xl-modal-content" ng-transclude ></div></div>\n' + '</div>');
}]);

angular.module('xl-widget-modal-transition', []).factory('$xlTransition', function ($q, $timeout, $rootScope) {

    var $xlTransition = function $xlTransition(element, trigger, options) {
        options = options || {};
        var deferred = $q.defer();
        var endEventName = $xlTransition[options.animation ? 'animationEndEventName' : 'transitionEndEventName'];

        var transitionEndHandler = function transitionEndHandler() {
            $rootScope.$apply(function () {
                element.unbind(endEventName, transitionEndHandler);
                deferred.resolve(element);
            });
        };

        if (endEventName) {
            element.bind(endEventName, transitionEndHandler);
        }

        $timeout(function () {
            if (angular.isString(trigger)) {
                element.addClass(trigger);
            } else if (angular.isFunction(trigger)) {
                trigger(element);
            } else if (angular.isObject(trigger)) {
                element.css(trigger);
            }
            //If browser does not support transitions, instantly resolve
            if (!endEventName) {
                deferred.resolve(element);
            }
        });

        deferred.promise.cancel = function () {
            if (endEventName) {
                element.unbind(endEventName, transitionEndHandler);
            }
            deferred.reject('Transition cancelled');
        };

        return deferred.promise;
    };

    var transElement = document.createElement('trans'); //eslint-disable-line
    var transitionEndEventNames = {
        'WebkitTransition': 'webkitTransitionEnd',
        'MozTransition': 'transitionend',
        'OTransition': 'oTransitionEnd',
        'transition': 'transitionend'
    };
    var animationEndEventNames = {
        'WebkitTransition': 'webkitAnimationEnd',
        'MozTransition': 'animationend',
        'OTransition': 'oAnimationEnd',
        'transition': 'animationend'
    };

    var findEndEventName = function findEndEventName(endEventNames) {
        _.forEach(endEventNames, function (name) {
            if (!_.isUndefined(transElement.style[name])) {
                return endEventNames[name];
            }
        });
    };

    $xlTransition.transitionEndEventName = findEndEventName(transitionEndEventNames);
    $xlTransition.animationEndEventName = findEndEventName(animationEndEventNames);
    return $xlTransition;
});
'use strict';

angular.module('xl.widget').directive('xlWidgetLabel', ['xlWidgetUtils', function (xlWidgetUtils) {
    return {
        restrict: 'A',
        scope: {
            ngModel: '=',
            options: '='
        },
        replace: true,
        templateUrl: 'src/xl-widget/xl-widget-label/xl-widget-label.html'
    };
}]);
'use strict';

angular.module('xl.widget').directive('xlWidgetItemSelector', function () {
    return {
        restrict: 'AE',
        scope: {
            ngModel: '=',
            handlers: '=',
            options: '=',
            objLabel: '@',
            emptyPlaceholder: '@'
        },
        templateUrl: 'src/xl-widget/xl-widget-item-selector/xl-widget-item-selector.html',
        link: function link(scope) {
            var handlers = scope.handlers || {};
            var itemSelectorHandler = handlers.itemSelectorHandler;
            scope.options.multiple = !_.isUndefined(scope.options.multiple) ? scope.options.multiple : true;
            scope.search = {};
            if (_.isUndefined(itemSelectorHandler)) {
                throw new Error('Handler itemSelectorHandler has to be defined');
            }

            var promise = itemSelectorHandler();
            if (promise && promise.then) {
                promise.then(function (data) {
                    scope.items = data;
                });
            }
        }
    };
});

angular.module('xl.widget').directive('checklistModel', function ($parse, $compile) {
    function contains(arr, item, comparaisonAttribute) {
        if (angular.isArray(arr)) {
            var itemValue = item[comparaisonAttribute];
            for (var i = 0; i < arr.length; i++) {
                if (arr[i][comparaisonAttribute] === itemValue) {
                    return true;
                }
            }
        }
        return false;
    }

    function add(arr, item) {
        arr = angular.isArray(arr) ? arr : [];
        for (var i = 0; i < arr.length; i++) {
            if (angular.equals(arr[i], item)) {
                return arr;
            }
        }
        arr.push(item);
        return arr;
    }

    function remove(arr, item) {
        if (angular.isArray(arr)) {
            for (var i = 0; i < arr.length; i++) {
                if (angular.equals(arr[i], item)) {
                    arr.splice(i, 1);
                    break;
                }
            }
        }
        return arr;
    }

    function postLinkFn(scope, elem, attrs) {
        $compile(elem)(scope);

        var getter = $parse(attrs.checklistModel);
        var setter = getter.assign;

        var value = $parse(attrs.checklistValue)(scope.$parent);
        var comparaisonAttribute = $parse(attrs.comparaisonAttribute)(scope.$parent);

        scope.$watch('checked', function (newValue, oldValue) {
            if (newValue === oldValue) {
                return;
            }
            var current = getter(scope.$parent);
            if (newValue === true) {
                if (scope.$parent.options && !scope.$parent.options.multiple) {
                    current.length = 0;
                }
                setter(scope.$parent, add(current, value));
            } else {
                setter(scope.$parent, remove(current, value));
            }
        });

        scope.$parent.$watch(attrs.checklistModel, function (newArr, oldArr) {
            scope.checked = contains(newArr, value, comparaisonAttribute);
        }, true);
    }

    return {
        restrict: 'A',
        priority: 1000,
        terminal: true,
        scope: true,
        compile: function compile(tElement, tAttrs) {
            if (tElement[0].tagName !== 'INPUT' || !tElement.attr('type', 'checkbox')) {
                throw 'checklist-model should be applied to `input[type="checkbox"]`.';
            }

            if (!tAttrs.checklistValue) {
                throw 'You should provide `checklist-value`.';
            }

            tElement.removeAttr('checklist-model');

            tElement.attr('ng-model', 'checked');

            return postLinkFn;
        }
    };
});
'use strict';

angular.module('xl.widget').directive('xlWidgetFilter', function () {
    return {
        scope: {
            ngModel: '='
        },
        templateUrl: 'src/xl-widget/xl-widget-filter/xl-widget-filter.html',
        link: _.noop
    };
});
'use strict';

angular.module('xl.widget').directive('xlWidgetFileUpload', function ($timeout, $interval, xlWidgetFileUploadService) {
    return {
        restrict: 'A',
        scope: {
            options: '=',
            isDisabled: '=',
            ngModel: '='
        },
        templateUrl: 'src/xl-widget/xl-widget-file-upload/xl-widget-file-upload.html',
        require: '?^form',
        link: function link(scope, element, attrs, parentForm) {
            scope.options = scope.options || {};
            scope.fileNameField = !_.isUndefined(scope.options.fileNameField) ? scope.options.fileNameField : 'fileName';
            scope.specClass = 'zone_' + new Date().getTime();
            scope.ngModel = { state: 'pristine', progress: 0, message: undefined, file: undefined };
            scope.selectedFile = '';

            var dataSubmitter, jqXHR;

            var init = function init() {
                dataSubmitter = null;
                jqXHR = null;
            };

            var pristine = function pristine() {
                init();
                scope.ngModel.state = 'pristine';
                scope.ngModel.progress = 0;
                scope.ngModel.message = undefined;
                scope.ngModel.file = undefined;
                scope.selectedFile = '';
            };
            scope.pristine = pristine;

            pristine();

            var upload = function upload() {
                scope.ngModel.state = 'progress';
                jqXHR = dataSubmitter.submit();

                if (scope.options.onStartUpload) {
                    scope.$apply(function () {
                        scope.options.onStartUpload(scope.selectedFile);
                    });
                }
            };

            function onFailure(e, data) {
                scope.$apply(function () {
                    scope.ngModel.state = 'error';
                    scope.ngModel.message = xlWidgetFileUploadService.getMessage(scope.ngModel.state);
                    if (scope.options.onError) {
                        var message = scope.options.onError(data, scope.selectedFile);
                        if (!_.isEmpty(message)) {
                            scope.ngModel.message = message;
                        }
                    }
                    init();
                });
            }

            function onSuccess(e, data) {
                scope.$apply(function () {
                    scope.ngModel.state = 'success';
                    scope.ngModel.message = xlWidgetFileUploadService.getMessage(scope.ngModel.state, data);
                    if (scope.options.onSuccess) {
                        var message = scope.options.onSuccess(data, scope.selectedFile);
                        if (!_.isEmpty(message)) {
                            scope.ngModel.message = message;
                        }
                    }
                    scope.selectedFile = '';
                    init();
                });
            }

            function onProgress(e, data) {
                var progress = parseInt(data.loaded / data.total * 100, 10);
                scope.$apply(function () {
                    scope.ngModel.message = xlWidgetFileUploadService.getMessage(scope.ngModel.state, data);
                    scope.ngModel.progress = progress;

                    if (scope.options.onProgress) {
                        scope.options.onProgress(data, scope.selectedFile);
                    }
                });
            }

            function add(e, data) {
                data.headers = { 'Accept': 'application/json' };
                if (data.files && data.files.length > 0) {
                    scope.$apply(function () {
                        scope.selectedFile = $.extend(true, {}, data.files[0]);
                        scope.ngModel.file = data.files[0];
                        if (parentForm) {
                            parentForm.$setDirty();
                        }
                    });
                }
                dataSubmitter = data;
                if (!_.isUndefined(scope.options.auto) ? scope.options.auto : true) {
                    upload();
                }
            }

            scope.browse = function () {
                $timeout(function () {
                    element.find('input:file').trigger('click');
                });
            };

            scope.cancel = function () {
                $timeout(function () {
                    if (jqXHR) {
                        jqXHR.abort();
                        pristine();
                    }
                });
            };

            scope.close = function () {
                pristine();
            };

            $timeout(function () {
                angular.element(element).fileupload({
                    replaceFileInput: true,
                    url: scope.options.uploadUrl,
                    dropZone: angular.element('.file-upload-drop-zone.' + scope.specClass),
                    add: add,
                    progressall: onProgress,
                    fail: onFailure,
                    done: onSuccess,
                    beforeSend: function beforeSend(xhr) {
                        if (scope.options.authorization) {
                            xhr.setRequestHeader('Authorization', scope.options.authorization);
                        }
                    }
                });
            });

            // Drag & drop zone configuration
            angular.element(document).bind('dragover', function (e) {
                var dropZone = angular.element('.file-upload-drop-zone');
                var foundDropzone = void 0,
                    timeout = $window.dropZoneTimeout;
                if (!timeout) {
                    dropZone.addClass('in');
                } else {
                    clearTimeout(timeout);
                }
                var found = false,
                    node = e.target;

                do {
                    if (angular.element(node).hasClass('file-upload-drop-zone')) {
                        found = true;
                        foundDropzone = angular.element(node);
                        break;
                    }
                    node = node.parentNode;
                } while (node !== null);

                dropZone.removeClass('hover');

                if (found) {
                    foundDropzone.addClass('hover');
                }

                $window.dropZoneTimeout = $interval(function () {
                    delete $window.dropZoneTimeout;
                    dropZone.removeClass('in hover');
                }, 100, 1);
            });
        }
    };
});
'use strict';

angular.module('xl.widget').factory('xlWidgetFileUploadService', function () {
    var humanFileSize = function humanFileSize(size) {
        var i = Math.floor(Math.log(size) / Math.log(1024));
        return ((size / Math.pow(1024, i)).toFixed(2) << 0) + ' ' + ['B', 'kB', 'MB', 'GB', 'TB'][i];
    };

    var getMessage = function getMessage(state, data) {
        var message = '';
        if (state === 'progress') {
            var progress = parseInt(data.loaded / data.total * 100, 10);
            //if (progress !== 100) {
            message = 'Imported ' + humanFileSize(data.loaded) + ' of ' + humanFileSize(data.total) + ' (' + progress + '%)';
            //}
        } else if (state === 'success') {
                message = 'Imported ' + humanFileSize(data.total);
            } else {
                message = 'Error';
            }
        return message;
    };

    return {
        getMessage: getMessage,
        humanFileSize: humanFileSize
    };
});
'use strict';

angular.module('xl.widget').directive('xlWidgetDate', ['xlWidgetUtils', function (xlWidgetUtils) {
    return {
        restrict: 'EA',
        replace: true,
        templateUrl: 'src/xl-widget/xl-widget-date/xl-widget-date.html',
        scope: {
            ngModel: '=',
            options: '=',
            displayMode: '='
        },
        require: 'ngModel',
        link: function link(scope, elem, attrs, ngModelCtrl) {
            scope.$watch('ngModel', function () {
                ngModelCtrl.$setValidity('required', !xlWidgetUtils.hasRequiredError(scope.ngModel, scope.options && scope.options.required));
            });
        },
        controller: function controller($scope) {
            $scope.$watch('options', function () {
                if (_.isUndefined($scope.ngModel) && !_.isUndefined($scope.options) && !_.isUndefined($scope.options['default']) && !_.isNull($scope.options['default'])) {
                    $scope.ngModel = $scope.options['default'];
                }
            });
        }
    };
}]);
'use strict';
/* jshint ignore:start */

angular.module('xl.widget.date.tooltip', ['xl.widget.date.dimensions']).provider('$tooltipLabs', function () {

    var defaults = this.defaults = {
        animation: 'am-fade',
        customClass: '',
        prefixClass: 'tooltip',
        prefixEvent: 'tooltip',
        container: false,
        target: false,
        placement: 'top',
        template: 'src/xl-widget/xl-widget-date/datepicker/tooltip.html',
        contentTemplate: false,
        trigger: 'hover focus',
        keyboard: false,
        html: false,
        show: false,
        title: '',
        type: '',
        delay: 0
    };

    this.$get = function ($window, $interval, $rootScope, $compile, $q, $templateCache, $http, $animate, xldimensions, $$rAF) {

        var trim = String.prototype.trim;
        var isTouch = 'createTouch' in $window.document;
        var htmlReplaceRegExp = /ng-bind="/ig;

        function TooltipFactory(element, config) {

            var $tooltipLabs = {};

            // Common vars
            var nodeName = element[0].nodeName.toLowerCase();
            var options = $tooltipLabs.$options = angular.extend({}, defaults, config);
            $tooltipLabs.$promise = fetchTemplate(options.template);
            var scope = $tooltipLabs.$scope = options.scope && options.scope.$new() || $rootScope.$new();
            if (options.delay && angular.isString(options.delay)) {
                options.delay = parseFloat(options.delay);
            }

            // Support scope as string options
            if (options.title) {
                $tooltipLabs.$scope.title = options.title;
            }

            // Provide scope helpers
            scope.$hide = function () {
                scope.$$postDigest(function () {
                    $tooltipLabs.hide();
                });
            };
            scope.$show = function () {
                scope.$$postDigest(function () {
                    $tooltipLabs.show();
                });
            };
            scope.$toggle = function () {
                scope.$$postDigest(function () {
                    $tooltipLabs.toggle();
                });
            };
            $tooltipLabs.$isShown = scope.$isShown = false;

            // Private vars
            var interval, hoverState;

            // Support contentTemplate option
            if (options.contentTemplate) {
                $tooltipLabs.$promise = $tooltipLabs.$promise.then(function (template) {
                    var templateEl = angular.element(template);
                    return fetchTemplate(options.contentTemplate).then(function (contentTemplate) {
                        var contentEl = findElement('[ng-bind="content"]', templateEl[0]);
                        if (!contentEl.length) contentEl = findElement('[ng-bind="title"]', templateEl[0]);
                        contentEl.removeAttr('ng-bind').html(contentTemplate);
                        return templateEl[0].outerHTML;
                    });
                });
            }

            // Fetch, compile then initialize tooltip
            var tipLinker, tipElement, tipTemplate, tipContainer;
            $tooltipLabs.$promise.then(function (template) {
                if (angular.isObject(template)) template = template.data;
                if (options.html) template = template.replace(htmlReplaceRegExp, 'ng-bind-html="');
                template = trim.apply(template);
                tipTemplate = template;
                tipLinker = $compile(template);
                $tooltipLabs.init();
            });

            $tooltipLabs.init = function () {

                // Options: delay
                if (options.delay && angular.isNumber(options.delay)) {
                    options.delay = {
                        show: options.delay,
                        hide: options.delay
                    };
                }

                // Replace trigger on touch devices ?
                // if(isTouch && options.trigger === defaults.trigger) {
                //   options.trigger.replace(/hover/g, 'click');
                // }

                // Options : container
                if (options.container === 'self') {
                    tipContainer = element;
                } else if (angular.isElement(options.container)) {
                    tipContainer = options.container;
                } else if (options.container) {
                    tipContainer = findElement(options.container);
                }

                // Options: trigger
                var triggers = options.trigger.split(' ');
                angular.forEach(triggers, function (trigger) {
                    if (trigger === 'click') {
                        element.on('click', $tooltipLabs.toggle);
                    } else if (trigger !== 'manual') {
                        element.on(trigger === 'hover' ? 'mouseenter' : 'focus', $tooltipLabs.enter);
                        element.on(trigger === 'hover' ? 'mouseleave' : 'blur', $tooltipLabs.leave);
                        nodeName === 'button' && trigger !== 'hover' && element.on(isTouch ? 'touchstart' : 'mousedown', $tooltipLabs.$onFocusElementMouseDown);
                    }
                });

                // Options: target
                if (options.target) {
                    options.target = angular.isElement(options.target) ? options.target : findElement(options.target);
                }

                // Options: show
                if (options.show) {
                    scope.$$postDigest(function () {
                        options.trigger === 'focus' ? element[0].focus() : $tooltipLabs.show();
                    });
                }
            };

            $tooltipLabs.destroy = function () {

                // Unbind events
                var triggers = options.trigger.split(' ');
                for (var i = triggers.length; i--;) {
                    var trigger = triggers[i];
                    if (trigger === 'click') {
                        element.off('click', $tooltipLabs.toggle);
                    } else if (trigger !== 'manual') {
                        element.off(trigger === 'hover' ? 'mouseenter' : 'focus', $tooltipLabs.enter);
                        element.off(trigger === 'hover' ? 'mouseleave' : 'blur', $tooltipLabs.leave);
                        nodeName === 'button' && trigger !== 'hover' && element.off(isTouch ? 'touchstart' : 'mousedown', $tooltipLabs.$onFocusElementMouseDown);
                    }
                }

                // Remove element
                if (tipElement) {
                    tipElement.remove();
                    tipElement = null;
                }

                // Cancel pending callbacks
                $interval.cancel(interval);

                // Destroy scope
                scope.$destroy();
            };

            $tooltipLabs.enter = function () {

                $interval.cancel(interval);
                hoverState = 'in';
                if (!options.delay || !options.delay.show) {
                    return $tooltipLabs.show();
                }

                interval = $interval(function () {
                    if (hoverState === 'in') $tooltipLabs.show();
                }, options.delay.show, 1);
            };

            $tooltipLabs.show = function () {

                scope.$emit(options.prefixEvent + '.show.before', $tooltipLabs);
                var parent = options.container ? tipContainer : null;
                var after = options.container ? null : element;

                // Hide any existing tipElement
                if (tipElement) tipElement.remove();
                // Fetch a cloned element linked from template
                tipElement = $tooltipLabs.$element = tipLinker(scope, _.noop);

                // Set the initial positioning.  Make the tooltip invisible
                // so IE doesn't try to focus on it off screen.
                tipElement.css({
                    top: '-9999px',
                    left: '-9999px',
                    display: 'block',
                    visibility: 'hidden'
                }).addClass(options.placement);

                // Options: animation
                if (options.animation) tipElement.addClass(options.animation);
                // Options: type
                if (options.type) tipElement.addClass(options.prefixClass + '-' + options.type);
                // Options: custom classes
                if (options.customClass) tipElement.addClass(options.customClass);

                $animate.enter(tipElement, parent, after).then(function () {
                    scope.$emit(options.prefixEvent + '.show', $tooltipLabs);
                });
                $tooltipLabs.$isShown = scope.$isShown = true;
                scope.$$phase || scope.$root && scope.$root.$$phase || scope.$digest();
                $$rAF(function () {
                    $tooltipLabs.$applyPlacement();

                    // Once placed, make the tooltip visible
                    tipElement.css({ visibility: 'visible' });
                }); // var a = bodyEl.offsetWidth + 1; ?

                // Bind events
                if (options.keyboard) {
                    if (options.trigger !== 'focus') {
                        $tooltipLabs.focus();
                        tipElement.on('keyup', $tooltipLabs.$onKeyUp);
                    } else {
                        element.on('keyup', $tooltipLabs.$onFocusKeyUp);
                    }
                }
            };

            $tooltipLabs.leave = function () {

                $interval.cancel(interval);
                hoverState = 'out';
                if (!options.delay || !options.delay.hide) {
                    return $tooltipLabs.hide();
                }
                interval = $interval(function () {
                    if (hoverState === 'out') {
                        $tooltipLabs.hide();
                    }
                }, options.delay.hide, 1);
            };

            $tooltipLabs.hide = function (blur) {
                if (!$tooltipLabs.$isShown) return;
                scope.$emit(options.prefixEvent + '.hide.before', $tooltipLabs);

                $animate.leave(tipElement).then(function () {
                    scope.$emit(options.prefixEvent + '.hide', $tooltipLabs);

                    // Allow to blur the input when hidden, like when pressing enter key
                    if (blur && options.trigger === 'focus') {
                        return element[0].blur();
                    }
                });

                $tooltipLabs.$isShown = scope.$isShown = false;
                scope.$$phase || scope.$root && scope.$root.$$phase || scope.$digest();

                // Unbind events
                if (options.keyboard && tipElement !== null) {
                    tipElement.off('keyup', $tooltipLabs.$onKeyUp);
                }
            };

            $tooltipLabs.toggle = function () {
                $tooltipLabs.$isShown ? $tooltipLabs.leave() : $tooltipLabs.enter();
            };

            $tooltipLabs.focus = function () {
                tipElement[0].focus();
            };

            // Protected methods

            $tooltipLabs.$applyPlacement = function () {
                if (!tipElement) return;

                // Get the position of the tooltip element.
                var elementPosition = getPosition();

                // Get the height and width of the tooltip so we can center it.
                var tipWidth = tipElement.prop('offsetWidth'),
                    tipHeight = tipElement.prop('offsetHeight');

                // Get the tooltip's top and left coordinates to center it with this directive.
                var tipPosition = getCalculatedOffset(options.placement, elementPosition, tipWidth, tipHeight);

                // Now set the calculated positioning.
                tipPosition.top += 'px';
                tipPosition.left += 'px';
                tipElement.css(tipPosition);
            };

            $tooltipLabs.$onKeyUp = function (evt) {
                if (evt.which === 27 && $tooltipLabs.$isShown) {
                    $tooltipLabs.hide();
                    evt.stopPropagation();
                }
            };

            $tooltipLabs.$onFocusKeyUp = function (evt) {
                if (evt.which === 27) {
                    element[0].blur();
                    evt.stopPropagation();
                }
            };

            $tooltipLabs.$onFocusElementMouseDown = function (evt) {
                evt.preventDefault();
                evt.stopPropagation();
                // Some browsers do not auto-focus buttons (eg. Safari)
                $tooltipLabs.$isShown ? element[0].blur() : element[0].focus();
            };

            // Private methods

            function getPosition() {
                if (options.container === 'body') {
                    return xldimensions.offset(options.target[0] || element[0]);
                } else {
                    return xldimensions.position(options.target[0] || element[0]);
                }
            }

            function getCalculatedOffset(placement, position, actualWidth, actualHeight) {
                var offset;
                var split = placement.split('-');

                switch (split[0]) {
                    case 'right':
                        offset = {
                            top: position.top + position.height / 2 - actualHeight / 2,
                            left: position.left + position.width
                        };
                        break;
                    case 'bottom':
                        offset = {
                            top: position.top + position.height,
                            left: position.left + position.width / 2 - actualWidth / 2
                        };
                        break;
                    case 'left':
                        offset = {
                            top: position.top + position.height / 2 - actualHeight / 2,
                            left: position.left - actualWidth
                        };
                        break;
                    default:
                        offset = {
                            top: position.top - actualHeight,
                            left: position.left + position.width / 2 - actualWidth / 2
                        };
                        break;
                }

                if (!split[1]) {
                    return offset;
                }

                // Add support for corners @todo css
                if (split[0] === 'top' || split[0] === 'bottom') {
                    switch (split[1]) {
                        case 'left':
                            offset.left = position.left;
                            break;
                        case 'right':
                            offset.left = position.left + position.width - actualWidth;
                    }
                } else if (split[0] === 'left' || split[0] === 'right') {
                    switch (split[1]) {
                        case 'top':
                            offset.top = position.top - actualHeight;
                            break;
                        case 'bottom':
                            offset.top = position.top + position.height;
                    }
                }

                return offset;
            }

            return $tooltipLabs;
        }

        // Helper functions

        function findElement(query, element) {
            return angular.element((element || document).querySelectorAll(query));
        }

        function fetchTemplate(template) {
            return $q.when($templateCache.get(template) || $http.get(template)).then(function (res) {
                if (angular.isObject(res)) {
                    $templateCache.put(template, res.data);
                    return res.data;
                }
                return res;
            });
        }

        return TooltipFactory;
    };
}).directive('bsTooltipLabs', function ($window, $location, $sce, $tooltipLabs, $$rAF) {

    return {
        restrict: 'EAC',
        scope: true,
        link: function postLink(scope, element, attr) {

            // Directive options
            var options = { scope: scope };
            angular.forEach(['template', 'contentTemplate', 'placement', 'container', 'target', 'delay', 'trigger', 'keyboard', 'html', 'animation', 'type', 'customClass'], function (key) {
                if (angular.isDefined(attr[key])) options[key] = attr[key];
            });

            // Observe scope attributes for change
            angular.forEach(['title'], function (key) {
                attr.$observe(key, function (newValue, oldValue) {
                    scope[key] = $sce.trustAsHtml(newValue);
                    angular.isDefined(oldValue) && $$rAF(function () {
                        tooltip && tooltip.$applyPlacement();
                    });
                });
            });

            // Support scope as an object
            attr.bsTooltipLabs && scope.$watch(attr.bsTooltipLabs, function (newValue, oldValue) {
                if (angular.isObject(newValue)) {
                    angular.extend(scope, newValue);
                } else {
                    scope.title = newValue;
                }
                angular.isDefined(oldValue) && $$rAF(function () {
                    tooltip && tooltip.$applyPlacement();
                });
            }, true);

            // Visibility binding support
            attr.bsShow && scope.$watch(attr.bsShow, function (newValue, oldValue) {
                if (!tooltip || angular.isUndefined(newValue)) return;
                if (angular.isString(newValue)) newValue = !!newValue.match(',?(tooltip),?');
                newValue === true ? tooltip.show() : tooltip.hide();
            });

            // Initialize popover
            var tooltip = $tooltipLabs(element, options);

            // Garbage collection
            scope.$on('$destroy', function () {
                if (tooltip) tooltip.destroy();
                options = null;
                tooltip = null;
            });
        }
    };
});
/* jshint ignore:end */
'use strict';

angular.module('xl.widget.date.dimensions', []).factory('xldimensions', function ($window) {

    var jqLite = angular.element;
    var fn = {};

    /**
     * Test the element nodeName
     * @param element
     * @param name
     */
    var nodeName = fn.nodeName = function (element, name) {
        return element.nodeName && element.nodeName.toLowerCase() === name.toLowerCase();
    };

    /**
     * Returns the element computed style
     * @param element
     * @param prop
     * @param extra
     */
    fn.css = function (element, prop, extra) {
        var value;
        if (element.currentStyle) {
            //IE
            value = element.currentStyle[prop];
        } else if ($window.getComputedStyle) {
            value = $window.getComputedStyle(element)[prop];
        } else {
            value = element.style[prop];
        }
        return extra === true ? parseFloat(value) || 0 : value;
    };

    /**
     * Provides read-only equivalent of jQuery's offset function:
     * @required-by bootstrap-tooltip, bootstrap-affix
     * @url http://api.jquery.com/offset/
     * @param element
     */
    fn.offset = function (element) {
        var boxRect = element.getBoundingClientRect();
        var docElement = element.ownerDocument;
        return {
            width: boxRect.width || element.offsetWidth,
            height: boxRect.height || element.offsetHeight,
            top: boxRect.top + ($window.pageYOffset || docElement.documentElement.scrollTop) - (docElement.documentElement.clientTop || 0),
            left: boxRect.left + ($window.pageXOffset || docElement.documentElement.scrollLeft) - (docElement.documentElement.clientLeft || 0)
        };
    };

    /**
     * Returns the closest, non-statically positioned offsetParent of a given element
     * @required-by fn.position
     * @param element
     */
    var offsetParent = function offsetParentElement(element) {
        var docElement = element.ownerDocument;
        var offsetParent = element.offsetParent || docElement;
        if (nodeName(offsetParent, '#document')) return docElement.documentElement;
        while (offsetParent && !nodeName(offsetParent, 'html') && fn.css(offsetParent, 'position') === 'static') {
            offsetParent = offsetParent.offsetParent;
        }
        return offsetParent || docElement.documentElement;
    };

    /**
     * Provides read-only equivalent of jQuery's position function
     * @required-by bootstrap-tooltip, bootstrap-affix
     * @url http://api.jquery.com/offset/
     * @param element
     */
    fn.position = function (element) {

        var offsetParentRect = { top: 0, left: 0 },
            offsetParentElement,
            offset;

        // Fixed elements are offset from window (parentOffset = {top:0, left: 0}, because it is it's only offset parent
        if (fn.css(element, 'position') === 'fixed') {

            // We assume that getBoundingClientRect is available when computed position is fixed
            offset = element.getBoundingClientRect();
        } else {

            // Get *real* offsetParentElement
            offsetParentElement = offsetParent(element);
            offset = fn.offset(element);

            // Get correct offsets
            offset = fn.offset(element);
            if (!nodeName(offsetParentElement, 'html')) {
                offsetParentRect = fn.offset(offsetParentElement);
            }

            // Add offsetParent borders
            offsetParentRect.top += fn.css(offsetParentElement, 'borderTopWidth', true);
            offsetParentRect.left += fn.css(offsetParentElement, 'borderLeftWidth', true);
        }

        // Subtract parent offsets and element margins
        return {
            width: element.offsetWidth,
            height: element.offsetHeight,
            top: offset.top - offsetParentRect.top - fn.css(element, 'marginTop', true),
            left: offset.left - offsetParentRect.left - fn.css(element, 'marginLeft', true)
        };
    };

    /**
     * Provides equivalent of jQuery's height function
     * @required-by bootstrap-affix
     * @url http://api.jquery.com/height/
     * @param element
     * @param outer
     */
    fn.height = function (element, outer) {
        var value = element.offsetHeight;
        if (outer) {
            value += fn.css(element, 'marginTop', true) + fn.css(element, 'marginBottom', true);
        } else {
            value -= fn.css(element, 'paddingTop', true) + fn.css(element, 'paddingBottom', true) + fn.css(element, 'borderTopWidth', true) + fn.css(element, 'borderBottomWidth', true);
        }
        return value;
    };

    /**
     * Provides equivalent of jQuery's width function
     * @required-by bootstrap-affix
     * @url http://api.jquery.com/width/
     * @param element
     * @param outer
     */
    fn.width = function (element, outer) {
        var value = element.offsetWidth;
        if (outer) {
            value += fn.css(element, 'marginLeft', true) + fn.css(element, 'marginRight', true);
        } else {
            value -= fn.css(element, 'paddingLeft', true) + fn.css(element, 'paddingRight', true) + fn.css(element, 'borderLeftWidth', true) + fn.css(element, 'borderRightWidth', true);
        }
        return value;
    };

    return fn;
});
'use strict';
/* jshint ignore:start */

angular.module('xl.widget.date', ['xl.widget.date.dateParser', 'xl.widget.date.tooltip']).provider('$datepickerLabs', function () {

    var defaults = this.defaults = {
        animation: 'am-fade',
        prefixClass: 'datepicker',
        placement: 'bottom-left',
        template: 'src/xl-widget/xl-widget-date/datepicker/datepicker.html',
        trigger: 'focus',
        container: false,
        keyboard: true,
        html: false,
        delay: 0,
        // lang: $locale.id,
        useNative: false,
        dateType: 'date',
        dateFormat: 'shortDate',
        modelDateFormat: null,
        dayFormat: 'dd',
        strictFormat: false,
        autoclose: false,
        minDate: -Infinity,
        maxDate: +Infinity,
        startView: 0,
        minView: 0,
        startWeek: 0,
        daysOfWeekDisabled: '',
        iconLeft: 'icon-left',
        iconRight: 'icon-right'
    };

    this.$get = function ($window, $interval, $document, $rootScope, $sce, $locale, dateFilter, datepickerViews, $tooltipLabs) {

        var bodyEl = angular.element($window.document.body);
        var isNative = /(ip(a|o)d|iphone|android)/ig.test($window.navigator.userAgent);
        var isTouch = 'createTouch' in $window.document && isNative;
        if (!defaults.lang) defaults.lang = $locale.id;

        function DatepickerFactory(element, controller, config) {

            var $datepickerLabs = $tooltipLabs(element, angular.extend({}, defaults, config));
            var parentScope = config.scope;
            var options = $datepickerLabs.$options;
            var scope = $datepickerLabs.$scope;
            if (options.startView) options.startView -= options.minView;

            // View vars

            var pickerViews = datepickerViews($datepickerLabs);
            $datepickerLabs.$views = pickerViews.views;
            var viewDate = pickerViews.viewDate;
            scope.$mode = options.startView;
            scope.$iconLeft = options.iconLeft;
            scope.$iconRight = options.iconRight;
            var $picker = $datepickerLabs.$views[scope.$mode];

            // Scope methods

            scope.$select = function (date) {
                $datepickerLabs.select(date);
            };
            scope.$selectPane = function (value) {
                $datepickerLabs.$selectPane(value);
            };
            scope.$toggleMode = function () {
                $datepickerLabs.setMode((scope.$mode + 1) % $datepickerLabs.$views.length);
            };

            // Public methods

            $datepickerLabs.update = function (date) {
                // console.warn('$datepickerLabs.update() newValue=%o', date);
                if (angular.isDate(date) && !isNaN(date.getTime())) {
                    $datepickerLabs.$date = date;
                    $picker.update.call($picker, date);
                }
                // Build only if pristine
                $datepickerLabs.$build(true);
            };

            $datepickerLabs.updateDisabledDates = function (dateRanges) {
                options.disabledDateRanges = dateRanges;
                for (var i = 0, l = scope.rows.length; i < l; i++) {
                    angular.forEach(scope.rows[i], $datepickerLabs.$setDisabledEl);
                }
            };

            $datepickerLabs.select = function (date, keep) {
                // console.warn('$datepickerLabs.select', date, scope.$mode);
                if (!angular.isDate(controller.$dateValue)) controller.$dateValue = new Date(date);
                if (!scope.$mode || keep) {
                    controller.$setViewValue(angular.copy(date));
                    controller.$render();
                    if (options.autoclose && !keep) {
                        $datepickerLabs.hide(true);
                    }
                } else {
                    angular.extend(viewDate, {
                        year: date.getFullYear(),
                        month: date.getMonth(),
                        date: date.getDate()
                    });
                    $datepickerLabs.setMode(scope.$mode - 1);
                    $datepickerLabs.$build();
                }
            };

            $datepickerLabs.setMode = function (mode) {
                // console.warn('$datepickerLabs.setMode', mode);
                scope.$mode = mode;
                $picker = $datepickerLabs.$views[scope.$mode];
                $datepickerLabs.$build();
            };

            // Protected methods

            $datepickerLabs.$build = function (pristine) {
                // console.warn('$datepickerLabs.$build() viewDate=%o', viewDate);
                if (pristine === true && $picker.built) return;
                if (pristine === false && !$picker.built) return;
                $picker.build.call($picker);
            };

            $datepickerLabs.$updateSelected = function () {
                for (var i = 0, l = scope.rows.length; i < l; i++) {
                    angular.forEach(scope.rows[i], updateSelected);
                }
            };

            $datepickerLabs.$isSelected = function (date) {
                return $picker.isSelected(date);
            };

            $datepickerLabs.$setDisabledEl = function (el) {
                el.disabled = $picker.isDisabled(el.date);
            };

            $datepickerLabs.$selectPane = function (value) {
                var steps = $picker.steps;
                var targetDate = new Date(Date.UTC(viewDate.year + (steps.year || 0) * value, viewDate.month + (steps.month || 0) * value, viewDate.date + (steps.day || 0) * value));
                angular.extend(viewDate, {
                    year: targetDate.getUTCFullYear(),
                    month: targetDate.getUTCMonth(),
                    date: targetDate.getUTCDate()
                });
                $datepickerLabs.$build();
            };

            $datepickerLabs.$onMouseDown = function (evt) {
                // Prevent blur on mousedown on .dropdown-menu
                evt.preventDefault();
                evt.stopPropagation();
                // Emulate click for mobile devices
                if (isTouch) {
                    var targetEl = angular.element(evt.target);
                    if (targetEl[0].nodeName.toLowerCase() !== 'button') {
                        targetEl = targetEl.parent();
                    }
                    targetEl.triggerHandler('click');
                }
            };

            $datepickerLabs.$onKeyDown = function (evt) {
                if (!/(38|37|39|40|13)/.test(evt.keyCode) || evt.shiftKey || evt.altKey) return;
                evt.preventDefault();
                evt.stopPropagation();

                if (evt.keyCode === 13) {
                    if (!scope.$mode) {
                        return $datepickerLabs.hide(true);
                    } else {
                        return scope.$apply(function () {
                            $datepickerLabs.setMode(scope.$mode - 1);
                        });
                    }
                }

                // Navigate with keyboard
                $picker.onKeyDown(evt);
                parentScope.$digest();
            };

            // Private

            function updateSelected(el) {
                el.selected = $datepickerLabs.$isSelected(el.date);
            }

            function focusElement() {
                element[0].focus();
            }

            // Overrides

            var _init = $datepickerLabs.init;
            $datepickerLabs.init = function () {
                if (isNative && options.useNative) {
                    element.prop('type', 'date');
                    element.css('-webkit-appearance', 'textfield');
                    return;
                } else if (isTouch) {
                    element.prop('type', 'text');
                    element.attr('readonly', 'true');
                    element.on('click', focusElement);
                }
                _init();
            };

            var _destroy = $datepickerLabs.destroy;
            $datepickerLabs.destroy = function () {
                if (isNative && options.useNative) {
                    element.off('click', focusElement);
                }
                _destroy();
            };

            var _show = $datepickerLabs.show;
            $datepickerLabs.show = function () {
                _show();
                $interval(function () {
                    $datepickerLabs.$element.on(isTouch ? 'touchstart' : 'mousedown', $datepickerLabs.$onMouseDown);
                    if (options.keyboard) {
                        element.on('keydown', $datepickerLabs.$onKeyDown);
                    }
                }, 1, 1);
            };

            var _hide = $datepickerLabs.hide;
            $datepickerLabs.hide = function (blur) {
                $datepickerLabs.$element.off(isTouch ? 'touchstart' : 'mousedown', $datepickerLabs.$onMouseDown);
                if (options.keyboard) {
                    element.off('keydown', $datepickerLabs.$onKeyDown);
                }
                _hide(blur);
            };

            return $datepickerLabs;
        }

        DatepickerFactory.defaults = defaults;
        return DatepickerFactory;
    };
}).directive('bsDatepickerLabs', function ($window, $parse, $q, $locale, dateFilter, $datepickerLabs, $dateParserLabs, $timeout) {

    var defaults = $datepickerLabs.defaults;
    var isNative = /(ip(a|o)d|iphone|android)/ig.test($window.navigator.userAgent);
    var isNumeric = function isNumeric(n) {
        return !isNaN(parseFloat(n)) && isFinite(n);
    };

    return {
        restrict: 'EAC',
        require: 'ngModel',
        link: function postLink(scope, element, attr, controller) {

            // Directive options
            var options = { scope: scope, controller: controller };
            angular.forEach(['placement', 'container', 'delay', 'trigger', 'keyboard', 'html', 'animation', 'template', 'autoclose', 'dateType', 'dateFormat', 'modelDateFormat', 'dayFormat', 'strictFormat', 'startWeek', 'startDate', 'useNative', 'lang', 'startView', 'minView', 'iconLeft', 'iconRight', 'daysOfWeekDisabled'], function (key) {
                if (angular.isDefined(attr[key])) options[key] = attr[key];
            });

            // Visibility binding support
            attr.bsShow && scope.$watch(attr.bsShow, function (newValue, oldValue) {
                if (!datepicker || angular.isUndefined(newValue)) return;
                if (angular.isString(newValue)) newValue = !!newValue.match(',?(datepicker),?');
                newValue === true ? datepicker.show() : datepicker.hide();
            });

            // Initialize datepicker
            var datepicker = $datepickerLabs(element, controller, options);
            options = datepicker.$options;
            // Set expected iOS format
            if (isNative && options.useNative) options.dateFormat = 'yyyy-MM-dd';

            // Observe attributes for changes
            angular.forEach(['minDate', 'maxDate'], function (key) {
                // console.warn('attr.$observe(%s)', key, attr[key]);
                angular.isDefined(attr[key]) && attr.$observe(key, function (newValue) {
                    // console.warn('attr.$observe(%s)=%o', key, newValue);
                    if (newValue === 'today') {
                        var today = new Date();
                        datepicker.$options[key] = +new Date(today.getFullYear(), today.getMonth(), today.getDate() + (key === 'maxDate' ? 1 : 0), 0, 0, 0, key === 'minDate' ? 0 : -1);
                    } else if (angular.isString(newValue) && newValue.match(/^".+"$/)) {
                        // Support {{ dateObj }}
                        datepicker.$options[key] = +new Date(newValue.substr(1, newValue.length - 2));
                    } else if (isNumeric(newValue)) {
                        datepicker.$options[key] = +new Date(parseInt(newValue, 10));
                    } else if (angular.isString(newValue) && 0 === newValue.length) {
                        // Reset date
                        datepicker.$options[key] = key === 'maxDate' ? +Infinity : -Infinity;
                    } else {
                        datepicker.$options[key] = +new Date(newValue);
                    }
                    // Build only if dirty
                    !isNaN(datepicker.$options[key]) && datepicker.$build(false);
                });
            });

            // Watch model for changes
            scope.$watch(attr.ngModel, function (newValue, oldValue) {
                datepicker.update(controller.$dateValue);
            }, true);

            // Normalize undefined/null/empty array,
            // so that we don't treat changing from undefined->null as a change.
            function normalizeDateRanges(ranges) {
                if (!ranges || !ranges.length) return null;
                return ranges;
            }

            if (angular.isDefined(attr.disabledDates)) {
                scope.$watch(attr.disabledDates, function (disabledRanges, previousValue) {
                    disabledRanges = normalizeDateRanges(disabledRanges);
                    previousValue = normalizeDateRanges(previousValue);

                    if (disabledRanges !== previousValue) {
                        datepicker.updateDisabledDates(disabledRanges);
                    }
                });
            }

            var dateParser = $dateParserLabs({
                format: options.dateFormat,
                lang: options.lang,
                strict: options.strictFormat
            });

            // viewValue -> $parsers -> modelValue
            controller.$parsers.unshift(function (viewValue) {
                // console.warn('$parser("%s"): viewValue=%o', element.attr('ng-model'), viewValue);
                // Null values should correctly reset the model value & validity
                if (!viewValue) {
                    controller.$setValidity('date', true);
                    return viewValue;
                }
                var parsedDate = dateParser.parse(viewValue, controller.$dateValue);
                if (!parsedDate || isNaN(parsedDate.getTime())) {
                    controller.$setValidity('date', false);
                    return;
                } else {
                    var isMinValid = isNaN(datepicker.$options.minDate) || parsedDate.getTime() >= datepicker.$options.minDate;
                    var isMaxValid = isNaN(datepicker.$options.maxDate) || parsedDate.getTime() <= datepicker.$options.maxDate;
                    var isValid = isMinValid && isMaxValid;
                    controller.$setValidity('date', isValid);
                    controller.$setValidity('min', isMinValid);
                    controller.$setValidity('max', isMaxValid);
                    // Only update the model when we have a valid date
                    if (isValid) controller.$dateValue = parsedDate;
                }
                if (options.dateType === 'string') {
                    return dateFilter(parsedDate, options.modelDateFormat || options.dateFormat);
                } else if (options.dateType === 'number') {
                    return controller.$dateValue.getTime();
                } else if (options.dateType === 'iso') {
                    return controller.$dateValue.toISOString();
                } else {
                    return new Date(controller.$dateValue);
                }
            });

            // modelValue -> $formatters -> viewValue
            controller.$formatters.push(function (modelValue) {
                // console.warn('$formatter("%s"): modelValue=%o (%o)', element.attr('ng-model'), modelValue, typeof modelValue);
                var date;
                if (angular.isUndefined(modelValue) || modelValue === null) {
                    date = NaN;
                } else if (angular.isDate(modelValue)) {
                    date = modelValue;
                } else if (options.dateType === 'string') {
                    date = dateParser.parse(modelValue, null, options.modelDateFormat);
                } else {
                    date = new Date(modelValue);
                }
                // Setup default value?
                // if(isNaN(date.getTime())) {
                //   var today = new Date();
                //   date = new Date(today.getFullYear(), today.getMonth(), today.getDate(), 0, 0, 0, 0);
                // }
                controller.$dateValue = date;
                return controller.$dateValue;
            });

            // viewValue -> element
            controller.$render = function () {
                // console.warn('$render("%s"): viewValue=%o', element.attr('ng-model'), controller.$viewValue);
                element.val(!controller.$dateValue || isNaN(controller.$dateValue.getTime()) ? '' : dateFilter(controller.$dateValue, options.dateFormat));
            };

            // Garbage collection
            scope.$on('$destroy', function () {
                if (datepicker) datepicker.destroy();
                options = null;
                datepicker = null;
            });
        }
    };
}).provider('datepickerViews', function () {

    var defaults = this.defaults = {
        dayFormat: 'dd',
        daySplit: 7
    };

    // Split array into smaller arrays
    function split(arr, size) {
        var arrays = [];
        while (arr.length > 0) {
            arrays.push(arr.splice(0, size));
        }
        return arrays;
    }

    // Modulus operator
    function mod(n, m) {
        return (n % m + m) % m;
    }

    this.$get = function ($locale, $sce, dateFilter) {

        return function (picker) {

            var scope = picker.$scope;
            var options = picker.$options;

            var weekDaysMin = $locale.DATETIME_FORMATS.SHORTDAY;
            var weekDaysLabels = weekDaysMin.slice(options.startWeek).concat(weekDaysMin.slice(0, options.startWeek));
            var weekDaysLabelsHtml = $sce.trustAsHtml('<th class="dow text-center">' + weekDaysLabels.join('</th><th class="dow text-center">') + '</th>');

            var startDate = picker.$date || (options.startDate ? new Date(options.startDate) : new Date());
            var viewDate = { year: startDate.getFullYear(), month: startDate.getMonth(), date: startDate.getDate() };
            var timezoneOffset = startDate.getTimezoneOffset() * 6e4;

            var views = [{
                format: options.dayFormat,
                split: 7,
                steps: { month: 1 },
                update: function update(date, force) {
                    if (!this.built || force || date.getFullYear() !== viewDate.year || date.getMonth() !== viewDate.month) {
                        angular.extend(viewDate, {
                            year: picker.$date.getFullYear(),
                            month: picker.$date.getMonth(),
                            date: picker.$date.getDate()
                        });
                        picker.$build();
                    } else if (date.getDate() !== viewDate.date) {
                        viewDate.date = picker.$date.getDate();
                        picker.$updateSelected();
                    }
                },
                build: function build() {
                    var firstDayOfMonth = new Date(viewDate.year, viewDate.month, 1),
                        firstDayOfMonthOffset = firstDayOfMonth.getTimezoneOffset();
                    var firstDate = new Date(+firstDayOfMonth - mod(firstDayOfMonth.getDay() - options.startWeek, 7) * 864e5),
                        firstDateOffset = firstDate.getTimezoneOffset();
                    var today = new Date().toDateString();
                    // Handle daylight time switch
                    if (firstDateOffset !== firstDayOfMonthOffset) firstDate = new Date(+firstDate + (firstDateOffset - firstDayOfMonthOffset) * 60e3);
                    var days = [],
                        day;
                    for (var i = 0; i < 42; i++) {
                        // < 7 * 6
                        day = new Date(firstDate.getFullYear(), firstDate.getMonth(), firstDate.getDate() + i);
                        days.push({
                            date: day,
                            isToday: day.toDateString() === today,
                            label: dateFilter(day, this.format),
                            selected: picker.$date && this.isSelected(day),
                            muted: day.getMonth() !== viewDate.month,
                            disabled: this.isDisabled(day)
                        });
                    }
                    scope.title = dateFilter(firstDayOfMonth, 'MMMM yyyy');
                    scope.showLabels = true;
                    scope.labels = weekDaysLabelsHtml;
                    scope.rows = split(days, this.split);
                    this.built = true;
                },
                isSelected: function isSelected(date) {
                    return picker.$date && date.getFullYear() === picker.$date.getFullYear() && date.getMonth() === picker.$date.getMonth() && date.getDate() === picker.$date.getDate();
                },
                isDisabled: function isDisabled(date) {
                    var time = date.getTime();

                    // Disabled because of min/max date.
                    if (time < options.minDate || time > options.maxDate) return true;

                    // Disabled due to being a disabled day of the week
                    if (options.daysOfWeekDisabled.indexOf(date.getDay()) !== -1) return true;

                    // Disabled because of disabled date range.
                    if (options.disabledDateRanges) {
                        for (var i = 0; i < options.disabledDateRanges.length; i++) {
                            if (time >= options.disabledDateRanges[i].start) {
                                if (time <= options.disabledDateRanges[i].end) return true;

                                // The disabledDateRanges is expected to be sorted, so if time >= start,
                                // we know it's not disabled.
                                return false;
                            }
                        }
                    }

                    return false;
                },
                onKeyDown: function onKeyDown(evt) {
                    var actualTime = picker.$date.getTime();
                    var newDate;

                    if (evt.keyCode === 37) newDate = new Date(actualTime - 1 * 864e5);else if (evt.keyCode === 38) newDate = new Date(actualTime - 7 * 864e5);else if (evt.keyCode === 39) newDate = new Date(actualTime + 1 * 864e5);else if (evt.keyCode === 40) newDate = new Date(actualTime + 7 * 864e5);

                    if (!this.isDisabled(newDate)) picker.select(newDate, true);
                }
            }, {
                name: 'month',
                format: 'MMM',
                split: 4,
                steps: { year: 1 },
                update: function update(date, force) {
                    if (!this.built || date.getFullYear() !== viewDate.year) {
                        angular.extend(viewDate, {
                            year: picker.$date.getFullYear(),
                            month: picker.$date.getMonth(),
                            date: picker.$date.getDate()
                        });
                        picker.$build();
                    } else if (date.getMonth() !== viewDate.month) {
                        angular.extend(viewDate, { month: picker.$date.getMonth(), date: picker.$date.getDate() });
                        picker.$updateSelected();
                    }
                },
                build: function build() {
                    var firstMonth = new Date(viewDate.year, 0, 1);
                    var months = [],
                        month;
                    for (var i = 0; i < 12; i++) {
                        month = new Date(viewDate.year, i, 1);
                        months.push({
                            date: month,
                            label: dateFilter(month, this.format),
                            selected: picker.$isSelected(month),
                            disabled: this.isDisabled(month)
                        });
                    }
                    scope.title = dateFilter(month, 'yyyy');
                    scope.showLabels = false;
                    scope.rows = split(months, this.split);
                    this.built = true;
                },
                isSelected: function isSelected(date) {
                    return picker.$date && date.getFullYear() === picker.$date.getFullYear() && date.getMonth() === picker.$date.getMonth();
                },
                isDisabled: function isDisabled(date) {
                    var lastDate = +new Date(date.getFullYear(), date.getMonth() + 1, 0);
                    return lastDate < options.minDate || date.getTime() > options.maxDate;
                },
                onKeyDown: function onKeyDown(evt) {
                    var actualMonth = picker.$date.getMonth();
                    var newDate = new Date(picker.$date);

                    if (evt.keyCode === 37) newDate.setMonth(actualMonth - 1);else if (evt.keyCode === 38) newDate.setMonth(actualMonth - 4);else if (evt.keyCode === 39) newDate.setMonth(actualMonth + 1);else if (evt.keyCode === 40) newDate.setMonth(actualMonth + 4);

                    if (!this.isDisabled(newDate)) picker.select(newDate, true);
                }
            }, {
                name: 'year',
                format: 'yyyy',
                split: 4,
                steps: { year: 12 },
                update: function update(date, force) {
                    if (!this.built || force || parseInt(date.getFullYear() / 20, 10) !== parseInt(viewDate.year / 20, 10)) {
                        angular.extend(viewDate, {
                            year: picker.$date.getFullYear(),
                            month: picker.$date.getMonth(),
                            date: picker.$date.getDate()
                        });
                        picker.$build();
                    } else if (date.getFullYear() !== viewDate.year) {
                        angular.extend(viewDate, {
                            year: picker.$date.getFullYear(),
                            month: picker.$date.getMonth(),
                            date: picker.$date.getDate()
                        });
                        picker.$updateSelected();
                    }
                },
                build: function build() {
                    var firstYear = viewDate.year - viewDate.year % (this.split * 3);
                    var years = [],
                        year;
                    for (var i = 0; i < 12; i++) {
                        year = new Date(firstYear + i, 0, 1);
                        years.push({
                            date: year,
                            label: dateFilter(year, this.format),
                            selected: picker.$isSelected(year),
                            disabled: this.isDisabled(year)
                        });
                    }
                    scope.title = years[0].label + '-' + years[years.length - 1].label;
                    scope.showLabels = false;
                    scope.rows = split(years, this.split);
                    this.built = true;
                },
                isSelected: function isSelected(date) {
                    return picker.$date && date.getFullYear() === picker.$date.getFullYear();
                },
                isDisabled: function isDisabled(date) {
                    var lastDate = +new Date(date.getFullYear() + 1, 0, 0);
                    return lastDate < options.minDate || date.getTime() > options.maxDate;
                },
                onKeyDown: function onKeyDown(evt) {
                    var actualYear = picker.$date.getFullYear(),
                        newDate = new Date(picker.$date);

                    if (evt.keyCode === 37) newDate.setYear(actualYear - 1);else if (evt.keyCode === 38) newDate.setYear(actualYear - 4);else if (evt.keyCode === 39) newDate.setYear(actualYear + 1);else if (evt.keyCode === 40) newDate.setYear(actualYear + 4);

                    if (!this.isDisabled(newDate)) picker.select(newDate, true);
                }
            }];

            return {
                views: options.minView ? Array.prototype.slice.call(views, options.minView) : views,
                viewDate: viewDate
            };
        };
    };
});
/* jshint ignore:end */
'use strict';

angular.module('xl.widget.date.dateParser', []).provider('$dateParserLabs', function ($localeProvider) {

    var proto = Date.prototype;

    function noop() {}

    function isNumeric(n) {
        return !isNaN(parseFloat(n)) && isFinite(n);
    }

    var defaults = this.defaults = {
        format: 'shortDate',
        strict: false
    };

    this.$get = function ($locale, dateFilter) {

        var DateParserFactory = function DateParserFactory(config) {

            var options = angular.extend({}, defaults, config);

            var $dateParserLabs = {};

            var regExpMap = {
                'sss': '[0-9]{3}',
                'ss': '[0-5][0-9]',
                's': options.strict ? '[1-5]?[0-9]' : '[0-9]|[0-5][0-9]',
                'mm': '[0-5][0-9]',
                'm': options.strict ? '[1-5]?[0-9]' : '[0-9]|[0-5][0-9]',
                'HH': '[01][0-9]|2[0-3]',
                'H': options.strict ? '1?[0-9]|2[0-3]' : '[01]?[0-9]|2[0-3]',
                'hh': '[0][1-9]|[1][012]',
                'h': options.strict ? '[1-9]|1[012]' : '0?[1-9]|1[012]',
                'a': 'AM|PM',
                'EEEE': $locale.DATETIME_FORMATS.DAY.join('|'),
                'EEE': $locale.DATETIME_FORMATS.SHORTDAY.join('|'),
                'dd': '0[1-9]|[12][0-9]|3[01]',
                'd': options.strict ? '[1-9]|[1-2][0-9]|3[01]' : '0?[1-9]|[1-2][0-9]|3[01]',
                'MMMM': $locale.DATETIME_FORMATS.MONTH.join('|'),
                'MMM': $locale.DATETIME_FORMATS.SHORTMONTH.join('|'),
                'MM': '0[1-9]|1[012]',
                'M': options.strict ? '[1-9]|1[012]' : '0?[1-9]|1[012]',
                'yyyy': '[1]{1}[0-9]{3}|[2]{1}[0-9]{3}',
                'yy': '[0-9]{2}',
                'y': options.strict ? '-?(0|[1-9][0-9]{0,3})' : '-?0*[0-9]{1,4}'
            };

            var setFnMap = {
                'sss': proto.setMilliseconds,
                'ss': proto.setSeconds,
                's': proto.setSeconds,
                'mm': proto.setMinutes,
                'm': proto.setMinutes,
                'HH': proto.setHours,
                'H': proto.setHours,
                'hh': proto.setHours,
                'h': proto.setHours,
                'EEEE': noop,
                'EEE': noop,
                'dd': proto.setDate,
                'd': proto.setDate,
                'a': function a(value) {
                    var hours = this.getHours();return this.setHours(value.match(/pm/i) ? hours + 12 : hours);
                },
                'MMMM': function MMMM(value) {
                    return this.setMonth($locale.DATETIME_FORMATS.MONTH.indexOf(value));
                },
                'MMM': function MMM(value) {
                    return this.setMonth($locale.DATETIME_FORMATS.SHORTMONTH.indexOf(value));
                },
                'MM': function MM(value) {
                    return this.setMonth(1 * value - 1);
                },
                'M': function M(value) {
                    return this.setMonth(1 * value - 1);
                },
                'yyyy': proto.setFullYear,
                'yy': function yy(value) {
                    return this.setFullYear(2000 + 1 * value);
                },
                'y': proto.setFullYear
            };

            var regex, setMap;

            $dateParserLabs.init = function () {
                $dateParserLabs.$format = $locale.DATETIME_FORMATS[options.format] || options.format;
                regex = regExpForFormat($dateParserLabs.$format);
                setMap = setMapForFormat($dateParserLabs.$format);
            };

            $dateParserLabs.isValid = function (date) {
                if (angular.isDate(date)) return !isNaN(date.getTime());
                return regex.test(date);
            };

            $dateParserLabs.parse = function (value, baseDate, format) {
                if (angular.isDate(value)) value = dateFilter(value, format || $dateParserLabs.$format);
                var formatRegex = format ? regExpForFormat(format) : regex;
                var formatSetMap = format ? setMapForFormat(format) : setMap;
                var matches = formatRegex.exec(value);
                if (!matches) return false;
                var date = baseDate || new Date(0, 0, 1);
                for (var i = 0; i < matches.length - 1; i++) {
                    if (formatSetMap[i]) {
                        formatSetMap[i].call(date, matches[i + 1]);
                    }
                }
                return date;
            };

            // Private functions

            function setMapForFormat(format) {
                var keys = _.keys(setFnMap),
                    i;
                var map = [],
                    sortedMap = [];
                // Map to setFn
                var clonedFormat = format;
                for (i = 0; i < keys.length; i++) {
                    if (format.split(keys[i]).length > 1) {
                        var index = clonedFormat.search(keys[i]);
                        format = format.split(keys[i]).join('');
                        if (setFnMap[keys[i]]) {
                            map[index] = setFnMap[keys[i]];
                        }
                    }
                }
                // Sort result map
                angular.forEach(map, function (v) {
                    // conditional required since angular.forEach broke around v1.2.21
                    // related pr: https://github.com/angular/angular.js/pull/8525
                    if (v) sortedMap.push(v);
                });
                return sortedMap;
            }

            function escapeReservedSymbols(text) {
                return text.replace(/\//g, '[\\/]').replace('/-/g', '[-]').replace(/\./g, '[.]').replace(/\\s/g, '[\\s]');
            }

            function regExpForFormat(format) {
                var keys = _.keys(regExpMap),
                    i;

                var re = format;
                // Abstract replaces to avoid collisions
                for (i = 0; i < keys.length; i++) {
                    re = re.split(keys[i]).join('${' + i + '}');
                }
                // Replace abstracted values
                for (i = 0; i < keys.length; i++) {
                    re = re.split('${' + i + '}').join('(' + regExpMap[keys[i]] + ')');
                }
                format = escapeReservedSymbols(format);

                return new RegExp('^' + re + '$', ['i']);
            }

            $dateParserLabs.init();
            return $dateParserLabs;
        };

        return DateParserFactory;
    };
});
/**
 * Country codes, altered for Angular use.
 *
 * Original license:
 *
 * country-region-dropdowns
 * ------------------------
 * v0.1.2
 * @author Ben Keen
 * @url https://github.com/benkeen/country-region-dropdowns
 * @licence MIT
 */
'use strict';

angular.module('xl.widget').directive('xlWidgetCountry', ['xlWidgetUtils', 'COUNTRY_CODES', '$q', function (xlWidgetUtils, COUNTRY_CODES, $q) {
    return {
        restrict: 'EA',
        replace: true,
        templateUrl: 'src/xl-widget/xl-widget-country/xl-widget-country.html',
        scope: {
            ngModel: '=',
            options: '=',
            displayMode: '='
        },
        require: 'ngModel',
        controller: function controller($scope) {

            function autocompleteHandler(metadata, options) {
                var differed = $q.defer();
                var candidates = _.filter(_.map(COUNTRY_CODES, _.head), function (item) {
                    return item.toLowerCase().indexOf(options.term.toLowerCase()) >= 0;
                });
                differed.resolve(candidates);
                return differed.promise;
            }

            $scope.autocompleteHandlers = {
                addCandidates: autocompleteHandler
            };
        }
    };
}]).directive('xlWidgetCountryState', ['xlWidgetUtils', 'COUNTRY_CODES', '$q', function (xlWidgetUtils, COUNTRY_CODES, $q) {
    return {
        restrict: 'EA',
        replace: true,
        templateUrl: 'src/xl-widget/xl-widget-country/xl-widget-country-state.html',
        scope: {
            ngModel: '=',
            options: '=',
            displayMode: '=',
            country: '='
        },
        require: 'ngModel',
        controller: function controller($scope) {
            var selectedCountry;
            var selectedStates = [];
            $scope.$watch('country', function () {
                selectedCountry = _.filter(COUNTRY_CODES, function (item) {
                    return $scope.country && item[0] === $scope.country;
                });
                if (selectedCountry.length) {
                    selectedStates = selectedCountry[0][2].split('|');
                } else {
                    selectedStates = [];
                    $scope.ngModel = '';
                }
            });

            function autocompleteHandler(metadata, options) {
                var differed = $q.defer();
                var candidates = _.filter(selectedStates, function (item) {
                    return item.toLowerCase().indexOf(options.term.toLowerCase()) >= 0;
                });
                differed.resolve(candidates);
                return differed.promise;
            }

            $scope.autocompleteHandlers = {
                addCandidates: autocompleteHandler
            };
        }
    };
}]).value('COUNTRY_CODES', [['United States', 'US', 'Alabama|Alaska|Arizona|Arkansas|California|Colorado|Connecticut|Delaware|District of Columbia|Florida|Georgia|Hawaii|Idaho|Illinois|Indiana|Iowa|Kansas|Kentucky|Louisiana|Maine|Maryland|Massachusetts|Michigan|Minnesota|Mississippi|Missouri|Montana|Nebraska|Nevada|New Hampshire|New Jersey|New Mexico|New York|North Carolina|North Dakota|Ohio|Oklahoma|Oregon|Pennsylvania|Rhode Island|South Carolina|South Dakota|Tennessee|Texas|Utah|Vermont|Virginia|Washington|West Virginia|Wisconsin|Wyoming'], ['France', 'FR', 'Alsace|Aquitaine|Auvergne|Basse-Normandie|Bourgogne|Bretagne|Centre|Champagne-Ardenne|Corse|Franche-Comte|Haute-Normandie|Ile-de-France|Languedoc-Roussillon|Limousin|Lorraine|Midi-Pyrenees|Nord-Pas-de-Calais|Pays de la Loire|Picardie|Poitou-Charentes|Provence-Alpes-Cote d\'Azur|Rhone-Alpes'], ['United Kingdom', 'GB', 'Barking and Dagenham|Barnet|Barnsley|Bath and North East Somerset|Bedfordshire|Bexley|Birmingham|Blackburn with ackpool|Bolton|Bournemouth|Bracknell Forest|Bradford|Brent|Brighton and ley|Buckinghamshire|Bury|Calderdale|Cambridgeshire|Camden|Cheshire|City of Bristol|City of Kingston upon Hull|City of rnwall|Coventry|Croydon|Cumbria|Darlington|Derby|Derbyshire|Devon|Doncaster|Dorset|Dudley|Durham|Ealing|East Riding of Yorkshire|East field|Essex|Gateshead|Gloucestershire|Greenwich|Hackney|Halton|Hammersmith and mpshire|Haringey|Harrow|Hartlepool|Havering|Herefordshire|Hertfordshire|Hillingdon|Hounslow|Isle of Wight|Islington|Kensington and ent|Kingston upon Thames|Kirklees|Knowsley|Lambeth|Lancashire|Leeds|Leicester|Leicestershire|Lewisham|Lincolnshire|Liverpool|Luton|Manchester|Medway|Middlesbrough|Milton Keynes|Newcastle upon Tyne|Newham|Norfolk|North East Lincolnshire|North Lincolnshire|North Somerset|North Tyneside|North |Northamptonshire|Northumberland|Nottingham|Nottinghamshire|Oldham|Oxfordshire|Peterborough|Plymouth|Poole|Portsmouth|Reading|Redbridge|Redcar and |Richmond upon Thames|Rochdale|Rotherham|Rutland|Salford|Sandwell|Sefton|Sheffield|Shropshire|Slough|Solihull|Somerset|South Gloucestershire|South Southampton|Southend-on-Sea|Southwark|St. Helens|Staffordshire|Stockport|Stockton-on-Tees|Stoke-on-folk|Sunderland|Surrey|Sutton|Swindon|Tameside|Telford and Wrekin|Thurrock|Torbay|Tower Hamlets|Trafford|Wakefield|Walsall|Waltham ndsworth|Warrington|Warwickshire|West Berkshire|West Sussex|Westminster|Wigan|Wiltshire|Windsor and d|Wirral|Wokingham|Wolverhampton|Worcestershire|York,Aberdeen City|Aberdeenshire|Angus|Argyll and Bute|City of Edinburgh|Clackmannanshire|Dumfries and Galloway|Dundee City|East Ayrshire|East shire|East Lothian|East Renfrewshire|Eilean Siar (Western Isles)|Falkirk|Fife|Glasgow City|Highland|Inverclyde|Midlothian|Moray|North Ayrshire|North re|Orkney Islands|Perth and Kinross|Renfrewshire|Shetland Islands|South Ayrshire|South Lanarkshire|Stirling|The Scottish Borders|West shire|West Lothian'], ['Germany', 'DE', 'Baden-Wuerttemberg|Bayern|Berlin|Brandenburg|Bremen|Hamburg|Hessen|Mecklenburg-Vorpommern|Niedersachsen|Nordrhein-Westfalen|Rheinland-Pfalz|Saarland|Sachsen|Sachsen-Anhalt|Schleswig-Holstein|Thueringen'], ['Netherlands', 'NL', 'Drenthe|Flevoland|Friesland|Gelderland|Groningen|Limburg|Noord-Brabant|Noord-Holland|Overijssel|Utrecht|Zeeland|Zuid-Holland'], ['Norway', 'NO', 'Akershus|Aust-Agder|Buskerud|Finnmark|Hedmark|Hordaland|More og Romsdal|Nord-Trondelag|Nordland|Oppland|Oslo|Ostfold|Rogaland|Sogn og Fjordane|Sor-|Telemark|Troms|Vest-Agder|Vestfold'], ['Sweden', 'SE', 'Blekinge|Dalarnas|Gavleborgs|Gotlands|Hallands|Jamtlands|Jonkopings|Kalmar|Kronobergs|Norrbottens|Orebro|Ostergotlands|Skane|Sodermanlands|Stockholm|Varmlands|Vasterbottens|Vasternorrlands|Vastmanlands|Vastra Gotalands'], ['Finland', 'FI', 'Aland|Etela-Suomen Laani|Ita-Suomen Laani|Lansi-Suomen Laani|Lappi|Oulun Laani'], ['Afghanistan', 'AF', 'Badakhshan|Badghis|Baghlan|Balkh|Bamyan|Daykundi|Farah|Faryab|Ghazni|Ghor|Helmand|Herat|Jowzjan|Kabul|Kandahar|Kapisa|Khost|Kunar|Kunduz|Laghman|Logar|Nangarhar|Nimruz|Nurestan|Oruzgan|Paktia|Paktika|Panjshir|Parvan|Samangan|Sare Pol|Takhar|Wardak|Zabul'], ['Åland Islands', 'AX', 'Brändö|Eckerö|Finström|Föglö|Geta|Hammarland|Jomala|Kumlinge|Kökar|Lemland|Lumparland|Mariehamn|Saltvik|Sottunga|Sund|Vårdö'], ['Albania', 'AL', 'Berat|Dibër|Durrës|Elbasan|Fier|Gjirokastër|Korçë|Kukës|Lezhë|Shkodër|Tirana|Vlorë'], ['Algeria', 'DZ', 'Adrar|Ain Defla|Ain Timouchent|Algiers|Annaba|Batna|Béchar|Béjaïa|Biskra|Blida|Bordj Bou Arréridj|Bouïra|Boumerdes|Chlef|Constantine|Djelfa|El Bayadh|El Taref|El Oued|Illizi|Ghardaia|GuelmaJijel|Khenchela|Laghouat|Mascara|Médéa|Mila|Mostaganem|M\'Sila|Naâma|Oran|Ouargla|Oum El Bouaghi|Saïda|Sétif|Sidi Bel Abbès|Skikda|Souk Ahras|Tamanrasset|Tébessa|Tiaret|Tindouf|Tipaza|Tissemsilt|Tizi Ouzou|Tlemcen'], ['American Samoa', 'AS', 'Tutuila|Aunu\'u|Ta\'ū|Ofu‑Olosega|Rose Atoll|Swains Island'], ['Andorra', 'AD', 'Andorra la Vella|Canillo|Encamp|Escaldes-Engordany|La Massana|Ordino|Sant Julià de Lòria'], ['Angola', 'AO', 'Bengo|Benguela|Bié|Cabinda|Cuando Cubango|Cuanza Norte|Cuanza Sul|Cunene|Huambo|Huila|Luanda|Lunda Norte|Lunda Sul|Malanje|Moxico|Namibe|Uíge|Zaire'], ['Anguilla', 'AI', 'Blowing Point|East End|George Hill|Island Harbour|North Hill|North Side|Sandy Ground|Sandy Hill|South Hill|Stoney Ground|The Farrington|The Quarter|The Valley|West End'], ['Antarctica', 'AQ', 'East Antarctica|West Antarctica'], ['Antigua and Barbuda', 'AG', 'Antigua|Barbuda|Bird Island|Bishop Island|Blake Island|Cinnamon Island|Codrington Island|Crump Island|Dulcina Island|Exchange Island|Five Islands|Great Bird Island|Green Island|Guiana Island|Hale Gate Island|Hawes Island|Henry Island|Johnson Island|Kid Island|Laviscounts Islan|Lobster Island|Long Island|Maid Island|Moor Island|Nanny Island|Pelican Island|Prickly Pear Island|Rabbit Island|Rat Island|Red Head Island|Redonda|Sandy Island|Smith Island|The Sisters|Vernon Island|Wicked Will Island|York Island'], ['Argentina', 'AR', 'Autonomous City of Buenos Airesa|Buenos Aires|Catamarca|Chaco|Chubut|Córdoba|Corrientes|Entre Ríos|Formosa|Jujuy|La Pampa|La Rioja|Mendoza|Misiones|Neuquén|Río Negrob|Salta|San Juan|San Luis|Santa Cruz|Santa Fe|Santiago del Estero|Tierra del Fuego, Antártida e Islas del Atlántico Surc|Tucumán'], ['Armenia', 'AM', 'Aragatsotn|Ararat|Armavir|Gegharkunik|Kotayk|Lori|Shirak|Syunik|Tavush|Vayots Dzor|Yerevan'], ['Aruba', 'AW', 'Aruba'], ['Australia', 'AU', 'Australian Capital Territory|New South Wales|Northern Territory|Queensland|South Australia|Tasmania|Victoria|Western Australia'], ['Austria', 'AT', 'Burgenland|Kaernten|Niederoesterreich|Oberoesterreich|Salzburg|Steiermark|Tirol|Vorarlberg|Wien'], ['Azerbaijan', 'AZ', 'Abseron Rayonu|Agcabadi Rayonu|Agdam Rayonu|Agdas Rayonu|Agstafa Rayonu|Agsu Rayonu|Ali Bayramli Sahari|Astara Rayonu|Baki Sahari|Balakan Rayonu|Barda Rayonu|Beylaqan Rayonu|Bilasuvar Rayonu|Cabrayil Rayonu|Calilabad Rayonu|Daskasan Rayonu|Davaci Rayonu|Fuzuli Rayonu|Gadabay Rayonu|Ganca Sahari|Goranboy Rayonu|Goycay Rayonu|Haciqabul Rayonu|Imisli Rayonu|Ismayilli Rayonu|Kalbacar Rayonu|Kurdamir Rayonu|Lacin Rayonu|Lankaran Rayonu|Lankaran Sahari|Lerik Rayonu|Masalli Rayonu|Mingacevir Sahari|Naftalan Sahari|Naxcivan Muxtar Respublikasi|Neftcala Rayonu|Oguz Rayonu|Qabala Rayonu|Qax Rayonu|Qazax Rayonu|Qobustan Rayonu|Quba Rayonu|Qubadli Rayonu|Qusar Rayonu|Saatli Rayonu|Sabirabad Rayonu|Saki Rayonu|Saki Sahari|Salyan Rayonu|Samaxi Rayonu|Samkir Rayonu|Samux Rayonu|Siyazan Rayonu|Sumqayit Sahari|Susa Rayonu|Susa Sahari|Tartar Rayonu|Tovuz Rayonu|Ucar Rayonu|Xacmaz Rayonu|Xankandi Sahari|Xanlar Rayonu|Xizi Rayonu|Xocali Rayonu|Xocavand Rayonu|Yardimli Rayonu|Yevlax Rayonu|Yevlax Sahari|Zangilan Rayonu|Zaqatala Rayonu|Zardab Rayonu'], ['Bahamas', 'BS', 'Acklins and Crooked Islands|Bimini|Cat Island|Exuma|Freeport|Fresh Creek|Governor\'s Harbour|Green Turtle Cay|Harbour Island|High Rock|Inagua|Kemps Bay|Long Island|Marsh Harbour|Mayaguana|New Providence|Nicholls Town and Berry Islands|Ragged Island|Rock Sound|San Salvador and Rum Cay|Sandy Point'], ['Bahrain', 'BH', 'Al Hadd|Al Manamah|Al Mintaqah al Gharbiyah|Al Mintaqah al Wusta|Al Mintaqah ash Shamaliyah|Al Muharraq|Ar Rifa\' wa al Mintaqah al Janubiyah|Jidd Hafs|Juzur Hawar|Madinat \'Isa|Madinat Hamad|Sitrah'], ['Bangladesh', 'BD', 'Barguna|Barisal|Bhola|Jhalokati|Patuakhali|Pirojpur|Bandarban|Brahmanbaria|Chandpur|Chittagong|Comilla|Cox\'s Bazar|Feni|Khagrachari|Lakshmipur|Noakhali|Rangamati|Dhaka|Faridpur|Gazipur|Gopalganj|Jamalpur|Kishoreganj|Madaripur|Manikganj|Munshiganj|Mymensingh|Narayanganj|Narsingdi|Netrokona|Rajbari|Shariatpur|Sherpur|Tangail|Bagerhat|Chuadanga|Jessore|Jhenaidah|Khulna|Kushtia|Magura|Meherpur|Narail|Satkhira|Bogra|Dinajpur|Gaibandha|Jaipurhat|Kurigram|Lalmonirhat|Naogaon|Natore|Nawabganj|Nilphamari|Pabna|Panchagarh|Rajshahi|Rangpur|Sirajganj|Thakurgaon|Habiganj|Maulvi bazar|Sunamganj|Sylhet'], ['Barbados', 'BB', 'Bridgetown|Christ Church|Saint Andrew|Saint George|Saint James|Saint John|Saint Joseph|Saint Lucy|Saint Michael|Saint Peter|Saint Philip|Saint Thomas'], ['Belarus', 'BY', 'Brestskaya (Brest)|Homyel\'skaya (Homyel\')|Horad Minsk|Hrodzyenskaya (Hrodna)|Mahilyowskaya (Mahilyow)|Minskaya|Vitsyebskaya (Vitsyebsk)'], ['Belgium', 'BE', 'Antwerpen|Brabant Wallon|Brussels Capitol Region|Hainaut|Liege|Limburg|Luxembourg|Namur|Oost-Vlaanderen|Vlaams Brabant|West-Vlaanderen'], ['Belize', 'BZ', 'Belize|Cayo|Corozal|Orange Walk|Stann Creek|Toledo'], ['Benin', 'BJ', 'Alibori|Atakora|Atlantique|Borgou|Collines|Couffo|Donga|Littoral|Mono|Oueme|Plateau|Zou'], ['Bermuda', 'BM', 'Devonshire|Hamilton|Paget|Pembroke|Saint George|Saint Georges|Sandys|Smiths|Southampton|Warwick'], ['Bhutan', 'BT', 'Bumthang|Chhukha|Chirang|Daga|Geylegphug|Ha|Lhuntshi|Mongar|Paro|Pemagatsel|Punakha|Samchi|Samdrup Jongkhar|Shemgang|Tashigang|Thimphu|Tongsa|Wangdi Phodrang'], ['Bolivia', 'BO', 'Beni|Chuquisaca|Cochabamba|La Paz|Oruro|Pando|Potosi|Santa Cruz|Tarija'], ['Bonaire, Sint Eustatius and Saba', 'BQ', 'Bonaire|Sint Eustatius|Saba'], ['Bosnia and Herzegovina', 'BA', 'Federation of Bosnia and Herzegovina|Republika Srpska'], ['Botswana', 'BW', 'Central|Chobe|Francistown|Gaborone|Ghanzi|Kgalagadi|Kgatleng|Kweneng|Lobatse|Ngamiland|North-East|Selebi-Pikwe|South-East|Southern'], ['Bouvet Island', 'BV', 'Bouvet Island'], ['Brazil', 'BR', 'Acre|Alagoas|Amapa|Amazonas|Bahia|Ceara|Distrito Federal|Espirito Santo|Goias|Maranhao|Mato Grosso|Mato Grosso do Sul|Minas Gerais|Para|Paraiba|Parana|Pernambuco|Piaui|Rio de Janeiro|Rio Grande do Norte|Rio Grande do Sul|Rondonia|Roraima|Santa Catarina|Sao Paulo|Sergipe|Tocantins'], ['British Indian Ocean Territory', 'IO', 'British Indian Ocean Territory'], ['Brunei Darussalam', 'BN', 'Belait|Brunei and Muara|Temburong|Tutong'], ['Bulgaria', 'BG', 'Blagoevgrad|Burgas|Dobrich|Gabrovo|Khaskovo|Kurdzhali|Kyustendil|Lovech|Montana|Pazardzhik|Pernik|Pleven|Plovdiv|Razgrad|Ruse|Shumen|Silistra|Sliven|Smolyan|Sofiya|Sofiya-Grad|Stara Zagora|Turgovishte|Varna|Veliko Turnovo|Vidin|Vratsa|Yambol'], ['Burkina Faso', 'BF', 'Bale|Bam|Banwa|Bazega|Bougouriba|Boulgou|Boulkiemde|Comoe|Ganzourgou|Gnagna|Gourma|Houet|Ioba|Kadiogo|Kenedougou|Komandjari|Kompienga|Kossi|Koupelogo|Kouritenga|Kourweogo|Leraba|Loroum|Mouhoun|Nahouri|Namentenga|Naumbiel|Nayala|Oubritenga|Oudalan|Passore|Poni|Samentenga|Sanguie|Seno|Sissili|Soum|Sourou|Tapoa|Tuy|Yagha|Yatenga|Ziro|Zondomo|Zoundweogo'], ['Burundi', 'BI', 'Bubanza|Bujumbura|Bururi|Cankuzo|Cibitoke|Gitega|Karuzi|Kayanza|Kirundo|Makamba|Muramvya|Muyinga|Mwaro|Ngozi|Rutana|Ruyigi'], ['Cambodia', 'KH', 'Banteay Mean Cheay|Batdambang|Kampong Cham|Kampong Chhnang|Kampong Spoe|Kampong Thum|Kampot|Kandal|Kaoh Kong|Keb|Kracheh|Mondol Kiri|Otdar Mean Cheay|Pailin|Phnum Penh|Pouthisat|Preah Seihanu (Sihanoukville)|Preah Vihear|Prey Veng|Rotanah Kiri|Siem Reab|Stoeng Treng|Svay Rieng|Takev'], ['Cameroon', 'CM', 'Adamaoua|Centre|Est|Extreme-Nord|Littoral|Nord|Nord-Ouest|Ouest|Sud|Sud-Ouest'], ['Canada', 'CA', 'Alberta|British Columbia|Manitoba|New Brunswick|Newfoundland|Northwest Territories|Nova Scotia|Nunavut|Ontario|Prince Edward Island|Quebec|Saskatchewan|Yukon Territory'], ['Cape Verde', 'CV', 'Boa Vista|Brava|Maio|Mosteiros|Paul|Porto Novo|Praia|Ribeira Grande|Sal|Santa Catarina|Santa Cruz|Sao Domingos|Sao Filipe|Sao Nicolau|Sao Vicente|Tarrafal'], ['Cayman Islands', 'KY', 'Creek|Eastern|Midland|South Town|Spot Bay|Stake Bay|West End|Western'], ['Central African Republic', 'CF', 'Bamingui-Bangoran|Bangui|Basse-Kotto|Gribingui|Haut-Mbomou|Haute-Kotto|Haute-Sangha|Kemo-Gribingui|Lobaye|Mbomou|Nana-Mambere|Ombella-Mpoko|Ouaka|Ouham|Ouham-Pende|Sangha|Vakaga'], ['Chad', 'TD', 'Batha|Biltine|Borkou-Ennedi-Tibesti|Chari-Baguirmi|Guera|Kanem|Lac|Logone Occidental|Logone Oriental|Mayo-Kebbi|Moyen-Chari|Ouaddai|Salamat|Tandjile'], ['Chile', 'CL', 'Aisen del General Carlos Ibanez del Campo|Antofagasta|Araucania|Atacama|Bio-Bio|Coquimbo|Libertador General Bernardo O\'Higgins|Los Lagos|Magallanes y de la Antartica Chilena|Maule|Region Metropolitana (Santiago)|Tarapaca|Valparaiso'], ['China', 'CN', 'Anhui|Beijing|Chongqing|Fujian|Gansu|Guangdong|Guangxi|Guizhou|Hainan|Hebei|Heilongjiang|Henan|Hubei|Hunan|Jiangsu|Jiangxi|Jilin|Liaoning|Nei Mongol|Ningxia|Qinghai|Shaanxi|Shandong|Shanghai|Shanxi|Sichuan|Tianjin|Xinjiang|Xizang (Tibet)|Yunnan|Zhejiang'], ['Christmas Island', 'CX', 'Christmas Island'], ['Cocos (Keeling) Islands', 'CC', 'Direction Island|Home Island|Horsburgh Island|North Keeling Island|South Island|West Island'], ['Colombia', 'CO', 'Amazonas|Antioquia|Arauca|Atlantico|Bolivar|Boyaca|Caldas|Caqueta|Casanare|Cauca|Cesar|Choco|Cordoba|Cundinamarca|Distrito Capital de Santa Fe de Bogota|Guainia|Guaviare|Huila|La Guajira|Magdalena|Meta|Narino|Norte de Santander|Putumayo|Quindio|Risaralda|San Andres y Providencia|Santander|Sucre|Tolima|Valle del Cauca|Vaupes|Vichada'], ['Comoros', 'KM', 'Anjouan (Nzwani)|Domoni|Fomboni|Grande Comore (Njazidja)|Moheli (Mwali)|Moroni|Moutsamoudou'], ['Congo', 'CG', 'Bouenza|Brazzaville|Cuvette|Kouilou|Lekoumou|Likouala|Niari|Plateaux|Pool|Sangha'], ['Congo, the Democratic Republic of the', 'CD', 'Bandundu|Bas-Congo|Équateur|Kasai-Occidental|Kasai-Oriental|Katanga|Kinshasa (city-province)|Maniema|North Kivu|Orientale|South Kivu'], ['Cook Islands', 'CK', 'Aitutaki|Atiu|Avarua|Mangaia|Manihiki|Manuae|Mauke|Mitiaro|Nassau Island|Palmerston|Penrhyn|Pukapuka|Rakahanga|Rarotonga|Suwarrow|Takutea'], ['Costa Rica', 'CR', 'Alajuela|Cartago|Guanacaste|Heredia|Limon|Puntarenas|San Jose'], ['Cote d\'Ivoire', 'CI', 'Abengourou|Abidjan|Aboisso|Adiake\'|Adzope|Agboville|Agnibilekrou|Ale\'pe\'|Bangolo|Beoumi|Biankouma|Bocanda|Bondoukou|Bongouanou|Bouafle|Bouake|Bouna|Boundiali|Dabakala|Dabon|Daloa|Danane|Daoukro|Dimbokro|Divo|Duekoue|Ferkessedougou|Gagnoa|Grand Bassam|Grand-Lahou|Guiglo|Issia|Jacqueville|Katiola|Korhogo|Lakota|Man|Mankono|Mbahiakro|Odienne|Oume|Sakassou|San-Pedro|Sassandra|Seguela|Sinfra|Soubre|Tabou|Tanda|Tiassale|Tiebissou|Tingrela|Touba|Toulepleu|Toumodi|Vavoua|Yamoussoukro|Zuenoula'], ['Croatia', 'HR', 'Bjelovarsko-Bilogorska Zupanija|Brodsko-Posavska Zupanija|Dubrovacko-Neretvanska Zupanija|Istarska Zupanija|Karlovacka Zupanija|Koprivnicko-Krizevacka Zupanija|Krapinsko-Zagorska Zupanija|Licko-Senjska Zupanija|Medimurska Zupanija|Osjecko-Baranjska Zupanija|Pozesko-Slavonska Zupanija|Primorsko-Goranska Zupanija|Sibensko-Kninska Zupanija|Sisacko-Moslavacka Zupanija|Splitsko-Dalmatinska Zupanija|Varazdinska Zupanija|Viroviticko-Podravska Zupanija|Vukovarsko-Srijemska Zupanija|Zadarska Zupanija|Zagreb|Zagrebacka Zupanija'], ['Cuba', 'CU', 'Camaguey|Ciego de Avila|Cienfuegos|Ciudad de La Habana|Granma|Guantanamo|Holguin|Isla de la Juventud|La Habana|Las Tunas|Matanzas|Pinar del Rio|Sancti Spiritus|Santiago de Cuba|Villa Clara'], ['Curaçao', 'CW', 'Curaçao'], ['Cyprus', 'CY', 'Famagusta|Kyrenia|Larnaca|Limassol|Nicosia|Paphos'], ['Czech Republic', 'CZ', 'Brnensky|Budejovicky|Jihlavsky|Karlovarsky|Kralovehradecky|Liberecky|Olomoucky|Ostravsky|Pardubicky|Plzensky|Praha|Stredocesky|Ustecky|Zlinsky'], ['Denmark', 'DK', 'Arhus|Bornholm|Fredericksberg|Frederiksborg|Fyn|Kobenhavn|Kobenhavns|Nordjylland|Ribe|Ringkobing|Roskilde|Sonderjylland|Storstrom|Vejle|Vestsjalland|Viborg'], ['Djibouti', 'DJ', '\'Ali Sabih|Dikhil|Djibouti|Obock|Tadjoura'], ['Dominica', 'DM', 'Saint Andrew|Saint David|Saint George|Saint John|Saint Joseph|Saint Luke|Saint Mark|Saint Patrick|Saint Paul|Saint Peter'], ['Dominican Republic', 'DO', 'Azua|Baoruco|Barahona|Dajabon|Distrito Nacional|Duarte|El Seibo|Elias Pina|Espaillat|Hato Mayor|Independencia|La Altagracia|La Romana|La Vega|Maria Trinidad Sanchez|Monsenor Nouel|Monte Cristi|Monte Plata|Pedernales|Peravia|Puerto Plata|Salcedo|Samana|San Cristobal|San Juan|San Pedro de Macoris|Sanchez Ramirez|Santiago|Santiago Rodriguez|Valverde'], ['Ecuador', 'EC', 'Azuay|Bolivar|Canar|Carchi|Chimborazo|Cotopaxi|El Oro|Esmeraldas|Galapagos|Guayas|Imbabura|Loja|Los Rios|Manabi|Morona-Santiago|Napo|Orellana|Pastaza|Pichincha|Sucumbios|Tungurahua|Zamora-Chinchipe'], ['Egypt', 'EG', 'Ad Daqahliyah|Al Bahr al Ahmar|Al Buhayrah|Al Fayyum|Al Gharbiyah|Al Iskandariyah|Al Isma\'iliyah|Al Jizah|Al Minufiyah|Al Minya|Al Qahirah|Al Qalyubiyah|Al Wadi al Jadid|As Suways|Ash Sharqiyah|Aswan|Asyut|Bani Suwayf|Bur Sa\'id|Dumyat|Janub Sina\'|Kafr ash Shaykh|Matruh|Qina|Shamal Sina\'|Suhaj'], ['El Salvador', 'SV', 'Ahuachapan|Cabanas|Chalatenango|Cuscatlan|La Libertad|La Paz|La Union|Morazan|San Miguel|San Salvador|San Vicente|Santa Ana|Sonsonate|Usulutan'], ['Equatorial Guinea', 'GQ', 'Annobon|Bioko Norte|Bioko Sur|Centro Sur|Kie-Ntem|Litoral|Wele-Nzas'], ['Eritrea', 'ER', 'Akale Guzay|Barka|Denkel|Hamasen|Sahil|Semhar|Senhit|Seraye'], ['Estonia', 'EE', 'Harjumaa (Tallinn)|Hiiumaa (Kardla)|Ida-Virumaa (Johvi)|Jarvamaa (Paide)|Jogevamaa (Jogeva)|Laane-Virumaa (Rakvere)|Laanemaa (Haapsalu)|Parnumaa (Parnu)|Polvamaa (Polva)|Raplamaa (Rapla)|Saaremaa (Kuessaare)|Tartumaa (Tartu)|Valgamaa (Valga)|Viljandimaa (Viljandi)|Vorumaa (Voru)'], ['Ethiopia', 'ET', 'Adis Abeba (Addis Ababa)|Afar|Amara|Dire Dawa|Gambela Hizboch|Hareri Hizb|Oromiya|Sumale|Tigray|YeDebub Biheroch Bihereseboch na Hizboch'], ['Falkland Islands (Islas Malvinas)', 'FK', 'Falkland Islands (Islas Malvinas)'], ['Faroe Islands', 'FO', 'Bordoy|Eysturoy|Mykines|Sandoy|Skuvoy|Streymoy|Suduroy|Tvoroyri|Vagar'], ['Fiji', 'FJ', 'Central|Eastern|Northern|Rotuma|Western'], ['French Guiana', 'GF', 'French Guiana'], ['French Polynesia', 'PF', 'Archipel des Marquises|Archipel des Tuamotu|Archipel des Tubuai|Iles du Vent|Iles Sous-le-Vent'], ['French Southern and Antarctic Lands', 'TF', 'Adelie Land|Ile Crozet|Iles Kerguelen|Iles Saint-Paul et Amsterdam'], ['Gabon', 'GA', 'Estuaire|Haut-Ogooue|Moyen-Ogooue|Ngounie|Nyanga|Ogooue-Ivindo|Ogooue-Lolo|Ogooue-Maritime|Woleu-Ntem'], ['Gambia, The', 'GM', 'Banjul|Central River|Lower River|North Bank|Upper River|Western'], ['Georgia', 'GE', 'Abashis|Abkhazia or Ap\'khazet\'is Avtonomiuri Respublika (Sokhumi)|Adigenis|Ajaria or Acharis Avtonomiuri Respublika (Bat\'umi)|Akhalgoris|Akhalk\'alak\'is|Akhalts\'ikhis|Akhmetis|Ambrolauris|Aspindzis|Baghdat\'is|Bolnisis|Borjomis|Ch\'khorotsqus|Ch\'okhatauris|Chiat\'ura|Dedop\'listsqaros|Dmanisis|Dushet\'is|Gardabanis|Gori|Goris|Gurjaanis|Javis|K\'arelis|K\'ut\'aisi|Kaspis|Kharagaulis|Khashuris|Khobis|Khonis|Lagodekhis|Lanch\'khut\'is|Lentekhis|Marneulis|Martvilis|Mestiis|Mts\'khet\'is|Ninotsmindis|Onis|Ozurget\'is|P\'ot\'i|Qazbegis|Qvarlis|Rust\'avi|Sach\'kheris|Sagarejos|Samtrediis|Senakis|Sighnaghis|T\'bilisi|T\'elavis|T\'erjolis|T\'et\'ritsqaros|T\'ianet\'is|Tqibuli|Ts\'ageris|Tsalenjikhis|Tsalkis|Tsqaltubo|Vanis|Zestap\'onis|Zugdidi|Zugdidis'], ['Ghana', 'GH', 'Ashanti|Brong-Ahafo|Central|Eastern|Greater Accra|Northern|Upper East|Upper West|Volta|Western'], ['Gibraltar', 'GI', 'Gibraltar'], ['Greece', 'GR', 'Aitolia kai Akarnania|Akhaia|Argolis|Arkadhia|Arta|Attiki|Ayion Oros (Mt. Athos)|Dhodhekanisos|Drama|Evritania|Evros|Evvoia|Florina|Fokis|Fthiotis|Grevena|Ilia|Imathia|Ioannina|Irakleion|Kardhitsa|Kastoria|Kavala|Kefallinia|Kerkyra|Khalkidhiki|Khania|Khios|Kikladhes|Kilkis|Korinthia|Kozani|Lakonia|Larisa|Lasithi|Lesvos|Levkas|Magnisia|Messinia|Pella|Pieria|Preveza|Rethimni|Rodhopi|Samos|Serrai|Thesprotia|Thessaloniki|Trikala|Voiotia|Xanthi|Zakinthos'], ['Greenland', 'GL', 'Avannaa (Nordgronland)|Kitaa (Vestgronland)|Tunu (Ostgronland)'], ['Grenada', 'GD', 'Carriacou and Petit Martinique|Saint Andrew|Saint David|Saint George|Saint John|Saint Mark|Saint Patrick'], ['Guadeloupe', 'GP', 'Basse-Terre|Grande-Terre|Iles de la Petite Terre|Iles des Saintes|Marie-Galante'], ['Guam', 'GU', 'Guam'], ['Guatemala', 'GT', 'Alta Verapaz|Baja Verapaz|Chimaltenango|Chiquimula|El Progreso|Escuintla|Guatemala|Huehuetenango|Izabal|Jalapa|Jutiapa|Peten|Quetzaltenango|Quiche|Retalhuleu|Sacatepequez|San Marcos|Santa Rosa|Solola|Suchitepequez|Totonicapan|Zacapa'], ['Guernsey', 'GG', 'Castel|Forest|St. Andrew|St. Martin|St. Peter Port|St. Pierre du Bois|St. Sampson|St. Saviour|Torteval|Vale'], ['Guinea', 'GN', 'Beyla|Boffa|Boke|Conakry|Coyah|Dabola|Dalaba|Dinguiraye|Dubreka|Faranah|Forecariah|Fria|Gaoual|Gueckedou|Kankan|Kerouane|Kindia|Kissidougou|Koubia|Koundara|Kouroussa|Labe|Lelouma|Lola|Macenta|Mali|Mamou|Mandiana|Nzerekore|Pita|Siguiri|Telimele|Tougue|Yomou'], ['Guinea-Bissau', 'GW', 'Bafata|Biombo|Bissau|Bolama-Bijagos|Cacheu|Gabu|Oio|Quinara|Tombali'], ['Guyana', 'GY', 'Barima-Waini|Cuyuni-Mazaruni|Demerara-Mahaica|East Berbice-Corentyne|Essequibo Islands-West Demerara|Mahaica-Berbice|Pomeroon-Supenaam|Potaro-Siparuni|Upper Demerara-Berbice|Upper Takutu-Upper Essequibo'], ['Haiti', 'HT', 'Artibonite|Centre|Grand\'Anse|Nord|Nord-Est|Nord-Ouest|Ouest|Sud|Sud-Est'], ['Heard Island and McDonald Islands', 'HM', 'Heard Island and McDonald Islands'], ['Holy See (Vatican City)', 'VA', 'Holy See (Vatican City)'], ['Honduras', 'HN', 'Atlantida|Choluteca|Colon|Comayagua|Copan|Cortes|El Paraiso|Francisco Morazan|Gracias a Dios|Intibuca|Islas de la Bahia|La Paz|Lempira|Ocotepeque|Olancho|Santa Barbara|Valle|Yoro'], ['Hong Kong', 'HK', 'Hong Kong'], ['Hungary', 'HU', 'Bacs-Kiskun|Baranya|Bekes|Bekescsaba|Borsod-Abauj-Zemplen|Budapest|Csongrad|Debrecen|Dunaujvaros|Eger|Fejer|Gyor|Gyor-Moson-Sopron|Hajdu-Bihar|Heves|Hodmezovasarhely|Jasz-Nagykun-Szolnok|Kaposvar|Kecskemet|Komarom-Esztergom|Miskolc|Nagykanizsa|Nograd|Nyiregyhaza|Pecs|Pest|Somogy|Sopron|Szabolcs-Szatmar-Bereg|Szeged|Szekesfehervar|Szolnok|Szombathely|Tatabanya|Tolna|Vas|Veszprem|Veszprem|Zala|Zalaegerszeg'], ['Iceland', 'IS', 'Akranes|Akureyri|Arnessysla|Austur-Bardhastrandarsysla|Austur-Hunavatnssysla|Austur-Skaftafellssysla|Borgarfjardharsysla|Dalasysla|Eyjafjardharsysla|Gullbringusysla|Hafnarfjordhur|Husavik|Isafjordhur|Keflavik|Kjosarsysla|Kopavogur|Myrasysla|Neskaupstadhur|Nordhur-Isafjardharsysla|Nordhur-Mulasys-la|Nordhur-Thingeyjarsysla|Olafsfjordhur|Rangarvallasysla|Reykjavik|Saudharkrokur|Seydhisfjordhur|Siglufjordhur|Skagafjardharsysla|Snaefellsnes-og Hnappadalssysla|Strandasysla|Sudhur-Mulasysla|Sudhur-Thingeyjarsysla|Vesttmannaeyjar|Vestur-Bardhastrandarsysla|Vestur-Hunavatnssysla|Vestur-Isafjardharsysla|Vestur-Skaftafellssysla'], ['India', 'IN', 'Andaman and Nicobar Islands|Andhra Pradesh|Arunachal Pradesh|Assam|Bihar|Chandigarh|Chhattisgarh|Dadra and Nagar Haveli|Daman and Diu|Delhi|Goa|Gujarat|Haryana|Himachal Pradesh|Jammu and Kashmir|Jharkhand|Karnataka|Kerala|Lakshadweep|Madhya Pradesh|Maharashtra|Manipur|Meghalaya|Mizoram|Nagaland|Orissa|Pondicherry|Punjab|Rajasthan|Sikkim|Tamil Nadu|Tripura|Uttar Pradesh|Uttaranchal|West Bengal'], ['Indonesia', 'ID', 'Aceh|Bali|Banten|Bengkulu|East Timor|Gorontalo|Irian Jaya|Jakarta Raya|Jambi|Jawa Barat|Jawa Tengah|Jawa Timur|Kalimantan Barat|Kalimantan Selatan|Kalimantan Tengah|Kalimantan Timur|Kepulauan Bangka Belitung|Lampung|Maluku|Maluku Utara|Nusa Tenggara Barat|Nusa Tenggara Timur|Riau|Sulawesi Selatan|Sulawesi Tengah|Sulawesi Tenggara|Sulawesi Utara|Sumatera Barat|Sumatera Selatan|Sumatera Utara|Yogyakarta'], ['Iran, Islamic Republic of', 'IR', 'Ardabil|Azarbayjan-e Gharbi|Azarbayjan-e Sharqi|Bushehr|Chahar Mahall va Bakhtiari|Esfahan|Fars|Gilan|Golestan|Hamadan|Hormozgan|Ilam|Kerman|Kermanshah|Khorasan|Khuzestan|Kohgiluyeh va Buyer Ahmad|Kordestan|Lorestan|Markazi|Mazandaran|Qazvin|Qom|Semnan|Sistan va Baluchestan|Tehran|Yazd|Zanjan'], ['Iraq', 'IQ', 'Al Anbar|Al Basrah|Al Muthanna|Al Qadisiyah|An Najaf|Arbil|As Sulaymaniyah|At Ta\'mim|Babil|Baghdad|Dahuk|Dhi Qar|Diyala|Karbala\'|Maysan|Ninawa|Salah ad Din|Wasit'], ['Ireland', 'IE', 'Carlow|Cavan|Clare|Cork|Donegal|Dublin|Galway|Kerry|Kildare|Kilkenny|Laois|Leitrim|Limerick|Longford|Louth|Mayo|Meath|Monaghan|Offaly|Roscommon|Sligo|Tipperary|Waterford|Westmeath|Wexford|Wicklow'], ['Isle of Man', 'IM', 'Isle of Man'], ['Israel', 'IL', 'Central|Haifa|Jerusalem|Northern|Southern|Tel Aviv'], ['Italy', 'IT', 'Abruzzo|Basilicata|Calabria|Campania|Emilia-Romagna|Friuli-Venezia Giulia|Lazio|Liguria|Lombardia|Marche|Molise|Piemonte|Puglia|Sardegna|Sicilia|Toscana|Trentino-Alto Adige|Umbria|Valle d\'Aosta|Veneto'], ['Jamaica', 'JM', 'Clarendon|Hanover|Kingston|Manchester|Portland|Saint Andrew|Saint Ann|Saint Catherine|Saint Elizabeth|Saint James|Saint Mary|Saint Thomas|Trelawny|Westmoreland'], ['Japan', 'JP', 'Aichi|Akita|Aomori|Chiba|Ehime|Fukui|Fukuoka|Fukushima|Gifu|Gumma|Hiroshima|Hokkaido|Hyogo|Ibaraki|Ishikawa|Iwate|Kagawa|Kagoshima|Kanagawa|Kochi|Kumamoto|Kyoto|Mie|Miyagi|Miyazaki|Nagano|Nagasaki|Nara|Niigata|Oita|Okayama|Okinawa|Osaka|Saga|Saitama|Shiga|Shimane|Shizuoka|Tochigi|Tokushima|Tokyo|Tottori|Toyama|Wakayama|Yamagata|Yamaguchi|Yamanashi'], ['Jersey', 'JE', 'Jersey'], ['Jordan', 'JO', '\'Amman|Ajlun|Al \'Aqabah|Al Balqa\'|Al Karak|Al Mafraq|At Tafilah|Az Zarqa\'|Irbid|Jarash|Ma\'an|Madaba'], ['Kazakhstan', 'KZ', 'Almaty|Aqmola|Aqtobe|Astana|Atyrau|Batys Qazaqstan|Bayqongyr|Mangghystau|Ongtustik Qazaqstan|Pavlodar|Qaraghandy|Qostanay|Qyzylorda|Shyghys Qazaqstan|Soltustik Qazaqstan|Zhambyl'], ['Kenya', 'KE', 'Central|Coast|Eastern|Nairobi Area|North Eastern|Nyanza|Rift Valley|Western'], ['Kiribati', 'KI', 'Abaiang|Abemama|Aranuka|Arorae|Banaba|Banaba|Beru|Butaritari|Central Gilberts|Gilbert Islands|Kanton|Kiritimati|Kuria|Line Islands|Line Islands|Maiana|Makin|Marakei|Nikunau|Nonouti|Northern Gilberts|Onotoa|Phoenix Islands|Southern Gilberts|Tabiteuea|Tabuaeran|Tamana|Tarawa|Tarawa|Teraina'], ['Korea, Democratic People\'s Republic of', 'KP', 'Chagang-do (Chagang Province)|Hamgyong-bukto (North Hamgyong Province)|Hamgyong-namdo (South Hamgyong Province)|Hwanghae-bukto (North Hwanghae Province)|Hwanghae-namdo (South Hwanghae Province)|Kaesong-si (Kaesong City)|Kangwon-do (Kangwon Province)|Namp\'o-si (Namp\'o City)|P\'yongan-bukto (North P\'yongan Province)|P\'yongan-namdo (South P\'yongan Province)|P\'yongyang-si (P\'yongyang City)|Yanggang-do (Yanggang Province)'], ['Korea, Republic of', 'KR', 'Ch\'ungch\'ong-bukto|Ch\'ungch\'ong-namdo|Cheju-do|Cholla-bukto|Cholla-namdo|Inch\'on-gwangyoksi|Kangwon-do|Kwangju-gwangyoksi|Kyonggi-do|Kyongsang-bukto|Kyongsang-namdo|Pusan-gwangyoksi|Soul-t\'ukpyolsi|Taegu-gwangyoksi|Taejon-gwangyoksi|Ulsan-gwangyoksi'], ['Kuwait', 'KW', 'Al \'Asimah|Al Ahmadi|Al Farwaniyah|Al Jahra\'|Hawalli'], ['Kyrgyzstan', 'KG', 'Batken Oblasty|Bishkek Shaary|Chuy Oblasty (Bishkek)|Jalal-Abad Oblasty|Naryn Oblasty|Osh Oblasty|Talas Oblasty|Ysyk-Kol Oblasty (Karakol)'], ['Lao People\'s Democratic Republic', 'LA', ''], ['Latvia', 'LV', 'Aizkraukles Rajons|Aluksnes Rajons|Balvu Rajons|Bauskas Rajons|Cesu Rajons|Daugavpils|Daugavpils Rajons|Dobeles Rajons|Gulbenes Rajons|Jekabpils Rajons|Jelgava|Jelgavas Rajons|Jurmala|Kraslavas Rajons|Kuldigas Rajons|Leipaja|Liepajas Rajons|Limbazu Rajons|Ludzas Rajons|Madonas Rajons|Ogres Rajons|Preilu Rajons|Rezekne|Rezeknes Rajons|Riga|Rigas Rajons|Saldus Rajons|Talsu Rajons|Tukuma Rajons|Valkas Rajons|Valmieras Rajons|Ventspils|Ventspils Rajons'], ['Lebanon', 'LB', 'Beyrouth|Ech Chimal|Ej Jnoub|El Bekaa|Jabal Loubnane'], ['Lesotho', 'LS', 'Berea|Butha-Buthe|Leribe|Mafeteng|Maseru|Mohales Hoek|Mokhotlong|Qacha\'s Nek|Quthing|Thaba-Tseka'], ['Liberia', 'LR', 'Bomi|Bong|Grand Bassa|Grand Cape Mount|Grand Gedeh|Grand Kru|Lofa|Margibi|Maryland|Montserrado|Nimba|River Cess|Sinoe'], ['Libya', 'LY', 'Ajdabiya|Al \'Aziziyah|Al Fatih|Al Jabal al Akhdar|Al Jufrah|Al Khums|Al Kufrah|An Nuqat al Khams|Ash Shati\'|Awbari|Az Zawiyah|Banghazi|Darnah|Ghadamis|Gharyan|Misratah|Murzuq|Sabha|Sawfajjin|Surt|Tarabulus|Tarhunah|Tubruq|Yafran|Zlitan'], ['Liechtenstein', 'LI', 'Balzers|Eschen|Gamprin|Mauren|Planken|Ruggell|Schaan|Schellenberg|Triesen|Triesenberg|Vaduz'], ['Lithuania', 'LT', 'Akmenes Rajonas|Alytaus Rajonas|Alytus|Anyksciu Rajonas|Birstonas|Birzu Rajonas|Druskininkai|Ignalinos Rajonas|Jonavos Rajonas|Joniskio Rajonas|Jurbarko Rajonas|Kaisiadoriu Rajonas|Kaunas|Kauno Rajonas|Kedainiu Rajonas|Kelmes Rajonas|Klaipeda|Klaipedos Rajonas|Kretingos Rajonas|Kupiskio Rajonas|Lazdiju Rajonas|Marijampole|Marijampoles Rajonas|Mazeikiu Rajonas|Moletu Rajonas|Neringa Pakruojo Rajonas|Palanga|Panevezio Rajonas|Panevezys|Pasvalio Rajonas|Plunges Rajonas|Prienu Rajonas|Radviliskio Rajonas|Raseiniu Rajonas|Rokiskio Rajonas|Sakiu Rajonas|Salcininku Rajonas|Siauliai|Siauliu Rajonas|Silales Rajonas|Silutes Rajonas|Sirvintu Rajonas|Skuodo Rajonas|Svencioniu Rajonas|Taurages Rajonas|Telsiu Rajonas|Traku Rajonas|Ukmerges Rajonas|Utenos Rajonas|Varenos Rajonas|Vilkaviskio Rajonas|Vilniaus Rajonas|Vilnius|Zarasu Rajonas'], ['Luxembourg', 'LU', 'Diekirch|Grevenmacher|Luxembourg'], ['Macao', 'MO', 'Macao'], ['Macedonia, Former Yugoslav Republic of', 'MK', 'Aracinovo|Bac|Belcista|Berovo|Bistrica|Bitola|Blatec|Bogdanci|Bogomila|Bogovinje|Bosilovo|Brvenica|Cair (Skopje)|Capari|Caska|Cegrane|Centar (SkopjeZupa|Cesinovo|Cucer-Sandevo|Debar|Delcevo|Delogozdi|Demir Hisar|Demir Kapija|Dobrusevo|Dolna Banjica|Dolneni|Dorce Petrov (Skopje)Dzepciste|Gazi Baba (Skopje)|Gevgelija|Gostivar|Gradsko|Ilinden|Izvor|Jegunovce|Kamenjane|Karbinci|Karpos (Skopje)|Kavadarci|Kicevo|Kisela Voda (lecevce|Kocani|Konce|Kondovo|Konopiste|Kosel|Kratovo|Kriva rivogastani|Krusevo|Kuklis|Kukurecani|Kumanovo|Labunista|Lipkovo|Lozovo|Lukovo|Makedonska Kamenica|Makedonski Brod|Mavrovi eista|Miravci|Mogila|Murtino|Negotino|Negotino-Poloska|Novaci|Novo Selo|Oblesevo|Ohrid|Orasac|Orizari|Oslomej|Pehcevo|Petrovec|Plasnia|Podares|Prilepp|Radovis|Rankovce|Resen|Rosoman|Rostusa|Samokov|Saraj|Sipkovica|Sopiste|Sopotnika|Srbinovo|Star Dojran|Staravina|Staro e|Stip|Struga|Strumica|Studenicani|Suto Orizari (Skopje)|Sveti Nikole|Tearce|Tetovo|Topolcani|Valandovo|Vasilevo|Veles|Velesta|Vevcani|Vinica|Vitolistica|Vrapciste|Vratnica|Vrutok|Zajas|Zelenikovo|Zileno|Zitose|Zletovo|Zrnovci'], ['Madagascar', 'MG', 'Antananarivo|Antsiranana|Fianarantsoa|Mahajanga|Toamasina|Toliara'], ['Malawi', 'MW', 'Balaka|Blantyre|Chikwawa|Chiradzulu|Chitipa|Dedza|Dowa|Karonga|Kasungu|Likoma|Lilongwe|Machinga (Kasupe)|Mchinji|Mulanje|Mwanza|Mzimba|Nkhata Bay|Nkhotakota|Nsanje|Ntcheu|Ntchisi|Phalombe|Rumphi|Salima|Thyolo|Zomba'], ['Malaysia', 'MY', 'Johor|Kedah|Kelantan|Labuan|Melaka|Negeri Sembilan|Pahang|Perak|Perlis|Pulau Pinang|Sabah|Sarawak|Selangor|Terengganu|Wilayah Persekutuan'], ['Maldives', 'MV', 'Alifu|Baa|Dhaalu|Faafu|Gaafu Alifu|Gaafu Dhaalu|Gnaviyani|Haa Alifu|Haa afu|Laamu|Lhaviyani|Maale|Meemu|Noonu|Raa|Seenu|Shaviyani|Thaa|Vaavu'], ['Mali', 'ML', 'Gao|Kayes|Kidal|Koulikoro|Mopti|Segou|Sikasso|Tombouctou'], ['Malta', 'MT', 'Valletta'], ['Marshall Islands', 'MH', 'Ailinginae|Ailinglaplap|Ailuk|Arno|Aur|Bikar|Bikini|Bokak|Ebon|Enewetak|Erikub|Jabat|Jaluit|Jemo|Kili|Kwajalein|Lae|Lib|Likiep|Majuro|Maloelap|Mejitorik|Namu|Rongelap|Rongrik|Toke|Ujae|Ujelang|Utirik|Wotho|Wotje'], ['Martinique', 'MQ', 'Martinique'], ['Mauritania', 'MR', 'Adrar|Assaba|Brakna|Dakhlet Nouadhibou|Gorgol|Guidimaka|Hodh Ech Chargui|Hodh El Gharbi|Inchiri|Nouakchott|Tagant|Tiris Zemmour|Trarza'], ['Mauritius', 'MU', 'Agalega Islands|Black River|Cargados Carajos Shoals|Flacq|Grand Port|Moka|Pamplemousses|Plaines Wilhems|Port Louis|Riviere du odrigues|Savanne'], ['Mayotte', 'YT', 'Mayotte'], ['Mexico', 'MX', 'Aguascalientes|Baja California|Baja California Sur|Campeche|Chiapas|Chihuahua|Coahuila de Zaragoza|Colima|Distrito urango|Guanajuato|Guerrero|Hidalgo|Jalisco|Mexico|Michoacan de Ocampo|Morelos|Nayarit|Nuevo Leon|Oaxaca|Puebla|Queretaro de Arteaga|Quintana Roo|San si|Sinaloa|Sonora|Tabasco|Tamaulipas|Tlaxcala|Veracruz-Llave|Yucatan|Zacatecas'], ['Micronesia, Federated States of', 'FM', 'Chuuk (Truk)|Kosrae|Pohnpei|Yap'], ['Moldova', 'MD', 'Balti|Cahul|Chisinau|Chisinau|Dubasari|Edinet|Gagauzia|Lapusna|Orhei|Soroca|Tighina|Ungheni'], ['Monaco', 'MC', 'Fontvieille|La Condamine|Monaco-Ville|Monte-Carlo'], ['Mongolia', 'MN', 'Arhangay|Bayan-Olgiy|Bayanhongor|Bulgan|Darhan|Dornod|Dornogovi|Dundgovi|Dzavhan|Erdenet|Govi-tiy|Hovd|Hovsgol|Omnogovi|Ovorhangay|Selenge|Suhbaatar|Tov|Ulaanbaatar|Uvs'], ['Montenegro', 'ME', 'Andrijevica|Bar|Berane|Bijelo Polje|Budva|Cetinje|Danilovgrad|Herceg Novi|Kolašin|Kotor|Mojkovac|Nikšić|Plav|Plužine|Pljevlja|Podgorica|Golubovci|Tuzi|Rožaje|Šavnik|Tivat|Ulcinj|Žablja'], ['Montserrat', 'MS', 'Saint Anthony|Saint Georges|Saint Peter\'s'], ['Morocco', 'MA', 'Agadir|Al Hoceima|Azilal|Ben Slimane|Beni Mellal|Boulemane|Casablanca|Chaouen|El Jadida|El Kelaa des Srarhna|Er Essaouira|Fes|Figuig|Guelmim|Ifrane|Kenitra|Khemisset|Khenifra|Khouribga|Laayoune|Larache|Marrakech|Meknes|Nador|Ouarzazate|Oujda|Rabat-|Settat|Sidi Kacem|Tan-Tan|Tanger|Taounate|Taroudannt|Tata|Taza|Tetouan|Tiznit'], ['Mozambique', 'MZ', 'Cabo Delgado|Gaza|Inhambane|Manica|Maputo|Nampula|Niassa|Sofala|Tete|Zambezia'], ['Myanmar', 'MM', 'Ayeyarwady Region|Bago Region|Chin State|Kachin State|Kayah State|Kayin State|Magway Region|Mandalay Region|Mon State|Rakhine State|Shan State|Sagaing Region|Tanintharyi Region|Yangon Region|Naypyidaw Union Territory|Danu Self-Administered Zone|Kokang Self-Administered Zone|Naga Self-Administered Zone|Pa-O Self-Administered Zone|Pa Laung Self-Administered Zone|Wa Self-Administered Division'], ['Namibia', 'NA', 'Caprivi|Erongo|Hardap|Karas|Khomas|Kunene|Ohangwena|Okavango|Omaheke|Omusati|Oshana|Oshikoto|Otjozondjupa'], ['Nauru', 'NR', 'Aiwo|Anabar|Anetan|Anibare|Baiti|Boe|Buada|Denigomodu|Ewa|Ijuw|Meneng|Nibok|Uaboe|Yaren'], ['Nepal', 'NP', 'Bagmati|Bheri|Dhawalagiri|Gandaki|Janakpur|Karnali|Kosi|Lumbini|Mahakali|Mechi|Narayani|Rapti|Sagarmatha|Seti'], ['New Caledonia', 'NC', 'Iles Loyaute|Nord|Sud'], ['New Zealand', 'NZ', 'Akaroa|Amuri|Ashburton|Bay of Islands|Bruce|Buller|Chatham heviot|Clifton|Clutha|Cook|Dannevirke|Egmont|Eketahuna|Ellesmere|Eltham|Eyre|Featherston|Franklin|Golden Bay|Great Barrier Island|Grey|Hauraki wera|Hawke\'s Bay|Heathcote|Hikurangi|Hobson|Hokianga|Horowhenua|Hurunui|Hutt|Inangahua|Inglewood|Kaikoura|Kairanga|Kiwitea|Lake|Mackenzie|Malvern|Manatu|Mangonui|Maniototo|Marlborough|Masterton|Matamata|Mount hinemuri|Opotiki|Oroua|Otamatea|Otorohanga|Oxford|Pahiatua|Paparua|Patea|Piako|Pohangina|Raglan|Rangiora|Rangitikei|Rodney|Rotorua|Runanga|Saint verpeaks|Southland|Stewart Island|Stratford|Strathallan|Taranaki|Taumarunui|Taupo|Tauranga|Thames-l|Tuapeka|Vincent|Waiapu|Waiheke|Waihemo|Waikato|Waikohu|Waimairi|Waimarino|Waimate|Waimate West|Waimea|Waipa|Waipawa|Waipukurau|Wairarapa rewa|Wairoa|Waitaki|Waitomo|Waitotara|Wallace|Wanganui|Waverley|Westland|Whakatane|Whangarei|Whangaroa|Woodville'], ['Nicaragua', 'NI', 'Atlantico Norte|Atlantico Sur|Boaco|Carazo|Chinandega|Chontales|Esteli|Granada|Jinotega|Leon|Madriz|Managua|Masaya|Matagalpa|Nueva Segovia|Rio San s'], ['Niger', 'NE', 'Agadez|Diffa|Dosso|Maradi|Niamey|Tahoua|Tillaberi|Zinder'], ['Nigeria', 'NG', 'Abia|Abuja Federal Capital Territory|Adamawa|Akwa Ibom|Anambra|Bauchi|Bayelsa|Benue|Borno|Cross River|Delta|Ebonyi|Edo|Ekiti|Enugu|Gombe|Imo|Jigawa|no|Katsina|Kebbi|Kogi|Kwara|Lagos|Nassarawa|Niger|Ogun|Ondo|Osun|Oyo|Plateau|Rivers|Sokoto|Taraba|Yobe|Zamfara'], ['Niue', 'NU', 'Niue'], ['Norfolk Island', 'NF', 'Norfolk Island'], ['Northern Mariana Islands', 'MP', 'Northern Islands|Rota|Saipan|Tinian'], ['Oman', 'OM', 'Ad Dakhiliyah|Al Batinah|Al Wusta|Ash Sharqiyah|Az Zahirah|Masqat|Musandam|Zufar'], ['Pakistan', 'PK', 'Balochistan|Federally Administered Tribal Areas|Islamabad Capital Territory|North-West Frontier Province|Punjab|Sindh'], ['Palau', 'PW', 'Aimeliik|Airai|Angaur|Hatobohei|Kayangel|Koror|Melekeok|Ngaraard|Ngarchelong|Ngardmau|Ngatpang|Ngchesar|Ngeremlengui|Ngiwal|Palau leliu|Sonsoral|Tobi'], ['Palestine, State of', 'PS', 'Palestine'], ['Panama', 'PA', 'Bocas del Toro|Chiriqui|Cocle|Colon|Darien|Herrera|Los Santos|Panama|San Blas|Veraguas'], ['Papua New Guinea', 'PG', 'Bougainville|Central|Chimbu|East New Britain|East Sepik|Eastern Highlands|Enga|Gulf|Madang|Manus|Milne Bay|Morobe|National Capital|New orthern|Sandaun|Southern Highlands|West New Britain|Western|Western Highlands'], ['Paraguay', 'PY', 'Alto Paraguay|Alto Parana|Amambay|Asuncion (city)|Caaguazu|Caazapa|Canindeyu|Central|Concepcion|Cordillera|Guaira|Itapua|Misiones|Neembucu|Paraguari|Presidente Hayes|San Pedro'], ['Peru', 'PE', 'Amazonas|Ancash|Apurimac|Arequipa|Ayacucho|Cajamarca|Callao|Cusco|Huancavelica|Huanuco|Ica|Junin|La Libertad|Lambayeque|Lima|Loreto|Madre de egua|Pasco|Piura|Puno|San Martin|Tacna|Tumbes|Ucayali'], ['Philippines', 'PH', 'Abra|Agusan del Norte|Agusan del Sur|Aklan|Albay|Angeles|Antique|Aurora|Bacolod|Bago|Baguio|Bais|Basilan|Basilan an|Batanes|Batangas|Batangas City|Benguet|Bohol|Bukidnon|Bulacan|Butuan|Cabanatuan|Cadiz|Cagayan|Cagayan de Oro|Calbayog|Caloocan|Camarines arines Sur|Camiguin|Canlaon|Capiz|Catanduanes|Cavite|Cavite City|Cebu|Cebu City|Cotabato|Dagupan|Danao|Dapitan|Davao City Davao|Davao del Sur|Davao Dipolog|Dumaguete|Eastern Samar|General Santos|Gingoog|Ifugao|Iligan|Ilocos Norte|Ilocos Sur|Iloilo|Iloilo City|Iriga|Isabela|Kalinga-Apayao|La a Union|Laguna|Lanao del Norte|Lanao del Sur|Laoag|Lapu-Lapu|Legaspi|Leyte|Lipa|Lucena|Maguindanao|Mandaue|Manila|Marawi|Marinduque|Masbate|Mindoro l|Mindoro Oriental|Misamis Occidental|Misamis Oriental|Mountain|Naga|Negros Occidental|Negros Oriental|North Cotabato|Northern Samar|Nueva va Vizcaya|Olongapo|Ormoc|Oroquieta|Ozamis|Pagadian|Palawan|Palayan|Pampanga|Pangasinan|Pasay|Puerto Princesa|Quezon|Quezon ino|Rizal|Romblon|Roxas|Samar|San Carlos (in Negros Occidental)|San Carlos (in Pangasinan)|San Jose|San Pablo|Silay|Siquijor|Sorsogon|South Southern Leyte|Sultan Kudarat|Sulu|Surigao|Surigao del Norte|Surigao del Sur|Tacloban|Tagaytay|Tagbilaran|Tangub|Tarlac|Tawitawi|Toledo|Trece Zambales|Zamboanga|Zamboanga del Norte|Zamboanga del Sur'], ['Pitcairn', 'PN', 'Pitcairn Islands'], ['Poland', 'PL', 'Dolnoslaskie|Kujawsko-|Lodzkie|Lubelskie|Lubuskie|Malopolskie|Mazowieckie|Opolskie|Podkarpackie|Podlaskie|Pomorskie|Slaskie|Swietokrzyskie|Warminsko-|Wielkopolskie|Zachodniopomorskie'], ['Portugal', 'PT', 'Acores (Azores)|Aveiro|Beja|Braga|Braganca|Castelo Branco|Coimbra|Evora|Faro|Guarda|Leiria|Lisboa|Madeira|Portalegre|Porto|Santarem|Setubal|Viana o|Vila Real|Viseu'], ['Puerto Rico', 'PR', 'Adjuntas|Aguada|Aguadilla|Aguas Buenas|Aibonito|Anasco|Arecibo|Arroyo|Barceloneta|Barranquitas|Bayamon|Cabo Rojo|Caguas|Camuy|Canovanas|Carolina|Cat|Ceiba|Ciales|Cidra|Coamo|Comerio|Corozal|Culebra|Dorado|Fajardo|Florida|Guanica|Guayama|Guayanilla|Guaynabo|Gurabo|Hatillo|Hormigueros|Humacao|Isabe|Juana Diaz|Juncos|Lajas|Lares|Las Marias|Las oiza|Luquillo|Manati|Maricao|Maunabo|Mayaguez|Moca|Morovis|Naguabo|Naranjito|Orocovis|Patillas|Penuelas|Ponce|Quebradillas|Rincon|Rio Grande|Sabana linas|San German|San Juan|San Lorenzo|San Sebastian|Santa Isabel|Toa Alta|Toa Baja|Trujillo Alto|Utuado|Vega Alta|Vega ues|Villalba|Yabucoa|Yauco'], ['Qatar', 'QA', 'Ad Dawhah|Al Ghuwayriyah|Al Jumayliyah|Al Khawr|Al Wakrah|Ar Rayyan|Jarayan al Batinah|Madinat ash Shamal|Umm Salal'], ['Réunion', 'RE', 'Réunion'], ['Romania', 'RO', 'Alba|Arad|Arges|Bacau|Bihor|Bistrita-Nasaud|Botosani|Braila|Brasov|Bucuresti|Buzau|Calarasi|Caras-luj|Constanta|Covasna|Dimbovita|Dolj|Galati|Giurgiu|Gorj|Harghita|Hunedoara|Ialomita|Iasi|Maramures|Mehedinti|Mures|Neamt|Olt|Prahova|Salaj|Satu u|Suceava|Teleorman|Timis|Tulcea|Vaslui|Vilcea|Vrancea'], ['Russian Federation', 'RU', 'Adygeya (Maykop)|Aginskiy Buryatskiy (Aginskoye)|Altay (Gorno-Altaysk)|Altayskiy (Barnaul)|Amurskaya (Blagoveshchensk)l\'skaya|Astrakhanskaya|Bashkortostan (Ufa)|Belgorodskaya|Bryanskaya|Buryatiya (Ulan-Ude)|Chechnya (Groznyy)|Chelyabinskaya|Chitinskaya|Chukotskiy (Chuvashiya (Cheboksary)|Dagestan (Makhachkala)|Evenkiyskiy (Tura)|Ingushetiya (Nazran\')|Irkutskaya|Ivanovskaya|Kabardino-Balkariya (Nal\'chik)radskaya|Kalmykiya (Elista)|Kaluzhskaya|Kamchatskaya (Petropavlovsk-Kamchatskiy)|Karachayevo-Cherkesiya (Cherkessk)|Kareliya (Petrozavodsk)kaya|Khabarovskiy|Khakasiya (Abakan)|Khanty-Mansiyskiy (Khanty-Mansiysk)|Kirovskaya|Komi (Syktyvkar)|Komi-Permyatskiy (Kudymkar)|Koryakskiy (Palana)kaya|Krasnodarskiy|Krasnoyarskiy|Kurganskaya|Kurskaya|Leningradskaya|Lipetskaya|Magadanskaya|Mariy-El (Yoshkar-Ola)|Mordoviya (Saransk)aya|Moskva (Moscow)|Murmanskaya|Nenetskiy (Nar\'yan-Mar)|Nizhegorodskaya|Novgorodskaya|Novosibirskaya|Omskaya|Orenburgskaya|Orlovskaya (Orel)aya|Permskaya|Primorskiy (Vladivostok)|Pskovskaya|Rostovskaya|Ryazanskaya|Sakha (Yakutsk)|Sakhalinskaya (Yuzhno-Sakhalinsk)|Samarskaya|Sankt- (Saint Petersburg)|Saratovskaya|Severnaya Osetiya-Alaniya [North Ossetia] (Vladikavkaz)|Smolenskaya|Stavropol\'skiy|Sverdlovskaya (Yekaterinburg)aya|Tatarstan (Kazan\')|Taymyrskiy (Dudinka)|Tomskaya|Tul\'skaya|Tverskaya|Tyumenskaya|Tyva (Kyzyl)|Udmurtiya (Izhevsk)|Ul\'yanovskaya|Ust\'-Ordynskiy y (Ust\'-Ordynskiy)|Vladimirskaya|Volgogradskaya|Vologodskaya|Voronezhskaya|Yamalo-Nenetskiy (Salekhard)|Yaroslavskaya|Yevreyskaya'], ['Rwanda', 'RW', 'Butare|Byumba|Cyangugu|Gikongoro|Gisenyi|Gitarama|Kibungo|Kibuye|Kigali Rurale|Kigali-ville|Ruhengeri|Umutara'], ['Saint Barthélemy', 'BL', 'Saint Barthélemy'], ['Saint Helena, Ascension and Tristan da Cunha', 'SH', 'Ascension|Saint Helena|Tristan da Cunha'], ['Saint Kitts and Nevis', 'KN', 'Christ Church Nichola Town|Saint Anne Sandy Point|Saint George Basseterre|Saint George Gingerland|Saint James Windward|Saint John Capisterre|Saint ree|Saint Mary Cayon|Saint Paul Capisterre|Saint Paul Charlestown|Saint Peter Basseterre|Saint Thomas Lowland|Saint Thomas Middle Island|Trinity Point'], ['Saint Lucia', 'LC', 'Anse-la-Raye|Castries|Choiseul|Dauphin|Dennery|Gros Islet|Laborie|Micoud|Praslin|Soufriere|Vieux Fort'], ['Saint Martin (French part)', 'MF', 'Saint Martin'], ['Saint Pierre and Miquelon', 'PM', 'Miquelon|Saint Pierre'], ['Saint Vincent and the Grenadines', 'VC', 'Charlotte|Grenadines|Saint Andrew|Saint David|Saint George|Saint Patrick'], ['Samoa', 'WS', 'A\'ana|Aiga-i-le-Tai|Atua|Fa\'asaleleaga|Gaga\'emauga|Gagaifomauga|Palauli|Satupa\'itea|Tuamasaga|Va\'a-o-Fonoti|Vaisigano'], ['San Marino', 'SM', 'Acquaviva|Borgo Maggiore|Chiesanuova|Domagnano|Faetano|Fiorentino|Monte Giardino|San Marino|Serravalle'], ['Sao Tome and Principe', 'ST', 'Principe|Sao Tome'], ['Saudi Arabia', 'SA', '\'Asir|Al Bahah|Al Hudud ash Shamaliyah|Al Jawf|Al Madinah|Al Qasim|Ar Riyad|Ash Sharqiyah (Eastern Province)|Ha\'il|Jizan|Makkah|Najran|Tabuk'], ['Senegal', 'SN', 'Dakar|Diourbel|Fatick|Kaolack|Kolda|Louga|Saint-Louis|Tambacounda|Thies|Ziguinchor'], ['Serbia', 'RS', 'Serbia'], ['Seychelles', 'SC', 'Anse aux Pins|Anse Boileau|Anse Etoile|Anse Louis|Anse Royale|Baie Lazare|Baie Sainte Anne|Beau Vallon|Bel Air|Bel Ombre|Cascade|Glacis|Grand\' Anse |Grand\' Anse (on Praslin)|La Digue|La Riviere Anglaise|Mont Buxton|Mont Fleuri|Plaisance|Pointe La Rue|Port Glaud|Saint Louis|Takamaka'], ['Sierra Leone', 'SL', 'Eastern|Northern|Southern|Western'], ['Singapore', 'SG', 'Singapore'], ['Sint Maarten (Dutch part)', 'SX', 'Sint Maarten'], ['Slovakia', 'SK', 'Banskobystricky|Bratislavsky|Kosicky|Nitriansky|Presovsky|Trenciansky|Trnavsky|Zilinsky'], ['Slovenia', 'SI', 'Ajdovscina|Beltinci|Bled|Bohinj|Borovnica|Bovec|Brda|Brezice|Brezovica|Cankova-Tisina|Celje|Cerklje na Gorenjskem|Cerknica|Cerkno|Crensovci|Crna na Crnomelj|Destrnik-Trnovska Vas|Divaca|Dobrepolje|Dobrova-Horjul-Polhov Gradec|Dol pri Ljubljani|Domzale|Dornava|Dravograd|Duplek|Gorenja Vas-orisnica|Gornja Radgona|Gornji Grad|Gornji Petrovci|Grosuplje|Hodos Salovci|Hrastnik|Hrpelje-Kozina|Idrija|Ig|Ilirska Bistrica|Ivancna ola|Jesenice|Jursinci|Kamnik|Kanal|Kidricevo|Kobarid|Kobilje|Kocevje|Komen|Koper|Kozje|Kranj|Kranjska o|Kungota|Kuzma|Lasko|Lenart|Lendava|Litija|Ljubljana|Ljubno|Ljutomer|Logatec|Loska Dolina|Loski e|Lukovica|Majsperk|Maribor|Medvode|Menges|Metlika|Mezica|Miren-Kostanjevica|Mislinja|Moravce|Moravske Toplice|Mozirje|Murska ta|Naklo|Nazarje|Nova Gorica|Novo Mesto|Odranci|Ormoz|Osilnica|Pesnica|Piran|Pivka|Podcetrtek|Podvelka-Ribnica|Postojna|Preddvor|Ptuj|Puconci|Race-ce|Radenci|Radlje ob Dravi|Radovljica|Ravne-Prevalje|Ribnica|Rogasevci|Rogaska Slatina|Rogatec|Ruse|Semic|Sencur|Sentilj|Sentjernej|Sentjur pri nica|Sezana|Skocjan|Skofja Loka|Skofljica|Slovenj Gradec|Slovenska Bistrica|Slovenske Konjice|Smarje pri Jelsah|Smartno ob anj|Starse|Store|Sveti Jurij|Tolmin|Trbovlje|Trebnje|Trzic|Turnisce|Velenje|Velike Lasce|Videm|Vipava|Vitanje|Vodice|Vojnik|Vrhnika|Vuzenica|Zagorje alec|Zavrc|Zelezniki|Ziri|Zrece'], ['Solomon Islands', 'SB', 'Bellona|Central|Choiseul (Lauru)|Guadalcanal|Honiara|Isabel|Makira|Malaita|Rennell|Temotu|Western'], ['Somalia', 'SO', 'Awdal|Bakool|Banaadir|Bari|Bay|Galguduud|Gedo|Hiiraan|Jubbada Dhexe|Jubbada Hoose|Mudug|Nugaal|Sanaag|Shabeellaha Dhexe|Shabeellaha l|Togdheer|Woqooyi Galbeed'], ['South Africa', 'ZA', 'Eastern Cape|Free State|Gauteng|KwaZulu-Natal|Mpumalanga|North-West|Northern Cape|Northern Province|Western Cape'], ['South Georgia and South Sandwich Islands', 'GS', 'Bird Island|Bristol Island|Clerke Rocks|Montagu Island|Saunders Island|South Georgia|Southern Thule|Traversay Islands'], ['South Sudan', 'SS', 'South Sudan'], ['Spain', 'ES', 'Andalucia|Aragon|Asturias|Baleares (Balearic Islands)|Canarias (Canary Islands)|Cantabria|Castilla y Leon|Castilla-La taluna|Ceuta|Communidad Valencian|Extremadura|Galicia|Islas Chafarinas|La Rioja|Madrid|Melilla|Murcia|Navarra|Pais Vasco (Basque Country)|Penon de |Penon de Velez de la Gomera'], ['Sri Lanka', 'LK', 'Central|Eastern|North Central|North Eastern|North Western|Northern|Sabaragamuwa|Southern|Uva|Western'], ['Sudan', 'SD', 'A\'ali an Nil|Al Bahr al Ahmar|Al Buhayrat|Al Jazirah|Al Khartum|Al Qadarif|Al Wahdah|An Nil al Abyad|An Nil al Azraq|Ash Shamaliyah|Bahr al rb al Istiwa\'iyah|Gharb Bahr al Ghazal|Gharb Darfur|Gharb Kurdufan|Janub Darfur|Janub Kurdufan|Junqali|Kassala|Nahr an Nil|Shamal Bahr al amal Darfur|Shamal Kurdufan|Sharq al Istiwa\'iyah|Sinnar|Warab'], ['Suriname', 'SR', 'Brokopondo|Commewijne|Coronie|Marowijne|Nickerie|Para|Paramaribo|Saramacca|Sipaliwini|Wanica'], ['Svalbard and Jan Mayen', 'SJ', 'Barentsoya|Bjornoya|Edgeoya|Hopen|Kvitoya|Nordaustandet|Prins Karls Forland|Spitsbergen'], ['Swaziland', 'SZ', 'Hhohho|Lubombo|Manzini|Shiselweni'], ['Switzerland', 'CH', 'Aargau|Ausser-Rhoden|Basel-Landschaft|Basel-Stadt|Bern|Fribourg|Geneve|Glarus|Graubunden|Inner-ra|Luzern|Neuchatel|Nidwalden|Obwalden|Sankt Gallen|Schaffhausen|Schwyz|Solothurn|Thurgau|Ticino|Uri|Valais|Vaud|Zug|Zurich'], ['Syrian Arab Republic', 'SY', 'Al Hasakah|Al Ladhiqiyah|Al Qunaytirah|Ar Raqqah|As Suwayda\'|Dar\'a|Dayr az Zawr|Dimashq|Halab|Hamah|Hims|Idlib|Rif Dimashq|Tartus'], ['Taiwan, Province of China', 'TW', 'Chang-hua|Chi-lung|Chia-i|Chia-i|Chung-hsing-hsin-ts\'un|Hsin-chu|Hsin-chu|Hua-lien|I-lan|Kao-hsiung|Kao-hsiung|Miao-li|Nan-t\'ou|P\'eng-hu|P\'ing--chung|T\'ai-chung|T\'ai-nan|T\'ai-nan|T\'ai-pei|T\'ai-pei|T\'ai-tung|T\'ao-yuan|Yun-lin'], ['Tajikistan', 'TJ', 'Viloyati Khatlon|Viloyati Leninobod|Viloyati Mukhtori Kuhistoni Badakhshon'], ['Tanzania, United Republic of', 'TZ', 'Arusha|Dar es Salaam|Dodoma|Iringa|Kagera|Kigoma|Kilimanjaro|Lindi|Mara|Mbeya|Morogoro|Mtwara|Mwanza|Pemba North|Pemba ni|Rukwa|Ruvuma|Shinyanga|Singida|Tabora|Tanga|Zanzibar Central/South|Zanzibar North|Zanzibar Urban/West'], ['Thailand', 'TH', 'Amnat Charoen|Ang Thong|Buriram|Chachoengsao|Chai Nat|Chaiyaphum|Chanthaburi|Chiang Mai|Chiang Rai|Chon Buri|Chumphon|Kalasin|Kamphaeng hanaburi|Khon Kaen|Krabi|Krung Thep Mahanakhon (Bangkok)|Lampang|Lamphun|Loei|Lop Buri|Mae Hong Son|Maha Sarakham|Mukdahan|Nakhon Nayok|Nakhon khon Phanom|Nakhon Ratchasima|Nakhon Sawan|Nakhon Si Thammarat|Nan|Narathiwat|Nong Bua Lamphu|Nong Khai|Nonthaburi|Pathum tani|Phangnga|Phatthalung|Phayao|Phetchabun|Phetchaburi|Phichit|Phitsanulok|Phra Nakhon Si Ayutthaya|Phrae|Phuket|Prachin Buri|Prachuap Khiri ng|Ratchaburi|Rayong|Roi Et|Sa Kaeo|Sakon Nakhon|Samut Prakan|Samut Sakhon|Samut Songkhram|Sara Buri|Satun|Sing ket|Songkhla|Sukhothai|Suphan Buri|Surat Thani|Surin|Tak|Trang|Trat|Ubon Ratchathani|Udon Thani|Uthai Thani|Uttaradit|Yala|Yasothon'], ['Timor-Leste', 'TL', 'Timor-Leste'], ['Togo', 'TG', 'De La Kara|Des Plateaux|Des Savanes|Du Centre|Maritime'], ['Tokelau', 'TK', 'Atafu|Fakaofo|Nukunonu'], ['Tonga', 'TO', 'Ha\'apai|Tongatapu|Vava\'u'], ['Trinidad and Tobago', 'TT', 'Arima|Caroni|Mayaro|Nariva|Port-of-Spain|Saint Andrew|Saint David|Saint George|Saint Patrick|San Fernando|Victoria|Tobago'], ['Tunisia', 'TN', 'Ariana|Beja|Ben Arous|Bizerte|El Kef|Gabes|Gafsa|Jendouba|Kairouan|Kasserine|Kebili|Mahdia|Medenine|Monastir|Nabeul|Sfax|Sidi Bou na|Sousse|Tataouine|Tozeur|Tunis|Zaghouan'], ['Turkey', 'TR', 'Adana|Adiyaman|Afyon|Agri|Aksaray|Amasya|Ankara|Antalya|Ardahan|Artvin|Aydin|Balikesir|Bartin|Batman|Bayburt|Bilecik|Bingol|Bitlis|Bolu|Burdur|Bursae|Cankiri|Corum|Denizli|Diyarbakir|Duzce|Edirne|Elazig|Erzincan|Erzurum|Eskisehir|Gaziantep|Giresun|Gumushane|Hakkari|Hatay|Icel|Igdir|Isparta|IstanbKahramanmaras|Karabuk|Karaman|Kars|Kastamonu|Kayseri|Kilis|Kirikkale|Kirklareli|Kirsehir|Kocaeli|Konya|Kutahya|Malatya|Manisa|Mardin|Mugla|Mus|NevsehOrdu|Osmaniye|Rize|Sakarya|Samsun|Sanliurfa|Siirt|Sinop|Sirnak|Sivas|Tekirdag|Tokat|Trabzon|Tunceli|Usak|Van|Yalova|Yozgat|Zonguldak'], ['Turkmenistan', 'TM', 'Ahal Welayaty|Balkan Welayaty|Dashhowuz Welayaty|Lebap Welayaty|Mary Welayaty'], ['Turks and Caicos Islands', 'TC', 'Turks and Caicos Islands'], ['Tuvalu', 'TV', 'Tuvalu'], ['Uganda', 'UG', 'Adjumani|Apac|Arua|Bugiri|Bundibugyo|Bushenyi|Busia|Gulu|Hoima|Iganga|Jinja|Kabale|Kabarole|Kalangala|Kampala|Kamuli|Kapchorwa|Kasese|Katakwi|Kibaleisoro|Kitgum|Kotido|Kumi|Lira|Luwero|Masaka|Masindi|Mbale|Mbarara|Moroto|Moyo|Mpigi|Mubende|Mukono|Nakasongola|Nebbi|Ntungamo|Pallisa|Rakai|Rukungirie|Soroti|Tororo'], ['Ukraine', 'UA', 'Avtonomna Respublika Krym (Simferopol\')|Cherkas\'ka (Cherkasy)|Chernihivs\'ka (Chernihiv)|Chernivets\'ka (Chernivtsi)|Dnipropetrovs\'ka (Dnipropetrovs\'kka (Donets\'k)|Ivano-Frankivs\'ka (Ivano-Frankivs\'k)|Kharkivs\'ka (Kharkiv)|Khersons\'ka (Kherson)|Khmel\'nyts\'ka (Khmel\'nyts\'kyy)|Kirovohrads\'ka (d)|Kyyiv|Kyyivs\'ka (Kiev)|L\'vivs\'ka (L\'viv)|Luhans\'ka (Luhans\'k)|Mykolayivs\'ka (Mykolayiv)|Odes\'ka (Odesa)|Poltavs\'ka (Poltava)|Rivnens\'ka (Rivne)ol\'|Sums\'ka (Sumy)|Ternopil\'s\'ka (Ternopil\')|Vinnyts\'ka (Vinnytsya)|Volyns\'ka (Luts\'k)|Zakarpats\'ka (Uzhhorod)|Zaporiz\'ka (Zaporizhzhya)s\'ka (Zhytomyr)'], ['United Arab Emirates', 'AE', '\'Ajman|Abu Zaby (Abu Dhabi)|Al Fujayrah|Ash Shariqah (Sharjah)|Dubayy (Dubai)|Ra\'s al Khaymah|Umm al Qaywayn'], ['United States Minor Outlying Islands', 'UM', 'Wake Island|Johnston Atoll|Midway Atoll|Kingman Reef|Palmyra Atoll|Jarvis Island|Baker Island|Howland Island|Navassa Island|Bajo Nuevo Bank|Serranilla Bank'], ['Uruguay', 'UY', 'Artigas|Canelones|Cerro Largo|Colonia|Durazno|Flores|Florida|Lavalleja|Maldonado|Montevideo|Paysandú|Río Negro|Rivera|Rocha|Salto|San José|Soriano|Tacuaremb|Treinta'], ['Uzbekistan', 'UZ', 'Andijon Wiloyati|Bukhoro Wiloyati|Farghona Wiloyati|Jizzakh Wiloyati|Khorazm Wiloyati (Urganch)|Namangan Wiloyati|Nawoiy Wiloyati|Qashqadaryo (Qarshi)|Qoraqalpoghiston (Nukus)|Samarqand Wiloyati|Sirdaryo Wiloyati (Guliston)|Surkhondaryo Wiloyati (Termiz)|Toshkent Shahri|Toshkent Wiloyati'], ['Vanuatu', 'VU', 'Malampa|Penama|Sanma|Shefa|Tafea|Torba'], ['Venezuela, Bolivarian Republic of', 'VE', 'Amazonas|Anzoategui|Apure|Aragua|Barinas|Bolivar|Carabobo|Cojedes|Delta Amacuro|Dependencias Federales|Distrito alcon|Guarico|Lara|Merida|Miranda|Monagas|Nueva Esparta|Portuguesa|Sucre|Tachira|Trujillo|Vargas|Yaracuy|Zulia'], ['Viet Nam', 'VN', 'An Giang|Ba Ria-Vung Tau|Bac Giang|Bac Kan|Bac Lieu|Bac Ninh|Ben Tre|Binh Dinh|Binh Duong|Binh Phuoc|Binh Thuan|Ca Mau|Can Tho|Cao Bang|Da Nang|Dac Nai|Dong Thap|Gia Lai|Ha Giang|Ha Nam|Ha Noi|Ha Tay|Ha Tinh|Hai Duong|Hai Phong|Ho Chi Minh|Hoa Binh|Hung Yen|Khanh Hoa|Kien Giang|Kon Tum|Lai Dong|Lang Son|Lao Cai|Long An|Nam Dinh|Nghe An|Ninh Binh|Ninh Thuan|Phu Tho|Phu Yen|Quang Binh|Quang Nam|Quang Ngai|Quang Ninh|Quang Tri|Soc  La|Tay Ninh|Thai Binh|Thai Nguyen|Thanh Hoa|Thua Thien-Hue|Tien Giang|Tra Vinh|Tuyen Quang|Vinh Long|Vinh Phuc|Yen Bai'], ['Virgin Islands, British', 'VG', 'Anegada|Jost Van Dyke|Tortola|Virgin Gorda'], ['Virgin Islands, U.S.', 'VI', 'St. Thomas|St. John|St. Croix'], ['Wallis and Futuna', 'WF', 'Alo|Sigave|Wallis'], ['Western Sahara', 'EH', 'Western Sahara'], ['Yemen', 'YE', '\'Adan|\'Ataq|Abyan|Al Bayda\'|Al Hudaydah|Al Jawf|Al Mahrah|Al Mahwit|Dhamar|Hadhramawt|Hajjah|Ibb|Lahij|Ma\'rib|Sa\'dah|San\'a\'|Ta\'izz'], ['Zambia', 'ZM', 'Central|Copperbelt|Eastern|Luapula|Lusaka|North-Western|Northern|Southern|Western'], ['Zimbabwe', 'ZW', 'Bulawayo|Harare|ManicalandMashonaland Central|Mashonaland East|Mashonaland West|Masvingo|Matabeleland North|Matabeleland South|Midlands']]);
'use strict';

angular.module('xl.widget').directive('xlWidgetContextMenu', function ($q, $timeout, uuid4) {
    function mapMenuItems(scope, items) {
        var result = {};
        _.forEach(items, function (item) {
            result[item.name] = {
                name: item.name,
                disabled: item.isDisabled && item.isDisabled(scope.contextMenuSourceData) || false,
                callback: function callback() {
                    return $q.when(item.action(scope.contextMenuSourceData)).then(function () {
                        return $timeout(_.noop);
                    });
                }
            };
        });
        return result;
    }

    function link(scope, element, attrs) {
        var id = attrs.id || uuid4.generate();
        element.attr('id', id);
        var items = scope.contextMenuProvider(scope.contextMenuSourceData);
        angular.element.contextMenu({
            selector: '#' + id,
            build: function build() {
                if (items.length === 0) {
                    return false;
                } else {
                    return {
                        items: mapMenuItems(scope, items)
                    };
                }
            }
        });
    }

    return {
        restrict: 'A',
        scope: {
            contextMenuProvider: '=',
            contextMenuSourceData: '='
        },
        link: link
    };
});
'use strict';

angular.module('xl.widget').directive('xlWidgetContenteditable', function (contentEditableService) {

    var passwordMask = '*************';

    return {
        restrict: 'A',
        scope: {
            ngModel: '=',
            password: '@password'
        },
        templateUrl: 'src/xl-widget/xl-widget-contenteditable/xl-widget-contenteditable.html',
        link: function link($scope, element, attr) {
            $scope.placeholder = '<span class="hidden-empty-value">' + attr.placeholder + '</span>';
        },
        controller: function controller($scope) {
            $scope.$watch('ngModel', function (value) {
                if ($scope.password === 'true') {
                    $scope.modelLabel = passwordMask;
                } else {
                    $scope.modelInternal = value !== '' ? contentEditableService.processInputString(value) : $scope.placeholder;
                    $scope.modelLabel = value !== '' ? contentEditableService.processInputString(value) : $scope.placeholder;
                }
            });

            this.change = function (value) {
                $scope.ngModel = value;
                if ($scope.password === 'true') {
                    $scope.modelInternal = passwordMask;
                } else {
                    $scope.modelInternal = value !== '' ? contentEditableService.processInputString(value) : $scope.placeholder;
                }
            };

            this.getExternalModel = function () {
                return $scope.ngModel;
            };

            this.getExternalView = function () {
                return $scope.modelLabel;
            };
        }
    };
});

angular.module('xl.widget').factory('contentEditableService', function () {
    var substituteSymbol = '<span class="xl-ws"> </span>';

    var processInputString = function processInputString(input) {
        var replace = input.toString().replace(/\s/gi, substituteSymbol);
        var re = new RegExp('([\\w])' + substituteSymbol + '([\\w])', 'g');
        return replace.toString().replace(re, '$1 $2');
    };

    return {
        processInputString: processInputString
    };
});

angular.module('xl.widget').directive('contenteditable', function ($document, $window) {
    return {
        restrict: 'A',
        require: ['?^ngModel', '?^xlWidgetContenteditable'],
        scope: {
            ngModel: '=',
            password: '@password'
        },
        link: function link(scope, element, attrs, constrollers) {
            var ngModel = constrollers[0];
            var widgetContenteditable = constrollers[1];

            if (_.isUndefined(ngModel) || _.isUndefined(widgetContenteditable)) {
                return;
            }

            if (!ngModel || !widgetContenteditable) return;

            // Specify how UI should be updated
            ngModel.$render = function () {
                return element.html(ngModel.$viewValue || '');
            };

            var doInit = function doInit() {
                scope.newValue = '';
                scope.selectedText = '';
                scope.updateValue = false;
                scope.initialValue = widgetContenteditable.getExternalModel();

                if (scope.password === 'true') {
                    scope.ngModel = '';
                    element.text(scope.ngModel || '');
                } else {
                    scope.ngModel = widgetContenteditable.getExternalModel();
                }
            };

            element.bind('click', function () {
                return scope.$apply(function () {
                    return doInit();
                });
            });

            element.bind('focusin', function () {
                return scope.$apply(function () {
                    return doInit();
                });
            });

            element.bind('keydown', function (event) {
                var keyCode = getKeyCode(event);
                var isPassword = scope.password === 'true';

                if (isPassword && (keyCode === 37 || keyCode === 38)) {
                    event.preventDefault();
                }

                if (keyCode === 8 && isPassword) {
                    var length = scope.newValue.length;

                    if (scope.selectedText) {
                        scope.newValue = '';
                    } else {
                        scope.newValue = scope.newValue.substring(0, length - 1);
                        doRestore(element[0], {
                            start: length,
                            end: length
                        });
                    }
                }

                scope.skipKeypress = keyCode < 48 && keyCode !== 32 || keyCode === 91 || keyCode === 92 || keyCode === 93 || keyCode >= 112 && keyCode <= 145;

                if (!scope.skipKeypress || keyCode === 8) {
                    scope.updateValue = true;
                }
            });

            element.bind('keypress', function (event) {
                var keyCode = getKeyCode(event);

                if (keyCode === 13 || keyCode === 0) {
                    event.preventDefault();
                    angular.element(event.currentTarget).trigger('blur');
                } else if (!scope.skipKeypress && scope.password === 'true') {
                    element[0].innerHTML = element[0].textContent + '*';
                    scope.newValue = scope.newValue + String.fromCharCode(keyCode);
                    event.preventDefault();

                    doRestore(element[0], {
                        start: element[0].innerHTML.length,
                        end: element[0].innerHTML.length
                    });
                }
            });

            element.on('keyup change', function () {
                return scope.$apply(read);
            });

            element.on('blur', function () {
                scope.$apply(function () {
                    var newValue = scope.initialValue;
                    if (scope.password === 'true') {
                        if (scope.newValue === '' && scope.updateValue) {
                            // user intentionally wants to have an empty value
                            widgetContenteditable.change('');
                            element.text('');
                        } else if (scope.newValue !== '' || scope.updateValue) {
                            // user updated the value
                            newValue = scope.newValue;
                            widgetContenteditable.change(newValue);
                            element.text(widgetContenteditable.getExternalView() || '');
                        } else if (scope.initialValue !== '' && !scope.updateValue) {
                            // user focused but didn't modify the value
                            element.text(widgetContenteditable.getExternalView() || '');
                        }
                    } else {
                        widgetContenteditable.change(scope.ngModel);
                    }
                });
            });

            var init = true;

            var read = function read() {
                getSelectedText();

                // hack with $pristine (=> set it to false just for $setViewValue() not alters the pristine of the parent form)
                if (init) {
                    ngModel.$pristine = false;
                }

                ngModel.$setViewValue(element.text());

                if (init) {
                    ngModel.$pristine = true;
                    init = false;
                }
            };

            var restoreSelection = void 0;

            if ($window.getSelection && $document[0].createRange) {
                restoreSelection = function restoreSelection(containerEl, savedSel) {
                    var charIndex = 0,
                        range = $document[0].createRange();
                    range.setStart(containerEl, 0);
                    range.collapse(true);
                    var nodeStack = [containerEl],
                        node = void 0,
                        foundStart = false,
                        stop = false;

                    while (!stop && (node = nodeStack.pop())) {
                        if (node.nodeType === 3) {
                            var nextCharIndex = charIndex + node.length;
                            if (!foundStart && savedSel.start >= charIndex && savedSel.start <= nextCharIndex) {
                                range.setStart(node, savedSel.start - charIndex);
                                foundStart = true;
                            }
                            if (foundStart && savedSel.end >= charIndex && savedSel.end <= nextCharIndex) {
                                range.setEnd(node, savedSel.end - charIndex);
                                stop = true;
                            }
                            charIndex = nextCharIndex;
                        } else {
                            var i = node.childNodes.length;
                            while (i--) {
                                nodeStack.push(node.childNodes[i]);
                            }
                        }
                    }

                    var sel = $window.getSelection();
                    sel.removeAllRanges();
                    sel.addRange(range);
                };
            } else if ($document[0].selection && $document[0].body.createTextRange) {
                restoreSelection = function restoreSelection(containerEl, savedSel) {
                    var textRange = $document[0].body.createTextRange();
                    textRange.moveToElementText(containerEl);
                    textRange.collapse(true);
                    textRange.moveEnd('character', savedSel.end);
                    textRange.moveStart('character', savedSel.start);
                    textRange.select();
                };
            }

            var getKeyCode = function getKeyCode(evt) {
                if (evt.keyCode > 0) {
                    return evt.keyCode;
                } else if (!_.isUndefined(evt.charCode)) {
                    return evt.charCode;
                }
                return undefined;
            };

            var doRestore = function doRestore(element, selection) {
                return restoreSelection(element, selection);
            };

            var getSelectedText = function getSelectedText() {
                if (!_.isUndefined($window.getSelection)) {
                    scope.selectedText = $window.getSelection().toString();
                } else if (!_.isUndefined($document[0].selection) && $document[0].selection.type === 'Text') {
                    scope.selectedText = $document[0].selection.createRange().text;
                }
            };

            read();
        }
    };
});
'use strict';

angular.module('xl.widget').directive('xlWidgetAutocomplete', function ($timeout) {
    return {
        restrict: 'A',
        scope: {
            ngModel: '=',
            metadata: '=',
            handlers: '=',
            selectMode: '=',
            isRequired: '=',
            isDisabled: '=',
            options: '=',
            openOnClick: '@',
            objLabel: '@',
            showAllOnClick: '@'
        },
        templateUrl: 'src/xl-widget/xl-widget-autocomplete/xl-widget-autocomplete.html',
        require: ['?^form', 'ngModel'],
        link: function link(scope, element, attr, ctrls) {
            var inputElement = element.find('input');
            var parentForm = ctrls[0];
            var directiveCtrl = ctrls[1];

            var handlers = scope.handlers || {};
            var newItemSuffix = attr.newItemSuffix ? ' ' + attr.newItemSuffix : ' (New Item)';

            var addCandidates = handlers.addCandidates;
            var onNewItem = handlers.onNewItem;
            var onSelect = handlers.onSelect;
            var onFormat = handlers.onFormat || function (value) {
                return value;
            };
            var showAllOnClick = String(scope.showAllOnClick) === 'true';

            var autoSelectOnExactMatch = attr.autoSelectOnExactMatch !== 'false';

            var setValidity = function setValidity(validationErrorKey, isValid) {
                if (_.has(scope, ['metadata', 'name']) && _.has(parentForm, scope.metadata.name)) {
                    var inputController = parentForm[scope.metadata.name];
                    inputController.$setValidity(validationErrorKey, isValid);
                }
                directiveCtrl.$setValidity(validationErrorKey, isValid);
            };

            if (_.isUndefined(addCandidates)) {
                throw new Error('Handler addCandidates has to be defined');
            }

            var formatItem = function formatItem(value) {
                return !!scope.objLabel ? value[scope.objLabel] : value;
            };

            /**
             * {label: 'A'} -> 'A'
             * 'A' -> 'A'
             * id===undefined -> ''
             */
            var unboxSelectedItem = function unboxSelectedItem(value) {
                return scope.ngModel ? formatItem(value) : '';
            };

            /**
             * 'A' -> {label: 'A'}
             * 'A' -> 'A'
             */
            var box = function box(value) {
                if (scope.objLabel && value) {
                    var rez = {};
                    rez[scope.objLabel] = value;
                    return rez;
                }

                return value;
            };

            if (_.isUndefined(scope.ngModel)) {
                scope.ngModel = box();
            }

            scope.ngModelIntern = unboxSelectedItem(scope.ngModel);

            scope.$watch('ngModel', function (value) {
                if (_.isUndefined(value)) {
                    scope.ngModel = box();
                }
                scope.ngModelIntern = unboxSelectedItem(value);
            });

            scope.onChange = function () {
                return scope.ngModel = box(scope.ngModelIntern);
            };

            scope.updateDirtyState = function () {
                if (parentForm) {
                    parentForm.$setDirty();
                }
            };

            scope.clearModelValues = function () {
                if (parentForm) {
                    parentForm.$setDirty();
                }
                if (scope.selectMode) {
                    setValidity('selectMode', true);
                }
                delete scope.ngModelIntern;
                delete scope.ngModel;
            };

            inputElement.autocomplete({
                delay: 200,
                minLength: 1,
                autoFocus: true,
                source: function source(params, add) {
                    if (_.startsWith(params.term, '____SEARCH____')) {
                        params.term = showAllOnClick ? '' : params.term.substr(14);
                    }
                    var promise = addCandidates(scope.metadata, { term: params.term });
                    if (promise && promise.then) {
                        promise.then(function (data) {
                            var exactMatching = _.find(data, function (item) {
                                return formatItem(item) === onFormat(params.term);
                            });

                            if (scope.selectMode) {
                                setValidity('selectMode', !!exactMatching);
                            }

                            if (exactMatching && autoSelectOnExactMatch) {
                                directiveCtrl.$setViewValue(exactMatching);
                            }

                            data = _.map(data, function (item) {
                                return {
                                    label: _.isUndefined(scope.objLabel) ? item : item[scope.objLabel],
                                    value: _.isUndefined(scope.objLabel) ? item : item[scope.objLabel],
                                    __initialObj__: item
                                };
                            });

                            if (onNewItem && params.term.length && !exactMatching) {
                                data.unshift({
                                    label: onFormat(params.term),
                                    value: onFormat(params.term),
                                    newElem: true,
                                    single: data.length === 0,
                                    __initialObj__: box(params.term)
                                });
                            }

                            return data;
                        }).then(add);
                    }
                },
                open: function open() {
                    $(this).autocomplete('widget').css({ //eslint-disable-line
                        'width': element.width()
                    });
                },
                select: function select(event, ui) {
                    scope.$apply(function () {
                        setValidity('selectMode', true);
                        if (parentForm) {
                            parentForm.$setDirty();
                        }

                        scope.ngModelIntern = ui.item.value;

                        directiveCtrl.$setViewValue(ui.item.__initialObj__);

                        if (ui.item.newElem) {
                            var newItem = onNewItem(scope.ngModel, scope.metadata, event);
                            if (newItem && newItem.then) {
                                newItem.then(function (updatedValue) {
                                    if (updatedValue) {
                                        inputElement.val(updatedValue);
                                    }

                                    inputElement.blur();
                                });
                            }
                        } else if (onSelect) {
                            onSelect(ui.item.__initialObj__);
                        }
                    });
                },
                highlightClass: 'xl-components xl-autocomplete-highlight'
            }).data('ui-autocomplete')._renderItem = function (ul, item) {
                angular.element(ul).addClass('xl-components xl-widget-autocomplete');

                if (this.term.length === 0) {
                    return angular.element('<li>').html(item.label).appendTo(ul);
                }
                var re = new RegExp('(' + this.term + ')', 'gi'),
                    cls = this.options.highlightClass,
                    template = '<span class=\'' + cls + '\'>$1</span>',
                    label = item.label.replace(re, template) + (item.newElem ? newItemSuffix : '');
                return angular.element('<li class=\'' + (item.newElem && !item.single ? ' xl-autocomplete-new-item' : '') + '\'>').html(label).appendTo(ul);
            };

            var wasOpened = true;
            var handlerElement = element.find('.xl-btn-handler');

            handlerElement.mousedown(function () {
                wasOpened = inputElement.autocomplete('widget').is(':visible');
            });

            handlerElement.bind('click focus', function () {
                $timeout(function () {
                    if (wasOpened) {
                        inputElement.autocomplete('close');
                    } else {
                        inputElement.autocomplete('search', '____SEARCH____' + inputElement.val());
                        inputElement.focus();
                    }
                });
            });

            if (String(scope.openOnClick) === 'true') {
                inputElement.bind('click focus', function () {
                    var that = $(this); //eslint-disable-line
                    $timeout(function () {
                        that.autocomplete('search', '____SEARCH____' + that.val());
                    });
                });
            }
        }
    };
});
'use strict';

angular.module('xl.widget').directive('xlWidgetAutocompleteType', function ($q) {
    return {
        scope: {
            xlWidgetAutocompleteType: '=',
            onSelect: '&',
            ngModel: '=',
            metadata: '=',
            options: '=',
            isRequired: '=',
            isDisabled: '='
        },
        templateUrl: 'src/xl-widget/xl-widget-autocomplete-type/xl-widget-autocomplete-type.html',
        require: ['?^form', 'ngModel'],
        link: {
            pre: function pre(scope) {
                scope.addCandidates = function (metadata, options) {
                    var differed = $q.defer();
                    var candidates = _.filter(scope.xlWidgetAutocompleteType, function (item) {
                        return item.toLowerCase().indexOf(options.term.toLowerCase()) >= 0;
                    });
                    differed.resolve(candidates);
                    return differed.promise;
                };

                scope.onSelectInternal = function (item) {
                    scope.onSelect({ type: item });
                };
            }
        }
    };
});
'use strict';

angular.module('xl.widget').directive('xlWidgetAlertMessage', function () {
    return {
        scope: {
            state: '=',
            options: '='
        },
        transclude: true,
        templateUrl: 'src/xl-widget/xl-widget-alert-message/xl-widget-alert-message.html',
        link: function link(scope, element, attr) {
            scope.$watch('state', function () {
                return scope.internalState = scope.state || 'error';
            });

            scope.showClose = !!attr.onDismiss;

            scope.dismiss = function () {
                scope.$parent.$eval(attr.onDismiss);
            };
        }
    };
});
'use strict';

angular.module('xl.tab-system', []).provider('$uiTabs', function () {

    var showWelcomePageByDefault = true;

    var tabs = {
        null: { template: '' }
    };

    this.tab = function (type, definition) {
        delete definition.locals;
        tabs[type] = angular.extend({}, definition);
        return this;
    };

    this.welcome = function (definition) {
        definition.isWelcome = true;
        if (!_.isUndefined(definition.showWelcomePageByDefault)) {
            showWelcomePageByDefault = definition.showWelcomePageByDefault;
        }
        return this.tab(null, definition);
    };

    var BaseTab = function BaseTab() {
        this.pristine = true;
        this.selected = false;
        this.initialized = false;
        this.volatile = true;
        this.$tabData = {};
    };

    BaseTab.prototype.onClose = ['$q', function ($q) {
        return function () {
            return $q.when();
        };
    }];

    this.onClose = function (handler) {
        BaseTab.prototype.onClose = handler;
    };

    var initTab = function initTab(type, options) {
        options = options || {};
        var tabProp = tabs[type];
        if (!tabProp) {
            return undefined;
        }
        tabProp.type = type;
        delete options.locals;
        return angular.extend(new BaseTab(), tabProp, options);
    };

    function filter(collection, callback) {
        var result = [];
        angular.forEach(collection, function (value, index, collection) {
            if (callback(value, index, collection)) {
                result.push(value);
            }
        });
        return result;
    }

    function remove(array, callback) {
        var index = -1,
            length = array ? array.length : 0,
            result = [];

        while (++index < length) {
            var value = array[index];
            if (callback(value, index, array)) {
                result.push(value);
                [].splice.call(array, index--, 1);
                length--;
            }
        }
        return result;
    }

    this.$get = function ($rootScope, $injector, $sce, $http, $q, $templateCache) {
        /**
         * Basically TABS.arr & TABS.map contain the same tabs objects
         */
        var TABS = {
            arr: [],
            map: {},
            history: []
        };

        /**
         * Return a tab object
         * @param id tab id
         * @returns {tab}
         */
        var getTab = function getTab(id) {
            return TABS.map[id];
        };

        /**
         * Select an existing tab
         * @param id tab id
         * @returns {tab}
         */
        var selectTab = function selectTab(id) {
            return updateTabs(id);
        };

        /*
         Private
         */
        var cleanTabScope = function cleanTabScope(tab) {
            if (tab.scope) {
                tab.scope.$destroy();
                tab.scope = null;
            }
        };

        /**
         * Add a new tab
         * @param type type of a tab described with $uiTabsProvider
         * @param options init tab options (title, disabled)
         * @param id (optional) tab's unique id. If 'id' exists, tab content of this tab will be replaced
         */
        var addTab = function addTab(type, options, id) {
            var newTab = initTab(type, options);

            if (!newTab) {
                throw new Error('Unknown tab type: ' + type);
            }

            newTab.$$id = id || Math.random().toString(16).substr(2);

            if (!!getTab(newTab.$$id)) {
                // Replace tab
                var currentTab = getTab(newTab.$$id);
                cleanTabScope(currentTab);
                angular.copy(newTab, currentTab);
            } else {
                // Add Tab
                if (type !== null) {
                    TABS.arr.push(newTab);
                }
                TABS.map[newTab.$$id] = newTab;
            }
            selectTab(newTab.$$id);
            return newTab.$$id;
        };

        var removeTabIntern = function removeTabIntern(tab) {
            remove(TABS.history, function (tabId) {
                return tabId === tab.$$id;
            });

            if (tab.selected && TABS.history.length > 0) {
                selectTab(TABS.history[TABS.history.length - 1]);
            }

            cleanTabScope(tab);
            TABS.arr.splice(TABS.arr.indexOf(tab), 1);
            delete TABS.map[tab.$$id];
            $rootScope.$broadcast('$tabRemoveSuccess', tab);
        };

        var removeTab = function removeTab(id) {
            var tab = getTab(id);

            return $q.when(tab).then(function () {
                if (tab.onClose) {
                    var fn = angular.isString(tab.onClose) ? $injector.get(tab.onClose) : $injector.invoke(tab.onClose);
                    return fn(tab);
                }
            }). // after tab close resolved
            then(function () {
                return removeTabIntern(tab);
            });
        };

        var getActiveTab = function getActiveTab() {
            var selectedTabs = filter(TABS.map, function (tab) {
                return tab.selected;
            });

            if (selectedTabs.length === 1) {
                return selectedTabs[0];
            } else if (selectedTabs.length === 0) {
                return undefined;
            } else if (selectedTabs.length > 1) {
                throw new Error('There should not be more than one selected tab at a time.');
            }
        };

        /**
         * Return all tabs
         * @returns {*}
         */
        var getTabs = function getTabs() {
            return TABS.arr;
        };

        /*
         Private
         */
        function updateTabs(nextTabId) {
            var next = getTab(nextTabId),
                last = getActiveTab();

            if (next && last && next.$$id === last.$$id) {
                $rootScope.$broadcast('$tabUpdate', last);
            } else if (next || last) {
                $rootScope.$broadcast('$tabChangeStart', next, last);

                $q.when(next).then(function () {
                    if (next) {
                        if (next.locals) {
                            // when switch
                            return $q.all(next.locals);
                        }

                        var locals = angular.extend({}, next.resolve),
                            template,
                            templateUrl;

                        angular.forEach(locals, function (value, key) {
                            locals[key] = angular.isString(value) ? $injector.get(value) : $injector.invoke(value);
                        });

                        if (angular.isDefined(template = next.template)) {
                            if (angular.isFunction(template)) {
                                template = template(next.params);
                            }
                        } else if (angular.isDefined(templateUrl = next.templateUrl)) {
                            if (angular.isFunction(templateUrl)) {
                                templateUrl = templateUrl(next.params);
                            }
                            templateUrl = $sce.getTrustedResourceUrl(templateUrl);
                            if (angular.isDefined(templateUrl)) {
                                next.loadedTemplateUrl = templateUrl;
                                template = $http.get(templateUrl, { cache: $templateCache }).then(function (response) {
                                    return response.data;
                                });
                            }
                        }
                        if (angular.isDefined(template)) {
                            locals.$template = template;
                        }
                        return $q.all(locals);
                    }
                }). // after tab change
                then(function (locals) {
                    angular.forEach(TABS.map, function (tab) {
                        tab.selected = false;
                    });

                    if (next) {
                        next.locals = locals;
                        next.selected = true;
                    }
                    remove(TABS.history, function (tabId) {
                        return tabId === nextTabId;
                    });
                    TABS.history.push(nextTabId);

                    $rootScope.$broadcast('$tabChangeSuccess', next, last);
                }, function (error) {
                    $rootScope.$broadcast('$tabChangeError', next, last, error);
                });
            }
            return next;
        }

        // Add welcome tab
        if (showWelcomePageByDefault) {
            addTab(null);
        }

        /**
         * Public API
         */
        return {
            getTabs: getTabs,
            addTab: addTab,
            getTab: getTab,
            removeTab: removeTab,
            selectTab: selectTab,
            getActiveTab: getActiveTab
        };
    };
}).directive('tabView', function ($uiTabs, $anchorScroll, $animate) {
    return {
        restrict: 'ECA',
        terminal: true,
        priority: 400,
        transclude: 'element',
        link: function link(scope, $element, attr, ctrl, $transclude) {
            var currentScope,
                currentElement,
                previousElement,
                autoScrollExp = attr.autoscroll,
                onloadExp = attr.onload || '',
                elems = {};

            function remove(event, tab) {
                if (tab.selected === false) {
                    var elem = elems[tab.$$id];
                    if (elem) {
                        delete elems[tab.$$id];
                        elem.remove();
                        elem = null;
                    }
                }
            }

            function cleanupLastView() {
                var id = currentElement && currentElement.data('$$id');
                var tab = $uiTabs.getTab(id);
                if (previousElement) {
                    previousElement.remove();
                    previousElement = null;
                }
                if (currentScope && _.isUndefined(tab)) {
                    currentScope.$destroy();
                    currentScope = null;
                }
                if (currentElement) {
                    if (tab) {
                        $animate.addClass(currentElement, 'ng-hide');
                        previousElement = null;
                    } else {
                        $animate.leave(currentElement).then(function () {
                            previousElement = null;
                        });
                        previousElement = currentElement;
                        currentElement = null;
                    }
                }
            }

            function update(event, currentTab) {
                var elem = elems[currentTab.$$id];
                if (elem) {
                    $animate.removeClass(elem, 'ng-hide');
                    cleanupLastView();
                    currentElement = elem;
                    return;
                }

                var locals = currentTab && currentTab.locals,
                    template = locals && locals.$template;

                if (angular.isDefined(template)) {
                    var newScope = scope.$new();
                    newScope.$$tabId = currentTab.$$id;

                    if (currentTab.volatile !== false) {
                        newScope.$tabData = currentTab.$tabData;
                    }

                    newScope.$setTabPristine = function () {
                        currentTab.pristine = false;
                    };

                    // Note: This will also link all children of tab-view that were contained in the original
                    // html. If that content contains controllers, ... they could pollute/change the scope.
                    // However, using ng-view on an element with additional content does not make sense...
                    // Note: We can't remove them in the cloneAttchFn of $transclude as that
                    // function is called before linking the content, which would apply child
                    // directives to non existing elements.
                    var clone = $transclude(newScope, function (clone) {
                        $animate.enter(clone, null, currentElement || $element).then(function onNgViewEnter() {
                            if (angular.isDefined(autoScrollExp) && (!autoScrollExp || scope.$eval(autoScrollExp))) {
                                $anchorScroll();
                            }
                        });

                        cleanupLastView();
                    });

                    currentElement = clone;

                    currentScope = newScope;
                    if (currentTab.volatile === false) {
                        currentElement.data('$$id', currentTab.$$id);
                        elems[currentTab.$$id] = currentElement;
                        currentTab.scope = newScope;
                    }

                    newScope.$emit('$tabContentLoaded');
                    newScope.$eval(onloadExp);
                } else {
                    cleanupLastView();
                }
            }

            scope.$on('$tabChangeSuccess', update);
            scope.$on('$tabRemoveSuccess', remove);
        }
    };
}).directive('tabView', function ($compile, $controller, $uiTabs) {
    return {
        restrict: 'ECA',
        priority: -400,
        link: function link(scope, $element) {
            var current = $uiTabs.getActiveTab(),
                locals = current.locals;

            $element.html(locals.$template);
            $element.addClass('xl-tab-system-view');

            var link = $compile($element.contents());

            if (current.controller) {
                locals.$scope = scope;
                var controller = $controller(current.controller, locals);
                if (current.controllerAs) {
                    scope[current.controllerAs] = controller;
                }
                $element.data('$ngControllerController', controller);
                $element.children().data('$ngControllerController', controller);
            }

            scope.$$currentTabId = current.$$id;

            link(scope);
        },
        controller: function controller($scope) {
            this.$$getCurrentTabId = function () {
                return $scope.$$currentTabId;
            };
        }
    };
}).directive('closeUiTab', function ($uiTabs) {
    return {
        restrict: 'ECA',
        priority: -400,
        require: '^tabView',
        link: function link(scope, $element, attr, tabViewCtrl) {
            $element.on('click', function () {
                $uiTabs.removeTab(tabViewCtrl.$$getCurrentTabId());
            });
        }
    };
}).directive('tabHeader', function ($interval, $rootScope, $uiTabs, $window) {
    return {
        restrict: 'ECA',
        priority: -400,
        templateUrl: 'src/xl-tab-system/xl-tab-system-header.html',
        scope: {
            showPlusTabType: '=',
            plusTabType: '=',
            showCloseTabIcon: '=',
            createTabIfLastClosed: '=',
            showOnlyActiveTab: '='
        }, link: function link(scope, elem, attr) {

            scope.safeApply = function (fn) {
                var phase = this.$root.$$phase;
                if (phase == '$apply' || phase == '$digest') {
                    if (fn && angular.isFunction(fn)) {
                        fn();
                    }
                } else {
                    this.$apply(fn);
                }
            };

            function debounce(func, wait, immediate) {
                var args, result, thisArg;

                var cancelId = void 0;

                function delayed() {
                    cancelId = null;
                    if (!immediate) {
                        result = func.apply(thisArg, args);
                    }
                }

                return function () {
                    var isImmediate = immediate && !cancelId;
                    args = arguments;
                    thisArg = this;

                    $interval.cancel(cancelId);
                    cancelId = $interval(delayed, wait, 1);

                    if (isImmediate) {
                        result = func.apply(thisArg, args);
                    }
                    return result;
                };
            }

            var container = elem.find('ul.ui-tab-header-container');
            var wrapper = elem.find('div.ui-tab-header-wrapper');
            var lastSelectedIndex;

            scope.tabs = $uiTabs.getTabs();

            scope.isShowPlusTab = function () {
                return !_.isUndefined(scope.showPlusTabType);
            };

            scope.isShowCloseTabIcon = function () {
                // by default show icon
                return scope.showCloseTabIcon !== false;
            };

            scope.addTab = function () {
                $uiTabs.addTab(scope.plusTabType);
            };

            var scrollToTab = function scrollToTab(index) {
                var left = void 0;
                if (container.outerWidth() + container.position().left < wrapper.innerWidth()) {
                    // Trim space in the right (when deletion or window resize)
                    left = Math.min(wrapper.innerWidth() - container.outerWidth(), 0);
                }

                scope.showTabMenuHandler = wrapper.innerWidth() < container.outerWidth();

                if (!_.isUndefined(index)) {
                    var li = elem.find('li.ui-tab-header-item:nth-child(' + (index + 1) + ')');
                    var leftOffset = container.position().left;

                    if (leftOffset + li.position().left < 0) {
                        // Scroll to active tab at left
                        left = -li.position().left;
                    } else {
                        // Scroll to active tab at right
                        var liOffset = li.position().left + li.outerWidth() + leftOffset;
                        if (liOffset > wrapper.innerWidth()) {
                            left = wrapper.innerWidth() + leftOffset - liOffset;
                        }
                    }
                }

                if (!_.isUndefined(left)) {
                    container.css({ left: left });
                }

                lastSelectedIndex = index;
            };

            scope.selectTab = function (tab, index) {
                $uiTabs.selectTab(tab.$$id);
                scrollToTab(index);
            };

            scope.$watch('showTabMenuHandler', function (visible) {
                angular.element('.ui-tab-add-tab').css('right', visible ? '24px' : 0);
            });

            scope.closeTab = function (tab) {
                $uiTabs.removeTab(tab.$$id);
            };

            scope.$watchCollection('tabs', function (tabs) {
                $interval(function () {
                    var index = tabs.indexOf($uiTabs.getActiveTab());
                    if (index !== -1) {
                        scrollToTab(index);
                    }
                    if (tabs.length === 0 && scope.createTabIfLastClosed) {
                        scope.addTab();
                    }
                }, 1, 1);
            });

            var w = angular.element($window);
            w.bind('resize', debounce(function (event) {
                scope.safeApply(scrollToTab(lastSelectedIndex));
            }, 200));
        }
    };
}).constant('dropdownConfig', {
    openClass: 'open'
}).service('dropdownService', ['$document', function ($document) {
    var openScope = null;

    var closeDropdown = function closeDropdown(evt) {
        // This method may still be called during the same mouse event that
        // unbound this event handler. So check openScope before proceeding.
        if (!openScope) {
            return;
        }

        var toggleElement = openScope.getToggleElement();
        if (evt && toggleElement && toggleElement[0].contains(evt.target)) {
            return;
        }

        openScope.$apply(function () {
            openScope.isOpen = false;
        });
    };

    var escapeKeyBind = function escapeKeyBind(evt) {
        if (evt.which === 27) {
            openScope.focusToggleElement();
            closeDropdown();
        }
    };

    this.open = function (dropdownScope) {
        if (!openScope) {
            $document.bind('click', closeDropdown);
            $document.bind('keydown', escapeKeyBind);
        }

        if (openScope && openScope !== dropdownScope) {
            openScope.isOpen = false;
        }

        openScope = dropdownScope;
    };

    this.close = function (dropdownScope) {
        if (openScope === dropdownScope) {
            openScope = null;
            $document.unbind('click', closeDropdown);
            $document.unbind('keydown', escapeKeyBind);
        }
    };
}]).controller('DropdownController', ['$scope', '$attrs', '$parse', 'dropdownConfig', 'dropdownService', '$animate', function ($scope, $attrs, $parse, dropdownConfig, dropdownService, $animate) {
    var vm = this;
    var scope = $scope.$new(),
        // create a child scope so we are not polluting original one
    openClass = dropdownConfig.openClass,
        getIsOpen,
        setIsOpen = angular.noop,
        toggleInvoker = $attrs.onToggle ? $parse($attrs.onToggle) : angular.noop;

    vm.init = function (element) {
        vm.$element = element;

        if ($attrs.isOpen) {
            getIsOpen = $parse($attrs.isOpen);
            setIsOpen = getIsOpen.assign;

            $scope.$watch(getIsOpen, function (value) {
                scope.isOpen = !!value;
            });
        }
    };

    vm.toggle = function (open) {
        scope.isOpen = arguments.length ? !!open : !scope.isOpen;
        return scope.isOpen;
    };

    // Allow other directives to watch status
    vm.isOpen = function () {
        return scope.isOpen;
    };

    scope.getToggleElement = function () {
        return vm.toggleElement;
    };

    scope.focusToggleElement = function () {
        if (vm.toggleElement) {
            vm.toggleElement[0].focus();
        }
    };

    scope.$watch('isOpen', function (isOpen, wasOpen) {
        $animate[isOpen ? 'addClass' : 'removeClass'](vm.$element, openClass);

        if (isOpen) {
            scope.focusToggleElement();
            dropdownService.open(scope);
        } else {
            dropdownService.close(scope);
        }

        setIsOpen($scope, isOpen);
        if (angular.isDefined(isOpen) && isOpen !== wasOpen) {
            toggleInvoker($scope, { open: !!isOpen });
        }
    });

    $scope.$on('$locationChangeSuccess', function () {
        scope.isOpen = false;
    });

    $scope.$on('$destroy', function () {
        scope.$destroy();
    });
}]).directive('uiTabMenuDropdown', function () {
    return {
        controller: 'DropdownController',
        link: function link(scope, element, attrs, dropdownCtrl) {
            dropdownCtrl.init(element);
        }
    };
}).directive('uiTabMenuDropdownToggle', function () {
    return {
        require: '?^uiTabMenuDropdown',
        priority: -500,
        link: function link(scope, element, attrs, dropdownCtrl) {
            if (!dropdownCtrl) {
                return;
            }

            dropdownCtrl.toggleElement = element;

            var toggleDropdown = function toggleDropdown(event) {
                event.preventDefault();

                if (!element.hasClass('disabled') && !attrs.disabled) {
                    scope.$apply(function () {
                        dropdownCtrl.toggle();
                    });
                }
            };

            element.bind('click', toggleDropdown);

            // WAI-ARIA
            element.attr({ 'aria-haspopup': true, 'aria-expanded': false });
            scope.$watch(dropdownCtrl.isOpen, function (isOpen) {
                element.attr('aria-expanded', !!isOpen);
            });

            scope.$on('$destroy', function () {
                element.unbind('click', toggleDropdown);
            });
        }
    };
});
'use strict';

angular.module('xl.form', ['xl.dip', 'xl.widget', 'ui.sortable']);
'use strict';

angular.module('xl.form').directive('xlDipDynamicForm', function () {
    return {
        restrict: 'A',
        scope: {
            ngModel: '=',
            metadata: '=',
            handlers: '=',
            displayMode: '=',
            form: '=formModel',
            collapsed: '=',
            objLabel: '@'
        },
        templateUrl: 'src/xl-form/xl-dip-dynamic-form/xl-dip-dynamic-form.html',
        link: function link(scope) {
            if (!scope.objLabel) {
                scope.objLabel = 'id';
            }
        }
    };
});
'use strict';

angular.module('xl.form').factory('xlDipDynamicFormService', function () {
    var groupByCategory = function groupByCategory(dynamicFormProperties, collapsed) {
        // we update the properties without category with default 'Common' category.
        var updatedFormProperties = _.map(dynamicFormProperties, function (prop) {
            if (!prop.category) {
                prop.category = 'Common';
            }
            return prop;
        });

        var groupedProperties = _.groupBy(updatedFormProperties, 'category');
        var arrayOfGroupedProperties = _.map(groupedProperties, function (group) {
            var isOpen = void 0;
            if (collapsed === 'auto') {
                var required = _.filter(group, { required: true });
                var hidden = _.filter(required, { hidden: true });
                isOpen = required - hidden !== 0;
            } else {
                isOpen = !_.isUndefined(collapsed) ? collapsed : true;
            }
            return { category: group[0].category, properties: group, isOpen: isOpen };
        });

        return _.sortBy(arrayOfGroupedProperties, function (group) {
            return group.category === 'Common' ? '!' : group.category.toLowerCase();
        });
    };

    var fileSelectorProperties = function fileSelectorProperties(metadata, displayMode) {
        if (displayMode !== 'creation' || _.isUndefined(displayMode)) {
            return undefined;
        }

        var hasFileSelector = _.includes(metadata.interfaces, 'udm.SourceArtifact');
        if (hasFileSelector) {
            return {
                label: 'Artifact data',
                description: 'The data attached to this ' + metadata.type,
                kind: '__UPLOAD_FILE__',
                name: '__upload_file__',
                required: true
            };
        }
        return undefined;
    };

    return {
        groupByCategory: groupByCategory,
        fileSelectorProperties: fileSelectorProperties
    };
});
'use strict';

angular.module('xl.form').directive('xlDipDynamicFormInner', function (xlDipDynamicFormService, $compile) {
    return {
        restrict: 'A',
        scope: {
            ngModel: '=',
            metadata: '=',
            handlers: '=',
            displayMode: '=',
            collapsed: '=',
            objLabel: '@'
        },
        templateUrl: 'src/xl-form/xl-dip-dynamic-form/xl-dip-dynamic-form-inner.html',
        compile: function compile(el) {
            var contents = el.contents().remove();
            var compiled;
            return function (scope, el, attr, ctrl) {
                if (!compiled) {
                    compiled = $compile(contents);
                }

                compiled(scope, function (clone) {
                    el.append(clone);
                });

                var isHidden = function isHidden(p) {
                    return _.isUndefined(p.hidden) || p.hidden === false;
                };

                var makeRequiredFalseIfAsContainment = function makeRequiredFalseIfAsContainment(p) {
                    p.required = p.asContainment ? false : p.required;
                    return p;
                };

                var _scope = scope;
                if (!scope.metadata) {
                    throw new Error('xlDipDynamicForm: metadata has to be defined');
                }
                scope.metadataClone = {};

                var prepareGroupedProperties = function prepareGroupedProperties() {
                    _scope.metadataClone = angular.copy(_scope.metadata);
                    _scope.metadataClone.properties = _(_scope.metadataClone.properties).filter(isHidden).map(makeRequiredFalseIfAsContainment).value();
                    _scope.groupedProperties = xlDipDynamicFormService.groupByCategory(_scope.metadataClone.properties, _scope.collapsed);
                };

                scope.$watch('metadata', prepareGroupedProperties);
                scope.$watch('displayMode', prepareGroupedProperties);

                scope.getLocalDisplayMode = function (asContainment) {
                    return asContainment ? 'visual' : scope.displayMode;
                };

                scope.getEntry = function (name) {
                    return name + '.entry';
                };
            };
        },
        controller: function controller($scope) {
            this.getModel = function () {
                return $scope.ngModel;
            };
        }
    };
});
'use strict';

angular.module('xl.form').directive('xlDipDeployableCi', function () {
    return {
        restrict: 'A',
        scope: {
            ngModel: '=',
            types: '=',
            handlers: '=',
            displayMode: '=',
            deployableCiForm: '=formModel',
            isSourceArtifact: '='
        },
        templateUrl: 'src/xl-form/xl-dip-deployable-ci/xl-dip-deployable-ci.html',
        link: _.noop,
        controller: function controller($scope, xlDipDeployableCiService) {
            $scope.metadata = {};
            if (!$scope.types || !angular.isArray($scope.types)) {
                throw new Error('Error: "types" should be defined and be an array of types');
            }

            if (!$scope.ngModel) {
                $scope.ngModel = {
                    form: {}
                };
            }
            if (!$scope.ngModel.form) {
                $scope.ngModel.form = {};
            }

            var filteredTypes = xlDipDeployableCiService.filterTypes($scope.types, $scope.isSourceArtifact);
            $scope.typeNames = _.map(filteredTypes, 'type');

            if ($scope.ngModel.type) {
                $scope.metadata = _.find(filteredTypes, { type: $scope.ngModel.type });
            }

            if ($scope.displayMode === 'creation') {
                $scope.$watch('ngModel.form.__upload_file__', function (fileWrapper, oldFileWrapper) {
                    if (!fileWrapper || !fileWrapper.file) {
                        $scope.ngModel.name = '';
                        return;
                    }

                    if (oldFileWrapper && oldFileWrapper.file && $scope.ngModel.name === oldFileWrapper.file.name || $scope.ngModel.name === '') {
                        $scope.ngModel.name = fileWrapper.file.name;
                    }
                }, true);
            }

            $scope.typeSelected = function (typeName) {
                $scope.metadata = _.find(filteredTypes, { type: typeName });

                var metadataProperties = $scope.metadata.properties;

                // Filter out dynamic 'Remote file Uri' cause we have it as static field for all types derived from file.File.
                for (var i = metadataProperties.length - 1; i >= 0; i--) {
                    if (metadataProperties[i].name === 'fileUri') {
                        metadataProperties.splice(i, 1);
                    }
                }
                $scope.metadata.properties = _.map(metadataProperties, function (property) {
                    if (property.asContainment) {
                        property.required = false;
                    }
                    return property;
                });
            };

            $scope.$watch('ngModel.type', function (type) {
                if (_.isUndefined(type)) {
                    $scope.metadata = {};
                }
            });
        }
    };
});
'use strict';

angular.module('xl.form').factory('xlDipDeployableCiService', function () {
    var filterTypes = function filterTypes(types, isSourceArtifact) {
        if (_.isUndefined(isSourceArtifact)) {
            return types;
        }
        return _.filter(types, function (metadata) {
            return !(_.includes(metadata.interfaces, 'udm.SourceArtifact') ^ !!isSourceArtifact);
        });
    };

    return {
        filterTypes: filterTypes
    };
});
'use strict';

angular.module('xl.dip', ['xl.widget', 'ui.sortable']);
'use strict';

angular.module('xl.dip').directive('xlDipString', function () {
    return {
        restrict: 'A',
        scope: {
            ngModel: '=',
            metadata: '=',
            displayMode: '=',
            isDisabled: '='
        },
        templateUrl: 'src/xl-dip/xl-dip-string/xl-dip-string.html',
        link: function link(scope) {
            scope.options = {};
            scope.$watch('metadata', function () {
                if (scope.metadata) {
                    scope.options.size = scope.metadata.size;
                    scope.options.required = scope.metadata.required;
                    scope.options.name = scope.metadata.name;
                    scope.options.label = scope.metadata.label;
                    scope.options.description = scope.metadata.description;
                    scope.options.password = scope.metadata.password;
                    scope.options['default'] = scope.metadata['default'];
                    scope.options.validator = scope.metadata.validator;
                }
            });
        }
    };
});
'use strict';

angular.module('xl.dip').directive('xlDipSet', function (xlWidgetUtils, $timeout) {

    function deduplicate(list, field) {
        if (!list) return list;
        return _.uniqBy(list, function (item) {
            return field ? item[field] : item.title;
        });
    }

    function onModify($scope) {
        $timeout(function () {
            if (!$scope.readOnly && $scope.onChange) {
                $scope.onChange();
            }
        });
    }

    return {
        restrict: 'A',
        scope: {
            ngModel: '=',
            metadata: '=',
            displayMode: '=?',
            readOnly: '=',
            onChange: '&'
        },
        templateUrl: function templateUrl(element, attrs) {
            return attrs.templateUrl || 'src/xl-dip/xl-dip-set/xl-dip-set.html';
        },
        require: ['?^form', 'ngModel'],
        link: function link(scope, elem, attrs, ctrls) {
            var parentForm = ctrls[0];
            var ngModelCtrl = ctrls[1];
            scope.isRequired = scope.metadata.required;

            if (!scope.ngModel || !angular.isArray(scope.ngModel)) {
                scope.ngModel = [];
            }

            scope.field = scope.metadata ? scope.metadata.field : undefined;

            if (!scope.field && !scope.readOnly) {
                scope.$watch('$set', function (value) {
                    if (!value) return;
                    // we do not want to loose scope.ngModel reference
                    var v = _.map(scope.$set, 'title');
                    angular.copy(v, scope.ngModel);
                    ngModelCtrl.$setValidity('required', !xlWidgetUtils.hasRequiredError(scope.ngModel, scope.isRequired));
                }, true);
            }

            scope.remove = function (item) {
                var index = _.indexOf(scope.$set, item);
                if (index > -1) {
                    scope.$set.splice(index, 1);
                    if (parentForm) {
                        parentForm.$setDirty();
                    }

                    onModify(scope);
                }
            };
        },
        controller: function controller($scope) {
            $scope.newItem = '';

            $scope.$watch('ngModel', function (newValue) {
                if (!$scope.field) {
                    $scope.$set = deduplicate(_.map($scope.ngModel, function (item) {
                        return { title: item };
                    }));
                } else {
                    $scope.$set = deduplicate($scope.ngModel, $scope.field);
                }
            });

            $scope.$watch('readOnly', function () {
                var readOnly = $scope.readOnly;
                $scope.displayMode = readOnly ? 'visual' : 'creation';
            }, true);

            $scope.$watch('metadata', function () {
                if ($scope.metadata) {
                    var defaultValue = $scope.metadata['default'];
                    if (!_.isNil(defaultValue) && !_.isNil(defaultValue)) {
                        defaultValue = defaultValue.substring(1, defaultValue.length - 1);
                        _.forEach(defaultValue.split(','), function (value) {
                            var item = {};
                            item[$scope.field ? $scope.field : 'title'] = value;
                            $scope.$set.push(item);
                        });
                        $scope.$set = deduplicate($scope.$set);
                    }
                }
            });

            $scope.addItem = function ($event) {
                if ($event && $event.isDefaultPrevented()) return;
                //For Firefox and IE8+
                //!$event For add button
                if (!$event || $event.keyCode === 13 || $event.keyCode === 0 || $event.which === 13 || $event.which === 0) {
                    if ($event) {
                        $event.preventDefault();
                    }
                    $scope.error = '';
                    if ($scope.newItem === '') return;
                    var existingItemTitles = _.map($scope.$set, function (item) {
                        return $scope.field ? item[$scope.field] : item.title;
                    });
                    if (existingItemTitles.indexOf($scope.newItem) >= 0) {
                        $scope.error = 'Value "' + $scope.newItem + '" is already used.';
                        return;
                    }
                    var item = {};
                    item[$scope.field ? $scope.field : 'title'] = $scope.newItem;

                    $scope.$set.push(item);
                    $scope.newItem = '';

                    onModify($scope);
                }
            };

            this.getScope = function () {
                return $scope;
            };
        }
    };
});
'use strict';

angular.module('xl.dip').directive('xlDipStringSet', function (xlWidgetUtils) {
    return {
        restrict: 'A',
        scope: {
            ngModel: '=',
            metadata: '=',
            displayMode: '='
        },
        require: ['?^form', 'ngModel'],
        templateUrl: 'src/xl-dip/xl-dip-set-of-strings/xl-dip-set-of-strings.html',
        link: function link(scope, elem, attrs, ctrls) {
            var ngModelCtrl = ctrls[1];
            scope.isRequired = scope.metadata.required;
            scope.$watch('ngModel', function (value) {
                if (!value) return;
                ngModelCtrl.$setValidity('required', !xlWidgetUtils.hasRequiredError(scope.ngModel, scope.isRequired));
            }, true);
        },
        controller: function controller($scope) {
            $scope.options = {};
            $scope.selectOptions = {};
            $scope.$watch('metadata', function () {
                if ($scope.metadata) {

                    $scope.options.name = $scope.metadata.name;
                    $scope.options.label = $scope.metadata.label;
                    $scope.options.description = $scope.metadata.description;
                    $scope.options.required = $scope.metadata.required;
                    $scope.selectOptions.allowClear = true;
                    $scope.selectOptions.width = 'resolve';

                    var defaultValue = $scope.metadata['default'];
                    if (!$scope.ngModel && defaultValue !== null) {
                        defaultValue = defaultValue.substring(1, defaultValue.length - 1);
                        $scope.ngModel = defaultValue.split(',');
                    }
                }
            });
        }
    };
});
'use strict';

var _typeof = typeof Symbol === "function" && typeof Symbol.iterator === "symbol" ? function (obj) { return typeof obj; } : function (obj) { return obj && typeof Symbol === "function" && obj.constructor === Symbol ? "symbol" : typeof obj; };

angular.module('xl.dip').directive('xlDipMapStringString', function (xlWidgetUtils, $timeout, $filter) {

    function onModify($scope) {
        $timeout(function () {
            if (!$scope.readOnly && $scope.onChange) {
                $scope.onChange();
            }
        });
    }

    return {
        restrict: 'A',
        scope: {
            ngModel: '=',
            metadata: '=',
            displayMode: '=?',
            entry: '=?',
            readOnly: '=',
            onChange: '&'
        },
        templateUrl: function templateUrl(element, attrs) {
            return attrs.templateUrl || 'src/xl-dip/xl-dip-map-string-string/xl-dip-map-string-string.html';
        },
        require: ['?^form', 'ngModel'],
        link: function link(scope, elem, attrs, ctrls) {
            var parentForm = ctrls[0];
            var ngModelCtrl = ctrls[1];

            var rowHeight = 35;
            scope.isSearching = false;
            scope.isRequired = scope.metadata.required;
            scope.password = scope.metadata.password;
            scope.entry = { key: '', value: '' };
            var minVisibleRow = scope.metadata.minVisibleRow || 8;

            scope.$watch('metadata', function () {
                if (scope.metadata) {
                    scope.isRequired = scope.metadata.required;

                    if (!scope.ngModel) {
                        scope.ngModel = {};
                    }

                    var defaultValue = scope.metadata['default'];
                    if (!_.isNil(defaultValue) && !_.isNil(defaultValue)) {
                        defaultValue = defaultValue.substring(1, defaultValue.length - 1);
                        _.forEach(defaultValue.split(','), function (entry) {
                            var keyValue = entry.split('=');
                            scope.ngModel[keyValue[0]] = keyValue[1];
                        });
                    }
                }
            });

            scope.isEmpty = function () {
                return _.isEmpty(scope.ngModel);
            };

            scope.switchInput = function (event, toNext, mustAdd) {
                var code = event.keyCode || event.which;
                if (code === 13) {
                    event.preventDefault();
                    var element = angular.element(event.target);
                    var inputs = element.closest('tr').find(':input');
                    var index = toNext === true ? 1 : -1;
                    inputs.eq(inputs.index(element) + index).focus();
                    if (mustAdd) {
                        scope.addRow();
                    }
                }
            };

            scope.addRow = function () {
                scope.error = '';
                if (_.has(scope.ngModel, scope.entry.key)) {
                    scope.error = 'Key "' + scope.entry.key + '" is already in use.';
                    return;
                }
                if (scope.entry.key.trim() === '') {
                    return;
                }

                if (angular.isUndefined(scope.ngModel)) {
                    scope.ngModel = {};
                }

                scope.ngModel[scope.entry.key] = scope.entry.value;
                scope.entry.key = '';
                scope.entry.value = '';

                if (parentForm) {
                    parentForm.$setDirty();
                }
                ngModelCtrl.$setValidity('required', !xlWidgetUtils.hasRequiredError(scope.ngModel, scope.isRequired));

                onModify(scope);
            };

            scope.deleteRow = function (key) {
                delete scope.ngModel[key];
                if (parentForm) {
                    parentForm.$setDirty();
                }

                onModify(scope);

                ngModelCtrl.$setValidity('required', !xlWidgetUtils.hasRequiredError(scope.ngModel, scope.isRequired));
            };

            scope.$watch('ngModel', function () {
                ngModelCtrl.$setValidity('required', !xlWidgetUtils.hasRequiredError(scope.ngModel, scope.isRequired));
            });

            scope.clearSearchWord = function () {
                scope.searchWord = '';
            };

            scope.inputValueType = function (password) {
                return password === true ? 'password' : 'text';
            };

            scope.keys = function (obj, searchWord) {
                return _.keys($filter('dipSearch')(obj || {}, searchWord));
            };

            var minHeight = rowHeight * minVisibleRow;

            angular.element('.table-scrollable-container').css('height', minHeight).resizable({
                grid: rowHeight,
                minHeight: minHeight - rowHeight
            });
        },
        controller: function controller($scope) {

            $scope.$watch('readOnly', function () {
                var readOnly = $scope.readOnly;
                $scope.displayMode = readOnly ? 'visual' : 'creation';
            }, true);

            this.setKey = function (key, oldKey) {
                if (key.length === 0 || !key.trim()) {
                    $scope.error = 'A key cannot be empty.';
                    return false;
                }
                if (_.has($scope.ngModel, key)) {
                    $scope.error = 'Key "' + key + '" is already in use.';
                    return false;
                }

                $scope.ngModel[key] = $scope.ngModel[oldKey];
                delete $scope.ngModel[oldKey];

                onModify($scope);

                return true;
            };
            this.setValue = function (value, key) {
                $scope.ngModel[key] = value;

                onModify($scope);
            };
        }
    };
});

angular.module('xl.dip').filter('dipSearch', function () {
    return function (values, searchWord) {
        if (values && searchWord) {
            var _ret = function () {
                var lowerCaseSearchKey = searchWord.toLowerCase();
                var result = {};
                _.forEach(values, function (value, key) {
                    if (_.includes(key.toLowerCase(), lowerCaseSearchKey) || _.includes(value.toString().toLowerCase(), lowerCaseSearchKey)) {
                        result[key] = value;
                    }
                });
                return {
                    v: result
                };
            }();

            if ((typeof _ret === 'undefined' ? 'undefined' : _typeof(_ret)) === "object") return _ret.v;
        }
        return values;
    };
});

angular.module('xl.dip').directive('xlWidgetContenteditableInt', function () {
    return {
        require: '^xlDipMapStringString',
        template: '<span xl-widget-contenteditable="true" data-ng-model="internal" data-password="{{password}}" placeholder="Empty value"></span>',
        scope: {
            ngModel: '=',
            password: '@password'
        },
        link: function link(scope, elm, attrs, ctrl) {
            scope.password = attrs.password;

            var type = attrs.type;
            scope.internal = scope.ngModel[type];
            scope.placeholder = attrs.placeholder;

            scope.$watch('internal', function (value, oldValue) {
                if (value === oldValue || value === scope.ngModel[attrs.type]) {
                    return;
                }

                var valid = true;
                if (attrs.type === 'key') {
                    valid = ctrl.setKey(value, oldValue);
                } else {
                    ctrl.setValue(value, scope.ngModel.key);
                }

                if (valid) {
                    scope.ngModel[attrs.type] = value;
                } else {
                    // rollback
                    scope.internal = scope.ngModel[attrs.type];
                }
            });
        }
    };
});

angular.module('xl.dip').directive('focusMe', function ($timeout, $parse) {
    return {
        link: function link(scope, element, attrs) {
            var model = $parse(attrs.focusMe);
            scope.$watch(model, function (value) {
                if (value === true) {
                    $timeout(function () {
                        element[0].focus();
                    });
                }
            });
        }
    };
});
'use strict';

var _slicedToArray = function () { function sliceIterator(arr, i) { var _arr = []; var _n = true; var _d = false; var _e = undefined; try { for (var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true) { _arr.push(_s.value); if (i && _arr.length === i) break; } } catch (err) { _d = true; _e = err; } finally { try { if (!_n && _i["return"]) _i["return"](); } finally { if (_d) throw _e; } } return _arr; } return function (arr, i) { if (Array.isArray(arr)) { return arr; } else if (Symbol.iterator in Object(arr)) { return sliceIterator(arr, i); } else { throw new TypeError("Invalid attempt to destructure non-iterable instance"); } }; }();

angular.module('xl.dip').directive('xlDipList', function (xlWidgetUtils, $interval) {

    var onModify = function onModify($scope) {
        $interval(function () {
            if (!$scope.readOnly && $scope.onChange) {
                $scope.onChange();
            }
        }, 0, 1);
    };

    return {
        restrict: 'A',
        scope: {
            ngModel: '=',
            metadata: '=',
            displayMode: '=?',
            readOnly: '=',
            onChange: '&'
        },
        templateUrl: function templateUrl(element, attrs) {
            return attrs.templateUrl || 'src/xl-dip/xl-dip-list/xl-dip-list.html';
        },
        require: ['?^form', 'ngModel'],
        link: function link(scope, elem, attrs, ctrls) {
            var _ctrls = _slicedToArray(ctrls, 2);

            var parentForm = _ctrls[0];
            var ngModelCtrl = _ctrls[1];

            scope.isRequired = scope.metadata.required;

            if (!scope.ngModel || !angular.isArray(scope.ngModel)) {
                scope.ngModel = [];
            }

            scope.sortableConfig = {
                tolerance: 'pointer',
                containment: 'parent'
            };

            scope.field = scope.metadata ? scope.metadata.field : undefined;

            if (!scope.field) {
                scope.$watch('$list', function (value, oldValue) {
                    if (!value) return;
                    // we do not want to loose scope.ngModel reference
                    var v = _.map(scope.$list, 'title');
                    angular.copy(v, scope.ngModel);
                    var hasErrors = !xlWidgetUtils.hasRequiredError(scope.ngModel, scope.isRequired);
                    ngModelCtrl.$setValidity('required', hasErrors);
                    var inputController = _.get(parentForm, scope.metadata.name);
                    if (inputController) {
                        inputController.$setValidity('required', hasErrors);
                    }

                    if (oldValue && value && !_.isEqual(_.map(value, 'title'), _.map(oldValue, 'title'))) {
                        onModify(scope);
                    }
                }, true);
            }

            scope.remove = function (item) {
                var index = _.indexOf(scope.$list, item);
                if (index > -1) {
                    scope.$list.splice(index, 1);
                    if (parentForm) {
                        parentForm.$setDirty();
                    }
                }
            };

            scope.inputValid = function () {
                var name = _.get(scope, ['metadata', 'name']);
                return _.result(parentForm, [name, '$valid'], true);
            };
        },
        controller: function controller($scope) {
            $scope.newItem = '';

            $scope.$watch('ngModel', function (newValue) {
                if (!$scope.field) {
                    $scope.$list = _.map(newValue, function (item) {
                        return { title: item };
                    });
                } else {
                    $scope.$list = newValue;
                }
            });

            $scope.$watch('readOnly', function () {
                $scope.displayMode = $scope.readOnly ? 'visual' : 'creation';
            }, true);

            $scope.$watch('metadata', function () {
                if ($scope.metadata) {
                    var defaultValue = $scope.metadata['default'];
                    if (!_.isNil(defaultValue) && !_.isNil(defaultValue)) {
                        defaultValue = defaultValue.substring(1, defaultValue.length - 1);
                        $scope.$list = _.concat($scope.$list, _.map(defaultValue.split(','), function (value) {
                            return { title: value };
                        }));
                    }
                }
            });

            $scope.addItem = function ($event) {
                var addButton = !$event || $event.keyCode === 13 || $event.keyCode === 0 || $event.which === 13 || $event.which === 0;
                if ($scope.inputValid() && addButton) {
                    if ($scope.newItem === '') return;
                    var item;
                    item = {};
                    if ($scope.field) {
                        item[$scope.field] = $scope.newItem;
                    } else {
                        item.title = $scope.newItem;
                    }

                    $scope.$list.push(item);
                    $scope.newItem = '';
                }
            };
        }
    };
});
'use strict';

angular.module('xl.dip').directive('xlDipInteger', function () {
    return {
        restrict: 'A',
        scope: {
            ngModel: '=',
            metadata: '=',
            displayMode: '='
        },
        templateUrl: 'src/xl-dip/xl-dip-integer/xl-dip-integer.html',
        require: 'ngModel',
        link: function link(scope, elem, attrs, ngModelCtrl) {
            scope.$watch('metadata', function () {
                if (scope.metadata) {
                    scope.isRequired = scope.metadata.required;
                }
            });

            if (_.isUndefined(scope.ngModel) && !_.isUndefined(scope.metadata['default']) && !_.isNull(scope.metadata['default'])) {
                scope.ngModel = scope.metadata['default'];
            }

            var previousVal = 0;

            scope.$watch('ngModel', function (newValue) {
                if (newValue === '') {
                    previousVal = 0;
                }
                var arr = String(newValue).split('');
                if (arr.length === 0) return;
                if (arr.length === 1 && arr[0] === '-') return;

                var stringNewValue = String(newValue);
                if (isNaN(newValue)) {
                    scope.ngModel = previousVal;
                } else if (newValue < -2147483648 || newValue > 2147483647) {
                    // -2147483648 > value < 2147483647 constraints for the integer
                    scope.ngModel = previousVal;
                } else if (stringNewValue.indexOf('.') > -1 || stringNewValue.indexOf(',') > -1) {
                    // different regional settings for delimeter
                    scope.ngModel = previousVal;
                } else {
                    scope.ngModel = Number(newValue);
                    previousVal = Number(newValue);
                }
            }, true);

            scope.erase = function () {
                ngModelCtrl.$setViewValue('');
            };
        },
        controller: function controller($scope) {
            // trying to prevent safari/ie bug which opens up a pop DEPL-7854
            $scope.setValue = function ($event) {
                if ($event && $event.isDefaultPrevented()) return;

                if ($event.keyCode === 13 || $event.keyCode === 0 || $event.which === 13 || $event.which === 0) {
                    $event.preventDefault();
                }
            };
        }
    };
});
'use strict';

angular.module('xl.dip').directive('xlDipFileSelector', function () {
    return {
        restrict: 'A',
        scope: {
            ngModel: '=',
            isDisabled: '=',
            metadata: '='
        },
        templateUrl: 'src/xl-dip/xl-dip-file-selector/xl-dip-file-selector.html',
        link: function link(scope) {
            scope.metadataClone = angular.copy(scope.metadata);
            scope.metadataClone.auto = false;
        }
    };
});
'use strict';

angular.module('xl.dip').directive('xlDipEnum', function ($q) {
    return {
        restrict: 'A',
        scope: {
            ngModel: '=',
            metadata: '=',
            displayMode: '='
        },
        templateUrl: 'src/xl-dip/xl-dip-enum/xl-dip-enum.html',
        link: {
            pre: function pre(scope) {
                scope.autocompleteHandlers = {
                    addCandidates: function addCandidates(dummy, options) {
                        var differed = $q.defer();
                        differed.resolve(_.filter(scope.metadata.enumValues, function (item) {
                            return item.toLowerCase().indexOf(options.term.toLowerCase()) >= 0;
                        }));
                        return differed.promise;
                    }
                };
            },
            post: function post(scope) {

                if (scope.metadata.default && _.isUndefined(scope.ngModel)) {
                    scope.ngModel = scope.metadata.default;
                }

                scope.$watch('metadata', function () {
                    if (scope.metadata) {
                        scope.isRequired = scope.metadata.required;
                    }
                });
            }

        }
    };
});
'use strict';

angular.module('xl.dip').directive('xlDipDatetime', function () {
    return {
        restrict: 'A',
        scope: {
            ngModel: '=',
            metadata: '=',
            displayMode: '='
        },
        templateUrl: 'src/xl-dip/xl-dip-datetime/xl-dip-datetime.html',
        controller: function controller($scope) {
            $scope.options = {
                date: {
                    format: 'shortDate'
                },
                time: {
                    format: 'shortTime'
                }
            };
            $scope.$watch('metadata', function () {
                if ($scope.metadata) {

                    $scope.options.date.required = $scope.metadata.required;
                    $scope.options.time.required = $scope.metadata.required;
                    $scope.options.name = $scope.metadata.name;
                    $scope.options.label = $scope.metadata.label;
                    $scope.options.description = $scope.metadata.description;
                    $scope.options['default'] = $scope.metadata['default'];
                    $scope.options.enableTime = true;

                    if (_.isUndefined($scope.ngModel) && !_.isUndefined($scope.metadata['default']) && !_.isNull($scope.metadata['default'])) {
                        $scope.ngModel = $scope.metadata['default'];
                    }
                }
            });
        }
    };
});
'use strict';

angular.module('xl.dip').directive('xlDipSetOfCi', function () {
    return {
        restrict: 'A',
        scope: {
            ngModel: '=',
            metadata: '=',
            displayMode: '=',
            handlers: '=',
            objLabel: '@',
            newItemSuffix: '@',
            openOnClick: '@',
            isDisabled: '=',
            isInputVisible: '=?',
            isFilterVisible: '=?',
            emptyPlaceholder: '@'
        },
        template: '<div class="xl-components xl-dip-set-of-ci" xl-dip-collection-of-ci ng-model="ngModel" metadata="metadata" display-mode="displayMode" handlers="handlers" obj-label="{{objLabel}}" new-item-suffix="{{newItemSuffix}}" open-on-click="{{openOnClick}}" is-disabled="isDisabled" is-unique="true" is-sortable="false" is-input-visible="isInputVisible" is-filter-visible="isFilterVisible" empty-placeholder="{{emptyPlaceholder}}"></div>'
    };
});
'use strict';

angular.module('xl.dip').directive('xlDipListOfCi', function () {
    return {
        restrict: 'A',
        scope: {
            ngModel: '=',
            metadata: '=',
            displayMode: '=',
            handlers: '=',
            objLabel: '@',
            newItemSuffix: '@',
            openOnClick: '@',
            isDisabled: '=',
            isInputVisible: '=?',
            isFilterVisible: '=?',
            emptyPlaceholder: '@'
        },
        template: '<div class="xl-components xl-dip-list-of-ci" xl-dip-collection-of-ci ng-model="ngModel" metadata="metadata" display-mode="displayMode" handlers="handlers" obj-label="{{objLabel}}" new-item-suffix="{{newItemSuffix}}" open-on-click="{{openOnClick}}" is-disabled="isDisabled" is-unique="false" is-sortable="true" is-input-visible="isInputVisible" is-filter-visible="isFilterVisible" empty-placeholder="{{emptyPlaceholder}}"></div>'
    };
});
'use strict';

angular.module('xl.dip').directive('xlDipCollectionOfCi', function ($timeout, $q, xlWidgetUtils) {
    return {
        restrict: 'A',
        scope: {
            ngModel: '=',
            metadata: '=',
            displayMode: '=',
            handlersInt: '=handlers',
            objLabel: '@',
            newItemSuffix: '@',
            openOnClick: '@',
            isSortable: '&',
            isUnique: '&',
            isDisabled: '=',
            isInputVisible: '=?',
            isFilterVisible: '=?',
            emptyPlaceholder: '@'
        },
        templateUrl: 'src/xl-dip/xl-dip-collection-of-ci/xl-dip-collection-of-ci.html',
        require: ['?^form', 'ngModel'],
        link: {
            pre: function pre(scope) {
                scope.search = {};
                scope.handlers = angular.copy(scope.handlersInt);
                scope.isInputVisible = angular.isDefined(scope.isInputVisible) ? scope.isInputVisible : true;
                scope.isFilterVisible = angular.isDefined(scope.isFilterVisible) ? scope.isFilterVisible : false;
                scope.handlers.onSelect = function (value) {
                    scope.addItem(value);
                };
                var oldNewItem = scope.handlers.onNewItem;
                if (oldNewItem) {
                    scope.handlers.onNewItem = function (value) {
                        $timeout(function () {
                            scope.binding = { newItem: {} };
                        });
                        oldNewItem(value);
                    };
                }
            },
            post: function post(scope, elem, attrs, ctrls) {
                scope.objLabel = scope.objLabel || 'id';

                var ERRORS = {
                    alreadyExists: 'Item already selected'
                };

                var parentForm = ctrls[0];
                var ngModelCtrl = ctrls[1];

                if (!scope.ngModel || !angular.isArray(scope.ngModel)) {
                    scope.ngModel = undefined;
                }
                scope.$list = scope.ngModel;
                scope.isRequired = scope.metadata.required;

                scope.remove = function (item) {
                    var index = _.indexOf(scope.$list, item);
                    if (index > -1) {
                        scope.$list.splice(index, 1);
                        scope.error = undefined;
                        if (parentForm) {
                            parentForm.$setDirty();
                        }
                    }
                    ngModelCtrl.$setValidity('required', !xlWidgetUtils.hasRequiredError(scope.ngModel, scope.isRequired));
                };

                var addNewItem = function addNewItem(value) {
                    if (_.isUndefined(scope.$list)) {
                        scope.ngModel = [];
                        scope.$list = scope.ngModel;
                    }
                    scope.$list.push(angular.copy(value));
                    $timeout(function () {
                        scope.binding.newItem = {};
                    });
                };

                scope.addItem = function (value) {
                    if (scope.binding.newItem === '') return;

                    if (scope.isUnique()) {
                        var exists = _.find(scope.$list, function (item) {
                            return value[scope.objLabel] === item[scope.objLabel];
                        });

                        if (!exists) {
                            addNewItem(value);
                        } else {
                            $timeout(function () {
                                scope.error = ERRORS.alreadyExists;
                            });
                        }
                    } else {
                        addNewItem(value);
                    }
                    ngModelCtrl.$setValidity('required', !xlWidgetUtils.hasRequiredError(scope.ngModel, scope.isRequired));
                };

                scope.onDeleteItem = function (item) {
                    if (scope.handlers.onDelete && angular.isFunction(scope.handlers.onDelete)) {
                        var rez = scope.handlers.onDelete(item);
                        $q.when(rez).then(function () {
                            scope.remove(item);
                        });
                    } else if (scope.handlers.onDelete === true) {
                        scope.remove(item);
                    }
                };

                scope.$watch('ngModel', function () {
                    ngModelCtrl.$setValidity('required', !xlWidgetUtils.hasRequiredError(scope.ngModel, scope.isRequired));
                });
            }
        },
        controller: function controller($scope) {
            $scope.binding = { newItem: '' };
            $scope.$watch('binding.newItem', function () {
                $scope.error = undefined;
            });

            $scope.eraseError = function () {
                $scope.error = undefined;
            };

            $scope.alertOptions = {
                onDismiss: function onDismiss() {
                    $scope.error = undefined;
                }
            };

            this.hasRequiredError = function () {
                if ($scope.hasRequiredError) return $scope.hasRequiredError($scope.ngModel, $scope.isRequired);
            };

            this.getScope = function () {
                return $scope;
            };
        }
    };
});
'use strict';

angular.module('xl.dip').directive('xlDipCi', function () {
    return {
        restrict: 'A',
        scope: {
            ngModel: '=',
            metadata: '=',
            handlers: '=',
            options: '=',
            isDisabled: '=',
            displayMode: '=',
            objLabel: '@'
        },
        templateUrl: 'src/xl-dip/xl-dip-ci/xl-dip-ci.html',
        link: {
            pre: function pre(scope) {
                if (scope.displayMode === 'visual') {
                    var noopHandlers = { addCandidates: angular.noop };
                    if (_.isUndefined(scope.handlers)) {
                        scope.handlers = noopHandlers;
                    } else if (_.isUndefined(scope.handlers.addCandidates)) {
                        scope.handlers.addCandidates = noopHandlers.addCandidates;
                    }
                }

                scope.$watch('ngModel', function () {
                    if (angular.isString(scope.ngModel) && scope.metadata && scope.metadata.referencedType) {
                        scope.ngModel = { type: scope.metadata.referencedType, id: scope.ngModel };
                    }
                });
            },
            post: function post(scope) {
                scope.$watch('metadata', function () {
                    if (scope.metadata) {
                        scope.isRequired = scope.metadata.required;
                    }
                });

                if (_.isUndefined(scope.ngModel) && !_.isUndefined(scope.metadata['default']) && !_.isNull(scope.metadata['default'])) {
                    scope.ngModel = scope.metadata['default'];
                }
            }
        }
    };
});
'use strict';

angular.module('xl.dip').directive('xlDipBoolean', function () {
    return {
        restrict: 'A',
        scope: {
            ngModel: '=',
            metadata: '=',
            displayMode: '='
        },
        templateUrl: 'src/xl-dip/xl-dip-boolean/xl-dip-boolean.html',
        require: 'ngModel',
        link: function link(scope, elem, attrs, ngModelCtrl) {
            scope.$watch('metadata', function () {
                if (scope.metadata) {
                    scope.isRequired = scope.metadata.required;
                }
            });

            if (_.isUndefined(scope.ngModel) && !_.isUndefined(scope.metadata['default']) && !_.isNull(scope.metadata['default'])) {
                scope.ngModel = scope.metadata['default'];
            }
        }
    };
});
'use strict';

angular.module('xl.components', ['xl.dip', 'xl.widget', 'xl.form', 'xl.tab-system', 'xl.buttons']);
'use strict';

angular.module('xl.buttons', []);
//# sourceMappingURL=xl-components.js.map

angular.module("xl.templates", []).run(["$templateCache", function($templateCache) {$templateCache.put("src/xl-dip/xl-dip-boolean/xl-dip-boolean.html","<div class=\"xl-components\" xl-random-id-generator=\"metadata\"><div ng-if=\"metadata.label\" xl-widget-label ng-model=\"ngModel\" options=\"metadata\"></div><!--\n --><div ng-class=\"metadata.label ? \'xl-components-input\' : \'xl-components-input-full\'\"><input dynamic-name=\"metadata.name\" type=\"checkbox\" xl-random-attr=\"id\" ng-model=\"ngModel\" ng-disabled=\"displayMode === \'visual\'\"></div></div>");
$templateCache.put("src/xl-dip/xl-dip-ci/xl-dip-ci.html","<div class=\"xl-components xl-dip-ci\" xl-random-id-generator=\"metadata\"><div ng-if=\"metadata.label\" xl-widget-label ng-model=\"ngModel\" options=\"metadata\"></div><!--\n --><div ng-show=\"displayMode !== \'visual\'\" ng-class=\"metadata.label ? \'xl-components-input\' : \'xl-components-input-full\'\"><div xl-widget-autocomplete ng-model=\"ngModel\" metadata=\"metadata\" handlers=\"handlers\" select-mode=\"options.selectMode\" show-all-on-click=\"true\" is-required=\"isRequired\" obj-label=\"{{objLabel}}\" is-disabled=\"isDisabled\"></div></div><!--\n --><span class=\"xl-components-display-mode\" ng-show=\"displayMode === \'visual\'\"><a href=\"\" ng-if=\"handlers.onClick\" ng-click=\"handlers.onClick(ngModel)\">{{ngModel.id}}</a> <span ng-if=\"!handlers.onClick\">{{ngModel.id}}</span></span></div>");
$templateCache.put("src/xl-dip/xl-dip-collection-of-ci/xl-dip-collection-of-ci.html","<div class=\"xl-components xl-dip-collection-of-ci\" xl-random-id-generator=\"metadata\"><div ng-if=\"metadata.label\" xl-widget-label ng-model=\"ngModel\" options=\"metadata\"></div><!--\n---><div ng-class=\"metadata.label ? \'xl-components-input\' : \'xl-components-input-full\'\"><div ng-if=\"isInputVisible\" class=\"dip-input\"><div xl-widget-autocomplete ng-model=\"binding.newItem\" metadata=\"metadata\" handlers=\"handlers\" select-mode=\"options.selectMode\" is-required=\"false\" obj-label=\"{{objLabel}}\" new-item-suffix=\"{{newItemSuffix}}\" open-on-click=\"{{openOnClick}}\" is-disabled=\"isDisabled\" options=\"{placeholder : emptyPlaceholder}\"></div><div ng-if=\"isUnique()\" xl-widget-alert-message ng-show=\"error\" on-dismiss=\"eraseError()\">{{ error }}</div></div><div ng-if=\"isFilterVisible\" class=\"dip-input\" xl-widget-filter ng-model=\"search[objLabel]\"></div><div class=\"xl-scrollable-panel\"><div ng-if=\"displayMode !== \'visual\'\"><ul ng-if=\"isSortable()\" ui-sortable=\"{ tolerance: \'pointer\', containment: \'parent\' }\" ng-model=\"$list\"><li class=\"xl-list-row\" ng-repeat=\"item in $list | filter:search\">{{ item[objLabel] }} <span class=\"ui-sortable-handler-icon\"></span> <span class=\"xl-btn-close\" data-ng-click=\"remove(item)\"></span></li></ul><ul ng-if=\"!isSortable()\"><li class=\"xl-list-row\" ng-repeat=\"item in $list | filter:search\">{{ item[objLabel] }} <a ng-if=\"handlers.onDelete\" href=\"\" class=\"glyphicon glyphicon-remove deleteHandler\" ng-click=\"onDeleteItem(item)\"><span>Delete</span></a> <a ng-if=\"handlers.onEdit\" href=\"\" class=\"glyphicon glyphicon-pencil editHandler\" ng-click=\"handlers.onEdit(item)\"><span>Edit</span></a> <span ng-if=\"!handlers.onDelete\" class=\"xl-btn-close\" data-ng-click=\"remove(item)\"></span></li></ul></div><div ng-if=\"displayMode === \'visual\'\"><ul class=\"xl-display-mode\"><li class=\"xl-simple-text\" ng-repeat=\"item in $list | filter:search\" ng-if=\"handlers.onClick\"><a href=\"\" ng-if=\"handlers.onClick\" ng-click=\"handlers.onClick(item)\">{{ item[objLabel] }}</a></li><li class=\"xl-simple-text\" ng-repeat=\"item in $list | filter:search\" ng-if=\"!handlers.onClick\">{{ item[objLabel] }}</li></ul></div></div></div></div>");
$templateCache.put("src/xl-dip/xl-dip-datetime/xl-dip-datetime.html","<div class=\"xl-components xl-dip-datetime\" xl-random-id-generator=\"metadata\"><div ng-if=\"metadata.label\" xl-widget-label ng-model=\"ngModel\" options=\"metadata\" class=\"datetime-label\"></div><!--\n---><div xl-widget-date ng-model=\"ngModel\" options=\"options.date\" display-mode=\"displayMode\" class=\"datetime-date\"></div><!--\n---><div xl-widget-time ng-model=\"ngModel\" options=\"options.time\" display-mode=\"displayMode\" class=\"datetime-time\"></div></div>");
$templateCache.put("src/xl-dip/xl-dip-enum/xl-dip-enum.html","<div class=\"xl-components xl-dip-enum\" xl-random-id-generator=\"metadata\"><div ng-if=\"metadata.label\" xl-widget-label options=\"metadata\"></div><!--\n --><div ng-class=\"metadata.label ? \'xl-components-input\' : \'xl-components-input-full\'\"><div xl-widget-autocomplete ng-model=\"ngModel\" handlers=\"autocompleteHandlers\" metadata=\"metadata\" select-mode=\"true\" show-all-on-click=\"true\" is-required=\"isRequired\" is-disabled=\"displayMode === \'visual\'\"></div></div></div>");
$templateCache.put("src/xl-dip/xl-dip-file-selector/xl-dip-file-selector.html","<div class=\"xl-components xl-dip-file-selector\" xl-random-id-generator=\"metadataClone\"><div ng-if=\"metadata.label\" xl-widget-label ng-model=\"ngModel.file\" options=\"metadataClone\"></div><!--\n --><div ng-class=\"metadata.label ? \'xl-components-input\' : \'xl-components-input-full\'\"><div xl-widget-file-upload options=\"metadataClone\" ng-model=\"ngModel\" is-disabled=\"isDisabled\"></div></div></div>");
$templateCache.put("src/xl-dip/xl-dip-integer/xl-dip-integer.html","<div class=\"xl-components xl-dip-integer\" xl-random-id-generator=\"metadata\"><div ng-if=\"metadata.label\" xl-widget-label ng-model=\"ngModel\" options=\"metadata\"></div><div ng-class=\"metadata.label ? \'xl-components-input\' : \'xl-components-input-full\'\"><input dynamic-name=\"metadata.name\" xl-random-attr=\"id\" type=\"text\" ng-model=\"ngModel\" ng-required=\"isRequired\" placeholder=\"{{options.placeholder}}\" ng-disabled=\"displayMode === \'visual\'\"> <span ng-if=\"displayMode !== \'visual\'\" class=\"xl-btn-close\" data-ng-show=\"ngModel\" data-ng-click=\"erase()\"></span><div class=\"xl-components-description\" ng-if=\"metadata.description\">{{metadata.description}}</div></div></div>");
$templateCache.put("src/xl-dip/xl-dip-list/xl-dip-list.html","<div class=\"xl-components xl-dip-list\" xl-random-id-generator=\"metadata\"><div ng-if=\"metadata.label\" xl-widget-label ng-model=\"ngModel\" options=\"metadata\"></div><div ng-class=\"metadata.label ? \'xl-components-input\' : \'xl-components-input-full\'\"><div ng-show=\"displayMode !== \'visual\' || $list.length === 0\"><button class=\"xl-button xl-primary xl-btn-add\" data-ng-disabled=\"!newItem || !inputValid()\" data-ng-click=\"addItem()\" ng-show=\"displayMode !== \'visual\'\">Add</button><div class=\"dip-input\" ng-init=\"ops = {name: metadata.name, description: metadata.description, validators: metadata.validators}\"><div xl-widget-string ng-model=\"newItem\" options=\"ops\" is-disabled=\"displayMode === \'visual\'\" key-down-handler=\"addItem\"></div></div></div><div class=\"xl-scrollable-panel\"><div><ul ng-show=\"displayMode !== \'visual\' && $list.length > 0\" ui-sortable=\"sortableConfig\" ng-model=\"$list\"><li class=\"xl-list-row\" ng-repeat=\"item in $list\"><span class=\"text-container\">{{ field ? item[field] : item[\'title\'] }}</span> <span class=\"ui-sortable-handler-icon\"></span> <span class=\"xl-btn-close\" data-ng-click=\"remove(item)\"></span></li></ul><ul class=\"xl-list-display-mode\" ng-show=\"displayMode === \'visual\' && $list.length > 0\" ng-model=\"$list\"><li class=\"xl-simple-text\" ng-repeat=\"item in $list\"><span class=\"text-container\">{{ field ? item[field] : item[\'title\'] }}</span></li></ul></div><div class=\"empty-placeholder\" ng-if=\"metadata.emptyPlaceholder && ($list === undefined || $list.length === 0)\">{{metadata.emptyPlaceholder}}</div></div></div></div>");
$templateCache.put("src/xl-dip/xl-dip-map-string-string/xl-dip-map-string-string.html","<div class=\"xl-components xl-map-string-string\" xl-random-id-generator=\"metadata\"><div ng-if=\"metadata.label\" xl-widget-label ng-model=\"ngModel\" options=\"metadata\"></div><!--\n  --><div ng-class=\"metadata.label ? \'xl-components-input\' : \'xl-components-input-full\'\"><table class=\"footer-table table visual-mode\" data-ng-show=\"displayMode === \'visual\'\"><tfoot><tr><td class=\"key-cell span5\"><b>Key</b></td><td class=\"value-cell span5\"><b>Value</b></td><td class=\"span1\"><div data-ng-click=\"clearSearchWord(); isSearching = !isSearching\"><span class=\"icon-magnifying-glass\"></span></div></td></tr></tfoot></table><table class=\"table\" data-ng-show=\"displayMode !== \'visual\'\"><tr><td class=\"xl-map-string-string-td\"><input class=\"input-key\" xl-random-attr=\"id\" ng-keydown=\"switchInput($event,true, false)\" data-ng-model=\"entry.key\" ng-trim=\"false\" type=\"text\" name=\"add-key\" placeholder=\"Key\" stop-pristine-propagation></td><td class=\"xl-map-string-string-td\"><input class=\"input-value\" ng-keydown=\"switchInput($event,false, true)\" type=\"{{inputValueType(metadata.password)}}\" data-ng-model=\"entry.value\" ng-trim=\"false\" name=\"add-value\" placeholder=\"Value\" stop-pristine-propagation></td><td><div class=\"xl-map-string-string-div-buttons\"><div data-ng-click=\"clearSearchWord(); isSearching = !isSearching\"><span class=\"icon-magnifying-glass\"></span></div><div style=\"text-align: right\" data-ng-click=\"addRow()\"><button ng-disabled=\"!entry.key\" class=\"add-row-button xl-button xl-primary\" tabindex=\"6\">Add</button></div></div></td></tr></table><div xl-widget-alert-message ng-show=\"error\" on-dismiss=\"error = undefined\">{{ error }}</div><table class=\"header-table\"><tbody data-ng-show=\"isSearching\"><tr><td colspan=\"3\" class=\"filter\"><span class=\"search-label\">Search</span> <input type=\"text\" data-ng-model=\"searchWord\" focus-me=\"isSearching\" stop-pristine-propagation> <span class=\"xl-btn-close\" data-ng-show=\"searchWord\" data-ng-click=\"clearSearchWord()\"></span></td></tr></tbody></table><div class=\"table-scrollable-container\" ng-hide=\"isEmpty()\"><div class=\"table-scrollable striped-background\"><table class=\"table-condensed striped-background\"><tbody><tr data-ng-repeat=\"key in keys(ngModel, searchWord)\"><td class=\"xl-map-key xl-map-string-string-td\" ng-init=\"obj= {\'key\': key, \'value\': ngModel[key]}\"><div ng-if=\"displayMode !== \'visual\'\" class=\"key-div\" xl-widget-contenteditable-int=\"true\" data-ng-model=\"obj\" data-type=\"key\"></div><div ng-if=\"displayMode === \'visual\'\" class=\"key-div-visual\">{{obj.key}}</div></td><td class=\"xl-map-value xl-map-string-string-td\"><div ng-if=\"displayMode !== \'visual\'\" class=\"value-div\" xl-widget-contenteditable-int=\"true\" data-ng-model=\"obj\" data-type=\"value\" data-password=\"{{password}}\" placeholder=\"Empty value\"></div><div ng-if=\"displayMode === \'visual\'\" class=\"value-div-visual\">{{obj.value}}</div></td><td ng-if=\"displayMode !== \'visual\'\"><div class=\"xl-map-string-string-div-buttons\"><div class=\"xl-btn-close\" data-ng-click=\"deleteRow(obj.key)\"></div></div></td></tr></tbody></table></div><div class=\"xl-components-description\" ng-if=\"metadata.description\">{{metadata.description}}</div></div></div></div>");
$templateCache.put("src/xl-dip/xl-dip-set-of-strings/xl-dip-set-of-strings.html","<div class=\"xl-dip-set-of-strings\" xl-random-id-generator=\"metadata\"><div xl-widget-tags ng-model=\"ngModel\" options=\"options\" select-options=\"selectOptions\" class=\"\" display-mode=\"displayMode\"></div></div>");
$templateCache.put("src/xl-dip/xl-dip-set/xl-dip-set.html","<div class=\"xl-components xl-dip-set\" xl-random-id-generator=\"metadata\"><div ng-if=\"metadata.label\" xl-widget-label ng-model=\"ngModel\" options=\"metadata\"></div><div ng-class=\"metadata.label ? \'xl-components-input\' : \'xl-components-input-full\'\"><div ng-show=\"displayMode !== \'visual\' || $set.length === 0\"><button class=\"xl-button xl-primary xl-btn-add\" data-ng-disabled=\"!newItem\" data-ng-click=\"addItem()\" ng-show=\"displayMode !== \'visual\'\">Add</button><div class=\"dip-input\"><input dynamic-name=\"metadata.name\" xl-random-attr=\"id\" type=\"text\" ng-model=\"newItem\" ng-keydown=\"addItem($event)\" ng-disabled=\"displayMode === \'visual\'\"> <span class=\"xl-btn-close\" data-ng-show=\"newItem\" data-ng-click=\"newItem=\'\'\"></span></div><div xl-widget-alert-message ng-show=\"error\" on-dismiss=\"error = undefined\">{{ error }}</div><div class=\"xl-components-description\" ng-if=\"metadata.description\">{{metadata.description}}</div></div><div class=\"xl-scrollable-panel\"><div><ul ng-show=\"displayMode !== \'visual\' && $set.length > 0\" class=\"xl-set-edit-mode\" ng-model=\"$set\"><li class=\"xl-list-row\" ng-repeat=\"item in $set\"><span class=\"text-container\">{{ field ? item[field] : item[\'title\'] }}</span> <span class=\"xl-btn-close\" data-ng-click=\"remove(item)\"></span></li></ul><ul class=\"xl-set-display-mode\" ng-show=\"displayMode === \'visual\' && $set.length > 0\" ng-model=\"$set\"><li class=\"xl-simple-text\" ng-repeat=\"item in $set\"><span class=\"text-container\">{{ field ? item[field] : item[\'title\'] }}</span></li></ul></div><div class=\"empty-placeholder\" ng-if=\"metadata.emptyPlaceholder && ($set === undefined || $set.length === 0)\">{{metadata.emptyPlaceholder}}</div></div></div></div>");
$templateCache.put("src/xl-dip/xl-dip-string/xl-dip-string.html","<div xl-widget-string ng-model=\"ngModel\" options=\"options\" display-mode=\"displayMode\" is-disabled=\"isDisabled\"></div>");
$templateCache.put("src/xl-form/xl-dip-deployable-ci/xl-dip-deployable-ci.html","<div class=\"xl-components xl-dip-deployable-ci\"><form name=\"deployableCiForm\" novalidate><!-- File selector --><div ng-if=\"isSourceArtifact && displayMode === \'creation\'\"><div xl-widget-label ng-model=\"ngModel.type\" options=\"{label: \'Indicate file location\', required: true}\"></div><div class=\"xl-dip-deployable-sub-elements\"><div xl-dip-file-selector ng-model=\"ngModel.form.__upload_file__\" is-disabled=\"ngModel.remoteFileUri\" metadata=\"{label: \'Choose file\', name: \'__upload_file__\'}\" class=\"xl-dip-deployable-ci-sub-element\"></div><div class=\"xl-dip-deployable-or-statement\">or</div><div xl-random-id-generator=\"{name: \'remoteFileUriSection\'}\" class=\"xl-dip-deployable-ci-sub-element xl-remote-file-uri-element\"><div class=\"xl-remote-file-uri-input\" class=\"xl-components-input\"><div xl-widget-string ng-model=\"ngModel.remoteFileUri\" options=\"{description: \'The URI pointing to the (remote) location of the file this artifact represents\', label: \'Remote file Uri\', name: \'remoteFileUri\'}\" is-disabled=\"ngModel.form.__upload_file__.file\"></div></div></div></div></div><br><!-- Name --><div xl-widget-string ng-model=\"ngModel.name\" options=\"{label: \'Name\', name: \'ciName\', required: true}\" class=\"xl-dip-deployable-ci-element\" is-disabled=\"displayMode !== \'creation\'\"></div><!-- Type --><div xl-random-id-generator=\"{name: \'ciType\'}\" class=\"xl-dip-deployable-ci-element last\"><div xl-widget-label ng-model=\"ngModel.type\" options=\"{label: \'Type\', required: true}\"></div><!--\n  ---><div class=\"xl-components-input\"><div xl-widget-autocomplete-type=\"typeNames\" ng-model=\"ngModel.type\" on-select=\"typeSelected(type)\" is-required=\"true\" is-disabled=\"displayMode !== \'creation\'\"></div></div></div><!-- Dynamic form --><div class=\"xl-dip-dynamic-form\"><div xl-dip-dynamic-form-inner ng-model=\"ngModel.form\" metadata=\"metadata\" handlers=\"handlers\" display-mode=\"displayMode\"></div></div></form></div>");
$templateCache.put("src/xl-form/xl-dip-dynamic-form/xl-dip-dynamic-form-inner.html","<div ng-repeat=\"categoryProperties in groupedProperties\" ng-class=\"{open: categoryProperties.isOpen}\"><div class=\"xl-category\" ng-click=\"categoryProperties.isOpen = !categoryProperties.isOpen\">{{categoryProperties.category}}</div><div class=\"xl-category-content\"><div ng-repeat=\"metaComponent in categoryProperties.properties\" class=\"xl-dip-dynamic-form-element\"><div ng-hide=\"metaComponent.hidden\" ng-switch on=\"metaComponent.kind\"><div ng-switch-when=\"STRING\"><div xl-dip-string ng-model=\"ngModel[metaComponent.name]\" metadata=\"metaComponent\" display-mode=\"getLocalDisplayMode(metaComponent.asContainment)\" is-disabled=\"metaComponent.asContainment\"></div></div><div ng-switch-when=\"INTEGER\"><div xl-dip-integer ng-model=\"ngModel[metaComponent.name]\" metadata=\"metaComponent\" display-mode=\"getLocalDisplayMode(metaComponent.asContainment)\" is-disabled=\"metaComponent.asContainment\"></div></div><div ng-switch-when=\"BOOLEAN\"><div xl-dip-boolean ng-model=\"ngModel[metaComponent.name]\" metadata=\"metaComponent\" display-mode=\"getLocalDisplayMode(metaComponent.asContainment)\" is-disabled=\"metaComponent.asContainment\"></div></div><div ng-switch-when=\"SET_OF_STRING\"><div xl-dip-string-set ng-model=\"ngModel[metaComponent.name]\" metadata=\"metaComponent\" display-mode=\"getLocalDisplayMode(metaComponent.asContainment)\" is-disabled=\"metaComponent.asContainment\"></div></div><div ng-switch-when=\"LIST_OF_STRING\"><div xl-dip-list ng-model=\"ngModel[metaComponent.name]\" metadata=\"metaComponent\" display-mode=\"getLocalDisplayMode(metaComponent.asContainment)\" is-disabled=\"metaComponent.asContainment\"></div></div><div ng-switch-when=\"SET_OF_CI\"><div xl-dip-set-of-ci ng-model=\"ngModel[metaComponent.name]\" metadata=\"metaComponent\" display-mode=\"getLocalDisplayMode(metaComponent.asContainment)\" is-disabled=\"metaComponent.asContainment\" handlers=\"handlers.SET_OF_CI\" obj-label=\"{{objLabel}}\"></div></div><div ng-switch-when=\"LIST_OF_CI\"><div xl-dip-list-of-ci ng-model=\"ngModel[metaComponent.name]\" metadata=\"metaComponent\" display-mode=\"getLocalDisplayMode(metaComponent.asContainment)\" is-disabled=\"metaComponent.asContainment\" handlers=\"handlers.LIST_OF_CI\" obj-label=\"{{objLabel}}\"></div></div><div ng-switch-when=\"CI\"><div xl-dip-ci ng-model=\"ngModel[metaComponent.name]\" metadata=\"metaComponent\" display-mode=\"getLocalDisplayMode(metaComponent.asContainment)\" is-disabled=\"metaComponent.asContainment\" handlers=\"handlers.CI\" obj-label=\"{{objLabel}}\" options=\"{selectMode: true}\"></div></div><div ng-switch-when=\"ENUM\"><div xl-dip-enum ng-model=\"ngModel[metaComponent.name]\" metadata=\"metaComponent\" display-mode=\"getLocalDisplayMode(metaComponent.asContainment)\" is-disabled=\"metaComponent.asContainment\"></div></div><div ng-switch-when=\"MAP_STRING_STRING\"><div xl-dip-map-string-string ng-model=\"ngModel[metaComponent.name]\" entry=\"entry\" metadata=\"metaComponent\" display-mode=\"getLocalDisplayMode(metaComponent.asContainment)\" is-disabled=\"metaComponent.asContainment\"></div></div><div ng-switch-when=\"DATE\"><div xl-dip-datetime ng-model=\"ngModel[metaComponent.name]\" metadata=\"metaComponent\" display-mode=\"getLocalDisplayMode(metaComponent.asContainment)\" is-disabled=\"metaComponent.asContainment\"></div></div></div></div></div></div>");
$templateCache.put("src/xl-form/xl-dip-dynamic-form/xl-dip-dynamic-form.html","<div class=\"xl-components xl-dip-dynamic-form\"><form name=\"form\" novalidate><div xl-dip-dynamic-form-inner ng-model=\"ngModel\" metadata=\"metadata\" obj-label=\"{{objLabel}}\" handlers=\"handlers\" display-mode=\"displayMode\" collapsed=\"collapsed\"></div></form></div>");
$templateCache.put("src/xl-tab-system/xl-tab-system-header.html","<div class=\"xl-components ui-tab-header\" ui-tab-menu-dropdown><div class=\"ui-tab-header-wrapper\"><ul class=\"ui-tab-header-container\"><li class=\"ui-tab-header-item\" ng-class=\"{active: tab.selected}\" data-ng-repeat=\"tab in tabs\" data-ng-click=\"selectTab(tab, $index)\"><span class=\"asterisk\" ng-show=\"tab.pristine === false\">*</span> <span class=\"ui-tab-header-title\">{{tab.title}}</span> <span class=\"ui-tab-header-close\" data-ng-click=\"closeTab(tab)\" ng-show=\"isShowCloseTabIcon()\"></span></li></ul></div><span class=\"ui-tab-add-tab\" data-ng-click=\"addTab()\" ng-show=\"isShowPlusTab()\">+</span> <span class=\"ui-tab-header-menu-toggle\" ui-tab-menu-dropdown-toggle ng-show=\"showTabMenuHandler\"></span><div class=\"ui-tab-header-menu\"><ul><li class=\"ui-tab-header-menu-item\" data-ng-repeat=\"tab in tabs\" data-ng-click=\"selectTab(tab, $index)\"><span class=\"ui-tab-header-menu-item-title\">{{tab.title}}</span></li></ul></div></div>");
$templateCache.put("src/xl-widget/xl-widget-alert-message/xl-widget-alert-message.html","<div class=\"xl-components xl-alert-message\"><div ng-if=\"options.indentForLabel\" class=\"indent-for-label\"></div><div class=\"message-panel\" ng-class=\"{\'xl-alert-error\': internalState === \'error\', \'xl-alert-warning\': internalState === \'warning\', \'xl-alert-success\': internalState === \'success\',\n     \'xl-alert-with-indent-for-label\' : options.indentForLabel}\"><span ng-transclude></span> <span class=\"xl-btn-close\" data-ng-show=\"showClose\" data-ng-click=\"dismiss()\"></span></div></div>");
$templateCache.put("src/xl-widget/xl-widget-autocomplete-type/xl-widget-autocomplete-type.html","<div class=\"xl-components xl-widget-autocomplete xl-widget-autocomplete-type\"><div xl-widget-autocomplete ng-model=\"ngModel\" metadata=\"metadata\" options=\"options\" handlers=\"{onSelect: onSelectInternal, addCandidates: addCandidates}\" select-mode=\"true\" is-required=\"isRequired\" is-disabled=\"isDisabled\" show-all-on-click=\"true\"></div></div>");
$templateCache.put("src/xl-widget/xl-widget-autocomplete/xl-widget-autocomplete.html","<div class=\"xl-components xl-widget-autocomplete\"><input ng-class=\"{\'xl-autocomplete-disabled\' : isDisabled}\" dynamic-name=\"metadata.name\" xl-random-attr=\"id\" type=\"text\" placeholder=\"{{options.placeholder}}\" ng-model=\"ngModelIntern\" ng-required=\"isRequired\" ng-change=\"onChange()\" ng-disabled=\"isDisabled\"> <span ng-class=\"{\'xl-btn-close\' : !isDisabled}\" data-ng-show=\"ngModelIntern\" data-ng-click=\"clearModelValues()\"></span> <span class=\"xl-btn-handler\" ng-class=\"{\'xl-autocomplete-disabled-btn\' : isDisabled, \'xl-autocomplete-enabled-btn\' : !isDisabled}\"></span><div class=\"xl-components-description\" ng-if=\"metadata.description\">{{metadata.description}}</div></div>");
$templateCache.put("src/xl-widget/xl-widget-contenteditable/xl-widget-contenteditable.html","<div class=\"xl-components xl-widget-contenteditable\" contenteditable=\"true\" data-ng-model=\"modelInternal\" data-password=\"{{password}}\">{{modelLabel}}</div>");
$templateCache.put("src/xl-widget/xl-widget-country/xl-widget-country-state.html","<div class=\"xl-components xl-widget-country-state\" xl-random-id-generator=\"options\"><div ng-if=\"metadata.label\" xl-widget-label ng-model=\"ngModel\" options=\"options\"></div><!--\n --><div ng-class=\"metadata.label ? \'xl-components-input\' : \'xl-components-input-full\'\"><div xl-widget-autocomplete ng-model=\"ngModel\" options=\"options\" handlers=\"autocompleteHandlers\" select-mode=\"true\" is-required=\"options.required\"></div></div><!--\n--></div>");
$templateCache.put("src/xl-widget/xl-widget-country/xl-widget-country.html","<div class=\"xl-components xl-widget-country\" xl-random-id-generator=\"options\"><div ng-if=\"metadata.label\" xl-widget-label ng-model=\"ngModel\" options=\"options\"></div><!--\n  --><div ng-class=\"metadata.label ? \'xl-components-input\' : \'xl-components-input-full\'\"><div xl-widget-autocomplete ng-model=\"ngModel\" options=\"options\" metadata=\"options\" handlers=\"autocompleteHandlers\" select-mode=\"true\" is-required=\"options.required\"></div></div><!--\n--></div>");
$templateCache.put("src/xl-widget/xl-widget-date/datepicker/datepicker.html","<div class=\"dropdown-menu datepicker\" ng-class=\"\'datepicker-mode-\' + $mode\" style=\"max-width: 200px\"><table style=\"table-layout: fixed; height: 100%; width: 100%\"><thead><tr class=\"text-center\"><th><button tabindex=\"-1\" type=\"button\" class=\"btn btn-default pull-left\" ng-click=\"$selectPane(-1)\"><i class=\"{{$iconLeft}}\"></i></button></th><th colspan=\"{{ rows[0].length - 2 }}\"><button tabindex=\"-1\" type=\"button\" class=\"btn btn-default btn-block text-strong\" ng-click=\"$toggleMode()\"><strong style=\"text-transform: capitalize\" ng-bind=\"title\"></strong></button></th><th><button tabindex=\"-1\" type=\"button\" class=\"btn btn-default pull-right\" ng-click=\"$selectPane(+1)\"><i class=\"{{$iconRight}}\"></i></button></th></tr><tr ng-show=\"showLabels\" ng-bind-html=\"labels\"></tr></thead><tbody><tr ng-repeat=\"(i, row) in rows\" height=\"{{ 100 / rows.length }}%\"><td class=\"text-center\" ng-repeat=\"(j, el) in row\"><button tabindex=\"-1\" type=\"button\" class=\"btn btn-default\" style=\"width: 100%\" ng-class=\"{\'btn-primary\': el.selected, \'btn-info btn-today\': el.isToday && !el.selected}\" ng-click=\"$select(el.date)\" ng-disabled=\"el.disabled\"><span ng-class=\"{\'text-muted\': el.muted}\" ng-bind=\"el.label\"></span></button></td></tr></tbody></table></div>");
$templateCache.put("src/xl-widget/xl-widget-date/datepicker/tooltip.html","<div class=\"tooltip in\" ng-show=\"title\"><div class=\"tooltip-arrow\"></div><div class=\"tooltip-inner\" ng-bind=\"title\"></div></div>");
$templateCache.put("src/xl-widget/xl-widget-date/xl-widget-date.html","<div class=\"xl-components xl-widget-date\"><div class=\"xl-components-input\"><input type=\"text\" xl-random-attr=\"id\" class=\"date-input\" bs-datepicker-labs date-format=\"{{options.format}}\" ng-model=\"ngModel\" autoclose=\"1\" start-week=\"1\" ng-disabled=\"displayMode === \'visual\'\"> <span class=\"svg-icon icon-calendar\"></span></div></div>");
$templateCache.put("src/xl-widget/xl-widget-file-upload/xl-widget-file-upload.html","<div class=\"xl-components xl-widget-file-upload\"><input class=\"file-selector-file\" type=\"file\" name=\"{{fileNameField}}\" ng-hide=\"true\"><div class=\"file-upload-browse-area\"><input dynamic-name=\"options.name\" class=\"file-name\" type=\"text\" ng-model=\"selectedFile.name\" ng-required=\"options.required\" disabled=\"disabled\"> <button type=\"button\" xl-random-attr=\"id\" class=\"xl-button xl-primary file-selector-button\" ng-click=\"browse()\" ng-disabled=\"ngModel.state ===\'progress\' || isDisabled\">Browse</button> <span class=\"xl-btn-close\" data-ng-show=\"selectedFile.name && options.auto === false\" data-ng-click=\"pristine()\"></span></div><div class=\"file-upload-upload-area\" ng-if=\"ngModel.state !==\'pristine\'\"><div ng-show=\"ngModel.state === \'progress\'\" xl-widget-progressbar current-progress=\"ngModel.progress\" options=\"{strictError: false, cancel: cancel, close: close}\" status=\"ngModel.state\"></div><div class=\"xl-upload-message\" ng-show=\"ngModel.message !== undefined && ngModel.state===\'progress\'\" ng-class=\"\'xl-fileupload-\'+ngModel.state\">{{ngModel.message}}</div><div id=\"xl-file-upload-error\" ng-show=\"ngModel.message !== undefined && ngModel.state!==\'progress\'\" on-dismiss=\"close()\" state=\"ngModel.state\" xl-widget-alert-message on-dismiss=\"ngModel.message=undefined\">{{ ngModel.message }}</div></div><div class=\"file-upload-drop-zone fade\" ng-class=\"specClass\"></div></div>");
$templateCache.put("src/xl-widget/xl-widget-filter/xl-widget-filter.html","<div class=\"xl-components xl-widget-filter\"><span class=\"icon-magnifying-glass\"></span> <span class=\"xl-btn-close\" data-ng-show=\"ngModel\" data-ng-click=\"ngModel=undefined\"></span> <input type=\"text\" ng-model=\"ngModel\"></div>");
$templateCache.put("src/xl-widget/xl-widget-item-selector/xl-widget-item-selector.html","<div class=\"xl-components xl-widget-item-selector\"><div class=\"xl-widget-item-selector-text\">{{options.description}}</div><div class=\"xl-widget-item-selector-input\" xl-widget-filter ng-model=\"search[objLabel]\"></div><div class=\"empty-placeholder\" ng-if=\"!items.length\">{{emptyPlaceholder}}</div><div class=\"xl-widget-item-selector-panel\" ng-class=\"{\'multiple\' : options.multiple}\"><label class=\"xl-widget-item-selector-label\" ng-repeat=\"item in items | filter:search\"><input ng-class=\"{\'act-as-radio\': !options.multiple}\" type=\"checkbox\" checklist-model=\"ngModel\" comparaison-attribute=\"objLabel\" checklist-value=\"item\"> {{item[objLabel]}}</label></div></div>");
$templateCache.put("src/xl-widget/xl-widget-label/xl-widget-label.html","<div class=\"xl-components-label\"><label xl-random-attr=\"for\">{{options.label}} <span class=\"required\" ng-if=\"options.required\">*</span></label></div>");
$templateCache.put("src/xl-widget/xl-widget-notify/xl-widget-notify.html","<div class=\"xl-components xl-widget-notify\"><div class=\"xl-widget-notification\" data-ng-class=\"notification.typeClass\" data-ng-repeat=\"notification in notifications\"><h4 class=\"title\" data-ng-if=\"notification.body.title\">{{ notification.body.title }}</h4><div class=\"message\">{{ notification.body.message }}</div><span class=\"xl-btn-close\" data-ng-click=\"dismiss(notification)\"></span></div></div>");
$templateCache.put("src/xl-widget/xl-widget-progressbar/xl-widget-progressbar.html","<div class=\"xl-components xl-widget-progressbar\" ng-class=\"{\'with-cancel-button\': isCancelable()}\"><div class=\"xl-progressbar\"><span ng-style=\"innerBarStyle\" class=\"{{innerBarClass}}\" ng-class=\"{\'xl-nostripes\': !options.stripped}\"></span></div><div class=\"xl-icon-container xl-icon-cancel\" ng-show=\"displayButtonByStatus(\'inProgress\')\" ng-click=\"cancelProgress()\"></div><div class=\"xl-icon-container xl-icon-success\" ng-show=\"displayButtonByStatus(\'success\')\" ng-click=\"applySuccessProgress()\"></div><div class=\"xl-icon-container xl-icon-error\" ng-show=\"displayButtonByStatus(\'error\')\" ng-click=\"applySuccessProgress()\"></div></div>");
$templateCache.put("src/xl-widget/xl-widget-select/xl-widget-select.html","<div class=\"xl-components xl-widget-select\"><div ng-if=\"metadata.label\" xl-widget-label ng-model=\"ngModel\" options=\"options\"></div><!--\n---><div ng-show=\"displayMode !== \'visual\'\" ng-class=\"metadata.label ? \'xl-components-input\' : \'xl-components-input-full\'\"><select class=\"select2-input\" ui-select2=\"selectOptions\" ng-model=\"ngModel\" id=\"{{options.name}}\"><option value=\"\"></option><option ng-repeat=\"data in selectData\" value=\"{{data}}\">{{data}}</option></select><div class=\"xl-components-description\" ng-if=\"options.description\">{{options.description}}</div></div><!--\n---><div class=\"xl-components-display-mode\" ng-show=\"displayMode === \'visual\'\" for=\"{{options.name}}\">{{ngModel}}</div></div>");
$templateCache.put("src/xl-widget/xl-widget-spinner/xl-widget-spinner.html","<div class=\"xl-widget-spinner\"><div class=\"cell\"><div></div></div></div>");
$templateCache.put("src/xl-widget/xl-widget-string/xl-widget-string.html","<div class=\"xl-components xl-widget-string\" xl-random-id-generator=\"options\"><div ng-if=\"options.label\" xl-widget-label ng-model=\"ngModel\" options=\"options\"></div><!--\n --><div ng-class=\"options.label ? \'xl-components-input\' : \'xl-components-input-full\'\"><input ng-keydown=\"setValue($event)\" dynamic-name=\"options.name + (isTextArea ? \'_NOT_SHOWN\' : \'\')\" xl-random-attr=\"id\" xl-random-suffix=\"{{isTextArea ? \'_NOT_SHOWN\' : \'\'}}\" type=\"{{ options.password ? \'password\' : \'text\' }}\" placeholder=\"{{options.placeholder}}\" ng-model=\"ngModel\" ng-show=\"!isTextArea\" ng-required=\"options.required\" ng-disabled=\"isDisabled || displayMode === \'visual\'\"><textarea ng-keydown=\"setValue($event)\" dynamic-name=\"options.name + (isTextArea ? \'\' : \'_NOT_SHOWN\')\" xl-random-attr=\"id\" xl-random-suffix=\"{{isTextArea ? \'\' : \'_NOT_SHOWN\'}}\" placeholder=\"{{options.placeholder}}\" ng-model=\"ngModel\" ng-show=\"isTextArea\" ng-required=\"options.required\" ng-disabled=\"isDisabled || displayMode === \'visual\'\"></textarea><span class=\"xl-btn-close\" data-ng-show=\"ngModel\" data-ng-click=\"erase()\" ng-if=\"!isDisabled && displayMode !== \'visual\'\"></span><div class=\"xl-components-description\" ng-if=\"options.description\" ng-bind=\"options.description\"></div></div></div>");
$templateCache.put("src/xl-widget/xl-widget-tags/xl-widget-tags.html","<div class=\"xl-components xl-widget-tags\"><div ng-if=\"options.label\" xl-widget-label ng-model=\"ngModel\" options=\"options\"></div><!--\n --><div ng-show=\"displayMode !== \'visual\'\" ng-class=\"options.label ? \'xl-components-input\' : \'xl-components-input-full\'\"><input class=\"select2-input\" ui-select2=\"selectOptions\" ng-model=\"ngModel\" id=\"{{options.name}}\"><div class=\"xl-components-description\" ng-if=\"options.description\">{{options.description}}</div></div><!--\n--><div ng-if=\"displayMode === \'visual\'\" class=\"xl-widget-tags tags-inline\"><ul><li class=\"xl-widget-tag-visual-mode\" ng-repeat=\"tag in ngModel\"><span>{{ tag }}</span></li></ul></div></div>");
$templateCache.put("src/xl-widget/xl-widget-time/timepicker/timepicker.html","<div class=\"dropdown-menu timepicker\" style=\"min-width: 0;width: auto\"><table height=\"100%\"><thead><tr class=\"text-center\"><th><button tabindex=\"-1\" type=\"button\" class=\"btn btn-default pull-left\" ng-click=\"$arrowAction(-1, 0)\"><i class=\"{{ $iconUp }}\"></i></button></th><th>&nbsp;</th><th><button tabindex=\"-1\" type=\"button\" class=\"btn btn-default pull-left\" ng-click=\"$arrowAction(-1, 1)\"><i class=\"{{ $iconUp }}\"></i></button></th></tr></thead><tbody><tr ng-repeat=\"(i, row) in rows\"><td class=\"text-center\"><button tabindex=\"-1\" style=\"width: 100%\" type=\"button\" class=\"btn btn-default\" ng-class=\"{\'btn-primary\': row[0].selected}\" ng-click=\"$select(row[0].date, 0)\" ng-disabled=\"row[0].disabled\"><span ng-class=\"{\'text-muted\': row[0].muted}\" ng-bind=\"row[0].label\"></span></button></td><td><span ng-bind=\"i == midIndex ? timeSeparator : \' \'\"></span></td><td class=\"text-center\"><button tabindex=\"-1\" ng-if=\"row[1].date\" style=\"width: 100%\" type=\"button\" class=\"btn btn-default\" ng-class=\"{\'btn-primary\': row[1].selected}\" ng-click=\"$select(row[1].date, 1)\" ng-disabled=\"row[1].disabled\"><span ng-class=\"{\'text-muted\': row[1].muted}\" ng-bind=\"row[1].label\"></span></button></td><td ng-if=\"showAM\">&nbsp;</td><td ng-if=\"showAM\"><button tabindex=\"-1\" ng-show=\"i == midIndex - !isAM * 1\" style=\"width: 100%\" type=\"button\" ng-class=\"{\'btn-primary\': !!isAM}\" class=\"btn btn-default\" ng-click=\"$switchMeridian()\" ng-disabled=\"el.disabled\">AM</button> <button tabindex=\"-1\" ng-show=\"i == midIndex + 1 - !isAM * 1\" style=\"width: 100%\" type=\"button\" ng-class=\"{\'btn-primary\': !isAM}\" class=\"btn btn-default\" ng-click=\"$switchMeridian()\" ng-disabled=\"el.disabled\">PM</button></td></tr></tbody><tfoot><tr class=\"text-center\"><th><button tabindex=\"-1\" type=\"button\" class=\"btn btn-default pull-left\" ng-click=\"$arrowAction(1, 0)\"><i class=\"{{ $iconDown }}\"></i></button></th><th>&nbsp;</th><th><button tabindex=\"-1\" type=\"button\" class=\"btn btn-default pull-left\" ng-click=\"$arrowAction(1, 1)\"><i class=\"{{ $iconDown }}\"></i></button></th></tr></tfoot></table></div>");
$templateCache.put("src/xl-widget/xl-widget-time/xl-widget-time.html","<div class=\"xl-components xl-widget-time\"><div class=\"xl-components-input\"><input type=\"text\" class=\"time-input\" bs-timepicker-labs time-format=\"{{options.format}}\" ng-model=\"ngModel\" autoclose=\"1\" ng-disabled=\"displayMode === \'visual\'\"> <span class=\"svg-icon icon-time\"></span></div></div>");
$templateCache.put("src/xl-widget/xl-widget-tree/tree-nodes.html","<ul vs-repeat><li ng-repeat=\"node in treeNodes\" id=\"{{node.idDom}}\"><span scroll-to scroll-to-me=\"node.scrollTo\" ng-if=\"node.scrollTo\"></span> <span class=\"treenode-line\" ng-class=\"isNodeSelected(node)\" ng-dblclick=\"dblClickAction(node)\" xl-right-click=\"selectNodeLabel(node)\" ng-click=\"selectNodeLabel(node)\" data-drag=\"{{draggable}}\" data-jqyoui-options=\"{revert: \'invalid\', helper: \'clone\', appendTo: \'body\'}\" jqyoui-draggable=\"{index:\'{{node.id}}\',placeholder:\'keep\',animate:true}\" ng-model=\"node\"><!-- on-single-click=\"...\" --><div class=\"whole-line\" ng-class=\"wholeLineClass(node, hoverOver)\" ng-mouseenter=\"hoverOver=true\" ng-mouseleave=\"hoverOver=false\"></div><i ng-show=\"node.children && !node.open\" ng-click=\"selectNodeHead(node, $event)\" ng-mouseenter=\"hoverOver=true\" ng-mouseleave=\"hoverOver=false\"><i class=\"arrow-right\"></i> <i class=\"collapsed\" ng-class=\"node.classType\" ng-style=\"classStyle(node)\"></i> </i><i ng-show=\"node.children && node.open\" ng-click=\"selectNodeHead(node, $event)\" ng-mouseenter=\"hoverOver=true\" ng-mouseleave=\"hoverOver=false\"><i class=\"arrow-down\"></i> <i class=\"expanded\" ng-class=\"node.classType\" ng-style=\"classStyle(node)\"></i> </i><i ng-hide=\"hasChildren(node)\"><div class=\"no-arrow\"></div><i class=\"normal\" ng-class=\"node.classType\" ng-style=\"classStyle(node)\"></i> </i><span class=\"tree-label\" ng-mouseenter=\"hoverOver=true\" ng-mouseleave=\"hoverOver=false\" title=\"{{node.label}}\">{{node.label}}</span></span><div ng-hide=\"!node.open\" tree-nodes=\"node.children\" config=\"config\" draggable=\"draggable\" current-node=\"currentNode\" broadcast-event-name=\"broadcastEventName\" datasource=\"datasource\" options=\"options\" class-style=\"classStyle\"></div></li></ul>");
$templateCache.put("src/xl-widget/xl-widget-tree/xl-widget-tree.html","<div class=\"xl-widget-tree\"><div tree-nodes=\"nodes\" draggable=\"draggable\" expand=\"expand\" config=\"config\" current-node=\"currentNode\" broadcast-event-name=\"broadcastEventName\" datasource=\"datasource\" options=\"options\" class-style=\"classStyle\"></div></div>");
$templateCache.put("src/xl-widget/xl-widget-wizard/xl-widget-wizard.html","<div><div class=\"xl-steps-header\"><ul class=\"steps-indicator steps-{{steps.length}}\" ng-show=\"!hideIndicators\"><li ng-class=\"{default: !step.completed && !step.selected, current: step.selected && !step.completed, done: step.completed && !step.selected, editing: step.selected && step.completed}\" ng-repeat=\"step in steps\"><div><a ng-show=\"navigable\" ng-click=\"goTo(step)\"><span>{{ $index + 1 }}</span> {{step.wzTitle}}</a> <a ng-hide=\"navigable\" class=\"xl-wizard-not-navigable\"><span>{{ $index + 1 }}</span> {{step.wzTitle}}</a></div></li></ul></div><div class=\"xl-steps-content\"><div class=\"steps\" ng-transclude></div></div></div>");}]);