from xld.kubernetes.pod.pod_helper import PodHelper
from kubernetes import client


class DeploymentConfigHelper(object):

    @staticmethod
    def read_deployment_config(deployed):
        for pod_template in deployed.template:
            template = PodHelper.read_pod(PodHelper(), deployed_pod=pod_template, pod=client.V1Pod())

        deployment_config = {
            "api_version": "v1",
            "kind": "DeploymentConfig",
            "metadata": {
                "name": DeploymentConfigHelper.get_deployment_config_name(deployed)
            },
            "spec": {
                "replicas": deployed.replicas,
                "paused": deployed.paused,
                "test": deployed.test,
                "selector": deployed.selector,
                "revision_history_limit": deployed.revision_history_limit,
                "min_ready_seconds": deployed.min_ready_seconds,
                "template": template,
                "triggers": DeploymentConfigHelper.__get_deployment_triggers(deployed.triggers)
            }
        }

        strategy = DeploymentConfigHelper.__get_deployment_strategy(deployed.strategy)

        if strategy is not None:
            deployment_config["spec"]["strategy"]=strategy

        return deployment_config

    @staticmethod
    def get_deployment_config_name(deployed):
        return deployed.deploymentConfigName if deployed.deploymentConfigName else deployed.name

    @staticmethod
    def enrich_app_selectors(deployed):
        for pod_template in deployed.template:
            if not pod_template.labels or 'app' not in pod_template.labels:
                pod_template.labels=dict(pod_template.labels, app = DeploymentConfigHelper.get_deployment_config_name(deployed))

    @staticmethod
    def __get_deployment_strategy(strategy_set):

        for strategy in strategy_set:
            if strategy.type == "openshift.CustomDeploymentStrategyParamsSpec":
                return {
                    "type": "Custom",
                    "annotations": strategy.annotations,
                    "labels": strategy.labels,
                    "custom_params": {
                        "command": strategy.command,
                        "image": strategy.image,
                        "environment": strategy.environment
                    }
                }
            elif strategy.type == "openshift.RollingDeploymentStrategyParamsSpec":
                strategy_spec = {
                    "type": "Rolling",
                    "rolling_params": {
                        "max_surge": strategy.max_surge,
                        "max_unavailable": strategy.max_unavailable,
                        "timeout_seconds": strategy.timeout_seconds,
                        "update_period_seconds": strategy.update_period_seconds,
                        "interval_seconds": strategy.interval_seconds
                    }
                }

                if DeploymentConfigHelper.__get_deployment_strategy_params(strategy.post) is not None:
                    strategy_spec["rolling_params"]["post"] = DeploymentConfigHelper.__get_deployment_strategy_params(strategy.post)

                return strategy_spec

            elif strategy.type == "openshift.RecreateDeploymentStrategyParamsSpec":
                strategy_spec =  {
                    "type": "Recreate",
                    "recreate_params": {
                        "timeout_seconds": strategy.timeout_seconds
                    }
                }

                if DeploymentConfigHelper.__get_deployment_strategy_params(strategy.pre) is not None:
                    strategy_spec["recreate_params"]["pre"] = DeploymentConfigHelper.__get_deployment_strategy_params(strategy.pre)

                if DeploymentConfigHelper.__get_deployment_strategy_params(strategy.post) is not None:
                    strategy_spec["recreate_params"]["post"] = DeploymentConfigHelper.__get_deployment_strategy_params(strategy.post)

                return strategy_spec

            else:
                return None

    @staticmethod
    def __get_deployment_triggers(triggers):
        list_of_triggers = []
        for trigger in triggers:
            list_of_triggers.append({
                "type": trigger.type_of_trigger,
                "image_change_params": {
                    "automatic": True if trigger.automatic else False,
                    "container_names": trigger.container_names,
                    "from": trigger.from_object,
                    "last_triggered_image": trigger.last_triggered_image
                }
            })

        return list_of_triggers

    @staticmethod
    def __get_deployment_strategy_params(during):
        return during if during else None
