import json
import os
import tempfile
from com.xebialabs.deployit.plugin.openshift import LoaderUtil
from java.nio.file import Files, Paths, StandardCopyOption

import yaml
from kubernetes.client.apis import core_v1_api
from kubernetes.client.configuration import ConfigurationObject
from openshift import client
from openshift.client import ApiClient
from resource_provider import get_resource


class OpenShiftConnector(object):
    def __init__(self, server):
        self.ca_bundle_path = self._set_ca_bundle_path()

        server_url = server.serverUrl
        token = server.openshiftToken

        apiclient_config = ConfigurationObject()
        apiclient_config.verify_ssl = server.verifyCertificates
        apiclient_config.api_key['authorization'] = "Bearer {0}".format(token)
        apiclient_config.ssl_ca_cert = self.ca_bundle_path

        if not server.verifyCertificates:
            self._disable_https_warnings()

        self.apiclient = ApiClient(host=server_url, config=apiclient_config)
        self.oapi = client.OapiApi(self.apiclient)
        self.k8sapi = core_v1_api.CoreV1Api(self.apiclient)

    def client(self):
        return self.apiclient

    def oapi_client(self):
        return self.oapi

    def create_resource_from_object(self, namespace, resource_object):
        provider = get_resource(resource_object["kind"], self)
        k8sapi_result = provider.create(namespace=namespace, resource_definition=resource_object)
        print("Deployment created. status='%s'" % str(k8sapi_result.status))
        return k8sapi_result

    def modify_resource_from_object(self, namespace, resource_object):
        provider = get_resource(resource_object["kind"], self)
        k8sapi_result = provider.modify(namespace=namespace, resource_definition=resource_object)
        print("Deployment modified. status='%s'" % str(k8sapi_result.status))
        return k8sapi_result

    def delete_resource_from_object(self, namespace, resource_object):
        provider = get_resource(resource_object["kind"], self)
        k8sapi_result = provider.delete(namespace=namespace, resource_definition=resource_object)
        return k8sapi_result

    def filter_resources_by_object(self, namespace, resource_object):
        provider = get_resource(resource_object["kind"], self)
        k8sapi_result = provider.filter_resources_by_definition(
            namespace=namespace,
            resource_definition=resource_object
        )
        return k8sapi_result

    def parse_resource_definition(self, definition_string):
        try:
            return json.loads(definition_string)
        except:
            try:
                return yaml.safe_load(definition_string)
            except:
                raise

    @staticmethod
    def _disable_https_warnings():
        import urllib3
        urllib3.disable_warnings()

    def _set_ca_bundle_path(self):
        ca_bundle_path = self._extract_file_from_jar("certifi/cacert.pem")
        os.environ['REQUESTS_CA_BUNDLE'] = ca_bundle_path
        return ca_bundle_path

    def _extract_file_from_jar(self, config_file):
        file_url = LoaderUtil.getResourceBySelfClassLoader(config_file)
        if file_url:
            tmp_file = None
            try:
                tmp_file, tmp_abs_path = tempfile.mkstemp()
                Files.copy(file_url.openStream(), Paths.get(tmp_abs_path), StandardCopyOption.REPLACE_EXISTING)
                return tmp_abs_path
            finally:
                if tmp_file:
                    tmp_file.close()
        else:
            return None
