from abc import ABCMeta, abstractmethod

from kubernetes.client import V1DeleteOptions


def get_resource(resource_type, connector):
    _supported_resources = "Pod", "Service", "Route", "ImageStream", "BuildConfig", "DeploymentConfig"

    if resource_type not in _supported_resources:
        raise ValueError("Unsupported resource type: {0}".format(resource_type))

    klass = globals()["{0}ResourceProvider".format(resource_type)]
    return klass(connector)


class Resource(object):
    __metaclass__ = ABCMeta

    def __init__(self, connector):
        self.k8sapi = connector.k8sapi
        self.oapi = connector.oapi

    @abstractmethod
    def create(self, namespace, resource_definition):
        pass

    @abstractmethod
    def modify(self, namespace, resource_definition):
        pass

    @abstractmethod
    def delete(self, namespace, resource_definition):
        pass

    @abstractmethod
    def filter_resources_by_definition(self, namespace, resource_definition):
        pass


class PodResourceProvider(Resource):
    def create(self, namespace, resource_definition):
        response = self.k8sapi.create_namespaced_pod(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.k8sapi.patch_namespaced_pod(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = V1DeleteOptions()
        response = self.k8sapi.delete_namespaced_pod(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.k8sapi.list_namespaced_pod(
            namespace=namespace,
            field_selector=field_selector
        )
        return response


class ServiceResourceProvider(Resource):
    def create(self, namespace, resource_definition):
        response = self.k8sapi.create_namespaced_service(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.k8sapi.patch_namespaced_service(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        response = self.k8sapi.delete_namespaced_service(
            namespace=namespace,
            name=resource_definition["metadata"]["name"]
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.k8sapi.list_namespaced_service(
            namespace=namespace,
            field_selector=field_selector
        )
        return response


class RouteResourceProvider(Resource):
    def create(self, namespace, resource_definition):
        response = self.oapi.create_namespaced_route(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.oapi.patch_namespaced_route(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = V1DeleteOptions()
        response = self.oapi.delete_namespaced_route(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.oapi.list_namespaced_route(
            namespace=namespace,
            field_selector=field_selector
        )
        return response


class ImageStreamResourceProvider(Resource):
    def create(self, namespace, resource_definition):
        response = self.oapi.create_namespaced_image_stream(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.oapi.patch_namespaced_image_stream(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = V1DeleteOptions()
        response = self.oapi.delete_namespaced_image_stream(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.oapi.list_namespaced_image_stream(
            namespace=namespace,
            field_selector=field_selector
        )
        return response


class BuildConfigResourceProvider(Resource):
    def create(self, namespace, resource_definition):
        response = self.oapi.create_namespaced_build_config(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.oapi.patch_namespaced_build_config(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = V1DeleteOptions()
        response = self.oapi.delete_namespaced_build_config(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.oapi.list_namespaced_image_stream(
            namespace=namespace,
            field_selector=field_selector
        )
        return response


class DeploymentConfigResourceProvider(Resource):
    def create(self, namespace, resource_definition):
        response = self.oapi.create_namespaced_deployment_config(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.oapi.patch_namespaced_deployment_config(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = V1DeleteOptions()
        response = self.oapi.delete_namespaced_deployment_config(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.oapi.list_namespaced_deployment_config(
            namespace=namespace,
            field_selector=field_selector
        )
        return response
