from kubernetes.client import V1DeleteOptions
from xld.kubernetes.resource.provider import Resource
from xld.openshift.connector import OpenShiftConnector
from abc import abstractmethod


class OpenShiftResource(Resource):
    def __init__(self, container):
        super(OpenShiftResource, self).__init__(container)
        connector = OpenShiftConnector(container.container)
        self.oapi = connector.oapi

    @abstractmethod
    def create(self, namespace, resource_definition):
        pass

    @abstractmethod
    def modify(self, namespace, resource_definition):
        pass

    @abstractmethod
    def delete(self, namespace, resource_definition):
        pass

    @abstractmethod
    def filter_resources_by_definition(self, namespace, resource_definition):
        pass


class RouteResourceProvider(OpenShiftResource):
    def create(self, namespace, resource_definition):
        response = self.oapi.create_namespaced_route(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.oapi.patch_namespaced_route(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = V1DeleteOptions()
        response = self.oapi.delete_namespaced_route(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.oapi.list_namespaced_route(
            namespace=namespace,
            field_selector=field_selector
        )
        return response


class ImageStreamResourceProvider(OpenShiftResource):
    def create(self, namespace, resource_definition):
        response = self.oapi.create_namespaced_image_stream(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.oapi.patch_namespaced_image_stream(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = V1DeleteOptions()
        response = self.oapi.delete_namespaced_image_stream(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.oapi.list_namespaced_image_stream(
            namespace=namespace,
            field_selector=field_selector
        )
        return response


class BuildConfigResourceProvider(OpenShiftResource):
    def create(self, namespace, resource_definition):
        response = self.oapi.create_namespaced_build_config(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.oapi.patch_namespaced_build_config(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = V1DeleteOptions()
        response = self.oapi.delete_namespaced_build_config(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.oapi.list_namespaced_image_stream(
            namespace=namespace,
            field_selector=field_selector
        )
        return response


class DeploymentConfigResourceProvider(OpenShiftResource):
    def create(self, namespace, resource_definition):
        response = self.oapi.create_namespaced_deployment_config(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.oapi.patch_namespaced_deployment_config(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = V1DeleteOptions()
        response = self.oapi.delete_namespaced_deployment_config(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.oapi.list_namespaced_deployment_config(
            namespace=namespace,
            field_selector=field_selector
        )
        return response
