from kubernetes.client import V1DeleteOptions
from openshift.dynamic import DynamicClient
from xld.kubernetes.resource.provider import Resource
from xld.kubernetes import client
from xld.openshift.connector import OpenShiftConnector
from abc import abstractmethod


class OpenShiftResource(Resource):
    def __init__(self, container, api_version='v1'):
        super(OpenShiftResource, self).__init__(container, api_version)
        connector = OpenShiftConnector(container.container)
        self.oapi = connector.oapi
        self.apiclient = connector.apiclient

    def get_api_client(self, kind):
        print ("[Using API version: {}]".format(self.api_version))
        dynamic_client = DynamicClient(self.oapi.api_client)
        return dynamic_client.resources.get(api_version=self.api_version, kind=kind)

    @abstractmethod
    def create(self, namespace, resource_definition):
        pass

    @abstractmethod
    def modify(self, namespace, resource_definition):
        pass

    @abstractmethod
    def delete(self, namespace, resource_definition):
        pass

    @abstractmethod
    def filter_resources_by_definition(self, namespace, resource_definition):
        pass

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        return self.apiclient.rest_client.pool_manager.__exit__(exc_type, exc_val, exc_tb)


class OpenshiftResourceProvider(OpenShiftResource):
    def __init__(self, container, kind, api_version):
        super(OpenshiftResourceProvider, self).__init__(container, api_version)
        self.kind=kind
        self.api_client = self.get_api_client(self.kind)
        self.patch_types = {
            "json": "application/json-patch+json",
            "merge": "application/merge-patch+json",
            "strategic": "application/strategic-merge-patch+json"
        }

    def create(self, namespace, resource_definition):
        return self.api_client.create(body=resource_definition, namespace=namespace)

    def modify(self, namespace, resource_definition, patch_type='strategic', update_method='patch'):
        if update_method == 'patch':
            if patch_type not in self.patch_types:
                raise RuntimeError("Merge patch type '{}' not supported. Please use 'json', 'merge' or 'strategic'.".format(patch_type))
            return self.api_client.patch(body=resource_definition, namespace=namespace, content_type=self.patch_types[patch_type])
        else:
            return self.api_client.replace(body=resource_definition, namespace=namespace)

    def delete(self, namespace, resource_definition, propagation_policy='Foreground'):
        return self.api_client.delete(name=resource_definition["metadata"]["name"], namespace=namespace, body=client.V1DeleteOptions(
            propagation_policy=propagation_policy
        ))

    def filter_resources_by_definition(self, namespace, resource_definition):
        return self.api_client.get(namespace=namespace, field_selector="metadata.name={}".format(
            resource_definition["metadata"]["name"]))


class RouteResourceProvider(OpenShiftResource):
    def create(self, namespace, resource_definition):
        response = self.oapi.create_namespaced_route(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.oapi.patch_namespaced_route(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = V1DeleteOptions()
        response = self.oapi.delete_namespaced_route(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.oapi.list_namespaced_route(
            namespace=namespace,
            field_selector=field_selector
        )
        return response


class ImageStreamResourceProvider(OpenShiftResource):
    def create(self, namespace, resource_definition):
        response = self.oapi.create_namespaced_image_stream(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.oapi.patch_namespaced_image_stream(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = V1DeleteOptions()
        response = self.oapi.delete_namespaced_image_stream(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.oapi.list_namespaced_image_stream(
            namespace=namespace,
            field_selector=field_selector
        )
        return response


class BuildConfigResourceProvider(OpenShiftResource):
    def create(self, namespace, resource_definition):
        response = self.oapi.create_namespaced_build_config(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.oapi.patch_namespaced_build_config(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = V1DeleteOptions()
        response = self.oapi.delete_namespaced_build_config(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.oapi.list_namespaced_image_stream(
            namespace=namespace,
            field_selector=field_selector
        )
        return response


class DeploymentConfigResourceProvider(OpenShiftResource):
    def create(self, namespace, resource_definition):
        response = self.oapi.create_namespaced_deployment_config(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.oapi.patch_namespaced_deployment_config(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = V1DeleteOptions()
        response = self.oapi.delete_namespaced_deployment_config(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.oapi.list_namespaced_deployment_config(
            namespace=namespace,
            field_selector=field_selector
        )
        return response
