package com.xebialabs.xldeploy.provisioner;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.services.SearchParameters;
import com.xebialabs.deployit.plugin.api.udm.*;
import com.xebialabs.deployit.plugin.api.xld.AppliedDistribution;
import com.xebialabs.deployit.plugin.api.xld.DistributionVersion;
import com.xebialabs.deployit.plugin.api.xld.Domain;

import java.util.*;
import java.util.stream.Collectors;

@Metadata(description = "A provisioned blueprint of a provisioning package to a provisioning environment.")
@TypeIcon("icons/types/upm.ProvisionedBlueprint.svg")
public class ProvisionedBlueprint extends AppliedDistribution {

    @Property(description = "The package that was provisioned to the provisioning environment.", candidateValuesFilter = "findProvisioningPackage")
    private ProvisioningPackage provisioningPackage;

    @Property(asContainment = true)
    private ProvisioningEnvironment provisioningEnvironment;

    @Property(description = "The provisioned items that were provisioned to the provisioning environment.")
    private Set<Deployed> provisioneds = new HashSet<>();

    @Property(description = "The environment that was provisioned.", required = false, asContainment = false, category = "Provisioning")
    private Environment provisionedEnvironment;

    @Property(description = "Resolved configuration items from templates on provisioning package", required = false, asContainment = false, category = "Provisioning")
    private Set<ConfigurationItem> boundConfigurationItems;

    @Property(description = "Placeholders provided by the user at provisioning.", required = false)
    private Map<String, String> unresolvedPlaceholders = new HashMap<>();

    @Property(description = "The id generated for by XLD to distinguish this provisioning", required = true, category = "Provisioning")
    private String provisioningId;

    @Property(description = "The environment name of the the to be provisioned environment", required = true, category = "Provisioning")
    private String environmentName;

    @Property(description = "The directory path where the provisioned environment and provisioned configuration items will be stored.", required = false, category = "Provisioning")
    private String directoryPath;

    @Override
    public DistributionVersion getVersion() {
        return provisioningPackage;
    }

    @Override
    public void setVersion(DistributionVersion version) {
        provisioningPackage = (ProvisioningPackage) version;
    }

    @Override
    public Domain getEnvironment() {
        return provisioningEnvironment;
    }

    @Override
    public void setEnvironment(Domain Domain) {
        provisioningEnvironment = (ProvisioningEnvironment) Domain;
    }

    @Override
    public Set<Deployed> getDeployeds() {
        return provisioneds;
    }

    @Override
    public void addDeployed(Deployed deployed) {
        provisioneds.add(deployed);
    }

    public ProvisioningPackage getProvisioningPackage() {
        return provisioningPackage;
    }

    public ProvisioningEnvironment getProvisioningEnvironment() {
        return provisioningEnvironment;
    }

    public Map<String, String> getUnresolvedPlaceholders() {
        return unresolvedPlaceholders;
    }

    public Map<String, String> getUnresolvedPlaceholdersWithValues() {
        return getUnresolvedPlaceholders().entrySet().stream().filter(e -> !e.getValue().isEmpty()).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
    }

    public void setProvisioningPackage(ProvisioningPackage provisioningPackage) {
        this.provisioningPackage = provisioningPackage;
    }

    public void setProvisioningEnvironment(ProvisioningEnvironment provisioningEnvironment) {
        this.provisioningEnvironment = provisioningEnvironment;
    }

    public void setProvisionedEnvironment(Environment environment) {
        this.provisionedEnvironment = environment;
    }

    public Environment getProvisionedEnvironment() {
        return provisionedEnvironment;
    }

    public void setUnresolvedPlaceholders(Map<String, String> unresolvedPlaceholders) {
        this.unresolvedPlaceholders = unresolvedPlaceholders;
    }

    @Override
    public void setDeployeds(Set<Deployed> deployeds) {
        this.provisioneds = deployeds;
    }

    @Override
    public void addDeployeds(Collection<Deployed> deployeds) {
        this.provisioneds.addAll(deployeds);
    }

    public Set<ConfigurationItem> getBoundConfigurationItems() {
        return boundConfigurationItems;
    }

    public void setBoundConfigurationItems(Set<ConfigurationItem> boundConfigurationItems) {
        this.boundConfigurationItems = boundConfigurationItems;
    }

    public void setProvisioningId(String provisioningId) {
        this.provisioningId = provisioningId;
    }

    public void setEnvironmentName(String environmentName) {
        this.environmentName = environmentName;
    }

    public void setDirectoryPath(String directoryPath) {
        this.directoryPath = directoryPath;
    }

    public String getEnvironmentName() {
        return environmentName;
    }

    public String getDirectoryPath() {
        return directoryPath;
    }

    public String getProvisioningId() {
        return provisioningId;
    }

    @CandidateValuesFilter(name = "findProvisioningPackage")
    public static SearchParameters findProvisioningPackage(ConfigurationItem context, PropertyDescriptor property) {
        ProvisionedBlueprint da = (ProvisionedBlueprint) context;
        SearchParameters params = new SearchParameters();
        params.setType(property.getReferencedType());
        ProvisioningPackage provisioningPackage = da.getProvisioningPackage();
        if (provisioningPackage != null) {
            params.setParent(provisioningPackage.getBlueprint().getId());
        }
        return params;
    }
}