import requests
import json


# noinspection PyPep8Naming
class Provisioner(object):
    def __init__(self):
        self.base_api_url = authConfig.apiUrl()
        self.auth = authConfig

    def initialProvisioning(self, packageId, provisioningEnvironmentId, placeholders=None):
        if not placeholders:
            placeholders = {}
        request_data = {
            "packageId": packageId,
            "provisioningEnvironmentId": provisioningEnvironmentId,
            "placeholders": placeholders
        }
        response = self._make_post_request("provisioning/initial", request_data)
        return self._process_response(response)

    def mapAllProvisionables(self, provision):
        response = self._make_post_request("provisioning/map/all", provision)
        return self._process_response(response)

    def mapSelectedProvisionables(self, selectedProvisionableIds, provision):
        request_data = {
            "selectedProvisionableIds": selectedProvisionableIds,
            "provision":provision
        }
        response = self._make_post_request("provisioning/map/selected", request_data)
        return self._process_response(response)

    def createProvisioningTask(self, provision):
        response = self._make_post_request("provisioning/task", provision)
        return self._process_response(response)

    def deprovision(self, provisionedBlueprintId):
        request_data = {
            "provisionedBlueprintId": provisionedBlueprintId
        }
        response = self._make_post_request("provisioning/destroy", request_data)
        return self._process_response(response)

    def validate(self, provision):
        response = self._make_post_request("provisioning/validate", provision)
        return self._process_response(response)

    def preview(self, provision, blockId=None, stepNumber=None):
        endpoint = "provisioning/preview"
        if blockId and stepNumber:
            endpoint = "provisioning/preview/{0}/{1}".format(blockId, stepNumber)
        response = self._make_post_request(endpoint, provision)
        return self._process_response(response)

    def _make_post_request(self, rest_endpoint, request_data):
        request_url = "{0}/{1}".format(self.base_api_url, rest_endpoint)
        headers = {"Content-Type": "application/json"}
        response = requests.post(request_url, headers=headers, auth=(self.auth.username(), self.auth.password()),
                                 data=json.dumps(request_data))
        return response

    def _process_response(self, response):
        if response.status_code == requests.codes.created or response.status_code == requests.codes.ok or response.status_code == 412:
            return self._parse_response(response)
        else:
            raise ProvisioningException(response.text)

    def _parse_response(self, response):
        return json.loads(response.text, object_hook=lambda dict: AttributeAccessDict(dict))


class AttributeAccessDict(dict):
    def __getattr__(self, key):
        return self[key]

    def __setattr__(self, key, value):
        self[key] = value

class ProvisioningException(Exception):
    def __init__(self, message):
        self.message = message

    def message(self):
        return self.message

    def __str__(self):
        return str(self.message)
