package com.xebialabs.xldeploy.auth.config

import ai.digital.config.ServerConfigurationHelper
import ai.digital.configuration.central.deploy.{ClientProperties, ServerSideProperties}
import com.xebialabs.deployit.core.auth.LoginMetadataService
import com.xebialabs.deployit.engine.api.distribution.TaskExecutionWorkerRepository
import com.xebialabs.deployit.plumbing.authentication.WithoutRedirectLoginSuccessHandler
import com.xebialabs.deployit.security.RoleService
import com.xebialabs.deployit.security.authentication.{BasicAuthWithRememberMeFilter, RememberMeAuthenticationProvider, XLCrowdAuthenticationProvider}
import com.xebialabs.deployit.taskexecution.security.{TaskWorkerAuthenticationFilter, TaskWorkerAuthenticationProvider}
import com.xebialabs.deployit.{LicenseExpiryCheckFilter, LogbackAccessSecurityAttributesSaveFilter}
import com.xebialabs.license.LicenseValidationFilter
import com.xebialabs.license.service.LicenseService
import com.xebialabs.xldeploy.auth.{BasicAuthOverridingHttpSessionSecurityContextRepository, XlPersistentTokenBasedRememberMeServices}
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.{Autowired, Qualifier, Value}
import org.springframework.context.ApplicationContext
import org.springframework.context.annotation.{Bean, Configuration, Lazy}
import org.springframework.http.HttpMethod
import org.springframework.security.authentication.ProviderManager
import org.springframework.security.web.util.matcher.AntPathRequestMatcher.antMatcher

import java.util
import scala.jdk.CollectionConverters.ListHasAsScala
import org.springframework.security.authentication.{AuthenticationManager, AuthenticationProvider}
import org.springframework.security.config.annotation.web.builders.{HttpSecurity, WebSecurity}
import org.springframework.security.config.http.SessionCreationPolicy
import org.springframework.security.ldap.authentication.LdapAuthenticationProvider
import org.springframework.security.web.access.AccessDeniedHandler
import org.springframework.security.web.access.intercept.RequestAuthorizationContext
import org.springframework.security.web.authentication.logout.HttpStatusReturningLogoutSuccessHandler
import org.springframework.security.web.authentication.{AuthenticationFailureHandler, DelegatingAuthenticationEntryPoint, UsernamePasswordAuthenticationFilter}
import org.springframework.security.web.context.request.async.WebAsyncManagerIntegrationFilter
import org.springframework.security.web.firewall.StrictHttpFirewall
import org.springframework.security.web.savedrequest.NullRequestCache

import scala.jdk.CollectionConverters._


@Configuration
abstract class DeploySecurityConfig {

  private val logger = LoggerFactory.getLogger(classOf[DeploySecurityConfig])

  @Value("${deploy.server.license.days-before-warning:5}")
  var daysBeforeWarning: Int = _

  @Autowired
  var applicationContext: ApplicationContext = _

  @Autowired
  var loginMetadataService: LoginMetadataService = _

  @Autowired
  var roleService: RoleService = _

  @Autowired
  var accessDeniedHandler: AccessDeniedHandler = _

  @Autowired
  @Qualifier("rememberMeKey")
  var rememberMeKey: String = _

  @Autowired
  var rememberMeServices: XlPersistentTokenBasedRememberMeServices = _

  @Autowired
  var withoutRedirectLogoutSuccessHandler: HttpStatusReturningLogoutSuccessHandler = _

  @Autowired
  var workerRepository: TaskExecutionWorkerRepository = _

  @Autowired
  var licenseService: LicenseService = _

  @Autowired
  var taskWorkerAuthenticationProvider: TaskWorkerAuthenticationProvider = _

  @Autowired(required = false)
  var ldapAuthenticationProvider: java.util.List[LdapAuthenticationProvider] = _

  @Autowired(required = false)
  var xlCrowdAuthenticationProvider: java.util.List[XLCrowdAuthenticationProvider] = _

  @Lazy
  @Autowired
  @Qualifier("xlAuthenticationProvider")
  var xlAuthenticationProvider: AuthenticationProvider = _

  @Autowired
  var delegatingAuthenticationEntryPoint: DelegatingAuthenticationEntryPoint = _

  @Autowired
  var delegatingSecurityContextRepository: BasicAuthOverridingHttpSessionSecurityContextRepository = _

  @Autowired
  var authenticationFailureHandler: AuthenticationFailureHandler = _

  @Autowired
  var serverSideConfiguration: ServerSideProperties = _

  @Autowired
  var clientProperties: ClientProperties = _

  @Bean
  @throws[Exception]
  def authenticationManager: AuthenticationManager = {
    val authenticationProviders = new util.ArrayList[AuthenticationProvider]()
    authenticationProviders.add(taskWorkerAuthenticationProvider)
    authenticationProviders.add(new RememberMeAuthenticationProvider())
    authenticationProviders.add(xlAuthenticationProvider)
    if (ldapAuthenticationProvider != null) {
      for (provider <- ldapAuthenticationProvider.asScala) {
        authenticationProviders.add(provider)
      }
    }
    if (xlCrowdAuthenticationProvider != null) {
      for (provider <- xlCrowdAuthenticationProvider.asScala) {
        authenticationProviders.add(provider)
      }
    }
    authenticationProviders.add(InMemoryConfigurer.centralConfigDaoProvider)
    new ProviderManager(authenticationProviders)
  }

  def configureSecurity(web: WebSecurity, prefix: String): Unit = {
    val firewall = new StrictHttpFirewall()
    firewall.setAllowUrlEncodedSlash(true)
    firewall.setAllowUrlEncodedPercent(true)

    if (serverSideConfiguration.http.allowedHosts.enabled) {
      firewall.setAllowedHostnames(serverSideConfiguration.http.allowedHosts.hostnames.contains)
    }

    web.httpFirewall(firewall)

    web.ignoring().requestMatchers(antMatcher("/productregistration"))
    web.ignoring().requestMatchers(antMatcher(s"/$prefix/internal/configuration/properties"))
    web.ignoring().requestMatchers(antMatcher(HttpMethod.GET, s"/$prefix/settings/general"))
    web.ignoring().requestMatchers(antMatcher(s"/$prefix/ha/health"))
    if (clientProperties.ignoreAuthCheckForServerStateApi)
      web.ignoring().requestMatchers(antMatcher(s"/$prefix/server/state"))
    web.ignoring().requestMatchers(antMatcher("/**.js**"))
    web.ignoring().requestMatchers(antMatcher("/**.html"))
    web.ignoring().requestMatchers(antMatcher("/**.css"))
    web.ignoring().requestMatchers(antMatcher("/favicon.ico"))
    web.ignoring().requestMatchers(antMatcher("/fonts/**"))
    web.ignoring().requestMatchers(antMatcher("/icons/**"))
    web.ignoring().requestMatchers(antMatcher("/images/**"))
  }

  def licenseValidationFilter: LicenseValidationFilter = {
    val filter = new LicenseValidationFilter()
    filter.setLicenseService(licenseService)
    filter
  }

  def configureSecurity(http: HttpSecurity, prefix: String): Unit = {
    val loginAuthorizationManager = new LoginAuthorizationManager[RequestAuthorizationContext]()
    loginAuthorizationManager.setWhitelistUrls(Set("/deployit/auth", "/xldeploy/auth").asJava)
    http
      .sessionManagement(smc =>
        smc
          .sessionCreationPolicy(SessionCreationPolicy.NEVER)
          .maximumSessions(serverSideConfiguration.session.maximumSessions)
          .maxSessionsPreventsLogin(serverSideConfiguration.session.exceptionIfMaximumExceeded)
      )
      .securityContext(sc =>
        sc
          .securityContextRepository(delegatingSecurityContextRepository)
          .requireExplicitSave(false)
          .configure(http)
      )
      .addFilterBefore(licenseValidationFilter, classOf[WebAsyncManagerIntegrationFilter])
      .addFilterAfter(new TaskWorkerAuthenticationFilter(authenticationManager, workerRepository), classOf[UsernamePasswordAuthenticationFilter])
      .addFilterBefore(new LogbackAccessSecurityAttributesSaveFilter(), classOf[TaskWorkerAuthenticationFilter])
      .addFilterAfter(new BasicAuthWithRememberMeFilter(authenticationManager, delegatingAuthenticationEntryPoint), classOf[TaskWorkerAuthenticationFilter])
      .addFilterAfter(new LicenseExpiryCheckFilter(licenseService, daysBeforeWarning), classOf[TaskWorkerAuthenticationFilter])
      .exceptionHandling(eh => eh.authenticationEntryPoint(delegatingAuthenticationEntryPoint))
      .formLogin(
        fl => fl
          .failureHandler(authenticationFailureHandler)
          .successHandler(new WithoutRedirectLoginSuccessHandler())
          .loginPage("/login")
      )
      .rememberMe(
        rm => rm
          .key(rememberMeKey)
          .rememberMeServices(rememberMeServices)
      )
      .logout(
        lc => lc.logoutSuccessHandler(withoutRedirectLogoutSuccessHandler)
      )
      .requestCache(rcc => rcc.requestCache(new NullRequestCache()))
      .exceptionHandling(ehc => ehc.accessDeniedHandler(accessDeniedHandler))
      .authorizeHttpRequests(
        arc => {
          arc
            .requestMatchers(antMatcher(s"/$prefix/auth")).permitAll()
            .requestMatchers(antMatcher("/centralConfiguration/**")).hasRole(ServerConfigurationHelper.ConfigRole)
            .requestMatchers(antMatcher("/actuator/**")).hasRole("ADMIN")
            .requestMatchers(antMatcher("/v1/roles/**")).denyAll()
            .requestMatchers(antMatcher("/ws")).fullyAuthenticated()
            .requestMatchers(antMatcher(s"/$prefix/**")).access(loginAuthorizationManager)
            .requestMatchers(antMatcher(s"/$prefix/**")).fullyAuthenticated()
            .requestMatchers(antMatcher("/ciExplorerDist/**")).permitAll()
            .requestMatchers(antMatcher("/stitchWorkbenchDist/**")).permitAll()
            .requestMatchers(antMatcher("/pluginManagerDist/**")).permitAll()
            .requestMatchers(antMatcher("/dist/**")).permitAll()
            .requestMatchers(antMatcher(HttpMethod.GET,"/")).permitAll()
            .requestMatchers(antMatcher("/**")).fullyAuthenticated()
        }
      )
    logger.info("Starting url prefix '{}' with default security config", prefix)
  }
}
