package com.xebialabs.xldeploy.auth.config

import ai.digital.configuration.central.deploy.ClientProperties
import com.xebialabs.deployit.filter.CsrfHeaderFilter
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.context.annotation.Configuration
import org.springframework.core.annotation.Order
import org.springframework.security.config.annotation.web.builders.{HttpSecurity, WebSecurity}
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity
import org.springframework.security.web.authentication.logout.{CookieClearingLogoutHandler, LogoutFilter, SecurityContextLogoutHandler}
import org.springframework.security.web.csrf.{CsrfFilter, CsrfLogoutHandler, HttpSessionCsrfTokenRepository}

@ConditionalOnProperty(name = Array("deploy.server.security.auth.provider"), havingValue = "default", matchIfMissing = true)
@Configuration
@EnableWebSecurity
@Order(org.springframework.core.Ordered.HIGHEST_PRECEDENCE)
class DeployCsrfSecurityConfig extends DeploySecurityConfig {

  @Autowired
  var tokenRepository: HttpSessionCsrfTokenRepository = _

  @Autowired
  var clientConfiguration: ClientProperties = _

  private val prefix = "xldeploy"

  @throws[Exception]
  override protected def configure(http: HttpSecurity): Unit = {
    http.requestMatchers().antMatchers(s"/$prefix/**")

    configureSecurity(http, prefix = prefix)

    http
      .addFilterAfter(new CsrfHeaderFilter(clientConfiguration), classOf[CsrfFilter])
      .addFilterBefore(logoutFilter, classOf[LogoutFilter])
      .csrf()
      .csrfTokenRepository(tokenRepository)
  }

  private def logoutFilter: LogoutFilter = {
    val logoutFilter = new LogoutFilter(
            withoutRedirectLogoutSuccessHandler,
            new CsrfLogoutHandler(tokenRepository),
            new CookieClearingLogoutHandler("cookiesToClear", "XSRF-TOKEN"),
            rememberMeServices,
            new SecurityContextLogoutHandler()
    )
    logoutFilter.setFilterProcessesUrl("/xldeploy/logout")
    logoutFilter
  }

  override def configure(web: WebSecurity): Unit = {
    configureSecurity(web, prefix = prefix)
  }
}
