package com.xebialabs.xldeploy.auth.config

import com.xebialabs.deployit.MissingLoginPermissionErrorHandler
import com.xebialabs.deployit.core.auth.LoginMetadataService
import com.xebialabs.deployit.plumbing.authentication.BasicAuthenticationForbiddenEntryPoint
import com.xebialabs.deployit.security.authentication.BasicAuthenticationEntryPoint
import com.xebialabs.deployit.security.RoleService
import com.xebialabs.xldeploy.auth.{BasicAuthOverridingHttpSessionSecurityContextRepository, JdbcRememberMeTokenRepositoryImpl, XlPersistentTokenBasedRememberMeServices}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.annotation.{Bean, Configuration}
import org.springframework.http.HttpStatus
import org.springframework.security.core.AuthenticationException
import org.springframework.security.core.userdetails.UserDetailsService
import org.springframework.security.web.AuthenticationEntryPoint
import org.springframework.security.web.access.AccessDeniedHandler
import org.springframework.security.web.authentication.logout._
import org.springframework.security.web.authentication.session.SessionAuthenticationException
import org.springframework.security.web.authentication.{AuthenticationFailureHandler, DelegatingAuthenticationEntryPoint, HttpStatusEntryPoint, SimpleUrlAuthenticationFailureHandler}
import org.springframework.security.web.util.matcher.{RequestHeaderRequestMatcher, RequestMatcher}

import java.io.IOException
import java.util.{UUID, LinkedHashMap => JLinkedHashMap}
import jakarta.servlet.ServletException
import jakarta.servlet.http.{HttpServletRequest, HttpServletResponse}
import org.springframework.security.authentication.{BadCredentialsException, DisabledException}

@Configuration
class XLSecurityConfig extends Logging {

  @Autowired
  var jdbcRememberMeTokenRepositoryImpl: JdbcRememberMeTokenRepositoryImpl = _

  @Autowired
  var userDetailsService: UserDetailsService = _

  val DEFAULT_LOGIN_URL = "login"

  val ERROR_PARAMETER_NAME = "error"

  @Bean
  def rememberMeKey: String = UUID.randomUUID().toString

  @Bean
  def withoutRedirectLogoutSuccessHandler: HttpStatusReturningLogoutSuccessHandler =
    new HttpStatusReturningLogoutSuccessHandler()

  @Bean
  def delegatingSecurityContextRepository: BasicAuthOverridingHttpSessionSecurityContextRepository =
    new BasicAuthOverridingHttpSessionSecurityContextRepository()

  @Bean
  def rememberMeServices: XlPersistentTokenBasedRememberMeServices = {
    val services = new XlPersistentTokenBasedRememberMeServices(
      rememberMeKey,
      userDetailsService,
      jdbcRememberMeTokenRepositoryImpl)
    services.setTokenValiditySeconds(31556926)
    services.setCookieName("remember-me")
    services
  }

  @Bean
  def accessDeniedHandler(loginMetadataService: LoginMetadataService, roleService: RoleService): AccessDeniedHandler = {
    new MissingLoginPermissionErrorHandler(loginMetadataService, roleService)
  }

  @Bean
  def delegatingAuthenticationEntryPoint: DelegatingAuthenticationEntryPoint = {
    val entryPoints = new JLinkedHashMap[RequestMatcher, AuthenticationEntryPoint]
    entryPoints.put(
      new RequestHeaderRequestMatcher("X-HTTP-Auth-Override", "true"),
      new BasicAuthenticationForbiddenEntryPoint()
    )
    entryPoints.put(
      new RequestHeaderRequestMatcher("Accept", "application/json"),
      new HttpStatusEntryPoint(HttpStatus.UNAUTHORIZED)
    )

    val point = new DelegatingAuthenticationEntryPoint(entryPoints)
    point.setDefaultEntryPoint(new BasicAuthenticationEntryPoint())
    point
  }

  @Bean(name = Array("authenticationFailureHandler"))
  def authenticationFailureHandler: AuthenticationFailureHandler = {
    new SimpleUrlAuthenticationFailureHandler() {
      @throws[IOException]
      @throws[ServletException]
      override def onAuthenticationFailure(request: HttpServletRequest, response: HttpServletResponse, exception: AuthenticationException): Unit = {
        exception match {
          case badCredentialsException: BadCredentialsException =>
            if(badCredentialsException.getMessage.startsWith("An internal user is available")) {
              response.sendError(HttpStatus.FORBIDDEN.value, badCredentialsException.getMessage)
            } else {
              response.sendError(HttpStatus.UNAUTHORIZED.value, badCredentialsException.getMessage)
            }
          case authException: DisabledException => response.sendError(HttpStatus.FORBIDDEN.value, authException.getMessage)
          case exception: SessionAuthenticationException => response.sendError(HttpStatus.UNAUTHORIZED.value, exception.getMessage)
          case _ => super.onAuthenticationFailure(request, response, exception)
        }
      }
    }
  }

  @Bean(name = Array("deployAuthenticationFailureHandler"))
  def deployAuthenticationFailureHandler: AuthenticationFailureHandler = {
    new SimpleUrlAuthenticationFailureHandler() {
      @throws[IOException]
      @throws[ServletException]
      override def onAuthenticationFailure(request: HttpServletRequest, response: HttpServletResponse, exception: AuthenticationException): Unit = {
        logger.debug(s"xld:auth: deployAuthenticationFailureHandler called - request URI: ${request.getRequestURI}, context path: ${request.getContextPath}, exception type: ${exception.getClass.getSimpleName}")

        exception match {
          case badCredentialsException: BadCredentialsException =>
            val referer = Option(request.getHeader("Referer")).getOrElse("")
             // robustly detect login referer even if referer contains query params or fragments
             val refererIsLogin = try {
               val uri = new java.net.URI(referer)
               val path = Option(uri.getPath).getOrElse("")
               logger.debug(s"xld:auth: deployAuthenticationFailureHandler called - request URI path: $path , fragment: ${uri.getFragment}")
               val frag = Option(uri.getFragment).getOrElse("") // fragment doesn't include '#'
               path.endsWith("/" + DEFAULT_LOGIN_URL) || frag.startsWith("/" + DEFAULT_LOGIN_URL)
             } catch {
               case _: Exception =>
                 logger.debug("xld:auth:URI error", exception)
                 // fallback to previous contains/endsWith checks
                 referer.endsWith("/" + DEFAULT_LOGIN_URL)
             }
            logger.debug(s"xld:auth: BadCredentialsException - referer='$referer', refererIsLogin=$refererIsLogin, exception=${badCredentialsException.getMessage}")
            if (refererIsLogin || badCredentialsException.getMessage.startsWith("An internal user is available")) {
              // include context path so redirects preserve application context
              val redirectUrl = request.getContextPath + "/" + DEFAULT_LOGIN_URL + "?" + ERROR_PARAMETER_NAME + "=" + badCredentialsException.getMessage
              logger.debug(s"xld:auth: BadCredentialsException - redirecting to: $redirectUrl")
              response.sendRedirect(redirectUrl)
            } else {
              logger.debug(s"xld:auth: BadCredentialsException (other) - delegating to parent handler")
              super.onAuthenticationFailure(request, response, exception)
            }
          case _ =>  
            // include context path so redirects preserve application context
            val redirectUrl = request.getContextPath + "/" + DEFAULT_LOGIN_URL + "?" + ERROR_PARAMETER_NAME + "=" + exception.getMessage
            logger.debug(s"xld:auth: Other authentication exception (${exception.getClass.getSimpleName}) - redirecting to: $redirectUrl")

            val accept = Option(request.getHeader("Accept")).getOrElse("")
            val xRequestedWith = Option(request.getHeader("X-Requested-With")).getOrElse("")
            val isAjax = xRequestedWith.equalsIgnoreCase("XMLHttpRequest") || accept.contains("application/json")

            logger.debug(s"xld:auth: handling authentication failure - requestURI=${request.getRequestURI}, contextPath=${request.getContextPath}, isAjax=$isAjax, exception=${exception.getClass.getSimpleName}")

            if (isAjax) {
              val rawMsg = Option(exception.getMessage).getOrElse("")
              val lower = rawMsg.toLowerCase
              val status = if (lower.contains("permission") || lower.contains("not allowed to login") || lower.contains("do not have permission") || lower.contains("login denied")) HttpServletResponse.SC_FORBIDDEN else HttpServletResponse.SC_UNAUTHORIZED
              logger.debug(s"xld:auth: AJAX branch - sending error status=$status, message=${if (rawMsg.length>200) rawMsg.substring(0,200) + "..." else rawMsg}")
              response.sendError(status, rawMsg)
            } else {
              logger.debug(s"xld:auth: non-AJAX branch - redirecting to: $redirectUrl")
              response.sendRedirect(redirectUrl)
            }
        }
        logger.debug("xld:auth: deployAuthenticationFailureHandler completed")
      }
    }
  }
}
