package com.xebialabs.deployit.core.service

import ai.digital.deploy.sql.http.enricher.PaginationService
import com.xebialabs.deployit.core.service.TasksService.FieldSorter
import com.xebialabs.deployit.core.service.TasksService.TaskSearchParameters
import com.xebialabs.deployit.core.service.TasksService.dateTimeOrdering
import com.xebialabs.deployit.core.service.TasksService.extractAndMergeMetadataFields
import com.xebialabs.deployit.core.service.TasksService.extractMetadata
import com.xebialabs.deployit.core.service.TasksService.toOrdering
import com.xebialabs.deployit.core.util.TaskFilterUtils
import com.xebialabs.deployit.engine.api.dto
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId
import com.xebialabs.deployit.engine.api.dto.Paging
import com.xebialabs.deployit.engine.api.execution._
import com.xebialabs.deployit.engine.tasker.repository.ActiveTaskRepository
import com.xebialabs.deployit.engine.tasker.TaskExecutionEngine
import com.xebialabs.deployit.engine.tasker.TaskId
import com.xebialabs.deployit.security.permission.PermissionHelper
import com.xebialabs.deployit.security.permission.PlatformPermissions
import com.xebialabs.deployit.spring.BeanWrapper
import com.xebialabs.deployit.task.TaskMetadata
import com.xebialabs.deployit.task.TaskType
import org.joda.time.DateTime
import org.joda.time.Interval
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util.function.BiConsumer
import scala.jdk.CollectionConverters._

object TasksService {
  implicit def dateTimeOrdering: Ordering[DateTime] = Ordering.fromLessThan { (left, right) =>
    Option(left) -> Option(right) match {
      case (Some(x), Some(y)) => x.isBefore(y)
      case (None, Some(_)) => false
      case (Some(_), None) => true
      case (None, None) => false
    }
  }

  case class TaskSearchParameters(begin: DateTime,
                                  end: DateTime,
                                  states: List[String],
                                  users: List[String],
                                  taskName: Option[String] = None,
                                  taskId: Option[String] = None,
                                  fetchMode: FetchMode,
                                  taskSet: TaskSet,
                                  paging: Option[Paging] = None,
                                  order: List[dto.Ordering],
                                  targets: List[String] = Nil,
                                  taskTypes: List[String] = Nil,
                                  onlySuccessful: Boolean = false,
                                  setTotalCount: BiConsumer[String, AnyRef],
                                  workerName: Option[String] = None,
                                  configurationItemIds: List[ConfigurationItemId] = Nil)

  case class FieldSorter[T](name: String, extractor: TaskWithBlock => T)(implicit ordering: Ordering[T]) {
    def ordering(order: dto.Ordering): Ordering[TaskWithBlock] = {
      val orderingBy = Ordering.by[TaskWithBlock, T](extractor)
      if (order.isAscending) orderingBy else orderingBy.reverse
    }
  }

  private def extractMetadata(field: String)(task: TaskWithBlock) = Option(task.getMetadata.get(field)).getOrElse("")

  private def extractAndMergeMetadataFields(fields: List[String], separator: String)(task: TaskWithBlock) =
    fields
      .flatMap(field => Option(task.getMetadata.get(field)))
      .mkString(separator)

  private def combineOrdering[T](o1: Ordering[T], o2: Ordering[T]) = new Ordering[T] {
    def compare(x: T, y: T): Int =
      o1.compare(x, y) match {
        case 0 => o2.compare(x, y)
        case n => n
      }
  }

  def toOrdering(order: List[dto.Ordering])(implicit sorters: Set[FieldSorter[_]]): Option[Ordering[TaskWithBlock]] = {
    val orderings = order.map { order =>
      val sorter = sorters.find(_.name == order.field).getOrElse {
        val availableFields = sorters.map(_.name).mkString(", ")
        throw new IllegalArgumentException(s"Cannot sort by field: '${order.field}'. Available fields: [$availableFields]")
      }
      sorter.ordering(order)
    }
    if (order.isEmpty)
      None
    else
      orderings.reduceLeftOption(combineOrdering)
  }
}

@Service
@Autowired
class TasksService(engine: BeanWrapper[TaskExecutionEngine],
                   paginationService: PaginationService,
                   activeTaskRepository: ActiveTaskRepository) {
  type TaskPredicate = TaskWithBlock => Boolean

  private val timeIntervalPredicate = (begin: DateTime, end: DateTime) => {
    lazy val filterInterval = new Interval(begin, end)
    task: TaskWithBlock => {
      Option(task.getStartDate) -> Option(task.getCompletionDate) match {
        case (Some(begin), Some(end)) => filterInterval.contains(new Interval(begin, end))
        case (Some(begin), None) => filterInterval.contains(begin)
        case (None, _) => true
      }
    }
  }

  private val taskTypePredicate = (taskTypes: List[String]) =>
    (task: TaskWithBlock) => taskTypes.contains(extractMetadata(TaskMetadata.TASK_TYPE)(task).toUpperCase())

  private val ciPermissionPredicate = (ciPathsFn: List[TaskId] => Map[TaskId, List[String]], tasks: List[TaskWithBlock]) => {
    val taskToCisMap = ciPathsFn(tasks.map(_.getId))
    val permissionsMap = PermissionHelper.hasPermission(PlatformPermissions.READ,
      taskToCisMap.values.flatten.toSet.toList.asJava)
    (task: TaskWithBlock) => {
      taskToCisMap.contains(task.getId) &&
        taskToCisMap.getOrElse(task.getId, Nil).map(ciPath =>
          permissionsMap.getOrDefault(ciPath.substring(1), false)).forall(_ == true)
    }
  }

  private val statePredicate = (states: List[String]) =>
    (task: TaskWithBlock) => {
      if (states.contains("UNKNOWN")) {
        val filteredStates = states.filterNot(_.equals("UNKNOWN"))
        states.isEmpty || filteredStates.map(TaskExecutionState.valueOf).contains(task.getState) || task.getState == null
      } else {
        states.isEmpty || states.map(TaskExecutionState.valueOf).contains(task.getState)
      }
    }

  private val taskNamePredicate = (taskName: Option[String]) =>
    (task: TaskWithBlock) =>
      taskName
        .flatMap(taskName => task.getMetadata.asScala.toMap.get(TaskMetadata.TASK_NAME).map(taskName -> _))
        .forall { case (taskName, currentTaskName) => currentTaskName.toUpperCase.contains(taskName.toUpperCase) }

  private val workerNamePredicate = (workerName: Option[String]) =>
    (task: TaskWithBlock) => {
      val currentWorkerName = task.getMetadata.asScala.toMap.getOrElse(TaskMetadata.WORKER_NAME, "")
      currentWorkerName.toUpperCase.contains(workerName.getOrElse("").toUpperCase)
    }

  private val taskSetPredicate = (taskSet: TaskSet, currentUserName: String) =>
    (task: TaskWithBlock) =>
      taskSet match {
        case TaskSet.ALL => true
        case TaskSet.MY => currentUserName.toUpperCase == task.getOwner.toUpperCase
      }

  private val usersPredicate = (users: List[String]) => {
    val upperCaseUsers = users.map(_.toUpperCase)
    task: TaskWithBlock => upperCaseUsers.isEmpty || upperCaseUsers.contains(task.getOwner.toUpperCase)
  }

  private val taskIdPredicate = (taskId: Option[String]) => (task: TaskWithBlock) => taskId.forall(task.getId.equals)

  private val configurationItemIdsPredicate = (ids: List[ConfigurationItemId]) => {
    val (applications, environments) = TaskFilterUtils.parseCiFilters(ids)
    task: TaskWithBlock => {
      val satisfiesApp = applications.isEmpty || applications.contains(extractMetadata(TaskMetadata.APPLICATION)(task))
      val satisfiesEnv = environments.isEmpty || environments.contains(extractMetadata(TaskMetadata.ENVIRONMENT_ID)(task))
      satisfiesApp && satisfiesEnv
    }
  }

  private val targetsPredicate = (targetCis: List[String]) =>
    (task: TaskWithBlock) =>
      targetCis.isEmpty || task.getMetadata.asScala.toMap
        .get(TaskMetadata.CONTROL_TASK_TARGET_CI)
        .forall { it: String =>
          targetCis.contains(it)
        }

  private def validateParams(params: TaskSearchParameters): Unit = {
    if (params.begin == null || params.end == null) {
      throw new IllegalArgumentException("'begin' and 'end' parameters are required.")
    }
    if (params.taskSet == null) {
      throw new IllegalArgumentException("'taskSet' parameter is required.")
    }
  }

  private def withPaging(paging: Option[Paging], stream: LazyList[TaskWithBlock]) = {
    val limited = paginationService.getLimitedPaging(paging.getOrElse(new Paging()))
    val firstRecord = (limited.page - 1) * limited.resultsPerPage
    stream
      .slice(firstRecord, firstRecord + limited.resultsPerPage)
  }

  private def withSorting(order: List[dto.Ordering], stream: LazyList[TaskWithBlock])(implicit sorters: Set[FieldSorter[_]]) =
    toOrdering(order)
      .map(stream.sorted(_))
      .getOrElse(stream)

  private def setTotalCountHeader(tasks: LazyList[TaskWithBlock], params: TaskSearchParameters): Unit = {
    val paging = params.paging.getOrElse(new Paging())
    val limited = paginationService.getLimitedPaging(paging)
    paginationService.addPagingHeaderIfNeeded(paging, limited, params.setTotalCount, () => tasks.length)
  }

  private def streamTasks(params: TaskSearchParameters, isDeploymentTask: Boolean)(implicit sorters: Set[FieldSorter[_]], predicate: TaskPredicate) = {
    validateParams(params)
    val allTasks = engine.get().getAllIncompleteTasks(params.fetchMode).asScala.toList

    val filtered = if (PermissionHelper.isCurrentUserAdmin) {
      allTasks.filter(predicate).to(LazyList)
    } else {
      val permPredicate = ciPermissionPredicate(
        if (isDeploymentTask) activeTaskRepository.ciPathsByDeployments else activeTaskRepository.ciPathsByControlTasks,
        allTasks)
      allTasks.filter(permPredicate).filter(predicate).to(LazyList)
    }
    setTotalCountHeader(filtered, params)

    val sorted = withSorting(params.order, filtered)
    withPaging(params.paging, sorted)
  }

  private def getStateName(task: SerializableTask): String = {
    val state = task.getState
    if (state != null) {
      return state.name()
    }
    TaskExecutionState.UNREGISTERED.name()
  }

  def streamControlTasks(params: TaskSearchParameters, currentUserName: String): LazyList[TaskWithBlock] = {
    val controlTaskParams = params.copy(taskTypes = List(TaskType.CONTROL.name(), TaskType.INSPECTION.name()))
    implicit val sorters: Set[FieldSorter[_]] = Set(
      FieldSorter("target", extractMetadata(TaskMetadata.CONTROL_TASK_TARGET_CI)),
      FieldSorter("taskName", extractMetadata(TaskMetadata.TASK_NAME)),
      FieldSorter("taskId", _.getId),
      FieldSorter("user", _.getOwner),
      FieldSorter("state", getStateName(_)),
      FieldSorter("begin", _.getStartDate),
      FieldSorter("end", _.getCompletionDate),
      FieldSorter("scheduled", _.getScheduledDate),
      FieldSorter("worker", extractMetadata(TaskMetadata.WORKER_NAME))
    )

    implicit val predicate: TaskPredicate = prepareFilters(params, controlTaskParams.taskTypes, currentUserName, false)

    streamTasks(controlTaskParams, false)
  }

  def streamDeploymentTasks(params: TaskSearchParameters, currentUserName: String): LazyList[TaskWithBlock] = {
    val types = TaskFilterUtils
      .buildTaskTypesFilter(params.taskTypes.asJava, params.onlySuccessful)
      .asScala
      .filter(!Seq(TaskType.CONTROL, TaskType.INSPECTION).contains(_))
      .map(_.name())
      .toList

    implicit val sorters: Set[FieldSorter[_]] = Set(
      FieldSorter("package", extractAndMergeMetadataFields(List(TaskMetadata.APPLICATION, TaskMetadata.VERSION), "/")),
      FieldSorter("environment", extractMetadata(TaskMetadata.ENVIRONMENT_ID)),
      FieldSorter("taskId", _.getId),
      FieldSorter("type", extractMetadata(TaskMetadata.TASK_TYPE)),
      FieldSorter("user", _.getOwner),
      FieldSorter("state", getStateName(_)),
      FieldSorter("begin", _.getStartDate),
      FieldSorter("end", _.getCompletionDate),
      FieldSorter("scheduled", _.getScheduledDate),
      FieldSorter("worker", extractMetadata(TaskMetadata.WORKER_NAME))
    )

    implicit val predicate: TaskPredicate = prepareFilters(params, types, currentUserName, true)

    streamTasks(params, true)
  }

  private def prepareFilters(params: TaskSearchParameters, taskTypes: List[String], currentUserName: String, isDeploymentTask: Boolean): TaskPredicate = {
    val commonFilters: Array[TaskWithBlock => Boolean] = Array(
      taskIdPredicate(params.taskId),
      timeIntervalPredicate(params.begin, params.end),
      taskTypePredicate(taskTypes),
      statePredicate(params.states),
      taskSetPredicate(params.taskSet, currentUserName),
      usersPredicate(params.users),
      workerNamePredicate(params.workerName)
    )

    val additionalFilters: Array[TaskPredicate] = if (isDeploymentTask) {
      Array(configurationItemIdsPredicate(params.configurationItemIds))
    } else {
      Array(taskNamePredicate(params.taskName), targetsPredicate(params.targets))
    }

    val filters: Array[TaskPredicate] = commonFilters ++ additionalFilters

    (task: TaskWithBlock) => filters.forall(_ (task))
  }
}
