package com.xebialabs.deployit.core.rest.api

import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.core.util.CiSugar._
import com.xebialabs.deployit.core.rest.api.DeploymentWriter.{convertDeployeds, createDeployment}
import com.xebialabs.deployit.engine.api.dto.Deployment
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm._
import com.xebialabs.deployit.repository.RepositoryService
import com.xebialabs.deployit.service.dependency.{DependencyService, DeploymentResult}
import com.xebialabs.deployit.service.deployment.{DeployedService, ResolvedPlaceholderGenerator}
import com.xebialabs.deployit.service.replacement.ConsolidatedDictionary
import com.xebialabs.deployit.service.replacement.Dictionaries.of
import com.xebialabs.deployit.service.version.VersionService
import com.xebialabs.xlplatform.utils.Implicits.listOfListToJavaListOfList
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.util.{Collections, Map => JMap}
import scala.jdk.CollectionConverters._

@Component
@Autowired
class DeploymentObjectGenerator(deployedService: DeployedService,
                                repositoryService: RepositoryService,
                                dependencyService: DependencyService,
                                versionService: VersionService,
                                permissionChecker: PermissionChecker) extends Logging {
  private def splitDeployments(deployments: List[List[Deployment]]): (Deployment, List[List[Deployment]]) = {
    val dependencies :+ (main :: _) = deployments
    (main, dependencies)
  }

  private def isDependencyResolutionEnabled = ServerConfiguration.getInstance.isServerResolveApplicationDependencies

  private def isProvisioning(version: Version) = Option(version) match {
    case Some(provisioningPackage) if provisioningPackage.instanceOf[ProvisioningPackage] =>
      provisioningPackage.getDeployables.asScala.exists(_.instanceOf[Provisionable])
    case _ => false
  }

  private def checkAndCast[T <: ConfigurationItem](ci: ConfigurationItem, clazz: Class[T]): T = {
    checkArgument(clazz.isAssignableFrom(ci.getClass), "%s is not a %s", ci, clazz.getSimpleName)
    clazz.cast(ci)
  }

  private def readDeploymentApplicationsMap(possibleDeployments: List[List[DeploymentResult]],
                                            preLoaded: Map[String, DeployedApplication]): Map[String, DeployedApplication] = {
    val idsToLoad = possibleDeployments
      .to(LazyList)
      .flatten
      .filter(deployment => deployment.hasDeployedApplication && !preLoaded.contains(deployment.deployedApplicationId))
      .map(_.deployedApplicationId)
      .toList

    val loaded: Map[String, DeployedApplication] =
      if (idsToLoad.isEmpty)
        Map()
      else
        repositoryService
          .read[ConfigurationItem](idsToLoad.asJava, Int.MaxValue, false)
          .asScala
          .map(ci => ci.getId -> checkAndCast(ci, classOf[DeployedApplication]))
          .toMap

    loaded ++ preLoaded
  }

  private def generateInitialDeployment(version: Version, env: Environment, consolidatedDictionary: ConsolidatedDictionary): Deployment = {
    val deployedApplication = deployedService.generateDeployedApplication(Type.valueOf(classOf[DeployedApplication]), version, env, consolidatedDictionary)
    val deployment = createDeployment(deployedApplication, Deployment.DeploymentType.INITIAL)
    ResolvedPlaceholderGenerator.generateAndAdd(deployment, consolidatedDictionary)
    deployment
  }

  private def generateUpdateDeployment(previousDeployedApplication: DeployedApplication, newVersion: Version, consolidatedDictionary: ConsolidatedDictionary): Deployment = {
    debug(s"Creating upgrade deployed application from: [$previousDeployedApplication] with new version [$newVersion]")

    val upgradedDeployeds = deployedService.generateUpgradedDeployeds(previousDeployedApplication, newVersion)
    val updatedDeployedApplication = deployedService.generateUpdateDeployedApplication(previousDeployedApplication, newVersion, consolidatedDictionary)

    debug(s"Generated upgrade deployed application [$updatedDeployedApplication] with version [${updatedDeployedApplication.getVersion}] and env [${updatedDeployedApplication.getEnvironment}]")
    debug(s"Upgrade deployed application contains [${updatedDeployedApplication.getDeployeds}] deployeds")

    val deployment = createDeployment(updatedDeployedApplication, Deployment.DeploymentType.UPDATE)
    permissionChecker.checkPermission(deployment)
    updatedDeployedApplication.get$validationMessages().addAll(previousDeployedApplication.get$validationMessages())
    deployment.setDeployeds(convertDeployeds(upgradedDeployeds.getDeployeds))
    ResolvedPlaceholderGenerator.generateAndAdd(deployment, consolidatedDictionary)
    deployment
  }

  private def deploymentResultToDeployment(deploymentResult: DeploymentResult,
                                           dictionary: ConsolidatedDictionary,
                                           environment: Environment,
                                           versionsMap: Map[String, ConfigurationItem],
                                           deployedApplicationsMap: Map[String, DeployedApplication]): Option[Deployment] = {
    val result = deploymentResult match {
      case possibleDeployment if possibleDeployment.hasDeployedApplication => for {
        deployedApplication <- deployedApplicationsMap.get(possibleDeployment.deployedApplicationId)
        version <- versionsMap.get(possibleDeployment.udmVersionId)
      } yield generateUpdateDeployment(deployedApplication, checkAndCast(version, classOf[Version]), dictionary)
      case possibleDeployment => for {
        version <- versionsMap.get(possibleDeployment.udmVersionId)
      } yield generateInitialDeployment(checkAndCast(version, classOf[Version]), environment, dictionary)
    }
    result
  }

  private def buildDeployments(newVersion: Version,
                               environment: Environment,
                               consolidatedDictionary: ConsolidatedDictionary,
                               entryPoint: Option[DeployedApplication] = None): List[List[Deployment]] = {
    try {

      val dependencyResolution = versionService.resolvePlaceHolder("dependencyResolution", newVersion, environment, consolidatedDictionary)
      val possibleDeployments = dependencyService.resolveMultiDeployment(environment.getId, newVersion, dependencyResolution)
      val deployedApplicationsMap = readDeploymentApplicationsMap(possibleDeployments, entryPoint.map(ci => ci.getId -> ci).toMap)
      val versionIds = possibleDeployments
        .flatMap(_.flatMap(possibleDeployment => Option(possibleDeployment.udmVersionId)))
        .filter(_ != newVersion.getId)
        .toSet
      val versions = (if (versionIds.isEmpty) Map[String, ConfigurationItem]() else repositoryService
        .read[ConfigurationItem](versionIds.toList.asJava, Int.MaxValue, false)
        .asScala
        .map(version => version.getId -> version)
        .toMap) + (newVersion.getId -> newVersion)

      possibleDeployments.map(_.flatMap(deploymentResultToDeployment(_, consolidatedDictionary, environment, versions, deployedApplicationsMap)))
    } catch {
      case e: IllegalArgumentException => throw new InvalidDeploymentException(e.getMessage)
    }
  }

  private def generateMultiDeployments(newVersion: Version,
                                       environment: Environment,
                                       consolidatedDictionary: ConsolidatedDictionary,
                                       entryPoint: Option[DeployedApplication] = None
                                       ): List[List[Deployment]] = {
    val result = buildDeployments(newVersion, environment, consolidatedDictionary, entryPoint)
    val (mainDeployment, _) = splitDeployments(result)
    permissionChecker.checkPermission(mainDeployment, result)
    result
  }

  /**
   * Extracts the main deployment
   * The main deployment goes the last in the reversed topological order of dependency graph
   *
   * @param deployments - dependency graph
   * @return main deployment instance
   */
  private def extractMainDeployment(deployments: List[List[Deployment]]): Deployment = {
    val (deployment, groupedRequiredDeployments) = splitDeployments(deployments)
    deployment.setGroupedRequiredDeployments(listOfListToJavaListOfList(groupedRequiredDeployments))
    deployment
  }

  /**
   * Generates a deployment object for an initial deployment
   *
   * @param version - version instance
   * @param env     - environment instance
   * @return generated Deployment object
   */
  def forInitial(version: Version, env: Environment): Deployment = {
    val dictionary = of(env).filterBy(version).consolidate()
    if (isDependencyResolutionEnabled)
      extractMainDeployment(generateMultiDeployments(version, env, dictionary))
    else
      generateInitialDeployment(version, env, dictionary)
  }

  /**
   * Generates a deployment object for an update deployment
   *
   * @param deployedApplication - entry point DeployedApplication
   * @param newVersion          - new application version
   * @param env                 - environment instance
   * @return generated Deployment object
   */
  def forUpdate(deployedApplication: DeployedApplication, newVersion: Version, env: Environment): Deployment = {
    val dictionary = of(env).filterBy(newVersion).consolidate()
    if (isDependencyResolutionEnabled)
      extractMainDeployment(generateMultiDeployments(newVersion, env, dictionary, Some(deployedApplication)))
    else
      generateUpdateDeployment(deployedApplication, newVersion, dictionary)
  }
}
