package com.xebialabs.deployit.core.rest.api;

import ai.digital.deploy.core.common.security.permission.DeployitPermissions;
import com.xebialabs.deployit.engine.api.execution.*;
import com.xebialabs.deployit.engine.spi.event.*;
import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.engine.spi.exception.HttpResponseCodeResult;
import com.xebialabs.deployit.engine.tasker.*;
import com.xebialabs.deployit.engine.tasker.log.StepLogRetriever;
import com.xebialabs.deployit.engine.tasker.repository.ActiveTask;
import com.xebialabs.deployit.engine.tasker.repository.ActiveTaskRepository;
import com.xebialabs.deployit.engine.tasker.repository.PendingTask;
import com.xebialabs.deployit.engine.tasker.repository.PendingTaskRepository;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.security.PermissionDeniedException;
import com.xebialabs.deployit.security.Permissions;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.service.controltask.ControlTaskService;
import com.xebialabs.deployit.spring.BeanWrapper;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.archive.ArchivedTask;
import com.xebialabs.deployit.task.archive.TaskArchive;
import com.xebialabs.deployit.task.archive.TaskArchiveQueue;
import com.xebialabs.deployit.task.archive.TaskReader;
import com.xebialabs.deployit.task.archive.sql.CachingTaskReader;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.LocalDate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import scala.Option;

import java.io.StringReader;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN;
import static ai.digital.deploy.tasker.common.TaskMetadata.*;
import static com.xebialabs.deployit.task.TaskMetadataModifier.*;
import static java.lang.String.format;

public abstract class AbstractTaskResource extends AbstractTaskRestrictedResource {

    protected static final Logger logger = LoggerFactory.getLogger(TaskResource.class);

    @Autowired
    protected BeanWrapper<TaskExecutionEngine> engine;

    @Autowired
    protected ActiveTaskRepository taskRepository;

    @Autowired
    protected StepLogRetriever stepLogRetriever;

    @Autowired
    protected PendingTaskRepository pendingTaskRepository;

    @Autowired
    protected TaskQueueService taskQueueService;

    @Autowired
    protected ControlTaskService controlTaskService;

    public void start(final String taskId) {
        checkOwnership(taskId);
        controlTaskService.start(taskId);
    }

    public void schedule(final String taskId, final DateTime time) {
        checkOwnership(taskId);
        checkNotNull(time, "No time specified for scheduling");

        DateTime utcTime = time.withZone(DateTimeZone.UTC);
        EventBusHolder.publish(new TaskScheduledEvent(taskId, utcTime));

        engine.get().schedule(taskId, utcTime);
    }

    public void cancel(final String taskId) {
        checkOwnership(taskId);
        engine.get().cancel(taskId);
        EventBusHolder.publish(new TaskCancelledEvent(taskId));
    }

    public void forceCancel(final String taskId) {
        checkForMissingPermission(!hasPermission(ADMIN), "Only user with ADMIN role can force-cancel a task.");
        handleForceCancel(taskId);
    }

    public void forceCancelTasks(final List<String> taskIds) {
        checkForMissingPermission(!hasPermission(ADMIN), "Only user with ADMIN role can force-cancel a task.");
        taskIds.forEach(this::handleForceCancel);
    }

    private void handleForceCancel(final String taskId) {
        try {
            engine.get().cancel(taskId, true);
        } catch (TaskNotFoundException tnf) {
            logger.debug(format("Deleting unknown task %s after force cancel.", taskId));
            taskRepository.delete(taskId);
        }
        EventBusHolder.publish(new TaskCancelledEvent(taskId));
    }

    public void archive(final String taskId) {
        checkOwnership(taskId);
        engine.get().archive(taskId);
        EventBusHolder.publish(new TaskArchivedEvent(taskId));
    }

    public void stop(final String taskId) {
        checkOwnership(taskId);
        try {
            engine.get().stop(taskId);
            EventBusHolder.publish(new TaskStoppedEvent(taskId));
        } catch (IllegalStateException ise) {
            logger.error(format("Could not stop task %s", taskId), ise);
            throw ise;
        } catch (Exception e) {
            logger.error(format("Could not stop task %s", taskId), e);
            throw e;
        }
    }

    public void abort(final String taskId) {
        checkOwnership(taskId);
        try {
            engine.get().abort(taskId);
            EventBusHolder.publish(new TaskAbortedEvent(taskId));
        } catch (IllegalStateException ise) {
            logger.error(format("Could not abort task %s", taskId), ise);
            throw ise;
        } catch (Exception e) {
            logger.error(format("Could not abort task %s", taskId), e);
            throw e;
        }
    }

    protected Stream<ArchivedTask> search(ArchivedTaskSearchParameters parameters, boolean loadSteps) {
        return doSearch(parameters, loadSteps, true);
    }

    public List<ArchivedTask> searchList(ArchivedTaskSearchParameters parameters, boolean loadSteps) {
        return doSearch(parameters, loadSteps, false).collect(Collectors.toList());
    }

    private Stream<ArchivedTask> doSearch(final ArchivedTaskSearchParameters parameters, final boolean loadSteps, final boolean streaming) {
        checkForMissingPermission(!hasPermission(ADMIN) && !hasPermission(DeployitPermissions.REPORT_VIEW()), "Only user with report#view permission or ADMIN can search the task archive.");

        return taskArchive.searchForTasks(parameters, streaming).map(
                t -> {
                    if (loadSteps) {
                        ArchivedTask task = t.fully();
                        if (!hasTaskPermission(task.getMetadata())) {
                            task.setBlock(new EmptyPhaseContainer(task.getBlock().getState()));
                        }
                        return task;
                    } else {
                        return t.withoutSteps();
                    }
                }
        );
    }

    public void purge(final String taskId) {
        checkForMissingPermission(!hasPermission(ADMIN), "Only user with ADMIN role can purge the task archive.");

        taskArchive.purgeTask(taskId);
    }

    protected TaskWithBlock doAssign(final String taskId, final String owner) {
        // Rules:
        // - admin can reassign everyone's tasks
        // - user can reassign his own tasks

        final SerializableTask task = getSerializableTask(taskId);
        checkArgument(task != null, "Could not find active task %s", taskId);

        UsernamePasswordAuthenticationToken token = new UsernamePasswordAuthenticationToken(owner, null);
        final TaskWithBlock assignedTask;
        if (hasPermission(ADMIN)) {
            assignedTask = engine.get().assign(taskId, token);
        } else {
            if (hasPermission(DeployitPermissions.TASK_ASSIGN()) && task.getOwner().equals(Permissions.getAuthenticatedUserName())) {
                assignedTask = engine.get().assign(taskId, token);
            } else {
                throw PermissionDeniedException.withMessage("Neither ADMIN nor TASK_ASSIGN permissions are granted to you or you are not the owner of the task");
            }
        }

        EventBusHolder.publish(new TaskAssignedEvent(taskId, owner));
        return assignedTask;
    }

    protected TaskWithBlock doTakeover(final String taskId, final String owner) {
        // Rules:
        // - admin can takeover everyone's tasks
        // - user with correct permission can takeover task if he knows current owner.

        String newOwner = Permissions.getAuthenticatedUserName();
        final SerializableTask task = getSerializableTask(taskId);
        checkArgument(task != null, "Could not find active task %s", taskId);

        checkPermission(DeployitPermissions.TASK_TAKEOVER(), task);
        final TaskWithBlock takeoveredTask;
        if (task.getOwner().equals(owner)) {
            takeoveredTask = engine.get().assign(taskId, Permissions.getAuthentication());
        } else {
            throw PermissionDeniedException.withMessage(format("You have not specified the correct owner for task [%s], not taking over.", owner));
        }

        EventBusHolder.publish(new TaskAssignedEvent(taskId, newOwner));
        return takeoveredTask;
    }

    protected SerializableTask getSerializableTask(final String taskId) {
        SerializableTask task = null;
        final Option<PendingTask> pendingTask = pendingTaskRepository.task(taskId, false);
        if (pendingTask.nonEmpty()) {
            task = pendingTask.get();
        } else {
            final Option<ActiveTask> activeTask = taskRepository.task(taskId, false);
            if (activeTask.nonEmpty()) {
                task = activeTask.get();
            }
        }
        return task;
    }

    protected void checkPermission(Permission permission, SerializableTask task) {
        if (metadataContainsKey(task, ENVIRONMENT_ID)) {
            String env = getMetadata(task, ENVIRONMENT_ID);
            checkPermission(permission, env);
        } else {
            checkPermission(permission);
        }
    }

    protected boolean hasBeenModifiedSince(StepState step, DateTime ifModifiedSince) {
        if (ifModifiedSince == null) {
            return true;
        }

        // Check either completionDate (when step is finished, or LMD when step is still in active task.
        DateTime lastModifiedCal = step.getCompletionDate();
        if (step instanceof TaskStep) {
            lastModifiedCal = ((TaskStep) step).getLastModificationDate();
        }
        // Set milliseconds to 0 because time resolution is to 1 second in HTTP format
        return lastModifiedCal.withMillis(0).isAfter(ifModifiedSince);
    }

    protected boolean isNotCalledByOwner(String taskId) {
        try {
            calledByOwner(pendingOrActiveOrArchivedTask(taskId));
            return false;
        } catch (PermissionDeniedException e) {
            return true;
        }
    }

    protected SerializableTask calledByOwner(SerializableTask t) {
        String currentUser = Permissions.getAuthenticatedUserName();
        checkForMissingPermission(!currentUser.equals(t.getOwner()) && !hasPermission(ADMIN), "Only owner or ADMIN can access the task.");
        return updateOwner(t);
    }

    private SerializableTask updateOwner(SerializableTask t) {
        if (t instanceof Task) {
            Task task = (Task) t;
            if (!(task.getAuthentication().isAuthenticated())) {
                task.setOwner(Permissions.getAuthentication());
            }
        }
        return t;
    }

    protected void checkForMissingPermission(boolean condition, String message) {
        if (condition) {
            throw PermissionDeniedException.withMessage(message);
        }
    }

    protected SerializableTask viewAbleTask(SerializableTask t) {
        String currentUser = Permissions.getAuthenticatedUserName();
        checkForMissingPermission(!currentUser.equals(t.getOwner()) && !hasPermission(ADMIN) && !hasPermission(DeployitPermissions.TASK_VIEW()), "Only user with task#view permission or owner or ADMIN can access the task");
        return t;
    }

    protected SerializableTask pendingOrLiveOrArchivedTask(String taskId, final boolean loadSteps) {
        try {
            return engine.get().retrieveTask(taskId);
        } catch (TaskNotFoundException l) {
            logger.debug(format("Task not found in Pending or Active state. Check if task %s is in archive queue", taskId));
            TaskReader taskReader = taskArchiveQueue.getTaskDetails(taskId)
                    .map(task ->
                            new CachingTaskReader(new StringReader(task)))
                    .getOrElse(() -> {
                        logger.debug(format("Task[%s] not found in Pending, Actor, Archive Queue and Archive repo", taskId));
                        return taskArchive.getTask(taskId);
                    });
            return loadSteps ? taskReader.fully() : taskReader.withoutSteps();
        }
    }

    protected SerializableTask pendingOrActiveOrArchivedTask(String taskId) {
        final Option<PendingTask> pendingTask = pendingTaskRepository.task(taskId, false);
        return pendingTask.isDefined() ? pendingTask.get() : taskRepository.task(taskId, true).getOrElse(() -> taskArchive.getTask(taskId).withoutSteps());
    }

    protected ArchivedTaskSearchParameters toSearchParameters(LocalDate begin, LocalDate end, boolean hasPaging) {
        ArchivedTaskSearchParameters parameters = new ArchivedTaskSearchParameters();
        if (hasPaging) {
            parameters.orderBy("startDate", true);
        }
        if (end == null) {
            end = new LocalDate();
        }
        if (begin != null) {
            parameters.createdBetween(begin.toDateTimeAtStartOfDay(), end.toDateTimeAtStartOfDay());
        }
        return parameters;
    }

    protected void checkOwnership(String taskId) {
        checkForMissingPermission(isNotCalledByOwner(taskId) && !hasPermission(ADMIN), "Only owner or ADMIN can archive the task.");
    }

    public void setTaskArchive(TaskArchive taskArchive) {
        this.taskArchive = taskArchive;
    }

    public void setTaskArchiveQueue(TaskArchiveQueue taskArchiveQueue) {
        this.taskArchiveQueue = taskArchiveQueue;
    }

    public void setEngine(BeanWrapper engine) {
        this.engine = engine;
    }

    public void setTaskRepository(ActiveTaskRepository taskRepository) {
        this.taskRepository = taskRepository;
    }

    protected StepState addLogs(String taskId, BlockPath stepPath, StepState step) {
        return stepLogRetriever.retrieveLogs(taskId, stepPath, step);
    }

    public void setPendingTaskRepository(final PendingTaskRepository pendingTaskRepository) {
        this.pendingTaskRepository = pendingTaskRepository;
    }

    @SuppressWarnings("serial")
    @HttpResponseCodeResult(statusCode = 304)
    protected static class NotModifiedException extends DeployitException {
    }

    private static class EmptyPhaseContainer implements PhaseContainerState {
        private final BlockExecutionState state;

        public EmptyPhaseContainer(final BlockExecutionState state) {
            this.state = state;
        }

        @Override
        public String getId() {
            return null;
        }

        @Override
        public String getDescription() {
            return "";
        }

        @Override
        public BlockExecutionState getState() {
            return state;
        }

        @Override
        public Iterable<PhaseState> getBlocks() {
            return Collections.emptyList();
        }

        @Override
        public Boolean hasSteps() {
            return Boolean.FALSE;
        }

        @Override
        public SatelliteConnectionState getSatelliteConnectionState() {
            return null;
        }
    }
}
