package com.xebialabs.deployit.core.rest.api.reports.widgets;

import ai.digital.deploy.sql.model.Report;
import ai.digital.deploy.sql.model.ReportLine;
import com.xebialabs.deployit.core.rest.api.reports.ReportUtils;
import com.xebialabs.deployit.task.archive.AverageDeploymentsDurationOverTimeReportLine;
import com.xebialabs.deployit.task.archive.TaskArchive;
import org.joda.time.DateTime;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import static java.util.stream.Collectors.averagingLong;
import static java.util.stream.Collectors.groupingBy;

/*********************
 * Deployment duration over time widget
 */
public class DeploymentTrendsPercentileWidget extends DashboardWidgetBase {

    private static final String TIME_SCALE = "timeScale";
    private static final String DEPLOYMENT_TIME = "deploymentTime";

    private enum ReportTimeScale {
        SECS, MINS, HRS
    }

    public DeploymentTrendsPercentileWidget(final TaskArchive taskArchive) {
        super(taskArchive);
    }

    @Override
    public Report getReport(final DateTime begin, final DateTime end) {
        return transformResult(taskArchive.averageDurationDeploymentsOverTime(begin, end), begin, end);
    }

    private Report transformResult(Collection<AverageDeploymentsDurationOverTimeReportLine> searchResult, DateTime begin, DateTime end) {
        Report report = new Report();
        if (end.isBefore(begin)) {
            return report;
        }

        final Map<Integer, Map<Integer, Double>> searchResultGroupedByYearMonth =
                searchResult.stream().collect(groupingBy(AverageDeploymentsDurationOverTimeReportLine::year,
                        groupingBy(AverageDeploymentsDurationOverTimeReportLine::monthno,
                                averagingLong(AverageDeploymentsDurationOverTimeReportLine::durationMillis))));

        ReportUtils.getMonthsOfInterval(begin, end).forEach(month -> {
            final Map<Integer, Double> monthDurationResult = searchResultGroupedByYearMonth.get(month.getYear());
            addReportLine(report, month.getMonthOfYear(), monthDurationResult);
        });

        return report;
    }

    private static ReportTimeScale findTimeScale(Map<Integer, Double> averageDurationByMonth) {
        long maxDeploymentTime = averageDurationByMonth == null
                ? 0L : Collections.max(averageDurationByMonth.values()).longValue();

        if (TimeUnit.MILLISECONDS.toSeconds(maxDeploymentTime) < 600) {
            return ReportTimeScale.SECS;
        } else if (TimeUnit.MILLISECONDS.toMinutes(maxDeploymentTime) < 600) {
            return ReportTimeScale.MINS;
        }
        return ReportTimeScale.HRS;
    }

    private static void addReportLine(Report report, int month, final Map<Integer, Double> monthDurationResult) {
        ReportLine line = report.addLine();

        line.addValue(MONTH, getFormattedMonth(month));
        final long averageDuration = (monthDurationResult == null || monthDurationResult.get(month) == null)
                ? 0L : monthDurationResult.get(month).longValue();
        final ReportTimeScale timeScale = findTimeScale(monthDurationResult);

        line.addValue(TIME_SCALE, timeScale.toString());
        switch (timeScale) {
            case SECS:
                line.addValue(DEPLOYMENT_TIME, formatToSecs(averageDuration));
                break;
            case MINS:
                line.addValue(DEPLOYMENT_TIME, formatToMins(averageDuration));
                break;
            default:
                line.addValue(DEPLOYMENT_TIME, formatToHours(averageDuration));
        }
    }
}
