package com.xebialabs.deployit.core.rest.json;

import org.springframework.stereotype.Component;

import jakarta.ws.rs.Produces;
import jakarta.ws.rs.WebApplicationException;
import jakarta.ws.rs.core.Context;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.MultivaluedMap;
import jakarta.ws.rs.ext.MessageBodyWriter;
import jakarta.ws.rs.ext.Provider;
import jakarta.ws.rs.ext.Providers;
import java.io.IOException;
import java.io.OutputStream;
import java.lang.annotation.Annotation;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.function.Consumer;
import java.util.stream.Stream;


@Component
@Provider
@Produces(MediaType.APPLICATION_JSON)
public class StreamJsonWriter implements MessageBodyWriter<Stream<Object>> {

    @Context
    private Providers providers;

    @Override
    public boolean isWriteable(final Class<?> type, final Type genericType, final Annotation[] annotations, final MediaType mediaType) {
        return Stream.class.isAssignableFrom(type) && genericType instanceof ParameterizedType;
    }

    @Override
    public long getSize(final Stream<Object> objectStream, final Class<?> type, final Type genericType, final Annotation[] annotations, final MediaType mediaType) {
        return -1;
    }

    @Override
    public void writeTo(final Stream<Object> stream, final Class<?> type, final Type genericType, final Annotation[] annotations, final MediaType mediaType, final MultivaluedMap<String, Object> httpHeaders, final OutputStream entityStream) throws IOException, WebApplicationException {
        final Type argumentType = ((ParameterizedType) genericType).getActualTypeArguments()[0];
        final MessageBodyWriter writer = providers.getMessageBodyWriter((Class) argumentType, argumentType, annotations, mediaType);

        try (Stream<Object> s = stream) {
            entityStream.write('[');
            s.forEachOrdered(new WritingConsumer(writer, argumentType, annotations, mediaType, httpHeaders, entityStream));
            entityStream.write(']');
        }
    }

    private static class WritingConsumer implements Consumer<Object> {

        private final MessageBodyWriter writer;
        private final Type argumentType;
        private final Annotation[] annotations;
        private final MediaType mediaType;
        private final MultivaluedMap<String, Object> httpHeaders;
        private final OutputStream entityStream;

        private boolean first = true;

        private WritingConsumer(final MessageBodyWriter writer, final Type argumentType, final Annotation[] annotations, final MediaType mediaType, final MultivaluedMap<String, Object> httpHeaders, final OutputStream entityStream) {
            this.writer = writer;
            this.argumentType = argumentType;
            this.annotations = annotations;
            this.mediaType = mediaType;
            this.httpHeaders = httpHeaders;
            this.entityStream = entityStream;
        }

        @Override
        public void accept(final Object o) {
            try {
                if (first) {
                    first = false;
                } else {
                    entityStream.write(',');
                }
                writer.writeTo(o, o.getClass(), argumentType, annotations, mediaType, httpHeaders, entityStream);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

}