package com.xebialabs.deployit.core.rest.xml;

import com.google.common.collect.Lists;
import com.thoughtworks.xstream.converters.Converter;
import com.xebialabs.xltype.serialization.xstream.XStreamReaderWriter;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;

import jakarta.annotation.PostConstruct;
import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.WebApplicationException;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.MultivaluedMap;
import jakarta.ws.rs.ext.Provider;
import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.annotation.Annotation;
import java.lang.reflect.Type;
import java.util.Collections;
import java.util.List;

import static com.xebialabs.deployit.checks.Checks.checkState;

@Component
@Provider
@Produces({"application/*+xml", "text/*+xml"})
@Consumes({"application/*+xml", "text/*+xml"})
public class SpringXStreamReaderWriter extends XStreamReaderWriter implements ApplicationContextAware {

    private static final List<String> blacklistPackages = Lists.newArrayList(
            "com.xebialabs.deployit.booter.remote.xml",
            "com.xebialabs.deployit.taskexecution.xml"
    );
    private ApplicationContext applicationContext;

    public SpringXStreamReaderWriter() {
    }

    @Override
    @PostConstruct
    protected void init() {
        if (applicationContext != null) {
            super.init();
            XStreamReaderWriter.registerConfigurationItemAliases();
        }
    }

    @Override
    protected Converter constructConverter(Class<?> clazz) {
        if (blacklistPackages.contains(clazz.getPackage().getName())) {
            return null;
        }
        try {
            Object bean = applicationContext.getBean(clazz);
            checkState(bean instanceof Converter, "[%s] should be a Converter", bean);
            return (Converter) bean;
        } catch (BeansException be) {
            return super.constructConverter(clazz);
        }
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @Override
    public Object readFrom(Class<Object> type, Type genericType, Annotation[] annotations, MediaType mediaType, MultivaluedMap<String, String> httpHeaders, InputStream entityStream) throws IOException, WebApplicationException {
        try {
            // Handle empty content for List<ConfigurationItemId> parameters
            if (List.class.isAssignableFrom(type)) {
                // Mark the stream and try to read one byte to check if it's empty
                if (!entityStream.markSupported()) {
                    // If mark is not supported, wrap in BufferedInputStream
                    entityStream = new BufferedInputStream(entityStream);
                }
                entityStream.mark(1);
                int firstByte = entityStream.read();
                if (firstByte == -1) {
                    // Stream is empty
                    return Collections.emptyList();
                }
                // Reset stream to beginning since we found content
                entityStream.reset();
            }
            return super.readFrom(type, genericType, annotations, mediaType, httpHeaders, entityStream);
        } catch (RuntimeException e) {
            XStreamCiConverterWithRepository.clear();
            throw e;
        } finally {
            XStreamCiConverterWithRepository.postProcess();
        }
    }
}
