package com.xebialabs.deployit.core.rest.api;

import ai.digital.deploy.settings.GeneralSettings;
import com.xebialabs.deployit.core.rest.api.support.PaginationSupport;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.core.service.GeneralSettingsService;
import com.xebialabs.deployit.engine.api.dto.Ordering;
import com.xebialabs.deployit.engine.api.dto.Paging;
import com.xebialabs.deployit.engine.api.security.User;
import com.xebialabs.deployit.engine.api.security.UserProfile;
import com.xebialabs.deployit.engine.spi.event.UserCreatedEvent;
import com.xebialabs.deployit.engine.spi.event.UserDeletedEvent;
import com.xebialabs.deployit.engine.spi.event.UserPasswordChangedEvent;
import com.xebialabs.deployit.engine.spi.event.UserUpdateEvent;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.security.Permissions;
import com.xebialabs.deployit.security.RepoUser;
import com.xebialabs.deployit.security.UserService;
import com.xebialabs.deployit.security.authentication.UserAlreadyExistsException;
import com.xebialabs.deployit.security.model.XldUserProfile;
import com.xebialabs.deployit.security.principaldata.PrincipalDataProvider;
import com.xebialabs.deployit.security.principaldata.UserData;
import com.xebialabs.deployit.security.service.UserGroupService;
import com.xebialabs.deployit.security.service.UserProfileService;
import com.xebialabs.deployit.security.validator.UserValidator;
import com.xebialabs.xlplatform.security.dto.PasswordValidationResult;
import org.jboss.resteasy.spi.HttpResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.session.SessionInformation;
import org.springframework.security.core.session.SessionRegistry;
import org.springframework.security.web.authentication.rememberme.PersistentTokenRepository;
import org.springframework.session.SessionRepository;
import org.springframework.stereotype.Controller;

import jakarta.ws.rs.core.Context;
import org.springframework.util.StringUtils;
import scala.collection.immutable.HashSet;


import java.util.List;
import java.util.Objects;
import java.util.Optional;

import static com.xebialabs.deployit.security.permission.PlatformPermissions.EDIT_SECURITY;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.VIEW_SECURITY;

@Controller
public class UserResource extends AbstractSecuredResource implements com.xebialabs.deployit.engine.api.DeployUserService {

    @Context
    private HttpResponse response;

    @Autowired
    UserService userService;

    @Autowired
    UserProfileService userProfileService;

    @Autowired
    UserValidator userValidator;

    @Autowired
    SessionRegistry sessionRegistry;

    @Autowired
    SessionRepository sessionRepository;

    @Autowired
    PrincipalDataProvider principalDataProvider;

    @Autowired
    PersistentTokenRepository persistentTokenRepository;

    @Autowired
    UserGroupService userGroupService;

    @Autowired
    GeneralSettingsService generalSettingsService;

    @Override
    public User create(final String username, final User user) {
        checkPermission(EDIT_SECURITY);
        sanityCheckUser(username, user);
        userValidator.check(user);
        validateUserProfile(user);

        checkNoUserProfileAlreadyExists(username);

        logger.debug("Creating user and UserProfile {}", username);
        userService.create(username, user.getPassword());
        XldUserProfile userProfile = getXldUserProfile(user);
        userProfileService.createOrUpdateUserProfile(userProfile);

        com.xebialabs.deployit.security.User u = userService.read(username);
        logger.debug("Created user {}", u.getUsername());
        XldUserProfile profile = userProfileService.findOne(userProfile.username());
        logger.debug("Created userProfile {}", profile.username());
        GeneralSettings generalSettings = generalSettingsService.getGeneralSettings();
        UserProfile userProfile1 = getUserProfile(profile,generalSettings);
        EventBusHolder.publish(new UserCreatedEvent(username));
        return new User(u.getUsername(), u.isAdmin(), userProfile1);
    }

    private void validateUserProfile(User user) {
        if(user.getUserProfile()!=null && user.getUserProfile().getFullName()!=null && !user.getUserProfile().getFullName().isEmpty()){
            userValidator.checkMaxLength(user.getUserProfile().getFullName(),"fullname");
        }
        if(user.getUserProfile()!=null && user.getUserProfile().getEmail()!=null && !user.getUserProfile().getEmail().isEmpty()){
            userValidator.checkMaxLength(user.getUserProfile().getEmail(),"email");
            userValidator.checkIsValidEmail(user.getUserProfile().getEmail());
        }
    }

    @Override
    public User read(final String username) {
        checkPermission(EDIT_SECURITY);

        logger.debug("Read userProfile {}", username);
        XldUserProfile profile = userProfileService.findOne(username);
        if (profile == null) {
            throw new NotFoundException("User {} does not exist", username);
        }
        GeneralSettings generalSettings = generalSettingsService.getGeneralSettings();
        UserProfile userProfile = getUserProfile(profile,generalSettings);
        logger.debug("Read user {}", username);
        com.xebialabs.deployit.security.User u;

        if(profile.isInternal()) {
            u = userService.read(username);
        } else {
            u = new RepoUser(profile.username(), Objects.equals(profile.username(), "admin"));
        }
        return new User(u.getUsername(), u.isAdmin(), userProfile);
    }

    @Override
    public List<String> listUserNames(final String username, final Paging paging, final Ordering order) {
        checkPermissions(EDIT_SECURITY, VIEW_SECURITY);
        List<String> usernames = userService.listUsernames(username, paging, order);
        PaginationSupport.addTotalCountHeader(userService.countUsers(username), response);
        return usernames;
    }

    @Override
    public void modifyPassword(final String username, final User user) {
        checkPermission(EDIT_SECURITY);
        sanityCheckUser(username, user);
        userValidator.checkPassword(user);
        Authentication authentication = Permissions.getAuthentication();
        if (username.equals(authentication.getName())) {
            if (user.getCurrentPassword() != null && !user.getCurrentPassword().trim().isEmpty()) {
                userService.modifyPassword(username, user.getPassword(), user.getCurrentPassword());
            } else {
                throw  new IllegalArgumentException("Current password is mandatory ");
            }
        } else {
            userService.modifyPassword(username, user.getPassword());
            expireUserSession(username);
        }
        EventBusHolder.publish(new UserPasswordChangedEvent(username));
    }

    @Override
    public List<PasswordValidationResult> validatePassword(User user) {
        String password = user.getPassword() == null ? "" : user.getPassword();
        return userValidator.validatePassword(password);
    }

    @Override
    public void delete(final String username) {
        checkPermission(EDIT_SECURITY);
        userService.delete(username);
        expireUserSession(username);
        logger.debug("Deleted user {}, updating group memberships for user {}", username, username);
        userGroupService.updateGroupsMembershipForUser(username, new HashSet<>());
        EventBusHolder.publish(new UserDeletedEvent(username));
    }

    @Override
    public void updateUser(String username, User user) {
        checkPermission(EDIT_SECURITY);
        sanityCheckUser(username, user);
        XldUserProfile profile = userProfileService.findOne(username);

        if (profile == null) {
          throw new NotFoundException("User {} does not exist", username);
        }
        validateUserProfile(user);

        XldUserProfile updatedProfile = new XldUserProfile(
                profile.username(),
                profile.analyticsEnabled(),
                new HashSet<>(),
                user.getUserProfile()!=null && user.getUserProfile().getFullName() != null ? user.getUserProfile().getFullName() : null,
                user.getUserProfile()!=null && user.getUserProfile().getEmail() != null ? user.getUserProfile().getEmail(): null,
                user.getUserProfile().getLoginAllowed(),
                profile.lastActive(),
                profile.isInternal(),
                profile.displayUsername()
        );
        userProfileService.createOrUpdateUserProfile(updatedProfile);

        // Expire session for internal/external users if user account is disabled or auto disabled on edit user when license exceeded
        XldUserProfile modifiedProfile = userProfileService.findOne(username);
        if (profile.loginAllowed() && !modifiedProfile.loginAllowed()) {
            expireUserSession(username);
        }
        if (user.getPassword()!=null && !user.getPassword().isEmpty() && user.getUserProfile().getIsInternal()) {
            modifyPassword(username, user);
        }
        EventBusHolder.publish(new UserUpdateEvent(username));
    }

    @Override
    public List<User> listAllUserProfiles(String username, String fullName, String email, Paging paging, Ordering order) {
        checkPermissions(EDIT_SECURITY, VIEW_SECURITY);
        List<XldUserProfile> xldUserProfiles = userProfileService.listAllUserProfiles(username, fullName, email, paging, order);
        GeneralSettings generalSettings = generalSettingsService.getGeneralSettings();
        List<User> usersList = xldUserProfiles.stream().map(xldUserProfile -> {
            UserProfile userProfile = getUserProfile(xldUserProfile,generalSettings);
            return new User(xldUserProfile.username(), false, userProfile);
        }).toList();
        PaginationSupport.addTotalCountHeader(userProfileService.countUsers(username, fullName, email), response);
        return usersList;
    }

    private void expireUserSession(String username) {
        for (SessionInformation information : sessionRegistry.getAllSessions(username, false)) {
            sessionRepository.deleteById(information.getSessionId());
        }
        persistentTokenRepository.removeUserTokens(username);
    }

    private static void sanityCheckUser(final String username, final User user) {
        if (user.getUsername() == null || user.getUsername().trim().isEmpty()) {
            throw new IllegalArgumentException("User name cannot be empty");
        }
        if (!user.getUsername().equals(username)) {
            throw new IllegalArgumentException("Username in URL is " + username + " but username in POST data is " + user.getUsername());
        }
    }

    void setUserService(UserService userService) {
        this.userService = userService;
    }

    void setUserValidator(UserValidator userValidator) {
        this.userValidator = userValidator;
    }
    void setPrincipalDataProvider(PrincipalDataProvider principalDataProvider) {
        this.principalDataProvider = principalDataProvider;
    }
    void setGeneralSettingsService(GeneralSettingsService generalSettingsService) {
        this.generalSettingsService = generalSettingsService;
    }

    void setUserProfileService(UserProfileService userProfileService) {
        this.userProfileService = userProfileService;
    }
    void setUserGroupService(UserGroupService userGroupService) {
        this.userGroupService = userGroupService;
    }
    void setSessionRegistry(SessionRegistry sessionRegistry) { this.sessionRegistry = sessionRegistry; }
    void setPersistentTokenRepository(PersistentTokenRepository persistentTokenRepository) {this.persistentTokenRepository = persistentTokenRepository; }
    private static final Logger logger = LoggerFactory.getLogger(UserResource.class);

    private UserProfile getUserProfile(XldUserProfile profile, GeneralSettings generalSettings ) {
        if (profile != null ) {           // to ensure that the settings are loaded
            boolean personalAccessTokenEnabled = generalSettings.patEnableForSSO() || (profile.isInternal() ||
                    Optional.ofNullable(principalDataProvider.getUserData(profile.username()))
                            .map(UserData::isFound)
                            .orElse(false));
            UserProfile userProfile = new UserProfile();
            userProfile.setEmail(profile.email() != null ? profile.email() : null);
            userProfile.setFullName(profile.fullName() != null ? profile.fullName() : null);
            userProfile.setIsInternal(profile.isInternal());
            userProfile.setLoginAllowed(profile.loginAllowed());
            userProfile.setLastActive(profile.lastActive());
            userProfile.setDisplayUsername(profile.displayUsername());
            userProfile.setPersonalAccessTokenEnabled(personalAccessTokenEnabled);
            return userProfile;
        }
        return null;
    }

    private static XldUserProfile getXldUserProfile(User user) {
        XldUserProfile userProfile = new XldUserProfile(
                user.getUsername(),
                true,
                new HashSet<>(),
                user.getUserProfile()!=null && StringUtils.hasText(user.getUserProfile().getFullName()) ? user.getUserProfile().getFullName() : null,
                user.getUserProfile()!=null && StringUtils.hasText(user.getUserProfile().getEmail()) ? user.getUserProfile().getEmail(): null,
                true,
                null,
                true,
                user.getUsername());
        return userProfile;
    }

    /**
     * Checks for existing external users before creating internal user.
     *
     * @param username
     */
    private void checkNoUserProfileAlreadyExists(String username) {
        try {
            boolean userProfileExists = userProfileService.findOne(username) != null;
            if (userProfileExists) {
                throw new UserAlreadyExistsException(username);
            }
        } catch (NotFoundException nof){
            logger.debug("User {} does not exist", username);
        }
    }
}

