package com.xebialabs.deployit.core.rest.api.reports.widgets;

import static com.google.common.collect.Lists.newArrayList;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Collection;
import java.util.List;
import java.util.Locale;

import com.xebialabs.deployit.core.api.dto.Report;
import com.xebialabs.deployit.core.api.dto.Report.ReportLine;
import com.xebialabs.deployit.core.api.resteasy.Date;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.DeploymentTaskInfo;
import com.xebialabs.deployit.task.TaskArchive;

public class DeploymentsDurationFrequencyWidget extends DashboardWidgetBase {

	private static final String INTERVAL = "interval";
	private static final String DEPLOYMENT_PERCENT = "deploymentPercent";
	private static final String NUM_OF_DEPLOYMENT = "numOfDeployment";

	private final DecimalFormat format = new DecimalFormat("#.##", DecimalFormatSymbols.getInstance(Locale.ENGLISH));

	public DeploymentsDurationFrequencyWidget(TaskArchive taskArchive) {
		this.taskArchive = taskArchive;
	}

	public final static Widget getInstance(final TaskArchive taskArchive) {
		return new DeploymentsDurationFrequencyWidget(taskArchive);
	};

	@Override
	public Report getReport(Date begin, Date end) {
		final ArchivedTaskSearchParameters sp = new ArchivedTaskSearchParameters().createdBetween(begin.getCalendar(), end.getCalendar()).thatCompleted();
		final Collection<DeploymentTaskInfo> foundTasks = taskArchive.searchTasksWithoutLoadingSteps(sp);
		if (foundTasks.size() == 0)
			return new Report();
		return generateReport(foundTasks);
	}

	private Report generateReport(Collection<DeploymentTaskInfo> foundTasks) {
		final long mean = mean(foundTasks);
		final long standardDeviation = standardDeviation(mean, foundTasks);
		final List<Long> clusterLimits = clusterLimits(mean, standardDeviation);
		return createDetails(foundTasks, clusterLimits);
	}

	private long mean(Collection<DeploymentTaskInfo> tasks) {
		long total = 0;
		for (DeploymentTaskInfo taskState : tasks) {
			total += taskState.getDurationInMillis();
		}
		return total / tasks.size();// Ignoring the fractions
	}

	private long standardDeviation(long mean, Collection<DeploymentTaskInfo> tasks) {
		if (tasks.size() > 1) {
			long total = 0;
			for (DeploymentTaskInfo taskState : tasks) {
				final long distance = taskState.getDurationInMillis() - mean;
				total += distance * distance;
			}
			return (long) Math.sqrt(total / (tasks.size() - 1));// Ignore the fractions
		} else
			return 0;
	}

	private List<Long> clusterLimits(long mean, long standardDeviation) {
		List<Long> clusterLimits = newArrayList();
		if ((standardDeviation > 0) && (mean - standardDeviation - standardDeviation > 0)) {
			clusterLimits.add(mean - standardDeviation - standardDeviation);
		}
		if ((standardDeviation > 0) && (mean - standardDeviation > 0)) {
			clusterLimits.add(mean - standardDeviation);
		}
		clusterLimits.add(mean);
		if (standardDeviation > 0) {
			clusterLimits.add(mean + standardDeviation);
			clusterLimits.add(mean + standardDeviation + standardDeviation);
		}
		return clusterLimits;
	}

	private Report createDetails(Collection<DeploymentTaskInfo> tasks, final List<Long> clusterLimits) {
		final Report report = new Report();
		final int[] clusters = cluster(tasks, clusterLimits);
		int totalDeployments = tasks.size();
		if (totalDeployments > 1) {
			ReportLine firstLine = report.addLine();
			firstLine.addValue(INTERVAL, "<" + formatToMinsAndSecs(clusterLimits.get(0)));
			firstLine.addValue(DEPLOYMENT_PERCENT, format.format(getPercentage(clusters[0], totalDeployments)));
			firstLine.addValue(NUM_OF_DEPLOYMENT, format.format(clusters[0]));
		}

		for (int i = 1; i < clusterLimits.size(); i++) {
			ReportLine line = report.addLine();
			line.addValue(INTERVAL, formatToMinsAndSecs(clusterLimits.get(i - 1)) + " - " + formatToMinsAndSecs(clusterLimits.get(i)));
			line.addValue(DEPLOYMENT_PERCENT, format.format(getPercentage(clusters[i], totalDeployments)));
			line.addValue(NUM_OF_DEPLOYMENT, format.format(clusters[i]));
		}

		ReportLine lastLine = report.addLine();
		lastLine.addValue(INTERVAL, ">" + formatToMinsAndSecs(clusterLimits.get(clusterLimits.size() - 1)));
		lastLine.addValue(DEPLOYMENT_PERCENT, format.format(getPercentage(clusters[clusters.length - 1], totalDeployments)));
		lastLine.addValue(NUM_OF_DEPLOYMENT, format.format(clusters[clusters.length - 1]));
		return report;
	}

	private int[] cluster(Collection<DeploymentTaskInfo> tasks, List<Long> clusterLimits) {
		int[] cluster = new int[clusterLimits.size() + 1];

		for (DeploymentTaskInfo taskState : tasks) {
			boolean aboveMaxLimit = true;
			for (int i = 0; i < clusterLimits.size(); i++) {
				Long limit = clusterLimits.get(i);
				if (taskState.getDurationInMillis() < limit) {
					cluster[i]++;
					aboveMaxLimit = false;
					break;
				}
			}
			if (aboveMaxLimit)
				cluster[clusterLimits.size()]++;
		}
		return cluster;
	}
}
