package com.xebialabs.deployit.core.rest.api.reports.widgets;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import com.xebialabs.deployit.core.api.dto.Report;
import com.xebialabs.deployit.core.api.dto.Report.ReportLine;
import com.xebialabs.deployit.core.api.resteasy.Date;
import com.xebialabs.deployit.jcr.grouping.Count;
import com.xebialabs.deployit.jcr.grouping.GroupBy;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;

public abstract class Top5DeploymentsByState extends DashboardWidgetBase {

	static final String FAILURE_COUNT = "failureCount";
	static final String STATE = "state";
	static final String COUNT = "count";
	static final String APPLICATION_NAME = "applicationName";
	static final String ENVIRONMENT = "environment";

	abstract boolean accept(Map<String, Object> line);

	@Override
	public Report getReport(final Date begin, final Date end) {
		ArchivedTaskSearchParameters sp = new ArchivedTaskSearchParameters();
		sp.createdBetween(begin.getCalendar(), end.getCalendar());
		final Collection<Map<String, Object>> searchResult = taskArchive.searchTasksWithoutLoadingSteps(sp, new GroupBy(newArrayList(APPLICATION_NAME, ENVIRONMENT, STATE, FAILURE_COUNT),
		        new Count(COUNT)));
		return getTop5(searchResult);
	}

	private Report getTop5(Collection<Map<String, Object>> searchResult) {
		final Map<String, Integer> totalsMap = calculateTotalsPerApplicationPerEnvironment(searchResult);
		final List<Map<String, Object>> list = filterByState(searchResult);
		Collections.sort(list, new Top5Comparator(totalsMap));

		DecimalFormat df = new DecimalFormat("#.##", DecimalFormatSymbols.getInstance(Locale.ENGLISH));
		Report report = new Report();
		for (Map<String, Object> line : list.subList(0, Math.min(list.size(), 5))) {
			ReportLine reportLine = report.addLine();
			String applicationOnEnvrionment = (String) line.get(APPLICATION_NAME) + "-" + (String) line.get(ENVIRONMENT);
			reportLine.addValue("applicationName", (String) line.get(APPLICATION_NAME));
			reportLine.addValue("environmentName", (String) line.get(ENVIRONMENT));
			reportLine.addValue("numOfDeployments", (String) line.get(COUNT).toString());
			reportLine.addValue("totalDeployments", (String) totalsMap.get(applicationOnEnvrionment).toString());
			reportLine.addValue("percentage", df.format(getPercentage((Integer) line.get(COUNT), totalsMap.get(applicationOnEnvrionment))));
		}
		return report;
	}

	private Map<String, Integer> calculateTotalsPerApplicationPerEnvironment(Collection<Map<String, Object>> resultList) {
		Map<String, Integer> totals = newHashMap();
		for (Map<String, Object> line : resultList) {
			String applicationOnEnvrionment = (String) line.get(APPLICATION_NAME) + "-" + (String) line.get(ENVIRONMENT);
			Integer total = totals.get(applicationOnEnvrionment);
			if (total == null)
				total = (Integer) line.get(COUNT);
			else
				total += (Integer) line.get(COUNT);
			totals.put(applicationOnEnvrionment, total);
		}
		return totals;
	}

	private List<Map<String, Object>> filterByState(Collection<Map<String, Object>> searchResult) {
		final List<Map<String, Object>> list = newArrayList();
		for (Map<String, Object> line : searchResult) {
			if (accept(line))
				list.add(line);
		}
		return list;
	}

	private static final class Top5Comparator implements Comparator<Map<String, Object>> {
		private final Map<String, Integer> totalsMap;

		private Top5Comparator(Map<String, Integer> totalsMap) {
			this.totalsMap = totalsMap;
		}

		@Override
		public int compare(Map<String, Object> o1, Map<String, Object> o2) {
			int comparisonResult = ((Integer) o1.get(COUNT)).compareTo((Integer) o2.get(COUNT));
			if (comparisonResult == 0) {
				final Integer total1 = totalsMap.get((String) o1.get(APPLICATION_NAME) + "-" + (String) o1.get(ENVIRONMENT));
				final Integer total2 = totalsMap.get((String) o2.get(APPLICATION_NAME) + "-" + (String) o2.get(ENVIRONMENT));
				comparisonResult = -(total1.compareTo(total2));
			}
			return -comparisonResult;// Descending sort
		}
	}
}
