package com.xebialabs.deployit.core.api;

import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.engine.api.dto.DeploymentInfo;
import com.xebialabs.deployit.engine.api.dto.Ordering;
import com.xebialabs.deployit.engine.api.dto.Paging;
import com.xebialabs.xlplatform.documentation.PublicApi;

import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import java.util.List;

/**
 * Retrieves information about Environments: which DeployedApplications and which Hosts are there in an environment?
 *
 * A User must have READ permission on the environment to obtain the requested info.
 */
@Path("/environment")
@PublicApi
@Produces({MediaType.APPLICATION_JSON})
public interface EnvironmentService {

    /**
     * Finds DeployedApplications on an environment. Optionally (a part of) an application name can be specified to narrow your
     * search results, e.g. specifying <i>deployedAppName=Depl</i> will find <i>XL Deploy</i>
     *
     * @param envId the environment ID to query
     * @param deployedAppName (Optional) (part of the) deployed application name to look for.
     * @param paging Which page and how many results per page to retrieve
     * @param order Which ordering to use when listing results.
     * @return a list of {@see com.xebialabs.deployit.engine.api.dto.ConfigurationItemId} objects corresponding to deployed
     * applications on the indicated environment
     */
    @GET
    @Path("{id:.+}/deployed-applications")
    List<ConfigurationItemId> getDeployedApplicationsByEnvironment(
            @PathParam("id") String envId,
            @QueryParam("deployedAppName") String deployedAppName,
            @BeanParam Paging paging,
            @QueryParam("order") Ordering order
    );

    /**
     * Finds Hosts that take part in the specified environment. Will return only hosts that the user has READ permission on.
     * Optionally, (part of) a host name can be specified to narrow search results. A list of DeploymentInfo objects will be
     * returned detailing the host ID, the host Type, and when and by whom a deployment to this host has taken place.
     *
     * @param envId the environment ID to query
     * @param hostName (Optional) (part of) the host name to look for
     * @param paging Which page and how many results per page to retrieve
     * @param order Which ordering to use when listing results
     * @return a list of {@see com.xebialabs.deployit.engine.api.dto.DeploymentInfo} objects
     */
    @GET
    @Path("{id:.+}/infrastructure")
    List<DeploymentInfo> getHostsByEnvironment(
            @PathParam("id") String envId,
            @QueryParam("name") String hostName,
            @BeanParam Paging paging,
            @QueryParam("order") Ordering order
    );
}
