package com.xebialabs.deployit.core.api;

import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.engine.api.dto.DeploymentInfo;
import com.xebialabs.deployit.engine.api.dto.Ordering;
import com.xebialabs.deployit.engine.api.dto.Paging;
import com.xebialabs.xlplatform.documentation.PublicApi;

import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;
import java.util.List;

/**
 * Retrieves information about Hosts: which DeployedApplications have deployeds on this host, and which environments does this
 * host take part in?
 *
 * The user must have READ access to the specified host.
 */
@Path("/host")
@Produces({MediaType.APPLICATION_JSON})
@PublicApi
public interface HostService {

    /**
     * Find deployedApplications that have deployeds on the specified host.
     *
     * @param hostId the host ID
     * @param deployedAppName (Optional) (Part of) the deployedApplication's to look for
     * @param paging How many results to display per page, and which page to retrieve
     * @param order How to order the search results.
     * @return A list of {@see com.xebialabs.deployit.engine.api.dto.ConfigurationItemId} objects corresponding to the
     * deployedApplications
     */
    @GET
    @Path("deployed-applications/{ID:.+}")
    List<ConfigurationItemId> findDeployedApplications(
            @PathParam("ID") String hostId,
            @QueryParam("deployedAppName") String deployedAppName,
            @BeanParam Paging paging,
            @QueryParam("order") Ordering order
    );

    /**
     * Gets all environments that the specified host is part of, including when and by whom this environment was last deployed to.
     * Lists only environments that the user has READ permission for.
     *
     * @param hostId The host ID to query for
     * @param environmentName (Optional) (part of) the name of the environment(s) to look for
     * @param paging How many results to display per page, and which page to retrieve
     * @param order How to order the search results.
     * @return a list of {@see com.xebialabs.deployit.engine.api.dto.DeploymentInfo} objects
     */
    @GET
    @Path("{id:.+}/environments")
    List<DeploymentInfo> getEnvironmentsByHost(@PathParam("id") String hostId,
                                                 @QueryParam("name") String environmentName,
                                                 @BeanParam Paging paging,
                                                 @QueryParam("order") Ordering order);

    /**
     * Count the number of environments that the user has READ access to and the specified host is part of, optionally filtered by
     * (part of) the environment name
     *
     * @param hostId The host ID to query
     * @param environmentName (Optional) (part of) the environment name(s) to look for.
     * @return The number of environments that this host is part of
     */
    @GET
    @Path("{id:.+}/environments/count")
    Integer countEnvironmentsByHost(@PathParam("id") String hostId,
                                 @QueryParam("name") String environmentName);

}
