package com.xebialabs.deployit.core.api.dto;

import java.util.Set;
import java.util.Map;
import java.util.Map.Entry;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElementRef;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlSeeAlso;
import javax.xml.bind.annotation.adapters.XmlJavaTypeAdapter;
import static java.lang.String.format;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import static com.google.common.collect.Maps.newTreeMap;

import com.xebialabs.deployit.core.api.jaxb.PrincipalPermissionEntry;
import com.xebialabs.deployit.core.api.jaxb.PrincipalPermissionList;
import com.xebialabs.deployit.core.api.jaxb.PrincipalPermissionsMapAdapter;

@XmlAccessorType(XmlAccessType.FIELD)
@XmlRootElement(name="security-settings")
@XmlSeeAlso({PrincipalPermissionList.class, PrincipalPermissionEntry.class})
public class PrincipalPermissions extends AbstractDTO {

	@XmlAttribute
	private String username;
	
	@XmlElementRef
	@XmlJavaTypeAdapter(value = PrincipalPermissionsMapAdapter.class)
	private Map<String, Set<String>> principalPermissions = newTreeMap();

	public PrincipalPermissions() {
	}

	public PrincipalPermissions(final String userName, final Map<String, Set<String>> principalPermissions) {
		this.username = userName;
		if (principalPermissions != null) {
			this.principalPermissions = principalPermissions;
		}
	}

	public String getUsername() {
		return username;
	}

	public void setUsername(final String userName) {
		this.username = userName;
	}

	public boolean hasPermissions() {
		return !principalPermissions.isEmpty();
	}

	public boolean hasPermission(final String permission) {
		return principalPermissions.containsKey(permission);
	}

	public boolean hasPermission(final String permission, final String target) {
		return principalPermissions.containsKey(permission) && (principalPermissions.get(permission).contains(target) || principalPermissions.get(permission).contains(""));
	}	

	public boolean isPermissionGlobal(final String permission) {
		return principalPermissions.containsKey(permission) && principalPermissions.get(permission).contains("");
	}

	public boolean setPermissions(final String permission, final Set<String> targets) {
		boolean validMapping = (permission != null) && (targets != null);
		if (validMapping) {
			principalPermissions.put(permission, targets);
		}
		return validMapping && this.principalPermissions.containsKey(permission);
	}

	public Set<String> getPermissions() {
		return principalPermissions.keySet();
	}

	public Set<String> getPermissionTargets(final String permission) {
		return principalPermissions.get(permission);
	}

	public Map<String, Set<String>> getValues() {
		return principalPermissions;
	}

	public Set<Entry<String, Set<String>>> getValuesAsSet() {
		return principalPermissions.entrySet();
	}

	public Map<String, Set<String>> asTargetsWithPermissions() {
		return swapPermissionsTargetsMap(principalPermissions.entrySet());
	}

	public String toString() {
		StringBuilder permissionString = new StringBuilder();
		if ((principalPermissions == null) || principalPermissions.isEmpty()) {
			permissionString.append("There are no permissions set for the requested user, or the requested user doesn't exist!\n");
		} else {
			for (String permissionTarget : principalPermissions.keySet()) {
				if (principalPermissions.get(permissionTarget).contains("")) {
					permissionString.append(format("\nGranted global permission \'%s\'\n", permissionTarget));
				} else {
					permissionString.append(format("\nGranted permission \'%s\' on:\n", permissionTarget));
					for (String value : principalPermissions.get(permissionTarget)) {
						permissionString.append(format("\t %s\n", value));
					}
					permissionString.append("\n");
				}
			}
		}
		return permissionString.toString();
	}

	private Map<String, Set<String>> swapPermissionsTargetsMap(final Set<Entry<String, Set<String>>> permissionsSet) {
		Map<String, Set<String>> targets = Maps.<String, Set<String>>newTreeMap();
		for (Entry<String, Set<String>> permission : permissionsSet) {
			for (String target : permission.getValue()) {
				if (!targets.containsKey(target)) {
					targets.put(target, Sets.<String>newTreeSet());
				}
				targets.get(target).add(permission.getKey());
			}
		}
		return targets;
	}
}
