package com.xebialabs.deployit.core.api.resteasy;

import static java.lang.String.format;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Locale;
import java.util.TimeZone;

/**
 * Can be used as a parameter in RestEASY, as it has a String argument constructor.
 */
public class Date implements Comparable<Date> {

	public static final String FORMAT = "yyyy-MM-dd'T'HH:mm:ss.SSSZ";

	/**
	 * List of possible RFC822 date formats. From <http://jimyjoshi.com/blog/2007/08/rfc822dateparsinginjava.html>.
	 */
	protected static final String rfc822DateFormats[] = new String[] { "EEE, dd MMM yy HH:mm:ss z", "EEE, dd MMM yy HH:mm z", "EEE, dd MMM yyyy HH:mm:ss z",
	        "EEE, dd MMM yyyy HH:mm z", "dd MMM yy HH:mm z", "dd MMM yy HH:mm:ss z", "dd MMM yyyy HH:mm z", "dd MMM yyyy HH:mm:ss z",
	        "EEE MMM dd HH:mm:ss zzz yyyy", "EEE, dd MMM yyyy z", FORMAT};

	private Calendar calendar;
	private String dateString;

	public Date(final Calendar date) {
		this(date, null);
	}

	public Date(final Calendar date, final String format) {
		this.calendar = date;
		if (this.calendar != null) {
			if (format != null) {
				dateString = new SimpleDateFormat(format).format(calendar.getTime());
			} else {
				dateString = formatter(rfc822DateFormats[7]).format(calendar.getTime());
			}
		}
	}

	private SimpleDateFormat formatter(String pattern) {
		final SimpleDateFormat simpleDateFormat = new SimpleDateFormat(pattern, Locale.ENGLISH);
		simpleDateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
		return simpleDateFormat;
	}

	public Date(String dateString) {
		this.dateString = dateString;
		if (dateString != null) {
			calendar = parse(dateString);
		}
	}

	public static Date valueOf(String dateString) {
		return new Date(dateString);
	}

	private Calendar parse(final String date) {
		if (date == null) {
			return null;
		}

		for (String each : rfc822DateFormats) {
			try {
				java.util.Date parsedDate = formatter(each).parse(date);
				Calendar cal = Calendar.getInstance(Locale.ENGLISH);
				cal.setTimeZone(TimeZone.getTimeZone("UTC"));
				cal.setTime(parsedDate);
				return cal;
			} catch (ParseException ignored) {
			}
		}
		throw new IllegalArgumentException(format("Cannot parse date %s", date));
	}

	public Calendar getCalendar() {
		return calendar;
	}

	@Override
	public int compareTo(Date arg0) {
		return this.calendar.compareTo(arg0.calendar);
	}
	
	@Override
	public String toString() {
		return dateString;
	}

}
