package com.xebialabs.xlplatform.rest.script.endpoints

import spray.routing._
import spray.routing.AuthenticationFailedRejection.{CredentialsRejected, CredentialsMissing}
import spray.http.StatusCodes
import spray.routing.authentication.{BasicAuth, UserPass}
import scala.concurrent.Future
import com.xebialabs.xldeploy.rest.SpringContextHolder
import org.springframework.security.authentication.{UsernamePasswordAuthenticationToken, AuthenticationManager}
import spray.routing.authentication.UserPass
import scala.Some
import akka.util.Timeout
import concurrent.duration._
import language.postfixOps

trait ExtendableRestApi extends HttpService with ScriptExtensionRoute with MetadataExtensionRoute {
  import scala.concurrent.ExecutionContext.Implicits.global

  override implicit val timeout: Timeout = Timeout(15 seconds)

  val customRejectionHandler = RejectionHandler {
    case AuthenticationFailedRejection(CredentialsMissing, headers) :: _ => complete(StatusCodes.Unauthorized, "Please login first")
    case AuthenticationFailedRejection(CredentialsRejected, headers) :: _ => complete(StatusCodes.Forbidden, "You do not have access")
    case MalformedRequestContentRejection(msg, _) :: _ => complete(StatusCodes.BadRequest, "Malformed Content")
  }

  private def clientAuthenticator(userPass: Option[UserPass]): Future[Option[AuthenticatedData]] = {
    userPass match {
      case None => Future { None }
      case Some(UserPass(user, pass)) =>
        val authManager = SpringContextHolder.getApplicationContext.getBean("authenticationManager").asInstanceOf[AuthenticationManager]
        val auth = authManager.authenticate(new UsernamePasswordAuthenticationToken(user, pass))

        if (!auth.isAuthenticated) Future{None}
        else Future{Some(AuthenticatedData(auth))}
    }
  }

  private val pingRoute = pathPrefix("ping") {
    get {
      complete("Pong!")
    }
  }

  private[endpoints] def defaultRoutes: Seq[AuthenticatedRoute] = Seq(customScriptEndpoints, Seq(metadataRoute)).flatten

  def extendableRoutes: Route = extendableRoutes(defaultRoutes)

  def extendableRoutes(apiRoutes: Seq[AuthenticatedRoute]): Route = customRoutes(clientAuthenticator)(apiRoutes)

  private[endpoints] def customRoutes(authenticator: Option[UserPass] => Future[Option[AuthenticatedData]])(apiRoutes: Seq[AuthenticatedRoute]) = handleRejections(customRejectionHandler) {
    pathPrefix("api") {
      authenticate(BasicAuth(authenticator, realm = "XLDeploy extension API")) {
        auth: AuthenticatedData =>
          apiRoutes.foldLeft(pingRoute)(_ ~ _(auth))
      }
    }
  }
}
