package com.xebialabs.xlplatform.rest.script.jython

import akka.actor._
import java.util.{Map => JMap}
import javax.script.{ScriptEngineManager, ScriptContext, SimpleScriptContext, ScriptEngine}
import spray.json._
import java.io._
import scala.util.Try
import org.python.core.{PyList, PyDictionary, PySyntaxError}
import com.xebialabs.xldeploy.rest.ServiceHolder
import com.xebialabs.xlplatform.rest.script.endpoints.AuthenticatedData
import org.springframework.security.core.context.SecurityContextHolder
import com.xebialabs.xldeploy.rest.script.jython.{JythonRequest, JythonResponse}
import java.net.URL
import com.xebialabs.deployit.booter.local.utils.Closeables
import JythonScriptExecutorActor.SCRIPT_ENGINE
import akka.event.LoggingAdapter
import scala.util.{Success, Failure}
import scala.Some
import com.xebialabs.xlplatform.rest.script.endpoints.json.ScriptRequest
import scala.collection.JavaConversions._

object JythonScriptExecutorActor {
  def props() = Props(classOf[JythonScriptExecutorActor])

  val SCRIPT_ENGINE =  new ScriptEngineManager().getEngineByName("python")
}

case class RunScript(scriptFileUrl: URL, request: ScriptRequest, auth: AuthenticatedData)

case class ScriptDone (entity: Any, stdout: String, stderr: String, statusCode: Option[Integer] = None, headers: Map[String, String] = Map(), exception: Option[Throwable] = None)

object ScriptDone {
  def apply(jr: JythonResponse, stdout: String, stderr: String, exception: Option[Throwable]): ScriptDone = {
    val entity = Option(jr.entity).map({
        case m: JMap[String, Any] => m.toMap
        case m => m
      }).getOrElse(Map())
    ScriptDone(entity, stdout, stderr, Option(jr.statusCode), jr.headers.toMap, exception)
  }
}

class InvalidScriptOutputException(msg: String) extends RuntimeException(msg)

class JythonScriptExecutorActor extends Actor with DefaultJsonProtocol with ActorLogging {

  override def receive(): Actor.Receive = {
    case RunScript(scriptUrl, request, auth) =>
      SecurityContextHolder.getContext.setAuthentication(auth.toAuthentication)
      val reader: Reader = new BufferedReader(new InputStreamReader(scriptUrl.openStream()))
      val contextHelper = ScriptContextHelper(SCRIPT_ENGINE).withRequest(request).withClients().withEmptyResponse().withLogger(log)
      Try(SCRIPT_ENGINE.eval(reader, contextHelper.context)) match {
        case Success(_) => sender ! ScriptDone(contextHelper.jythonResponse, contextHelper.stdOut, contextHelper.stdErr, None)
        case Failure(e) if e.getCause.isInstanceOf[PySyntaxError] =>
          log.error(e.getCause, "Jython syntax error occurred.")
          sender ! ScriptDone(contextHelper.jythonResponse, contextHelper.stdOut, contextHelper.stdErr, Some(e.getCause))
        case Failure(e) =>
          log.error(e, "Jython runtime exception occurred.")
          sender ! ScriptDone(contextHelper.jythonResponse, contextHelper.stdOut, contextHelper.stdErr, Some(e.getCause))
      }

      Closeables.closeQuietly(reader)
      SecurityContextHolder.getContext.setAuthentication(null)
      self ! PoisonPill
  }
}

object ScriptContextHelper {
  def apply(se: ScriptEngine) = new ScriptContextHelper(se)
}

class ScriptContextHelper(engine: ScriptEngine) {

  final val outputParamName = "output"
  final val inputParamName = "input"
  final val requestParamName = "request"
  final val responseParamName = "response"
  final val headersParamName = "headers"
  final val statusCodeParamName = "statusCode"

  val outWriter: StringWriter = new StringWriter()
  val errWriter: StringWriter = new StringWriter()

  val context = createContext

  def createContext = {
    val c = new SimpleScriptContext
    c.setWriter(new PrintWriter(outWriter))
    c.setErrorWriter(new PrintWriter(errWriter))
    c.setBindings(engine.getContext.getBindings(ScriptContext.ENGINE_SCOPE), ScriptContext.GLOBAL_SCOPE)
    c
  }


  def jythonResponse = Option(context.getAttribute(responseParamName)).map(_.asInstanceOf[JythonResponse]).getOrElse(new JythonResponse())


  def stdOut = outWriter.getBuffer.toString

  def stdErr = errWriter.getBuffer.toString

  def withRequest(req: ScriptRequest): ScriptContextHelper = {
    def mapAsJava(jsv: JsValue): Object = jsv match {
      case JsString(value) => value
      case JsNumber(value) if value.isValidInt => Int.box(value.toInt)
      case JsNumber(value) if value.isValidLong => Long.box(value.toLong)
      case JsNumber(value) => Double.box(value.toDouble)
      case JsArray(elems) =>
        val pyList = new PyList()
        pyList.addAll(asJavaCollection(elems.map(mapAsJava(_))))
        pyList
      case JsObject(fields) =>
        val pyDict = new PyDictionary()
        pyDict.putAll(mapAsJavaMap(fields.mapValues(mapAsJava(_))))
        pyDict
      case JsTrue => Boolean.box(true)
      case JsFalse => Boolean.box(false)
      case _ => null
    }

    val simplifiedParams = req.query.map {
      case (x: String,  List("")) =>(x, null)
      case (x: String, h :: Nil) => (x, h)
      case (x, hh:List[String]) => (x, seqAsJavaList(hh))
      case (x, hh) => (x, hh)
    }

    val queryPyDict = new PyDictionary()
    queryPyDict.putAll(mapAsJavaMap(simplifiedParams))

    val jreq = new JythonRequest(mapAsJava(req.entity), queryPyDict)
    context.setAttribute(requestParamName, jreq, ScriptContext.ENGINE_SCOPE)
    this
  }

  def withEmptyResponse(): ScriptContextHelper = {
    context.setAttribute(responseParamName, new JythonResponse(), ScriptContext.ENGINE_SCOPE)
    this
  }

  def withLogger(log:LoggingAdapter): ScriptContextHelper = {
    context.setAttribute("logger", log, ScriptContext.ENGINE_SCOPE)
    this
  }

  def withClients(): ScriptContextHelper = {
    context.setAttribute("controlService", ServiceHolder.getControlService, ScriptContext.ENGINE_SCOPE)
    context.setAttribute("deploymentService", ServiceHolder.getDeploymentService, ScriptContext.ENGINE_SCOPE)
    context.setAttribute("inspectionService", ServiceHolder.getInspectionService, ScriptContext.ENGINE_SCOPE)
    context.setAttribute("metadataService", ServiceHolder.getMetadataService, ScriptContext.ENGINE_SCOPE)
    context.setAttribute("packageService", ServiceHolder.getPackageService, ScriptContext.ENGINE_SCOPE)
    context.setAttribute("permissionService", ServiceHolder.getPermissionService, ScriptContext.ENGINE_SCOPE)
    context.setAttribute("repositoryService", ServiceHolder.getRepositoryService, ScriptContext.ENGINE_SCOPE)
    context.setAttribute("roleService", ServiceHolder.getRoleService, ScriptContext.ENGINE_SCOPE)
    context.setAttribute("serverService", ServiceHolder.getServerService, ScriptContext.ENGINE_SCOPE)
    context.setAttribute("taskService", ServiceHolder.getTaskService, ScriptContext.ENGINE_SCOPE)
    context.setAttribute("taskBlockService", ServiceHolder.getTaskBlockService, ScriptContext.ENGINE_SCOPE)
    context.setAttribute("userService", ServiceHolder.getUserService, ScriptContext.ENGINE_SCOPE)
    this
  }


}
