package com.xebialabs.deployit.cluster

import akka.actor.ActorSystem
import com.typesafe.config.{Config, ConfigFactory}
import com.xebialabs.xlplatform.cluster.ClusterMode._
import com.xebialabs.xlplatform.cluster.{ClusterConfig, NodeState, XlCluster}
import com.xebialabs.xlplatform.config.ProductConfiguration
import com.xebialabs.deployit.ServerConfiguration
import grizzled.slf4j.Logging
import org.springframework.stereotype.Component

import scala.util.Try

@Component
class DeployClusterActorSystem(xldProductConfiguration: ProductConfiguration) extends Logging {

  def boot(): Unit = {
    val fallbackSettings = ConfigFactory.parseString(
      """
        | name = "deploy-cluster"
        | membership {
        |   heartbeat = 10 seconds
        |   ttl = 60 seconds
        |   jdbc {
        |     pool-name = "ClusterPool"
        |     max-pool-size = 1
        |     minimum-idle = 1
        |     connection-timeout = 30 seconds
        |     max-life-time = 30 minutes
        |     idle-timeout = 10 minutes
        |     leak-connection-threshold = 15 seconds
        |   }
        | }
        | akka {
        |   actor.provider = "akka.cluster.ClusterActorRefProvider"
        |   loggers = ["akka.event.slf4j.Slf4jLogger"]
        |   loglevel = "INFO"
        | }
      """.stripMargin)

    val systemClusterConfig = Try(
      xldProductConfiguration.configuration.getConfig("deploy.cluster")
    ).getOrElse({
      logger.info("No cluster settings found in deploy-cluster.yaml - assuming Standalone")
      ConfigFactory.empty()
    })
    boot(systemClusterConfig.withFallback(fallbackSettings))
  }

  private def resolveAkkaHostnameWithFallback(config: Config): Config = {
    val confWithHostname = {
      if (xldProductConfiguration.configuration.hasPath(ServerConfiguration.KEY_CLUSTER_NODE_HOSTNAME))
        config
          .withValue("akka.remote.artery.canonical.hostname", xldProductConfiguration.configuration.getValue(ServerConfiguration.KEY_CLUSTER_NODE_HOSTNAME))
      else if (xldProductConfiguration.configuration.hasPath(ServerConfiguration.KEY_SERVER_HOSTNAME))
        config
          .withValue("akka.remote.artery.canonical.hostname", xldProductConfiguration.configuration.getValue(ServerConfiguration.KEY_SERVER_HOSTNAME))
      // For upgrade from 10.2.x to latest
      else if (xldProductConfiguration.configuration.hasPath("deploy.server.hostname"))
        config
          .withValue("akka.remote.artery.canonical.hostname", xldProductConfiguration.configuration.getValue("deploy.server.hostname"))
      else
        config.withFallback(ConfigFactory.parseString(s"akka.remote.artery.canonical.hostname= " + s"${ServerConfiguration.getInstance().getServerHostname()}"))
    }
    confWithHostname
  }

  private def resolveAkkaBindHostnameWithFallback(config: Config): Config = {
    val confWithHostname = {
      if (xldProductConfiguration.configuration.hasPath(ServerConfiguration.KEY_CLUSTER_NODE_BIND_HOSTNAME))
        config
          .withValue("akka.remote.artery.bind.hostname", xldProductConfiguration.configuration.getValue(ServerConfiguration.KEY_CLUSTER_NODE_BIND_HOSTNAME))
      else
        config.withFallback(ConfigFactory.parseString(s"akka.remote.artery.bind.hostname= " + s"0.0.0.0"))
    }
    confWithHostname
  }

  private def resolveAkkaPortWithFallback(configWithHostname: Config): Config = {
    val confWithPort =
      if (xldProductConfiguration.configuration.hasPath(ServerConfiguration.KEY_CLUSTER_NODE_PORT))
        configWithHostname
          .withValue("akka.remote.artery.canonical.port", xldProductConfiguration.configuration.getValue(ServerConfiguration.KEY_CLUSTER_NODE_PORT))
          .withValue("akka.remote.artery.bind.port", xldProductConfiguration.configuration.getValue(ServerConfiguration.KEY_CLUSTER_NODE_PORT))
      else
        configWithHostname
          .withFallback(ConfigFactory.parseString(s"akka.remote.artery.canonical.port=" + s"${ServerConfiguration.getInstance().getClusterNodePort().toString}"))
    confWithPort;
  }

  def boot(config: Config): Unit = {
    val clusterConfig = ClusterConfig(resolveAkkaPortWithFallback(resolveAkkaBindHostnameWithFallback(resolveAkkaHostnameWithFallback(config))))
    logger.info(s"mode is ${clusterConfig.mode}")
    clusterConfig.mode match {
      case Standalone =>
        NodeState.setActive(true)
      case HotStandby | Full =>
        val mgmtActorSystem = ActorSystem(clusterConfig.name, clusterConfig.config)
        XlCluster.init(clusterConfig)
        XlCluster(mgmtActorSystem).start(() => {})
      case other =>
        throw new IllegalArgumentException(s"Cluster mode $other is not supported.")
    }
  }
}
