package datasets

import com.xebialabs.deployit.plugin.api.udm._
import com.xebialabs.deployit.plugin.test.v3.{PortalWar, Portlet, PortletAttributes, PortletSettings}
import com.xebialabs.deployit.repository._
import com.xebialabs.deployit.rest.test.api.DataSet
import ai.digital.deploy.tasker.common.TaskType
import dsl.{CiUtils, EarUtils, EnvironmentUtils, TaskUtils, Creator => DslCreator}
import org.joda.time.DateTime
import org.slf4j.{Logger, LoggerFactory}

import java.util.UUID
import scala.jdk.CollectionConverters._

object RiaTestSet extends DataSet with EarUtils with EnvironmentUtils with TaskUtils with CiUtils {

  def name() = "RiaTestSet"

  def generateSet(args_unused: List[String]): Unit = {
    create { c =>
      createEnvironments(c)
      createPackages(c)
      c.directory("Applications/dropFolder1")
      c.directory("Applications/dropFolder1/dropFolder2")
      c.dictionary("PlaceholderDictionary", Map("htmlPlaceholder" -> "SampleApp", "config2SecondPlaceholder" -> "config2SecondPlaceholderValue").asJava)
      c.dictionary("dict1", Map("key1" -> "true", "key2" -> "bbbbb", "key3" -> "ccccc").asJava)
      c.dictionary("dict2", Map("key1" -> "true", "key2" -> "ddddd", "key4" -> "xxxxx").asJava)
      c.dictionary("dict3", Map("key1" -> "true", "key2" -> "eeeee", "key5" -> "false").asJava)
      c.pipeline("AnimalZoo-pipeline")
      c.template("cloud-template-small", null, 2)
      c.template("cloud-template-medium", "This is a medium dummy template. This is a medium dummy template. This is a medium dummy template. This is a medium dummy template. This is a medium dummy template. This is a medium dummy template. ", 3)
      c.template("cloud-template-large", "This is a very large dummy template. This is a very large dummy template. This is a very large dummy template. This is a very large dummy template. This is a very large dummy template. This is a very large dummy template. This is a very large dummy template. This is a very large dummy template. This is a very large dummy template. This is a very large dummy template. This is a very large dummy template. ", 4)
    }

    createUsers()
    generateTasksForReports()
    importPackages()
    configurePipelines()

    val startDate1 = new DateTime(2010, 11, 10, 9, 30, 10)
    val startDate2 = new DateTime(2010, 11, 15, 7, 20, 56)

    val completionDate1 = new DateTime(2010, 11, 10, 10, 30, 45)
    val completionDate2 = new DateTime(2010, 11, 15, 10, 30, 45)

    deploy("Applications/tinyApp/1.0", "Environments/0tinyEnv0")
    deploy("Applications/tinyApp/1.0", "Environments/1newEnv1")
    deploy("Applications/smallApp/1.0", "Environments/0tinyEnv0")
    deploy("Applications/smallApp/1.0", "Environments/1newEnv1")

    // used in HttpUnit tests

    val repository = RepositoryServiceHolder.getRepositoryService

    val appCis = readCisAsMap(List("tinyApp", "smallApp").map("Applications/" + _), repository)
    val envCis = readCisAsMap(List("0tinyEnv0", "1newEnv1").map("Environments/" + _), repository)

    task { t =>
      t.success.forApp("tinyApp", appCis("tinyApp").get$internalId).forVersion("1.0").forEnv("0tinyEnv0", envCis("0tinyEnv0").get$internalId).startedAt(startDate1).completedAt(completionDate1).withId(UUID.randomUUID().toString).withDefaultBlock
    }

    task { t =>
      t.success.forApp("tinyApp", appCis("tinyApp").get$internalId).forVersion("1.0").forEnv("1newEnv1", envCis("1newEnv1").get$internalId).startedAt(startDate1).completedAt(completionDate1).withId(UUID.randomUUID().toString).withDefaultBlock
    }

    task { t =>
      t.success.forApp("smallApp", appCis("smallApp").get$internalId).forVersion("1.0").forEnv("0tinyEnv0", envCis("0tinyEnv0").get$internalId).startedAt(startDate2).completedAt(completionDate2).withId(UUID.randomUUID().toString).withDefaultBlock
    }

    task { t =>
      t.success.forApp("smallApp", appCis("smallApp").get$internalId).forVersion("1.0").forEnv("1newEnv1", envCis("1newEnv1").get$internalId).startedAt(startDate2).completedAt(completionDate2).withId(UUID.randomUUID().toString).withDefaultBlock
    }

    task { t =>
      t.success.forCloudOperation("instantiate")
        .forCloudEnv("Environments/SmallCloud")
        .withCloudEnvTemplate("Configuration/SmallCloudEnvTemplate")
        .startedAt(startDate1)
        .completedAt(completionDate1).withId(UUID.randomUUID().toString)
        .forDeploymentType(TaskType.CONTROL)
        .withDefaultBlock
    }
    task { t =>
      t.success.forCloudOperation("destroy")
        .forCloudEnv("Environments/SmallCloud")
        .withCloudEnvTemplate("Configuration/SmallCloudEnvTemplate")
        .startedAt(startDate2)
        .completedAt(completionDate2).withId(UUID.randomUUID().toString)
        .forDeploymentType(TaskType.CONTROL)
        .withDefaultBlock
    }
  }

  private def createPackages(c: DslCreator): Unit = {
    createEmptyPackage(c, "EmptyPackage", nrPackages = 1)
    createPackageWithDummyEar(c, "newApp", nrPackages = 10, packageSize = 1, "newEar")
    createPackageWithDummyEar(c, "oldApp", nrPackages = 10, packageSize = 3, "oldEar")
    createPackageWithDummyEar(c, "otherApp", nrPackages = 10, packageSize = 5, "otherEar")
    createPackageWithDummyEarAndConfigurationFile(c, "tinyApp", nrPackages = 10, packageSize = 1, "tinyEar")
    createPackageWithDummyEar(c, "smallApp", nrPackages = 100, packageSize = 2, "smallEar")
    createPackageWithDummyEar(c, "largeApp", nrPackages = 100, packageSize = 20, "largeEar")
    createPackageWithDummyEarWithProperties(c, "dictionaryApp", nrPackages = 10, packageSize = 2, "dictionaryEar")
    createPackageWithDummyEar(c, "renameApp", nrPackages = 2, packageSize = 1, "renameEar")
    createPackageWithDummyEar(c, "dragApp", nrPackages = 2, packageSize = 1, "dragEar")
    createPackageWithDummyEar(c, "parallelApp", nrPackages = 3, packageSize = 3, "parallelEar")
    createPackageWithDummyEarWithFailingStep(c, "parallelFailingApp", nrPackages = 3, packageSize = 3, "parallelEar")
    createPackageWithPortal(c, "portalApp", nrPackages = 4, nrPortals = 2, nrPortlets = 4, nrSettings = 2, nrAttributes = 3, "portalApp")
    createPackageWithDependencies(c, appName = "mainApp", version = "1.0", dependencies = Map("dependentAppA" -> "[0.0,0.0]"))
    createPackageWithDependencies(c, appName = "dependentAppA", version = "0.0", dependencies = Map("dependentAppB" -> "[0.0,0.0]"))
    createPackageWithDummyEar(c, "dependentAppB", nrPackages = 1, packageSize = 1, "dependantAppB")

    def createPackageWithPortal(c: DslCreator, appName: String, nrPackages: Int, nrPortals: Int, nrPortlets: Int, nrSettings: Int, nrAttributes: Int, portalName: String): Unit = {
      createPackage(c, appName, nrPackages) { dp =>
        for (j <- 0 until nrPortals) {
          createPortal(c, portalName + j, nrPortlets, nrSettings, nrAttributes, dp)
          ::
        }
        Nil
      }
    }

    def createPortal(c: DslCreator, portalName: String, nrPortlets: Int, nrSettings: Int, nrAttributes: Int, dp: DeploymentPackage): Deployable = {
      c.portal(portalName, dp) { p =>
        p.getPortlets.addAll(createPortlets(c, "portlet", nrPortlets, nrSettings, nrAttributes, p).asJava)
      }
    }

    def createPortlets(c: DslCreator, portletName: String, nrPortlets: Int, nrSettings: Int, nrAttributes: Int, pw: PortalWar) = {
      for (j <- 0 until nrPortlets) yield {
        val portlet: Portlet = c.portlet(portletName + j, pw) {
          p =>
            p.getPortletSettings.addAll(createSettings(c, "setting", nrSettings, p).asJava)
            p.getPortletAttributes.asScala.addAll(createAttributes(c, "attribute", nrAttributes, p))
        }
        portlet.setPortletName("myPortlet" + j)
        portlet
      }
    }

    def createSettings(c: DslCreator, settingsName: String, nrSettings: Int, pl: Portlet) = {
      for (j <- 0 until nrSettings) yield {
        val s: PortletSettings = c.settings(settingsName + j, pl)
        s.setMySetting("myValue" + j)
        s
      }
    }

    def createAttributes(c: DslCreator, attributesName: String, nrAttributes: Int, pl: Portlet) = {
      for (j <- 0 until nrAttributes) yield {
        val s: PortletAttributes = c.attributes(attributesName + j, pl)
        s.setMyAttribute("myAttributeValue" + j)
        s
      }
    }

  }

  private def createUsers(): Unit = {
    user("mary")
    user("john")
    user("tom")
    user("harry")
    user("bob")
    role("senior-deployer", List("mary", "john"))
    role("junior-deployer", List("tom"))
    role("senior-developer", List("harry"))
    role("junior-developer", List("bob"))
    role("administrator", List("bob"))
    val grantSeniorDeployer = grant("senior-deployer") _
    grantSeniorDeployer("login", "")
    grantSeniorDeployer("report#view", "")
    grantSeniorDeployer("repo#edit", "Applications")
    grantSeniorDeployer("repo#edit", "Environments")
    grantSeniorDeployer("repo#edit", "Infrastructure")
    grantSeniorDeployer("import#initial", "Applications")
    grantSeniorDeployer("deploy#initial", "Environments")
    grantSeniorDeployer("read", "Applications")
    grantSeniorDeployer("read", "Environments")
    grantSeniorDeployer("read", "Infrastructure")
    grantSeniorDeployer("read", "Configuration")
    //TODO:  Jeroen how must this be setup for new security system?
    //grantMary("read", "Applications/smallApp")
    //grantMary("read", "Applications/largeApp")
    grantSeniorDeployer("deploy#initial", "Environments")
    val grantJuniorDeployer = grant("junior-deployer") _
    grantJuniorDeployer("login", "")
    grantJuniorDeployer("report#view", "")
    grantJuniorDeployer("read", "Applications")
    grantJuniorDeployer("read", "Environments")
    grantJuniorDeployer("read", "Infrastructure")
    grantJuniorDeployer("discovery", "")
    val grantSeniorDevelopor = grant("senior-developer") _
    grantSeniorDevelopor("login", "")
    grantSeniorDevelopor("repo#edit", "Applications")
    grantSeniorDevelopor("repo#edit", "Environments")
    grantSeniorDevelopor("repo#edit", "Infrastructure")
    grantSeniorDevelopor("import#initial", "Applications")
    grantSeniorDevelopor("deploy#initial", "Environments")
    grantSeniorDevelopor("task#assign", "")
    grantSeniorDevelopor("read", "Applications")
    grantSeniorDevelopor("read", "Environments")
    grantSeniorDevelopor("read", "Infrastructure")
    grantSeniorDevelopor("read", "Configuration")
  }

  private def importPackages(): Unit = {
    importPackage("AnimalZoo-ear/1.0")
    importPackage("AnimalZoo-ear/1.5")
    importPackage("AnimalZoo-ear/2.0")
    importPackage("AnimalZoo-ear/3.0")
    importPackage("AnimalZoo-ear/4.0")
    importPackage("SampleDarWithConfig/1.0")
    importPackage("DragOnEnvApp/1.0")
    importPackage("DragOnEnvApp/2.0")
    importPackage("Person/1.0")
  }

  private def configurePipelines(): Unit = {
    val repository = RepositoryServiceHolder.getRepositoryService

    val animalZoo: ConfigurationItem = repository.read("Applications/AnimalZoo-ear", false)
    val newEnv: Environment = repository.read("Environments/0newEnv0")
    val largeEnv: Environment = repository.read("Environments/0largeEnv0")
    val envs: java.util.List[Environment] = List(newEnv, largeEnv).asJava
    val pipeline: ConfigurationItem = repository.read("Configuration/AnimalZoo-pipeline")

    // Configure package
    pipeline.setProperty("pipeline", envs)
    repository.update(pipeline)

    animalZoo.setProperty("pipeline", pipeline)
    repository.update(animalZoo)

    // Configure Environments
    newEnv.setProperty("requiresReleaseNotes", true)
    newEnv.setProperty("requiresSigningOffManager", true)
    newEnv.setProperty("requiresTicketNumber", true)
    repository.update(newEnv)
    largeEnv.setProperty("requiresDocumentationDoneDev", true)
    repository.update(largeEnv)
  }

  import dsl.TaskState._

  private def generateTasksForReports(): Unit = {
    val repository = RepositoryServiceHolder.getRepositoryService

    val appIds = List(
      "newApp",
      "dictionaryApp",
      "otherApp",
      "oldApp",
      "tinyApp"
    ).map("Applications/" + _)

    val envIds = List(
      "0tinyEnv0",
      "0smallEnv0",
      "0largeEnv0"
    ).map("Environments/" + _)

    val appCis = readCisAsMap(appIds, repository)
    val envCis = readCisAsMap(envIds, repository)

    createCurrentTask(appCis("newApp"), "1.0", envCis("0tinyEnv0"), days = 30, durationInMinutes = 3, SUCCESSFUL, TaskType.INITIAL)
    createCurrentTask(appCis("dictionaryApp"), "1.0", envCis("0tinyEnv0"), days = 30, durationInMinutes = 11, ABORTED, TaskType.INITIAL)
    createCurrentTask(appCis("otherApp"), "1.0", envCis("0smallEnv0"), days = 30, durationInMinutes = 11, FAILED, TaskType.INITIAL)
    createTask(appCis("oldApp"), "1.0", envCis("0tinyEnv0"), month = 2, durationInMinutes = 7, SUCCESSFUL, TaskType.INITIAL)
    createTask(appCis("otherApp"), "1.0", envCis("0tinyEnv0"), month = 3, durationInMinutes = 11, ABORTED, TaskType.INITIAL)
    createTask(appCis("tinyApp"), "1.0", envCis("0smallEnv0"), month = 4, durationInMinutes = 4, FAILED, TaskType.INITIAL)
    createTask(appCis("dictionaryApp"), "1.0", envCis("0smallEnv0"), month = 4, durationInMinutes = 3, SUCCESSFUL, TaskType.INITIAL)
    createTask(appCis("newApp"), "1.0", envCis("0smallEnv0"), month = 5, durationInMinutes = 6, ABORTED, TaskType.INITIAL)
    createTask(appCis("oldApp"), "1.0", envCis("0smallEnv0"), month = 6, durationInMinutes = 17, FAILED, TaskType.INITIAL)
    createTask(appCis("newApp"), "1.0", envCis("0largeEnv0"), month = 6, durationInMinutes = 14, SUCCESSFUL, TaskType.INITIAL)

    createTask(appCis("dictionaryApp"), "1.0", envCis("0smallEnv0"), month = 7, durationInMinutes = 10, SUCCESSFUL, TaskType.INITIAL)
    createTask(appCis("dictionaryApp"), "1.0", envCis("0smallEnv0"), month = 6, durationInMinutes = 20, SUCCESSFUL, TaskType.ROLLBACK)

    createCurrentTask(appCis("newApp"), "1.0", envCis("0largeEnv0"), days = 3, durationInMinutes = 10, SUCCESSFUL, TaskType.INITIAL)
    createCurrentTask(appCis("newApp"), "2.0", envCis("0largeEnv0"), days = 2, durationInMinutes = 20, SUCCESSFUL, TaskType.UPGRADE)
    createCurrentTask(appCis("newApp"), "2.0", envCis("0largeEnv0"), days = 1, durationInMinutes = 30, SUCCESSFUL, TaskType.ROLLBACK)

    createCurrentTask(appCis("oldApp"), "1.0", envCis("0smallEnv0"), days = 1, durationInMinutes = 17, ABORTED, TaskType.INITIAL)

    createTaskWithDependencies(appCis("mainApp"), "1.0", envCis("0hostEnv0"), new DateTime(2014, 9, 10, 9, 30, 10), durationInMinutes = 22, SUCCESSFUL, TaskType.INITIAL, Map("dependentAppB" -> "0.0", "dependentAppA" -> "0.0"))

    createCurrentProvisoningTask("instantiate", "Configuration/CloudEnvTemplate", "Environments/BigCloud", days = 1, durationInMinutes = 17, SUCCESSFUL)
    createCurrentProvisoningTask("destroy", "Configuration/CloudEnvTemplate", "Environments/BigCloud", days = 1, durationInMinutes = 17, SUCCESSFUL)

    createControlTask("someControlTask", appCis("otherApp"), days = 1, durationInMinutes = 10, state = SUCCESSFUL)
    createControlTask("anotherControlTask", appCis("otherApp"), days = 1, durationInMinutes = 10, state = SUCCESSFUL, "control task with description")
  }

  val logger: Logger = LoggerFactory.getLogger("RiaTestSet")
}
