package com.xebialabs.deployit.rest.test.api.datasets

import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.deployit.repository.RepositoryServiceHolder
import com.xebialabs.deployit.rest.test.api.DataSet
import ai.digital.deploy.tasker.common.TaskType
import dsl.TaskState._
import dsl.{CiUtils, EarUtils, TaskUtils}
import org.joda.time.DateTime

object CiExplorerSet extends DataSet with TaskUtils with CiUtils with EarUtils {
  override def name() = "CiExplorer"

  private def generateEnvironments() = create { c =>
    c.directory("Environments/TestDirEnv_Report")
    c.environment("Env1_Report") {
      List.empty
    }
    c.environment("Env2_Report") {
      List.empty
    }
    c.environment("TestDirEnv_Report/Env2_Report") {
      List.empty
    }
  }

  private def generateApplications() = create { c =>
    createEmptyPackage(c, "LongRunningApp_Report", 1)
    createEmptyPackage(c, "TestApp_Report", 1)

    for (i <- 0 to 50) {
      createEmptyPackage(c, s"TestApp${i}_Report", 1)
    }
  }

  def generateDeploymentsForMonth(successful: Int, failed: Int, aborted: Int, rollback: Int, date: DateTime, duration: Int): Unit = {
    val repository = RepositoryServiceHolder.getRepositoryService
    val app = repository.read[BaseConfigurationItem]("/Applications/LongRunningApp_Report")
    val env = repository.read[BaseConfigurationItem]("/Environments/Env1_Report")
    val generateBatch = (amount: Int, taskType: TaskType, state: TaskState) => {
      (0 until amount).foreach { (_) =>
        createTaskWithDependencies(app, "1.0", env, date, duration, state, taskType, Map())
      }
    }
    generateBatch(successful, TaskType.INITIAL, SUCCESSFUL)
    generateBatch(failed, TaskType.INITIAL, FAILED)
    generateBatch(aborted, TaskType.INITIAL, ABORTED)
    generateBatch(rollback, TaskType.ROLLBACK, SUCCESSFUL)
  }

  override def generateSet(args_unused: List[String] = List()): Unit = {
    generateApplications()
    generateEnvironments()
    val repository = RepositoryServiceHolder.getRepositoryService

    val appIds = (List("LongRunningApp_Report", "TestApp_Report") ++ (0 to 50).map(i => s"TestApp${i}_Report")).map("/Applications/" + _)
    val envIds = List("Env1_Report", "Env2_Report", "TestDirEnv_Report/Env2_Report").map("/Environments/" + _)
    val apps = readCisAsMap(appIds, repository)
    val envs = readCisAsMap(envIds, repository)
    createCurrentTask(apps("LongRunningApp_Report"), "1.0", envs("Env1_Report"), 6, 24, SUCCESSFUL, TaskType.INITIAL)
    createCurrentTask(apps("TestApp_Report"), "1.0", envs("Env1_Report"), 4, 36, ABORTED, TaskType.INITIAL)
    createCurrentTask(apps("LongRunningApp_Report"), "1.0", envs("Env1_Report"), 3, 2, SUCCESSFUL, TaskType.UNDEPLOY)

    //Create some historical tasks for sorting
    createTaskWithDependencies(apps("LongRunningApp_Report"), "1.0", envs("Env1_Report"), new DateTime(2010, 1, 1, 0, 0), 5, SUCCESSFUL, TaskType.INITIAL, Map())
    createTaskWithDependencies(apps("LongRunningApp_Report"), "1.0", envs("Env2_Report"), new DateTime(2010, 1, 1, 0, 0), 5, SUCCESSFUL, TaskType.INITIAL, Map())
    createTaskWithDependencies(apps("TestApp_Report"), "1.0", envs("Env1_Report"), new DateTime(2010, 1, 1, 0, 0), 5, SUCCESSFUL, TaskType.INITIAL, Map())
    createTaskWithDependencies(apps("TestApp_Report"), "1.0", envs("Env2_Report"), new DateTime(2010, 1, 1, 0, 0), 5, SUCCESSFUL, TaskType.INITIAL, Map())
    createTaskWithDependencies(apps("LongRunningApp_Report"), "1.0", envs("Env1_Report"), new DateTime(2010, 1, 2, 0, 0), 5, SUCCESSFUL, TaskType.INITIAL, Map())
    createTaskWithDependencies(apps("TestApp_Report"), "1.0", envs("Env1_Report"), new DateTime(2010, 1, 2, 0, 0), 5, SUCCESSFUL, TaskType.INITIAL, Map())
    createTaskWithDependencies(apps("LongRunningApp_Report"), "1.0", envs("Env1_Report"), new DateTime(2010, 1, 2, 0, 0), 5, ABORTED, TaskType.INITIAL, Map())
    createTaskWithDependencies(apps("TestApp_Report"), "1.0", envs("Env1_Report"), new DateTime(2010, 1, 3, 0, 0), 5, SUCCESSFUL, TaskType.INITIAL, Map())
    createTaskWithDependencies(apps("LongRunningApp_Report"), "1.0", envs("Env2_Report"), new DateTime(2010, 1, 3, 0, 0), 5, SUCCESSFUL, TaskType.UNDEPLOY, Map())
    createTaskWithDependencies(apps("TestApp_Report"), "1.0", envs("Env2_Report"), new DateTime(2010, 1, 3, 0, 0), 5, SUCCESSFUL, TaskType.UPGRADE, Map())

    //Create historical tasks for trend statistics
    val startOfThisMonth = new DateTime().withDayOfMonth(1)
    generateDeploymentsForMonth(successful = 20, failed = 3, aborted = 11, rollback = 5, date = startOfThisMonth.minusMonths(1), duration = 200)
    generateDeploymentsForMonth(successful = 3, failed = 2, aborted = 0, rollback = 11, date = startOfThisMonth.minusMonths(2), duration = 60)
    generateDeploymentsForMonth(successful = 31, failed = 4, aborted = 2, rollback = 0, date = startOfThisMonth.minusMonths(3), duration = 220)
    generateDeploymentsForMonth(successful = 50, failed = 20, aborted = 1, rollback = 12, date = startOfThisMonth.minusMonths(4), duration = 150)
    generateDeploymentsForMonth(successful = 10, failed = 8, aborted = 6, rollback = 2, date = startOfThisMonth.minusMonths(5), duration = 300)
    generateDeploymentsForMonth(successful = 18, failed = 12, aborted = 11, rollback = 0, date = startOfThisMonth.minusMonths(6), duration = 250)

    //Create 50 successful deployments not later than 90 days
    val successfulStartDate = new DateTime()
    for (i <- 0 to 50) {
      createTaskWithDependencies(apps(s"TestApp${i}_Report"), "1.0", envs("TestDirEnv_Report/Env2_Report"), successfulStartDate.minusDays(i), 10, SUCCESSFUL, TaskType.INITIAL, Map())
      createTaskWithDependencies(apps(s"TestApp${i}_Report"), "1.0", envs("TestDirEnv_Report/Env2_Report"), successfulStartDate.minusDays(i), 5, ABORTED, TaskType.INITIAL, Map())
      createTaskWithDependencies(apps(s"TestApp${i}_Report"), "1.0", envs("TestDirEnv_Report/Env2_Report"), successfulStartDate.minusDays(i), 5, FAILED, TaskType.ROLLBACK, Map())
      createTaskWithDependencies(apps(s"TestApp${i}_Report"), "1.0", envs("TestDirEnv_Report/Env2_Report"), successfulStartDate.minusDays(i), 5, FAILED, TaskType.INITIAL, Map())
    }

    //Create more then 100 tasks in a specific time range to test pagination
    val startDate = new DateTime(2005, 1, 1, 12, 0)
    for (i <- 0 to 121)
      createTaskWithDependencies(apps("TestApp_Report"), "1.0", envs("Env1_Report"), startDate.plusDays(i), 10, SUCCESSFUL, TaskType.INITIAL, Map())
  }
}
