package com.xebialabs.deployit.rest.test.api {

  import java.io.File
  import java.util
  import java.util.concurrent.CountDownLatch
  import java.util.{List => JavaList, Map => JMap, Set => JSet}

  import ai.digital.deploy.tasker.common.{TaskMetadata, TaskType}
  import com.xebialabs.deployit.ServerConfiguration
  import com.xebialabs.deployit.deployment.planner.StepPlan.Checkpoint
  import com.xebialabs.deployit.deployment.planner._
  import com.xebialabs.deployit.engine.api.dto.Deployment
  import com.xebialabs.deployit.engine.api.execution._
  import com.xebialabs.deployit.engine.spi.execution.{ExecutionStateListener, StepExecutionStateEvent, TaskExecutionStateEvent}
  import com.xebialabs.deployit.engine.tasker.Phase.PhaseBuilder
  import com.xebialabs.deployit.engine.tasker._
  import com.xebialabs.deployit.plugin.TestServiceHolder
  import com.xebialabs.deployit.plugin.api.deployment.specification.Operation
  import com.xebialabs.deployit.plugin.api.udm._
  import com.xebialabs.deployit.repository._
  import com.xebialabs.deployit.security.permission.Permission
  import com.xebialabs.deployit.security.{PermissionEditor, Role, RoleService}
  import com.xebialabs.deployit.service.deployment._
  import com.xebialabs.deployit.service.deployment.setter.DefaultPropertySetter
  import com.xebialabs.deployit.service.importer.source.FileSource
  import com.xebialabs.deployit.service.replacement.Dictionaries
  import com.xebialabs.deployit.task.archive.ArchivedTask
  import com.xebialabs.deployit.test.deployment.{DeltaSpecifications, DeployitTester}
  import com.xebialabs.xlplatform.satellite.Satellite
  import dsl.Creator
  import org.joda.time.DateTime
  import org.springframework.security.authentication.UsernamePasswordAuthenticationToken

  import scala.collection.mutable
  import scala.collection.mutable.ListBuffer
  import scala.jdk.CollectionConverters._
  import scala.language.implicitConversions

  trait Generator {

    lazy val tester = new DeployitTester()

    lazy val workDir = new WorkDir(new File("build/work"), "data")

    def shutdown(): Unit = tester.shutdown()

    def user(name: String): Unit = TestServiceHolder.getUserService.create(name, name)

    def role(name: String, principals: List[String]): Unit = {
      val service: RoleService = TestServiceHolder.getRoleService
      val assignments: JavaList[Role] = service.readRoleAssignments(null, null, null)
      val role = new Role(name)
      role.getPrincipals.asScala.addAll(principals)
      assignments.add(role)
      service.writeRoleAssignments(assignments)
    }

    def create(c: Creator => Unit): Unit = {
      val changeSet: ChangeSet = new ChangeSet
      val creator = new Creator(changeSet)
      c.apply(creator)
      RepositoryServiceHolder.getRepositoryService.execute(changeSet)
      creator.cleanUp()
    }

    private def read[T](id: String, workDir: WorkDir): T = {
      RepositoryServiceHolder.getRepositoryService.read(id, workDir)
    }

    def grant(roleName: String)(perm: String, onCi: String = ""): Unit = {
      val editor: PermissionEditor = TestServiceHolder.getPermissionEditor
      val permissions: JMap[Role, JSet[Permission]] = editor.readPermissions(onCi)
      val role = TestServiceHolder.getRoleService.getRoleForRoleName(roleName)
      if (!permissions.containsKey(role)) permissions.put(role, new util.HashSet[Permission]())
      val rolePermissions: JSet[Permission] = permissions.get(role)
      rolePermissions.add(Permission.find(perm))
      editor.editPermissions(onCi, permissions)
    }

    def task(c: ArchivedTask => Unit): Unit = {
      val taskForReporting: ArchivedTask = new ArchivedTask()
      taskForReporting.getMetadata.put(TaskMetadata.TASK_TYPE, TaskType.INITIAL.name())
      taskForReporting.setOwner("admin")
      c.apply(taskForReporting)
      TestServiceHolder.getTaskArchiveStore.archive(taskForReporting)
    }

    def deploy(dpId: String, envId: String): String = {
      val workDir = TestServiceHolder.getWorkDirFactory.newWorkDir()
      val dp: DeploymentPackage = read(dpId, workDir)
      val env: Environment = read(envId, workDir)
      val deployeds = generateDeployeds(dp, env)
      val specification = createSpecification(dp, env, deployeds)

      val resolve: Plan = tester.resolve(specification)

      def toPhaseContainer(plan: Plan): PhaseContainer = plan match {
        case executable: ExecutablePlan => toExecutableBlock(executable).build()
        case planPhase: PlanPhase => BlockBuilders.phases(planPhase.getDescription, toPhase(planPhase)).build()
        case phasedPlan: PhasedPlan => BlockBuilders.phases(phasedPlan.getDescription, phasedPlan.phases.asScala.map(toPhase).toList).build()
      }

      def toPhase(planPhase: PlanPhase): PhaseBuilder = planPhase match {
        case planPhase: PlanPhase => BlockBuilders.phase(planPhase.getDescription, planPhase.getDescription, toExecutableBlock(planPhase.plan))

      }

      def toExecutableBlock(plan: ExecutablePlan): ExecutableBlockBuilder = plan match {
        case p: ParallelPlan => BlockBuilders.parallel(p.getDescription, Option.empty[Satellite], p.getSubPlans.asScala.map(toExecutableBlock).toList)
        case p: SerialPlan => BlockBuilders.serial(p.getDescription, Option.empty[Satellite], p.getSubPlans.asScala.map(toExecutableBlock).toList)
        case p: StepPlan => BlockBuilders.steps(p.getDescription, p.satellite, p.getStepsWithPlanningInfo.asScala.map[StepState](s => {
          val step = new TaskStep(s.getStep)
          var i: Int = 0
          s.getDeltas.asScala.foreach(d => {
            step.getMetadata.put("deployed_%d".format(i), (if (d.getOperation == Operation.DESTROY) d.getPrevious else d.getDeployed).getId)
            i += 1
          })
          step
        }).toList.asJava)
      }

      val phaseContainer = toPhaseContainer(resolve)

      val latch = new CountDownLatch(1)

      val specification1 = new TaskSpecification("deployment", new UsernamePasswordAuthenticationToken("admin", "s3cret"), workDir, phaseContainer)
      specification1.getMetadata.put(TaskMetadata.ENVIRONMENT, env.getName)
      specification1.getMetadata.put(TaskMetadata.ENVIRONMENT_ID, env.getId)
      specification1.getMetadata.put(TaskMetadata.APPLICATION, dp.getApplication.getName)
      specification1.getMetadata.put(TaskMetadata.VERSION, dp.getName)
      specification1.getMetadata.put(TaskMetadata.TASK_TYPE, TaskType.INITIAL.name())
      specification1.getListeners.add(new ExecutionStateListener {
        def taskStateChanged(event: TaskExecutionStateEvent): Unit = {
          if (event.currentState() == TaskExecutionState.EXECUTED || event.currentState() == TaskExecutionState.STOPPED) {
            latch.countDown()
          }
        }

        def stepStateChanged(event: StepExecutionStateEvent): Unit = {}
      })

      val multiDeltaSpecification: MultiDeltaSpecification = new MultiDeltaSpecification(specification)
      specification1.getListeners.add(new ReferentialIntegrityTrigger(multiDeltaSpecification))
      specification1.getListeners.add(new CheckPointManagerListener(multiDeltaSpecification, new util.ArrayList[Checkpoint]()))
      val taskId = TestServiceHolder.getExecutionEngine.register(specification1)

      TestServiceHolder.getExecutionEngine.execute(taskId)
      latch.await()
      if (TestServiceHolder.getExecutionEngine.retrieveTaskSummary(taskId).getState == TaskExecutionState.EXECUTED) {
        TestServiceHolder.getExecutionEngine.archive(taskId)
      }

      taskId
    }

    private def generateDeployeds(dp: DeploymentPackage, env: Environment): Set[Deployed[_ <: Deployable, _ <: Container]] = {
      val resolver: DeployedArtifactPlaceholdersResolver = new DeployedArtifactPlaceholdersResolver()
      ServerConfiguration.setInstance(new ServerConfiguration)
      val defaultSetter = new DefaultPropertySetter(RepositoryServiceHolder.getRepositoryService)
      val deployedPropertySetter = new DeployedPropertySetter(defaultSetter)
      val deployedProcessorFactory = new DeployedProcessorsFactory(resolver, deployedPropertySetter)
      val dictionaries = Dictionaries.of(env)
      val generatedDeployed: GeneratedDeployeds = new GeneratedDeployeds()
      val deployedProcessor = deployedProcessorFactory.createCoreDeployedGenerator()
      val deployment = new Deployment()
      deployment.setDeployedApplication(new DeployedApplication())
      val deplContext = deployedProcessorFactory.createContextWithCalculatedType(deployment, dictionaries)
      for (
        deployable <- dp.getDeployables.asScala;
        container <- env.getMembers.asScala
      ) {
        generatedDeployed.merge(deployedProcessor.generateDeployed(deplContext, deployable, container))
      }
      generatedDeployed.getDeployeds.asScala.toSet
    }

    private def createSpecification(dp: DeploymentPackage, env: Environment, set: Set[Deployed[_ <: Deployable, _ <: Container]]) = {
      val deployedApp = DeltaSpecifications.createDeployedApplication(dp, env)
      val builder: DeltaSpecificationBuilder = DeltaSpecificationBuilder.newSpecification.initial(deployedApp)
      DeploymentOperationCalculator.calculate(builder, mutable.Set.empty[Deployed[_ <: Deployable, _ <: Container]].asJava, set.asJava)
      builder.build()
    }

    def importPackage(id: String): Unit = {
      val dir: File = TestServiceHolder.getImporterService.getImportablePackageDirectory
      TestServiceHolder.getImporterService.importPackage(new FileSource(new File(dir, id).getPath, false))
    }


    implicit def task2dsl(task: ArchivedTask): TaskDsl = new TaskDsl(task)

    class TaskDsl(task: ArchivedTask) {
      def withId(id: String): ArchivedTask = {
        task.setId(id)
        task
      }

      def completedAt(date: DateTime): ArchivedTask = {
        task.setCompletionDate(date)
        task
      }

      def startedAt(date: DateTime): ArchivedTask = {
        task.setStartDate(date)
        task
      }

      def forEnv(env: String, internalId: Int): ArchivedTask = {
        task.getMetadata.put(TaskMetadata.ENVIRONMENT, env)
        task.getMetadata.put(TaskMetadata.ENVIRONMENT_ID, s"Environments/$env")
        task.getMetadata.put(TaskMetadata.ENVIRONMENT_INTERNAL_ID, internalId.toString)
        task.getMetadata.put(TaskMetadata.ENVIRONMENT_SECURED_CI, "0")
        task
      }

      def forApp(app: String, internalId: Int): ArchivedTask = {
        task.getMetadata.put(TaskMetadata.APPLICATION, app)
        task.getMetadata.put(TaskMetadata.APPLICATION_INTERNAL_ID, internalId.toString)
        task.getMetadata.put(TaskMetadata.APPLICATION_SECURED_CI, "0")
        task
      }

      def forVersion(version: String): ArchivedTask = {
        task.getMetadata.put(TaskMetadata.VERSION, version)
        task
      }

      def forDeploymentType(taskType: TaskType): ArchivedTask = {
        task.getMetadata.put(TaskMetadata.TASK_TYPE, taskType.toString)
        task
      }

      def withDependency(app: String, version: String): ArchivedTask = {
        task.getPackageDependencies.add(new TaskPackageDependency(app, version))
        task
      }

      def forCloudOperation(operation: String): ArchivedTask = {
        task.getMetadata.put(TaskMetadata.CLOUD_OPERATION, operation)
        task
      }

      def forCloudEnv(environmentId: String): ArchivedTask = {
        task.getMetadata.put(TaskMetadata.CLOUD_ENVIRONMENT_ID, environmentId)
        task
      }

      def withCloudEnvTemplate(cloudEnvironmentTemplateId: String): ArchivedTask = {
        task.getMetadata.put(TaskMetadata.CLOUD_ENVIRONMENT_TEMPLATE_ID, cloudEnvironmentTemplateId)
        task
      }

      def success: ArchivedTask = {
        task.setState(TaskExecutionState.DONE)
        task
      }

      def successWithFails: ArchivedTask = {
        task.setState(TaskExecutionState.DONE)
        task.setFailureCount(1)
        task
      }

      def failed: ArchivedTask = {
        task.setState(TaskExecutionState.CANCELLED)
        task.setFailureCount(1)
        task
      }

      def withDefaultBlock: ArchivedTask = {
        val block = BlockBuilders.steps("Empty block", null, ListBuffer.empty[StepState].asJava).build()
        block.newState(BlockExecutionState.DONE)
        task.setBlock(block)
        task
      }

      def withBlocks(depths: List[Int], stepsPerLeafBlock: Int, description: String = "top level block"): ArchivedTask = {
        val block = blockBuilder(depths, stepsPerLeafBlock, description).build()
        block.newState(BlockExecutionState.DONE)
        task.setBlock(block)
        task
      }

      @SuppressWarnings(Array("UnsafeTraversableMethods"))
      def blockBuilder(depths: List[Int], stepsPerLeafBlock: Int, description: String): ExecutableBlockBuilder = {
        if (depths.isEmpty)
          BlockBuilders.steps(s"Leaf block with $stepsPerLeafBlock steps", None, stepBlock(stepsPerLeafBlock))
        else {
          val builders = for (i <- 1 to depths.head) yield
            blockBuilder(depths.tail, stepsPerLeafBlock, s"parallel block, level ${depths.length}, #$i")
          BlockBuilders.parallel(description, null, builders.asJava)
        }
      }

      private def stepBlock(numSteps: Int): List[StepState] = (for (i <- 1 to numSteps) yield {
        val timestamp = new DateTime().getMillis
        new StepState {
          override def getStartDate: DateTime = task.getStartDate

          override def getMetadata: JMap[StepId, StepId] = Map[String, String]().asJava

          override def getLog: String = timestamp + loremIpsum2k

          override def getPreviousAttemptsLogs: JavaList[String] = (1 to i).map(n => s"failure $n, argh! :-(").asJava

          override def getCompletionDate: DateTime = task.getStartDate.plus(1000L * 60 * 60)

          override def getDescription = s"Step $i of this marvellous step block, yay!"

          override def getState: StepExecutionState = StepExecutionState.DONE

          override def getFailureCount: Int = i

          override def getSkippable: Boolean = true

        }
      }).toList

      def withDescription(description: String): ArchivedTask = {
        task.setDescription(description)
        task
      }

      def addMetadata(kvs: (String, String)*): ArchivedTask = {
        val metadata: util.Map[String, String] = kvs.toMap.asJava
        task.getMetadata.putAll(metadata)
        task
      }
    }

    val loremIpsum2k: String =
      """Lorem ipsum dolor sit amet, consectetur adipiscing elit. Phasellus et sem ipsum. Integer sagittis odio in magna elementum facilisis. Quisque lectus velit, laoreet vitae tincidunt non, mattis vel elit. Aenean ac lacus eu neque sagittis rutrum eget ut enim. Vestibulum quam nulla, blandit eu eros eget, hendrerit finibus ipsum. Duis sollicitudin ultricies odio. Praesent finibus malesuada tellus eget congue. Phasellus posuere risus non suscipit elementum. Aenean eu turpis tincidunt, vestibulum risus a, faucibus metus. Etiam at massa in sapien pretium sodales. Proin pellentesque sagittis odio maximus egestas. Pellentesque eget pellentesque neque. Sed facilisis tincidunt felis eu consequat. Ut nec elementum felis, non iaculis dolor. Suspendisse consectetur consectetur aliquet.

Nam sed luctus leo. Phasellus et faucibus lacus. Nullam nec feugiat ante. Nunc vitae ultricies augue, sit amet congue tortor. Vestibulum vitae tristique orci, quis pellentesque justo. Etiam fermentum orci augue, sit amet elementum libero dapibus ut. Nulla finibus accumsan mi in accumsan. Vestibulum interdum massa sit amet ipsum laoreet consectetur. Curabitur tempus lacus at massa placerat, eu hendrerit diam tincidunt. Praesent eu finibus arcu.

Sed sit amet quam vel arcu ultrices tempus. Cras euismod nisl et lectus feugiat porttitor. Duis aliquet ut orci nec consectetur. Nam suscipit, augue vitae convallis euismod, neque felis ullamcorper odio, et tempor mauris diam sit amet dolor. Quisque eu placerat felis. Vestibulum at tortor et justo facilisis sollicitudin. Sed et facilisis velit. Donec at lectus vel mauris mollis fermentum sed in libero. Mauris maximus sapien eu mauris fermentum, eu convallis tortor mollis. Nulla non sem a metus tincidunt ultrices. Integer porta lacus in vehicula vestibulum. Cras eget tincidunt ex.
"""
  }

}


