/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.api.reflect;

import java.util.*;
import java.util.concurrent.Callable;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import static java.lang.String.format;

public class DescriptorRegistry {
    private static ReentrantReadWriteLock readWriteLock = new ReentrantReadWriteLock();

    private static final Map<DescriptorRegistryId, IDescriptorRegistry> REGISTRIES = new LinkedHashMap<>();

    protected static IDescriptorRegistry getInstance() {
        return getDefaultDescriptorRegistry();
    }

    private static <T> T withReadLock(Callable<T> callable) {
        readWriteLock.readLock().lock();
        try {
            return callable.call();
        } catch (RuntimeException e) {
            throw e;
        } catch (Exception e) {
            throw new IllegalStateException("Unexpected exception", e);
        } finally {
            readWriteLock.readLock().unlock();
        }
    }

    public static <T> T withWriteLock(Callable<T> callable) {
        readWriteLock.writeLock().lock();
        try {
            return callable.call();
        } catch (RuntimeException e) {
            throw e;
        } catch (Exception e) {
            throw new IllegalStateException("Unexpected exception", e);
        } finally {
            readWriteLock.writeLock().unlock();
        }
    }

    /**
     * Registers a DescriptorRegistry to a global map.
     */
    public static void add(IDescriptorRegistry registry) {
        withWriteLock(() -> {
            final DescriptorRegistryId registryId = registry.getId();
            if (REGISTRIES.containsKey(registry.getId())) {
                throw new IllegalStateException(format("There is a DescriptorRegistry booted with id [%s]", registryId));
            }
            if (registry.isLocal()) {
                DescriptorRegistryId otherLocal = lookupLocalRegistry();
                if (otherLocal != null) {
                    throw new IllegalStateException(format("Tried to load another local DescriptorRegistry under [%s], already present under [%s]", registryId, otherLocal));
                }
            }
            REGISTRIES.put(registry.getId(), registry);
            return null;
        });
    }

    /**
     * Unregisters the descriptor registry with the specified id.
     */
    public static void remove(DescriptorRegistryId id) {
        withWriteLock(() -> {
            REGISTRIES.remove(id);
            return null;
        });
    }

    public static void replaceRegistry(IDescriptorRegistry registry) {
        withWriteLock(() -> {
            DescriptorRegistryId registryId = registry.getId();
            if (registry.isLocal()) {
                DescriptorRegistryId otherLocal = lookupLocalRegistry();
                if (otherLocal != null) {
                    throw new IllegalStateException(format("Tried to load another local DescriptorRegistry under [%s], already present under [%s]", registryId, otherLocal));
                }
            }
            REGISTRIES.put(registryId, registry);
            return null;
        });
    }


    private static DescriptorRegistryId lookupLocalRegistry() {
        return withReadLock(() -> {
            for (IDescriptorRegistry descriptorRegistry : REGISTRIES.values()) {
                if (descriptorRegistry.isLocal()) {
                    return descriptorRegistry.getId();
                }
            }
            return null;
        });
    }

    /**
     * Gets a DescriptorRegistry by id.
     */
    public static IDescriptorRegistry getDescriptorRegistry(DescriptorRegistryId id) {
        return withReadLock(() -> REGISTRIES.get(id));
    }

    /**
     * @return The default registered DescriptorRegistry (the LOCAL registry if one is present, otherwise the first one).
     */
    static IDescriptorRegistry getDefaultDescriptorRegistry() {
        return withReadLock(() -> {
            if (REGISTRIES.keySet().isEmpty()) {
                throw new IllegalStateException("No DescriptorRegistries have been loaded.");
            }

            // from list of registries take default ones and sort them by priority, and then take the 1st
            IDescriptorRegistry defaultRegistry = REGISTRIES.values().stream()
                    .filter(IDescriptorRegistry::isDefault).min(Comparator.comparingInt(IDescriptorRegistry::getOrder))
                    .orElseGet(() -> REGISTRIES.values().iterator().next());

            return defaultRegistry;
        });
    }

    static IDescriptorRegistry getDescriptorRegistry(Type type) {
        return withReadLock(() -> {
            if (type.getTypeSource() == null) {
                throw new NullPointerException(format("The type [%s] is not registered with any DescriptorRegistry", type));
            }
            return REGISTRIES.get(type.getTypeSource());
        });
    }

    /**
     * @param type The type of a CI.
     * @return The descriptor of the specified CI type.
     */
    public static Descriptor getDescriptor(Type type) {
        return withReadLock(() -> getDescriptorRegistry(type)._getDescriptor(type));
    }

    /**
     * @param prefixedName The prefixed type name of a CI.
     * @return The descriptor of the specified CI type.
     */
    public static Descriptor getDescriptor(String prefixedName) {
        return getDescriptor(Type.valueOf(prefixedName));
    }

    /**
     * @param prefix The prefix of a CI.
     * @param name   The simple name of a CI.
     * @return The descriptor of the specified CI type.
     */
    public static Descriptor getDescriptor(String prefix, String name) {
        return withReadLock(() -> getDescriptor(Type.valueOf(prefix, name)));
    }

    /**
     * @return A collection of all the registered type descriptors.
     */
    public static Collection<Descriptor> getDescriptors() {
        return withReadLock(() -> getDefaultDescriptorRegistry()._getDescriptors());
    }

    public static Collection<Descriptor> getDescriptors(DescriptorRegistryId registryId) {
        return withReadLock(() -> {
            final IDescriptorRegistry descriptorRegistry = REGISTRIES.get(registryId);
            if (null == descriptorRegistry) {
                final String msg = String.format("No DescriptorRegistry with key %s has been loaded.", registryId.toString());
                throw new IllegalStateException(msg);
            }
            return descriptorRegistry._getDescriptors();
        });
    }

    /**
     * @param type The type of a CI.
     * @return Whether the given type can be found in the registry.
     */
    public static boolean exists(Type type) {
        return withReadLock(() -> getDescriptorRegistry(type)._exists(type));
    }

    /**
     * @param supertype The super type to search against.
     * @return A collection of subtypes of the given super type.
     */
    public static Collection<Type> getSubtypes(Type supertype) {
        return withReadLock(() -> Collections.unmodifiableCollection(getDescriptorRegistry(supertype)._getSubtypes(supertype)));
    }

}
