/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.api.reflect;

import com.xebialabs.deployit.plugin.api.udm.Prefix;
import com.xebialabs.xlplatform.documentation.PublicApiRef;

import java.util.*;

import static java.lang.String.format;

/**
 * A registry for the {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem}s the
 * XL Deploy system should know about. Every CI registry must extend this class, and it can be added to the collection
 * of known registries, which is kept in this class.
 *
 * One special descriptor registry is the LocalDescriptorRegistry, which is usually the default one. Another descriptor
 * registry is the RemoteDescriptorRegistry.
 */
@PublicApiRef
public abstract class DescriptorRegistry {
    private static final Map<DescriptorRegistryId, DescriptorRegistry> REGISTRIES = new LinkedHashMap<>();

    private final Map<String, Type> NAME_TYPE_CACHE = new HashMap<>();

    private final Map<Class<?>, Type> CLASS_TYPE_CACHE = new HashMap<>();

    private final DescriptorRegistryId id;

    protected DescriptorRegistry(DescriptorRegistryId id) {
        if (id == null) {
            throw new NullPointerException("The id of a DescriptorRegistry cannot be null.");
        }
        this.id = id;
    }

    /**
     * Registers a DescriptorRegistry to a global map.
     */
    public static void add(DescriptorRegistry registry) {
        if (REGISTRIES.containsKey(registry.id)) {
            throw new IllegalStateException(format("There is a DescriptorRegistry booted with id [%s]", registry.id));
        }
        if (registry.isLocal()) {
            DescriptorRegistryId otherLocal = lookupLocalRegistry();
            if (otherLocal != null) {
                throw new IllegalStateException(format("Tried to load another local DescriptorRegistry under [%s], already present under [%s]", registry.id, otherLocal));
            }
        }
        REGISTRIES.put(registry.id, registry);
    }

    private static DescriptorRegistryId lookupLocalRegistry() {
        for (DescriptorRegistry descriptorRegistry : REGISTRIES.values()) {
            if (descriptorRegistry.isLocal()) {
                return descriptorRegistry.id;
            }
        }
        return null;
    }

    /**
     * Unregisters the descriptor registry with the specified id.
     */
    public static void remove(DescriptorRegistryId id) {
        REGISTRIES.remove(id);
    }

    /**
     * @return A collection of all the registered type descriptors.
     */
    public static Collection<Descriptor> getDescriptors() {
        return getDefaultDescriptorRegistry()._getDescriptors();
    }

    /**
     * @return The default registered DescriptorRegistry (the LOCAL registry if one is present, otherwise the first one).
     */
    static DescriptorRegistry getDefaultDescriptorRegistry() {
        if (REGISTRIES.keySet().isEmpty()) {
            throw new IllegalStateException("No DescriptorRegistries have been loaded.");
        }

        // Prefer the local registry
        DescriptorRegistryId defaultRegistryId = lookupLocalRegistry();
        if (defaultRegistryId != null) {
            return getDescriptorRegistry(defaultRegistryId);
        }

        // Get the first one.
        return REGISTRIES.values().iterator().next();
    }

    /**
     * @param supertype The super type to search against.
     * @return A collection of subtypes of the given super type.
     */
    public static Collection<Type> getSubtypes(Type supertype) {
        return Collections.unmodifiableCollection(getDescriptorRegistry(supertype)._getSubtypes(supertype));
    }

    /**
     * @param prefixedName The prefixed type name of a CI.
     * @return The descriptor of the specified CI type.
     */
    public static Descriptor getDescriptor(String prefixedName) {
        return getDescriptor(Type.valueOf(prefixedName));
    }

    /**
     * @param prefix The prefix of a CI.
     * @param name The simple name of a CI.
     * @return The descriptor of the specified CI type.
     */
    public static Descriptor getDescriptor(String prefix, String name) {
        return getDescriptor(Type.valueOf(prefix, name));
    }

    /**
     * @param type The type of a CI.
     * @return The descriptor of the specified CI type.
     */
    public static Descriptor getDescriptor(Type type) {
        return getDescriptorRegistry(type)._getDescriptor(type);
    }

    private static DescriptorRegistry getDescriptorRegistry(Type type) {
        if (type.getTypeSource() == null) {
            throw new NullPointerException(format("The type [%s] is not registered with any DescriptorRegistry", type));
        }
        return REGISTRIES.get(type.getTypeSource());
    }

    /**
     * @param type The type of a CI.
     * @return Whether the given type can be found in the registry.
     */
    public static boolean exists(Type type) {
        return getDescriptorRegistry(type)._exists(type);
    }

    protected static DescriptorRegistry getInstance() {
        return getDefaultDescriptorRegistry();
    }

    /**
     * Gets a DescriptorRegistry by id.
     */
    public static DescriptorRegistry getDescriptorRegistry(DescriptorRegistryId id) {
        return REGISTRIES.get(id);
    }

    /**
     * Searches in the registry for a CI type by CI class.
     *
     * @param ciClass A class instance of a CI.
     * @return The type for the given CI class.
     */
    public Type lookupType(Class<?> ciClass) {
        if (ciClass == null) {
            throw new NullPointerException("Type name may not be null");
        }

        if (CLASS_TYPE_CACHE.containsKey(ciClass)) {
            return CLASS_TYPE_CACHE.get(ciClass);
        }

        Package ciPackage = ciClass.getPackage();
        Prefix prefix = ciPackage.getAnnotation(Prefix.class);
        if (prefix == null) {
            throw new NullPointerException(format("Package [%s] should have an @Prefix annotation for ci-class [%s]", ciPackage.getName(), ciClass.getName()));
        }
        String simpleName = ciClass.getSimpleName();
        if (simpleName.isEmpty()) {
            throw new IllegalArgumentException(format("Could not get a typename for ci-class [%s]", ciClass.getName()));
        }
        Type type = lookupType(prefix.value(), simpleName);
        CLASS_TYPE_CACHE.put(ciClass, type);
        return type;
    }

    /**
     * Searches in the registry for a CI type by CI prefixed type name.
     *
     * @param typeName The prefixed type name of a CI.
     * @return The type of the CI.
     */
    public Type lookupType(String typeName) {
        if (typeName == null) {
            throw new NullPointerException("Type name may not be null");
        }

        if (NAME_TYPE_CACHE.containsKey(typeName)) {
            return NAME_TYPE_CACHE.get(typeName);
        }

        if (typeName.indexOf('.') == -1) {
            throw new IllegalArgumentException(format("Type %s does not contain a prefix", typeName));
        }
        int indexOfLastDot = typeName.lastIndexOf('.');
        Type t = lookupType(typeName.substring(0, indexOfLastDot), typeName.substring(indexOfLastDot + 1));
        NAME_TYPE_CACHE.put(typeName, t);
        return t;
    }

    /**
     * Searches in the registry for a CI type by CI prefix and simple name.
     *
     * @param prefix The prefix of a CI.
     * @param simpleName The simple name of a CI.
     * @return The type of the CI.
     */
    public Type lookupType(String prefix, String simpleName) {
        return new Type(prefix, simpleName, id);
    }

    protected abstract boolean isLocal();

    protected abstract Collection<Descriptor> _getDescriptors();

    protected abstract Collection<Type> _getSubtypes(Type supertype);

    protected abstract Descriptor _getDescriptor(Type type);

    protected abstract boolean _exists(Type type);
}
