/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.api.semver;

import com.github.zafarkhaja.semver.util.Stream;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class Lexer {

    static class Token {

        enum Type implements Stream.ElementType<Token> {

            NUMERIC("0|[1-9][0-9]*"),
            HYPHEN("-[0-9A-Za-z]*"),
            DOT("\\."),
            WHITESPACE("\\s+"),
            EOI("?!"),
            LEFT_CLOSED("\\["),
            RIGHT_CLOSED("\\]"),
            LEFT_OPEN("\\("),
            RIGHT_OPEN("\\)"),
            COMMA("\\,");


            final Pattern pattern;

            private Type(String regexp) {
                pattern = Pattern.compile("^(" + regexp + ")");
            }

            @Override
            public String toString() {
                return name() + "(" + pattern + ")";
            }

            @Override
            public boolean isMatchedBy(Token token) {
                return token != null && this == token.type;
            }
        }

        final Type type;

        final String lexeme;

        final int position;

        public Token(Type type, String lexeme, int position) {
            this.type = type;
            this.lexeme = (lexeme == null) ? "" : lexeme;
            this.position = position;
        }

        @Override
        public String toString() {
            return String.format(
                    "%s(%s) at position %d",
                    type.name(),
                    lexeme, position
            );
        }

        public Type getType() {
            return type;
        }

        public String getLexeme() {
            return lexeme;
        }

        public int getPosition() {
            return position;
        }
    }


    public Lexer() {

    }

    public Stream<Token> tokenize(String input) {
        String origInput = input;
        List<Token> tokens = new ArrayList<>();
        int tokenPos = 0;
        while (!input.isEmpty()) {
            boolean matched = false;
            for (Token.Type tokenType : Token.Type.values()) {
                Matcher matcher = tokenType.pattern.matcher(input);
                if (matcher.find()) {
                    matched = true;
                    input = matcher.replaceFirst("");
                    if (tokenType != Token.Type.WHITESPACE) {
                        Token token = new Token(
                                tokenType,
                                matcher.group(),
                                tokenPos
                        );
                        tokens.add(token);
                    }
                    tokenPos += matcher.end();
                    break;
                }
            }
            if (!matched) {
                throw new IllegalArgumentException(String.format("Invalid range: '%s'.", origInput));
            }
        }
        tokens.add(new Token(Token.Type.EOI, null, tokenPos));
        return new Stream<>(tokens.toArray(new Token[tokens.size()]));
    }
}

