/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.api.validation;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;

/**
 * The annotated element must not contain the specified values.
 * <p>
 * Supported types are:
 * <ul>
 *      <li>{@code STRING}</li>
 *      <li>{@code SET_OF_STRING}</li>
 *      <li>{@code LIST_OF_STRING}</li>
 *      <li>{@code MAP_STRING_STRING}</li>
 * </ul>
 * </p>
 * {@code null} elements are considered valid.
 */
@Retention(RetentionPolicy.RUNTIME)
@Rule(clazz = NotContain.Validator.class, type = "not-contain")
@ApplicableTo({PropertyKind.STRING, PropertyKind.SET_OF_STRING, PropertyKind.LIST_OF_STRING, PropertyKind.MAP_STRING_STRING})
@Target(ElementType.FIELD)
public @interface NotContain {
    String DEFAULT_MESSAGE = "The property must not contain %s.";

    String message() default DEFAULT_MESSAGE;

    String[] value() default {};

    class Validator implements com.xebialabs.deployit.plugin.api.validation.Validator<Object> {
        private String message = NotContain.DEFAULT_MESSAGE;

        private String[] value = {};

        @Override
        public void validate(final Object object, final ValidationContext context) {
            List<String> values = Arrays.asList(value);
            String logFriendlyValues = String.join(", ", values).replaceFirst(",(?!.*,)", " and");
            if (object != null) {
                if (object instanceof String) {
                    String s = (String) object;
                    if (values.stream().anyMatch(s::contains)) {
                        context.error(message, logFriendlyValues);
                    }
                } else if (object instanceof Collection) {
                    if (containsAny(((Collection<String>) object), values)) {
                        context.error(message, logFriendlyValues);
                    }
                } else if (object instanceof Map) {
                    if (containsAny(((Map<String, String>) object).keySet(), values)) {
                        context.error(message, logFriendlyValues);
                    }
                }
            }
        }

        private boolean containsAny(Collection<String> c1, Collection<String> c2) {
            return c1.stream().anyMatch(c2::contains);
        }

        public String getMessage() {
            return message;
        }

        public String[] getValue() {
            return value;
        }
    }
}
