/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.overthere;

import com.xebialabs.deployit.plugin.api.flow.StagingTarget;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.TypeIcon;
import com.xebialabs.deployit.plugin.api.udm.base.BaseContainer;
import com.xebialabs.deployit.plugin.credentials.HostCredentials;
import com.xebialabs.overthere.ConnectionOptions;
import com.xebialabs.overthere.OperatingSystemFamily;
import com.xebialabs.overthere.Overthere;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.xlplatform.satellite.Satellite;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;

import static com.xebialabs.overthere.ConnectionOptions.JUMPSTATION;
import static com.xebialabs.overthere.ConnectionOptions.TEMPORARY_DIRECTORY_PATH;
import static com.xebialabs.overthere.cifs.CifsConnectionBuilder.WINRS_PROXY_CONNECTION_OPTIONS;
import static com.xebialabs.overthere.cifs.CifsConnectionBuilder.WINRS_PROXY_PROTOCOL;
import static com.xebialabs.xlplatform.utils.Strings.mkString;

/**
 * A machine that runs middleware.
 */
@SuppressWarnings("serial")
@Metadata(root = Metadata.ConfigurationItemRoot.INFRASTRUCTURE, virtual = true, description = "Machine that runs middleware, on which scripts can be executed, etc.")
@TypeIcon("icons/types/overthere.Host.svg")
public class Host extends BaseContainer implements HostContainer, StagingTarget {

    @Property(description = "Protocol to use when connecting to this host", hidden = true)
    private String protocol;

    @Property(label = "Operating system", description = "Operating system the host runs")
    private OperatingSystemFamily os;

    @Property(label = "Executable shell", description = "Executable shell program", required = false, hidden = true)
    private String shell;

    @Property(category = "Advanced", description = "Directory into which temporary files are stored. Will be cleaned up when the connection is closed.", required = false)
    private String temporaryDirectoryPath;

    @Property(category = "Advanced", description = "Directory into which staged files are stored. Will be cleaned up when the task is finished.", required = false)
    private String stagingDirectoryPath;

    @Property(label = "Satellite or Satellite Group", category = "Advanced", description = "Instance of xl-satellite that can manage this host.", required = false)
    private Satellite satellite;

    public OverthereConnection getConnection() {
        logger.debug("Using connection protocol {}", protocol);
        return Overthere.getConnection(protocol, getConnectionOptions());
    }

    @Override
    public Host getHost() {
        return this;
    }

    public OperatingSystemFamily getOs() {
        return os;
    }

    public void setOs(OperatingSystemFamily os) {
        this.os = os;
    }

    public String getTemporaryDirectoryPath() {
        return temporaryDirectoryPath;
    }

    public void setTemporaryDirectoryPath(String temporaryDirectoryPath) {
        this.temporaryDirectoryPath = temporaryDirectoryPath;
    }

    public String getStagingDirectoryPath() {
        return stagingDirectoryPath;
    }

    public void setStagingDirectoryPath(String stagingDirectoryPath) {
        this.stagingDirectoryPath = stagingDirectoryPath;
    }

    private ConnectionOptions getConnectionOptions() {
        return new ConnectionOptionsBuilder().getConnectionOptions(this);
    }

    public String getShell() {
        return shell;
    }

    public static class ConnectionOptionsBuilder {
        private List<String> jumpStationsSeen = new ArrayList<>();
        private List<String> winrsProxiesSeen = new ArrayList<>();

        public ConnectionOptions getConnectionOptions(Host host) {
            ConnectionOptions options = new ConnectionOptions();

            copyPropertiesToConnectionOptions(options, host);

            // set the temporary directory path if it has been set
            String temporaryDirectoryPath = host.getTemporaryDirectoryPath();
            if (temporaryDirectoryPath != null && !temporaryDirectoryPath.trim().isEmpty()) {
                setConnectionOption(options, TEMPORARY_DIRECTORY_PATH, temporaryDirectoryPath);
            }

            applyCredentials(options);

            logger.debug("Created connection options: {}", options);

            return options;
        }

        protected void copyPropertiesToConnectionOptions(ConnectionOptions options, ConfigurationItem ci) {
            // copy all CI properties to connection properties
            for (PropertyDescriptor pd : ci.getType().getDescriptor().getPropertyDescriptors()) {
                Object value = pd.get(ci);
                setConnectionOption(options, pd.getName(), value);
            }
        }

        private void applyCredentials(ConnectionOptions options) {
            Object credentials = options.getOptional("credentials");
            if (credentials != null) {
                ((HostCredentials) credentials).applyProperties((key, value) -> setConnectionOption(options, key, value));
            }
        }

        private void setConnectionOption(ConnectionOptions options, String key, Object value) {
            if (key.equals("temporaryDirectoryPath")) {
                return;
            }

            if (value == null || value.toString().isEmpty()) {
                return;
            }

            if (value instanceof Integer && ((Integer) value).intValue() == 0) {
                logger.debug("Activating workaround for DEPLOYITPB-4775: Integer with value of 0 not passed to Overthere.");
                return;
            }

            if (key.equals(JUMPSTATION)) {
                ConfigurationItem item = (ConfigurationItem) value;
                checkCircularReference((ConfigurationItem) value, jumpStationsSeen, "jumpstations");

                ConnectionOptions jumpstationOptions = new ConnectionOptions();
                copyPropertiesToConnectionOptions(jumpstationOptions, item);
                applyCredentials(jumpstationOptions);
                options.set(key, jumpstationOptions);
            } else if (key.equals("winrsProxy")) {
                ConfigurationItem item = (ConfigurationItem) value;
                checkCircularReference((ConfigurationItem) value, winrsProxiesSeen, "winrs proxies");

                options.set(WINRS_PROXY_PROTOCOL, item.getProperty("protocol"));
                ConnectionOptions winrsProxyOptions = new ConnectionOptions();
                copyPropertiesToConnectionOptions(winrsProxyOptions, item);
                applyCredentials(winrsProxyOptions);
                options.set(WINRS_PROXY_CONNECTION_OPTIONS, winrsProxyOptions);
            } else if ("metadataSshKeysProvider".equals(key)) {
                ConfigurationItem item = (ConfigurationItem) value;
                copyPropertiesToConnectionOptions(options, item);
            } else {
                options.set(key, value);
            }
        }

        private void checkCircularReference(ConfigurationItem value, List<String> list, final String what) {
            if (list.contains(value.getId())) {
                list.add(value.getId());

                String loop = mkString(list, " -> ");
                throw new IllegalStateException("Detected loop in " + what + ": " + loop);
            }
            list.add(value.getId());
        }
    }

    public Satellite getSatellite() {
        return satellite;
    }

    private static Logger logger = LoggerFactory.getLogger(Host.class);

}
