/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.api.udm;


import com.xebialabs.deployit.plugin.api.validation.ApplicationDependenciesFormat;
import com.xebialabs.xlplatform.documentation.PublicApiRef;

import java.util.*;

/**
 * A particular version of an {@link Application} consisting of both artifacts and resource specifications
 * (a set of {@link Deployable}s) that can be deployed onto a {@link Container}.
 */
@PublicApiRef
@SuppressWarnings("serial")
@Metadata(description = "A package of a certain version of an application. Groups all deployable artifact CIs for an application for a certain version. "
        + "Is contained by the Application CI.")
public class DeploymentPackage extends Version {

    @Property(description = "The set of deployables that are part of this package", asContainment = true, required = false)
    private Set<Deployable> deployables = new HashSet<>();

    @Property(description = "The versions of other applications that this package depends on, as a mapping from application name to version range", required = false, category = "Dependencies")
    @ApplicationDependenciesFormat()
    private Map<String, String> applicationDependencies;

    @Property(description = "The resolution to use for application dependencies when deploying this application. Should be 'LATEST' or 'EXISTING'. A placeholder can be used.", defaultValue = "LATEST", category = "Dependencies")
    private String dependencyResolution = DependencyResolution.LATEST;

    @Property(description = "Whether to undeploy direct or transient dependencies of this application. Dependency is undeployed if there are not other application that depend on it. Should be 'true' or 'false'. A placeholder can be used.",
            required = false, defaultValue = "false", category = "Dependencies",label = "Undeploy Unused Dependencies")
    private String undeployDependencies;

    @Property(description = "The set of CI templates.", asContainment = true, required = false, category = "Provisioning")
    private Set<Template> templates = new HashSet<>();

    @Property(description = "The set of CI bound templates.", required = false, candidateValuesFilter = "filterBySelf", category = "Provisioning")
    private Set<Template> boundTemplates = new HashSet<>();

    public DeploymentPackage() {
        // no op
    }

    /**
     * @param deployables The {@link Deployable}s that are part of this package.
     */
    public DeploymentPackage(Deployable... deployables) {
        this.deployables = new HashSet<>(Arrays.asList(deployables));
    }

    public Set<Template> getTemplates() {
        return templates;
    }

    public void setTemplates(Set<Template> templates) {
        this.templates = templates;
    }

    public Set<Template> getBoundTemplates() {
        return boundTemplates;
    }

    public void setBoundTemplates(Set<Template> boundTemplates) {
        this.boundTemplates = boundTemplates;
    }

    /**
     * @return The set of {@link Deployable}s that are part of this package.
     */
    public Set<Deployable> getDeployables() {
        return Collections.unmodifiableSet(deployables);
    }

    /**
     * @param deployables The set of {@link Deployable}s that are part of this package.
     */
    public void setDeployables(Set<Deployable> deployables) {
        this.deployables = deployables;
    }

    /**
     * Adds a {@link Deployable} The set of Deployables that are part of this package.
     */
    public void addDeployable(Deployable deployable) {
        if (deployable == null) {
            throw new NullPointerException("Cannot add a null deployable");
        }
        deployables.add(deployable);
    }

    /**
     * Gets the application versions that the this package depends on. This is a mapping from application names or id's to version ranges.
     * A version range is a string in OSGi Semantic Versioning format.
     *
     * @return a Map&lt;String, String&gt;
     */
    public Map<String, String> getApplicationDependencies() {
        return applicationDependencies;
    }

    /**
     * Sets the application versions that this package depends on. This is a mapping from application names or id's (e.g. "Application/PetClinic")
     * to version ranges. A version range is a string in OSGi Semantic Versioning format, e.g. "[1.5,3)"
     *
     * @param applicationDependencies the map of dependencies for this application version.
     */
    public void setApplicationDependencies(Map<String, String> applicationDependencies) {
        this.applicationDependencies = applicationDependencies;
    }

    /**
     * Indicates how application dependencies will be resolved when deploying an application.
     */
    public interface DependencyResolution {

        /**
         * Deploy the latest version of any dependency.
         */
        String LATEST = "LATEST";

        /**
         * Use existing versions where possible, i.e. do a minimal change.
         */
        String EXISTING = "EXISTING";
    }

    public String getDependencyResolution() {
        return dependencyResolution;
    }

    public void setDependencyResolution(String dependencyResolution) {
        this.dependencyResolution = dependencyResolution;
    }

    public String getUndeployDependencies() {
        return undeployDependencies;
    }

    public void setUndeployDependencies(String undeployDependencies) {
        this.undeployDependencies = undeployDependencies;
    }
}
