package com.xebialabs.deployit.plugin.api.reflect;

import com.google.common.base.Function;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import com.xebialabs.deployit.plugin.api.inspection.InspectionProperty;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Property;
import org.w3c.dom.Element;

import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.util.*;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.difference;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.*;
import static com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.*;
import static java.lang.Character.isUpperCase;
import static java.lang.Character.toUpperCase;
import static java.lang.String.format;

public class PropertyDescriptor {

	private Descriptor declaringDescriptor;
	private String name;

    private Field field;
	private boolean asContainment;
	private String category;
	private String description;
	private String label;
	private boolean password;
	private boolean required;
    private Property.Size size;
    private PropertyKind kind;
    private List<String> enumValues;
    private Class<?> enumClass;
    private Type referencedType;
    private List<PropertyDescriptor> listDescriptors;
    private boolean hidden;
	private boolean inspectionProperty;
	private boolean requiredForInspection;
	private boolean isTransient;

	private PropertyDescriptor(Field field) {
		this.name = field.getName();
		this.field = field;
		field.setAccessible(true);
	}

	private PropertyDescriptor(String name) {
		this.name = name;
		this.field = null;
	}

	static PropertyDescriptor from(Descriptor descriptor, Field field) {
        return from(descriptor, field, true);
    }

	static PropertyDescriptor from(Descriptor descriptor, Field field, boolean topLevel) {
		Property annotation = field.getAnnotation(Property.class);

		PropertyDescriptor propertyDescriptor = new PropertyDescriptor(field);
		propertyDescriptor.declaringDescriptor = descriptor;
		propertyDescriptor.initMetadata(annotation);
		propertyDescriptor.initType(annotation, topLevel);
		initInspectionMetadata(field, propertyDescriptor);
		check(propertyDescriptor);
		return propertyDescriptor;
	}

	static PropertyDescriptor from(Descriptor descriptor, Element propertyElement) {
        String name = getRequiredStringAttribute(propertyElement, "name");
        
		PropertyDescriptor propertyDescriptor = new PropertyDescriptor(name);
		propertyDescriptor.declaringDescriptor = descriptor;
		propertyDescriptor.initSynthetic(propertyElement);
		check(propertyDescriptor);
		return propertyDescriptor;
	}

	private static void check(PropertyDescriptor propertyDescriptor) {
		if (propertyDescriptor.hidden) {
			checkState(propertyDescriptor.isTransient, "Hidden property %s should be transient", propertyDescriptor);
		}
	}

	private static void initInspectionMetadata(Field field, PropertyDescriptor propertyDescriptor) {
		if (field.isAnnotationPresent(InspectionProperty.class)) {
			propertyDescriptor.inspectionProperty = true;
			propertyDescriptor.requiredForInspection = field.getAnnotation(InspectionProperty.class).required();
		}
	}

	private void initMetadata(Property annotation) {
		category = annotation.category();
		label = isBlank(annotation.label()) ? deCamelize(name) : annotation.label();
		description = isBlank(annotation.description()) ? label : annotation.description();
		password = annotation.password();
		required = annotation.required();
		size = annotation.size();
        hidden = annotation.hidden();
		if (hidden) {
			isTransient = true;
		} else {
			isTransient = annotation.isTransient();
		}
	}

	private void initType(Property annotation, boolean topLevel) {
		Class<?> type = field.getType();
		if (type == boolean.class) {
			kind = BOOLEAN;
		} else if (type == int.class) {
			kind = INTEGER;
		} else if (type == String.class) {
			kind = STRING;
		} else if (type.isEnum()) {
			kind = ENUM;
			initEnumValues(field.getType());
		} else if (ConfigurationItem.class.isAssignableFrom(type)) {
			kind = CI;
			referencedType = Type.valueOf(type);
			asContainment = annotation.asContainment();
		} else if (Set.class.isAssignableFrom(type)) {
			initSetType(annotation);
		} else if (Map.class.isAssignableFrom(type)) {
			initMapType(annotation);
		} else {
			throw new IllegalArgumentException(format("Type of %s not supported as an @Property field, found on %s.%s", type.getName(), field
			        .getDeclaringClass().getName(), name));
		}

        GlobalContext.register(this, annotation.defaultValue());
	}

	private Object convertValue(String val) {
        if (Strings.isNullOrEmpty(val)) return null;
        switch (kind) {
            case BOOLEAN:
                return Boolean.parseBoolean(val);
            case INTEGER:
                return Integer.parseInt(val);
            case STRING:
                return val;
            case ENUM:
                for (Enum<?> enumConstant : (Enum<?>[]) enumClass.getEnumConstants()) {
                    if (enumConstant.name().equalsIgnoreCase(val)) {
                        return enumConstant;
                    }
                }
                throw new IllegalArgumentException("Value " + val + " not a member of enum " + field.getType());
            case SET_OF_STRING:
            case SET_OF_CI:
            case CI:
	        case MAP_STRING_STRING:
            default:
                throw new IllegalArgumentException("Property " + name + " of kind " + kind + " cannot be converted from a string value");
        }
    }

	private void initSetType(Property annotation) {
		Class<?> setType = getGenericType(Set.class, 1, 0);
		if (setType == String.class) {
			kind = SET_OF_STRING;
		} else if (ConfigurationItem.class.isAssignableFrom(setType)) {
			kind = SET_OF_CI;
			referencedType = Type.valueOf(setType);
			asContainment = annotation.asContainment();
		} else {
			throw new IllegalStateException(format("Unsupported Set type encountered for [%s]. Only support String and ConfigurationItem", name));
		}
	}

	private void initMapType(Property annotation) {
		checkArgument(getGenericType(Map.class, 2, 0) == String.class, "Property %s.%s of type Map should be Map<String, String>", field.getDeclaringClass().getName(), name);
		checkArgument(getGenericType(Map.class, 2, 1) == String.class, "Property %s.%s of type Map should be Map<String, String>", field.getDeclaringClass().getName(), name);
		kind = MAP_STRING_STRING;
	}

	private Class<?> getGenericType(Class<?> collectionClass, int nrExpectedTypes, int indexOfType) {
		java.lang.reflect.Type genericType = field.getGenericType();
		checkArgument(genericType instanceof ParameterizedType, "The field %s.%s is a %s but it isn't a generic type (%s)",
		        field.getDeclaringClass().getName(), name, collectionClass, genericType);
		java.lang.reflect.Type[] actualTypeArguments = ((ParameterizedType) genericType).getActualTypeArguments();
		checkArgument(actualTypeArguments.length == nrExpectedTypes, "The field %s is a %s.%s but it doesn't have the right generic type (%s)", field.getDeclaringClass()
		        .getName(), name, collectionClass, actualTypeArguments);
		checkArgument(actualTypeArguments[indexOfType] instanceof Class, "The field %s.%s is a %s but it is not a concrete subclass (%s)", field.getDeclaringClass()
		        .getName(), name, collectionClass, Arrays.toString(actualTypeArguments));
		return (Class<?>) actualTypeArguments[indexOfType];

	}

	private void initEnumValues(Class<?> enumClass) {
		enumValues = newArrayList();
        this.enumClass = enumClass;
		for (Enum<?> enumValue : (Enum<?>[]) enumClass.getEnumConstants()) {
			enumValues.add(enumValue.name());
		}
	}

    void overrideWith(PropertyDescriptor superDescriptor) {
        checkArgument(superDescriptor.getKind() == this.getKind(),"Type '%s' attempts to overrides property '%s' declared in type '%s', but kind attribute does not match. Derived kind: '%s'. Super kind: '%s'.", getDeclaringDescriptor().getType(), getName(), superDescriptor.getDeclaringDescriptor().getType(), getKind(), superDescriptor.getKind());
        checkOverrideArgument(superDescriptor.isAsContainment() == this.isAsContainment(), "asContainment", superDescriptor.isAsContainment(), superDescriptor);
        checkOverrideArgument(superDescriptor.getReferencedType() == this.getReferencedType(), "referenceType", superDescriptor.getReferencedType(), superDescriptor);
        checkOverrideArgument(superDescriptor.enumClass == this.enumClass, "enumClass", superDescriptor.enumClass, superDescriptor);
        checkOverrideArgument(superDescriptor.isPassword() == this.isPassword(), "password", superDescriptor.isPassword(), superDescriptor);
        if (superDescriptor.isRequired()) {
            checkOverrideArgument(this.isRequired(), "required", superDescriptor.isRequired(), superDescriptor);
        }

        field = superDescriptor.field;
    }

    private void checkOverrideArgument(boolean condition, String attribute, Object expectedValue, PropertyDescriptor superDescriptor) {
        final String attributeErrorTemplate = "Type '%s' attempts to overrides property '%s' declared in type '%s', but '%s' attribute does not match that in the super type. Should be set to %s.";
        checkArgument(condition, attributeErrorTemplate, getDeclaringDescriptor().getType(), getName(), superDescriptor.getDeclaringDescriptor().getType(), attribute, expectedValue);
    }

	private void initSynthetic(Element propertyElement) {
		kind = PropertyKind.valueOf(getOptionalStringAttribute(propertyElement, "kind", STRING.name()).toUpperCase());
		category = getOptionalStringAttribute(propertyElement, "category", "Common");
		label = getOptionalStringAttribute(propertyElement, "label", deCamelize(name));
		description = getOptionalStringAttribute(propertyElement, "description", label);
		required = getOptionalBooleanAttribute(propertyElement, "required", true);
		password = getOptionalBooleanAttribute(propertyElement, "password", false);
		asContainment = getOptionalBooleanAttribute(propertyElement, "as-containment", false);
		size = Property.Size.valueOf(getOptionalStringAttribute(propertyElement, "size", Property.Size.MEDIUM.name()).toUpperCase());
        String defaultValueAttr = getOptionalStringAttribute(propertyElement, "default", "");
        hidden = getOptionalBooleanAttribute(propertyElement, "hidden", false);
		if (hidden) {
			isTransient = true;
		} else {
            isTransient = getOptionalBooleanAttribute(propertyElement, "transient", false);
		}

		if (kind == ENUM) {
			String enumClassAttributeValue = getRequiredStringAttribute(propertyElement, "enum-class", "for property " + name + " of kind " + kind);
			try {
				enumClass = Class.forName(enumClassAttributeValue);
				if (!enumClass.isEnum()) {
					throw new IllegalArgumentException("enum-class supplied for property " + name + " of kind " + kind + " is not an enum: "
					        + enumClassAttributeValue);
				}
				initEnumValues(enumClass);
			} catch (ClassNotFoundException exc) {
				throw new IllegalArgumentException("Unknown enum-class supplied for property " + name + " of kind " + kind + ": " + enumClassAttributeValue,
				        exc);
			}
		}

		if (kind == CI || kind == SET_OF_CI) {
            referencedType = Type.valueOf(getRequiredStringAttribute(propertyElement, "referenced-type", "for property " + name + " of kind " + kind));
		}

        GlobalContext.register(this, defaultValueAttr);
	}

	static PropertyDescriptor generateDeployableFrom(Descriptor descriptor, PropertyDescriptor deployedPropertyDescriptor) {
		PropertyDescriptor propertyDescriptor = new PropertyDescriptor(deployedPropertyDescriptor.getName());
		propertyDescriptor.declaringDescriptor = descriptor;
		propertyDescriptor.generateDeployable(deployedPropertyDescriptor);
		return propertyDescriptor;
	}

	private void generateDeployable(PropertyDescriptor deployedPropertyDescriptor) {
        // Generated 'simple' Deployable properties always are of kind String so that placeholder/dictionaries values can be filled.
		kind = deployedPropertyDescriptor.getKind().isSimple() ? PropertyKind.STRING : deployedPropertyDescriptor.getKind();
		category = deployedPropertyDescriptor.getCategory();
		label = deployedPropertyDescriptor.getLabel();
		description = deployedPropertyDescriptor.getDescription();
		required = false;
		password = deployedPropertyDescriptor.isPassword();
		size = deployedPropertyDescriptor.getSize();
        field = null;
    }

	public String getName() {
		return name;
	}

	public Field getField() {
		return field;
	}

	public String getDescription() {
		return description;
	}

	public boolean isAsContainment() {
		return asContainment;
	}

	public String getCategory() {
		return category;
	}

	public String getLabel() {
		return label;
	}

	public boolean isPassword() {
		return password;
	}

	public boolean isRequired() {
		return required;
	}

	public Property.Size getSize() {
		return size;
	}

	public PropertyKind getKind() {
		return kind;
	}

	public List<String> getEnumValues() {
		return enumValues;
	}

	public Descriptor getDeclaringDescriptor() {
		return declaringDescriptor;
	}

	public List<PropertyDescriptor> getListDescriptors() {
		return listDescriptors;
	}

	public Type getReferencedType() {
		return referencedType;
	}

    public Object getDefaultValue() {
        return convertValue(GlobalContext.lookup(this));
    }

    public boolean isHidden() {
        return hidden;
    }

    public Object get(ConfigurationItem item) {
		if (field != null) {
			try {
				return field.get(item);
			} catch (IllegalAccessException e) {
				throw new RuntimeException("Cannot get field " + field, e);
			}
		} else {
			return item.getSyntheticProperty(name);
		}
	}

	public void set(ConfigurationItem item, Object value) {
        if (kind.isSimple() && value instanceof String) {
            value = convertValue((String) value);
        } else if (value == null) {
	        value = getDefaultValue();
        }

		try {
			if (field != null) {
				if (value == null) {
					return;
				}
				field.set(item, value);
			} else {
				item.putSyntheticProperty(name, value);
			}
		} catch (IllegalAccessException e) {
			throw new RuntimeException("Cannot set field " + field, e);
		}
	}

	public boolean areEqual(ConfigurationItem item, ConfigurationItem other) {
        return areEqual(item, other, new HashSet<String>());
	}

	boolean areEqual(ConfigurationItem item, ConfigurationItem other, Set<String> itemsBeingCompared) {
        Object left = get(item);
        Object right = get(other);
        return areValuesEqual(left, right, itemsBeingCompared);
	}

    @SuppressWarnings("unchecked")
	private boolean areValuesEqual(Object itemValue, Object otherValue, Set<String> itemsBeingCompared) {

		if (itemValue == null) {
			return otherValue == null;
		}

		switch (kind) {
		case SET_OF_STRING:
			return difference((Set<String>) itemValue, (Set<String>) otherValue).isEmpty();
		case SET_OF_CI:
            Function<ConfigurationItem, String> f = new Function<ConfigurationItem, String>() {
				@Override
				public String apply(ConfigurationItem from) {
					return from.getName();
				}
			};

            Iterable<ConfigurationItem> cis = Iterables.concat((Set<ConfigurationItem>) itemValue, (Set<ConfigurationItem>) otherValue);
            Multimap<String,ConfigurationItem> index = Multimaps.index(cis, f);
            for(String key : index.keySet()) {
                Collection<ConfigurationItem> cisToCompare = index.get(key);
                if (cisToCompare.size() != 2) {
                    return false;
                }
                Iterator<ConfigurationItem> itemIterator = cisToCompare.iterator();
                ConfigurationItem lhs = itemIterator.next();
                ConfigurationItem rhs = itemIterator.next();
                if (lhs.getType().equals(rhs.getType())) {
                    Descriptor descriptor = DescriptorRegistry.getDescriptor(lhs.getType());
                    if (descriptor.areEqual(lhs, rhs, itemsBeingCompared)) {
                        continue;
                    }
                }

                return false;
            }
            return true;
		case CI:
			ConfigurationItem itemValueAsCi = (ConfigurationItem) itemValue;
			ConfigurationItem otherValueAsCi = (ConfigurationItem) otherValue;
            if (otherValueAsCi != null && itemValueAsCi.getName().equals(otherValueAsCi.getName())) {
                Descriptor descriptor = DescriptorRegistry.getDescriptor(itemValueAsCi.getType());
                return descriptor.areEqual(itemValueAsCi, otherValueAsCi, itemsBeingCompared);
            }
            return false;
		case MAP_STRING_STRING:
			Map<String, String> left = (Map<String, String>) itemValue;
			Map<String, String> right = (Map<String, String>) otherValue;
			return Maps.difference(left, right).areEqual();
		default:
			return itemValue.equals(otherValue);
		}
	}

	private static boolean isBlank(String s) {
		return s == null || s.trim().isEmpty();
	}

	private static String deCamelize(String fieldName) {
		StringBuilder buf = new StringBuilder();
		for (int i = 0; i < fieldName.length(); i++) {
			char c = fieldName.charAt(i);
			if (i == 0) {
				c = toUpperCase(c);
			} else if (isUpperCase(c)) {
				buf.append(" ");
			}
			buf.append(c);

		}
		return buf.toString();
	}

	public boolean isInspectionProperty() {
		return inspectionProperty;
	}

	public boolean isRequiredForInspection() {
		return requiredForInspection;
	}

	public boolean isTransient() {
		return isTransient;
	}

	@Override
	public String toString() {
		return declaringDescriptor.getType() + "." + name;
	}

}
