package com.xebialabs.deployit.plugin.api.reflect;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static com.google.common.base.Predicates.equalTo;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newLinkedHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_CI;
import static com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.filterChildElementsByName;
import static com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.forEach;
import static com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.getOptionalBooleanAttribute;
import static com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.getOptionalStringAttribute;
import static com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.getOptionalTypeAttribute;
import static com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.getRequiredStringAttribute;
import static com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.getRequiredTypeAttribute;
import static java.lang.reflect.Modifier.isAbstract;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.w3c.dom.Element;

import com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.Closure;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Prefix;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployed;


public class Descriptor {

	static final String PLACEHOLDERS_FIELD = "placeholders";

	private Type type;
	private Class<? extends ConfigurationItem> clazz;
	private String description;
	private Metadata.ConfigurationItemRoot root;
	private List<Type> superclasses = newArrayList();
	private Set<Type> interfaces = newHashSet();
	private boolean virtual;

	private Map<String, PropertyDescriptor> properties = newLinkedHashMap();
	private Type deployableType;
	private Type containerType;

	private Type generatedDeployableType;
	private Type generatedDeployableBase;
	private String generatedDeployableDescription;

	private boolean hierarchyInitialized = false;


	private Descriptor(Class<? extends ConfigurationItem> clazz) {
		this.type = Type.valueOf(clazz);
		this.clazz = clazz;
	}

	private Descriptor(Type type) {
		this.type = type;
	}

	static Descriptor from(Class<? extends ConfigurationItem> clazz) {
		try {
			Descriptor descriptor = new Descriptor(clazz);
			descriptor.initMetadata();
			descriptor.scanClass();
			return descriptor;
		} catch (RuntimeException e) {
			throw new DescriptorException("Could not create descriptor for: " + clazz.getName(), e);
		}
	}

	private void initMetadata() {
		Metadata annotation = checkNotNull(clazz.getAnnotation(Metadata.class), "Class " + clazz.getName() + " or one of its ancestors does not have a @Metadata annotation");
		description = annotation.description();
		root = annotation.root();
		virtual = annotation.virtual() || Modifier.isAbstract(clazz.getModifiers());
	}

	private void scanClass() {
		findProperties();
		findInterfaces();

		Class<?> superclass = clazz.getSuperclass();
		if (ConfigurationItem.class.isAssignableFrom(superclass)) {
			Type supertype = Type.valueOf(superclass);
			addSuperClass(supertype);
		}

		initDeployableAndContainerTypes();
	}

	private void findProperties() {
		for (Field field : clazz.getDeclaredFields()) {
			if (field.isAnnotationPresent(Property.class)) {
				PropertyDescriptor propertyDescriptor = PropertyDescriptor.from(this, field);
				addProperty(propertyDescriptor);
			}
		}
	}

	private void findInterfaces() {
        Class<?>[] clazzInterfaces = clazz.getInterfaces();
        List<Class<?>> allInterfacesFound = newArrayList();
        findAllSuperInterfaces(clazzInterfaces, allInterfacesFound);
		for (Class<?> clazzInterface : allInterfacesFound) {
			if (clazzInterface.getPackage().isAnnotationPresent(Prefix.class)) {
				addInterface(Type.valueOf(clazzInterface));
			}
		}
	}

    private void findAllSuperInterfaces(Class<?>[] childInterfaces, List<Class<?>> allInterfacesFound) {
        for (Class<?> childInterface : childInterfaces) {
            allInterfacesFound.add(childInterface);
            findAllSuperInterfaces(childInterface.getInterfaces(), allInterfacesFound);
        }
    }

    private void initDeployableAndContainerTypes() {
		if (Deployed.class.isAssignableFrom(clazz)) {
			@SuppressWarnings({ "unchecked", "rawtypes" })
			List<Class<?>> typeArguments = ClassUtils.getActualTypeArguments((Class<? extends Deployed>) clazz, Deployed.class);
			checkArgument(typeArguments.size() == 2, "Expected exactly a Deployable and a Container, but got %s", typeArguments);

			Class<?> deployableClass = typeArguments.get(0);
			if (deployableClass != null) {
				checkArgument(Deployable.class.isAssignableFrom(deployableClass), "Expected first item to be a deployable");
				deployableType = Type.valueOf(deployableClass);
			} else {
				deployableType = null;
			}

			Class<?> containerClass = typeArguments.get(1);
			if (containerClass != null) {
				checkArgument(Container.class.isAssignableFrom(containerClass), "Expected second item to be a container");
				containerType = Type.valueOf(containerClass);
			} else {
				containerType = null;
			}
		} else {
			deployableType = null;
			containerType = null;
		}
	}

	static Descriptor from(Element typeElement) {
		Type type = getRequiredTypeAttribute(typeElement, "type");

		Descriptor descriptor = new Descriptor(type);
		descriptor.initSynthetic(typeElement);

		return descriptor;
	}

	private void initSynthetic(Element typeElement) {
		description = getOptionalStringAttribute(typeElement, "description", "Description unavailable");
		virtual = getOptionalBooleanAttribute(typeElement, "virtual", false);

		Type superType = getRequiredTypeAttribute(typeElement, "extends");
		addSuperClass(superType);

		parseSyntheticDeployableAndContainerType(typeElement);
		parseTypeModification(typeElement);
	}

	private void parseSyntheticDeployableAndContainerType(Element typeElement) {
        deployableType = getOptionalTypeAttribute(typeElement, "deployable-type");
        containerType = getOptionalTypeAttribute(typeElement, "container-type");
        List<Element> generateElements = filterChildElementsByName(typeElement, equalTo("generate-deployable"));
        if (!generateElements.isEmpty()) {
            Element generateDeployable = generateElements.get(0);
            generatedDeployableType = getRequiredTypeAttribute(generateDeployable, "type");
            generatedDeployableBase = getRequiredTypeAttribute(generateDeployable, "extends");
            generatedDeployableDescription= getOptionalStringAttribute(generateDeployable, "description", "Description unavailable");
        }
	}

    void parseTypeModification(Element element) {
        forEach(filterChildElementsByName(element, equalTo("property")), new Closure<Element>() {
            @Override
            public void call(Element element) {
                String name = getRequiredStringAttribute(element, "name");
                PropertyDescriptor newDesc = PropertyDescriptor.from(Descriptor.this, element);
                PropertyDescriptor oldDesc = properties.get(name);
                if (oldDesc != null) {
                    newDesc.overrideWith(oldDesc);
                }
                addProperty(newDesc);
            }
        });
    }

	static Descriptor from(Descriptor deployedDescriptor) {
        Descriptor deployableDescriptor = new Descriptor(deployedDescriptor.generatedDeployableType);
        deployableDescriptor.addSuperClass(deployedDescriptor.generatedDeployableBase);
		deployableDescriptor.initDeployableFromDeployed(deployedDescriptor, deployedDescriptor.generatedDeployableDescription);
        deployableDescriptor.initHierarchy();
		return deployableDescriptor;
	}

	private void initDeployableFromDeployed(Descriptor deployedDescriptor, String generatedDeployableDescription) {
		if (generatedDeployableDescription == null || generatedDeployableDescription.equals("Description unavailable")) {
			description = deployedDescriptor.getDescription() + " (deployable)";
		} else {
			description = generatedDeployableDescription;
		}

		for (PropertyDescriptor pd : deployedDescriptor.getPropertyDescriptors()) {
			boolean isUdmField = pd.getName().equals(BaseDeployed.DEPLOYABLE_FIELD) || pd.getName().equals(BaseDeployed.CONTAINER_FIELD)
			        || pd.getName().equals(PLACEHOLDERS_FIELD);
			boolean isReferenceField = pd.getKind() == CI || pd.getKind() == SET_OF_CI;
			if (isUdmField || isReferenceField || pd.isHidden()) {
				continue;
			}

			addProperty(PropertyDescriptor.generateDeployableFrom(this, pd));
		}
	}

	void initHierarchy() {
		if (hierarchyInitialized || superclasses.isEmpty()) {
			return;
		}

		Type toInitFrom = superclasses.get(0);

		do {
			Descriptor superDesc = DescriptorRegistry.getDescriptor(toInitFrom);
			if(superDesc == null) {
				throw new IllegalStateException("Cannot build type hierarchy for " + getType() + " because one of its supertypes cannot be found: " + toInitFrom + " not found");
			}

			// For synthetic types, root and clazz might not be set yet.
			if (root == null) {
				root = superDesc.getRoot();
			}

			if (clazz == null) {
				clazz = superDesc.clazz;
			}

			inheritPropertyDescriptors(properties, superDesc.properties);
			for (Type superclass : superDesc.superclasses) {
				addSuperClass(superclass);
			}
			for (Type intf : superDesc.interfaces) {
				addInterface(intf);
			}
			toInitFrom = superDesc.superclasses.isEmpty() || superDesc.hierarchyInitialized ? null : superDesc.getSuperClasses().get(0);

            if (deployableType == null) {
                deployableType = superDesc.getDeployableType();
            }

            if (containerType == null) {
                containerType = superDesc.getContainerType();
            }
		} while (toInitFrom != null);

		validate();

		hierarchyInitialized = true;
	}

	private void validate() {
	    validateReferenceTypes();
	    validateNonArtifactDoesNotHavePlaceholders();
	    validateArtifactInterfaces();
	    if(!isVirtual()) {
	    	validateJavaClassIsNotAbstract();
	    	validateDeployedHasDeployableAndContainerType();
			validateHiddenRequiredPropertiesHaveDefaultValue();
		}
    }

	private void validateReferenceTypes() {
	    for(PropertyDescriptor p : properties.values()) {
	    	if(p.getKind() == CI || p.getKind() == SET_OF_CI) {
	    		checkState(isValidReferencedType(p.getReferencedType()), "Property %s of type %s refers to non-existing type %s", p.getName(), p.getDeclaringDescriptor().getType(), p.getReferencedType());
	    	}
	    }
    }

	private boolean isValidReferencedType(Type referencedType) {
	    if(DescriptorRegistry.exists(referencedType))
	    	return true;
	    
	    for(Descriptor d : DescriptorRegistry.getDescriptors()) {
	    	if(d.getSuperClasses().contains(referencedType)) { 
	    		return true;
	    	}
	    	if(d.getInterfaces().contains(referencedType)) { 
	    		return true;
	    	}
	    }
	    
	    return false;
    }

	private void validateNonArtifactDoesNotHavePlaceholders() {
		if (!isAssignableTo(Artifact.class)) {
			for (PropertyDescriptor propertyDescriptor : properties.values()) {
				checkState(!propertyDescriptor.getName().equals(PLACEHOLDERS_FIELD), "Type %s that does not implemnt the udm.Artifact interface must not have a field called 'placeholders'", type);
			}
		}
	}

	private void validateArtifactInterfaces() {
		if(isAssignableTo(Deployable.class) && isAssignableTo(Artifact.class)) {
			checkState(isAssignableTo(SourceArtifact.class), "Type %s that implements the udm.Deployable and udm.Artifact interface must also implement the udm.SourceArtifact interface", type);
			checkState(isAssignableTo(DeployableArtifact.class), "Type %s that implements the udm.Deployable and udm.Artifact interface must also implement the udm.DeployableArtifact interface", type);
		}
		if(isAssignableTo(Deployed.class) && isAssignableTo(Artifact.class)) {
			checkState(isAssignableTo(DerivedArtifact.class), "Type %s that implements the udm.Deployed and udm.Artifact interface must also implement the udm.DerivedArtifact interface", type);
		}
    }

	private void validateJavaClassIsNotAbstract() {
		checkState(!isAbstract(clazz.getModifiers()), "Non-virtual type %s has an abstract Java class %s", getType(), clazz.getName());
    }

	private void validateDeployedHasDeployableAndContainerType() {
		Type deployedType = Type.valueOf(Deployed.class);
		if(isAssignableTo(deployedType)) {
			checkState(getDeployableType() != null, "Non-virtual type %s is a sub-type of %s but does not have a deployable-type", getType(), deployedType);
			checkState(getContainerType() != null, "Non-virtual type %s is a sub-type of %s but does not have a container-type", getType(), deployedType);
		}
    }

	private void validateHiddenRequiredPropertiesHaveDefaultValue() {
	    for(PropertyDescriptor p : properties.values()) {
	    	if(p.isHidden() && p.isRequired()) {
	    		checkState(p.getDefaultValue() != null, "Hidden required property %s of non-virtual type %s must have a default value", p.getName(), getType());
	    	}
	    }
    }

    private void inheritPropertyDescriptors(Map<String, PropertyDescriptor> dest, Map<String, PropertyDescriptor> source) {
        for (String sourceKey : source.keySet()) {
            if (!dest.containsKey(sourceKey)) {
                dest.put(sourceKey, new PropertyDescriptor(source.get(sourceKey), this));
            } else {
                dest.get(sourceKey).overrideWith(source.get(sourceKey));
            }
        }
    }

    private void addSuperClass(Type supertype) {
		superclasses.add(supertype);
		DescriptorRegistry.registerSubtype(supertype, type);
	}

	private void addInterface(Type intf) {
		DescriptorRegistry.registerSubtype(intf, type);
		interfaces.add(intf);
	}

	void addProperty(PropertyDescriptor propertyDescriptor) {
		properties.put(propertyDescriptor.getName(), propertyDescriptor);
	}

	public Type getType() {
		return type;
	}

	public Class<?> getClazz() {
		return clazz;
	}

	public String getDescription() {
		return description;
	}

	public Metadata.ConfigurationItemRoot getRoot() {
		return root;
	}

	public Collection<PropertyDescriptor> getPropertyDescriptors() {
		return properties.values();
	}

	public PropertyDescriptor getPropertyDescriptor(String name) {
		return properties.get(name);
	}

	public boolean isAssignableTo(Class<?> clazz) {
		return isAssignableTo(Type.valueOf(clazz));
	}

	public boolean isAssignableTo(Type type) {
		return DescriptorRegistry.getSubtypes(type).contains(this.type) || this.type.equals(type);
	}

	public List<Type> getSuperClasses() {
		return superclasses;
	}

	public Set<Type> getInterfaces() {
		return interfaces;
	}

	public boolean isVirtual() {
		return virtual;
	}

    public boolean areEqual(ConfigurationItem item, ConfigurationItem other) {
    	return areEqual(item, other, new HashSet<String>());
    }

    boolean areEqual(ConfigurationItem item, ConfigurationItem other, Set<String> itemsBeingCompared) {
        if (item == null) {
            return other == null;
        }

        if (!getType().equals(item.getType()) || !getType().equals(other.getType())) {
            return false;
        }

        if(!itemsBeingCompared.add(item.getId())) {
        	return true;
        }

        for (PropertyDescriptor pd : getPropertyDescriptors()) {
            if (!pd.areEqual(item, other, itemsBeingCompared)) {
                return false;
            }
        }
        return true;
    }

    boolean shouldGenerateDeployableType() {
		return generatedDeployableType != null;
	}

	@SuppressWarnings("unchecked")
	public <T extends ConfigurationItem> T newInstance() {
		if (virtual) {
			throw new IllegalArgumentException("Cannot instantiate class for " + type + " because it is virtual");
		}

		try {
			Field typeField = searchField(clazz, ConfigurationItem.TYPE_FIELD);
			typeField.setAccessible(true);
			T t = (T) clazz.newInstance();
			typeField.set(t, type);

			prefillDefaultProperties(t);

			return t;
		} catch (InstantiationException exc) {
			throw new RuntimeException("Cannot instantiate class " + clazz.getName(), exc);
		} catch (IllegalAccessException exc) {
			throw new RuntimeException("Cannot instantiate class " + clazz.getName(), exc);
		}
	}

	private <T extends ConfigurationItem> void prefillDefaultProperties(T t) {
		for (PropertyDescriptor pd : getPropertyDescriptors()) {
			if (pd.getDefaultValue() != null) {
				pd.set(t, pd.getDefaultValue());
			}
		}
	}

	private Field searchField(Class<?> clazz, String name) {
		for (Class<?> scan = clazz; !scan.equals(Object.class); scan = scan.getSuperclass()) {
			try {
				return scan.getDeclaredField(name);
			} catch (NoSuchFieldException e) {
				// scan up the tree
			}
		}
		throw new IllegalArgumentException("Cannot find '" + name + "' field on " + clazz.getName());
	}

	@Override
	public String toString() {
		return "Descriptor[" + type + "]";
	}

	public Type getDeployableType() {
		return deployableType;
	}

	public Type getContainerType() {
		return containerType;
	}

    @SuppressWarnings("serial")
	private static class DescriptorException extends RuntimeException {
		public DescriptorException(String s, RuntimeException e) {
			super(s, e);
		}
	}
}
