package com.xebialabs.deployit.plugin.api.reflect;

import com.google.common.collect.Iterators;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Enumeration;
import java.util.List;

import static com.google.common.base.Predicates.equalTo;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.iterateChildElementsByName;
import static java.lang.Thread.currentThread;
import static javax.xml.XMLConstants.W3C_XML_SCHEMA_NS_URI;

public class SyntheticInitializer {
    private static final DocumentBuilderFactory DOCUMENT_BUILDER_FACTORY = createDocumentBuilderFactory();

    public static void init() {
        scanSynthetic();
    }

    static void scanSynthetic() {
        try {
            List<Element> typeDefinitionElements = newArrayList();
            List<Element> typeModificationElements = newArrayList();

            readSynthetics("synthetic.xml", typeDefinitionElements, typeModificationElements);
            readSynthetics("synthetic-test.xml", typeDefinitionElements, typeModificationElements);

            parseAllSyntheticTypeDefinitions(typeDefinitionElements);
            parseAllSyntheticTypeModifications(typeModificationElements);
        } catch (IOException exc) {
            throw new RuntimeException("Cannot read synthetic configurations", exc);
        }

    }

	private static void readSynthetics(String name, List<Element> typeDefinitionElements, List<Element> typeModificationElements) throws IOException {
	    ClassLoader cl = currentThread().getContextClassLoader();
	    Enumeration<URL> syntheticXMLs = cl.getResources(name);
	    while (syntheticXMLs.hasMoreElements()) {
	        final URL syntheticXML = syntheticXMLs.nextElement();
	        Element docElement = readSyntheticDocument(syntheticXML).getDocumentElement();
		    Iterators.addAll(typeDefinitionElements,iterateChildElementsByName(docElement, equalTo("type")));
	        Iterators.addAll(typeModificationElements, iterateChildElementsByName(docElement, equalTo("type-modification")));
	    }
    }

    protected static void parseAllSyntheticTypeDefinitions(List<Element> typeElements) {
        for (Element e : typeElements) {
            DescriptorRegistry.registerSyntheticTypeDefinition(e);
        }
    }

    protected static void parseAllSyntheticTypeModifications(List<Element> typeModificationElements) {
        for (Element e : typeModificationElements) {
            DescriptorRegistry.registerSyntheticTypeModification(e);
        }
    }

    private static DocumentBuilderFactory createDocumentBuilderFactory() {
        DocumentBuilderFactory documentBuilderFactory;
        try {
            SchemaFactory schemaFactory = SchemaFactory.newInstance(W3C_XML_SCHEMA_NS_URI);
            Schema syntheticSchema = schemaFactory.newSchema(currentThread().getContextClassLoader().getResource("synthetic.xsd"));
            documentBuilderFactory = DocumentBuilderFactory.newInstance();
            documentBuilderFactory.setNamespaceAware(true);
            documentBuilderFactory.setSchema(syntheticSchema);
        } catch (SAXException exc) {
            throw new RuntimeException("Cannot read schema synthetic.xsd", exc);
        }
        return documentBuilderFactory;
    }

    private static Document readSyntheticDocument(final URL syntheticXML) throws IOException {
        InputStream syntheticXMLStream = syntheticXML.openStream();
        try {
            final boolean validationErrorsFound[] = new boolean[1];
            DocumentBuilder builder = DOCUMENT_BUILDER_FACTORY.newDocumentBuilder();
            builder.setErrorHandler(new ErrorHandler() {
                @Override
                public void warning(SAXParseException exc) throws SAXException {
                    logger.warn("Warning while parsing " + syntheticXML, exc);
                }

                @Override
                public void error(SAXParseException exc) throws SAXException {
                    logger.error("Error while parsing " + syntheticXML, exc);
                    validationErrorsFound[0] = true;
                }

                @Override
                public void fatalError(SAXParseException exc) throws SAXException {
                    logger.error("Fatal error while parsing " + syntheticXML, exc);
                    validationErrorsFound[0] = true;
                }
            });
            Document doc = builder.parse(syntheticXMLStream);
            if (validationErrorsFound[0]) {
                throw new RuntimeException("One or more errors were found while parsing " + syntheticXML);
            }

            return doc;
        } catch (RuntimeException exc) {
            throw new RuntimeException("Cannot read synthetic configuration " + syntheticXML, exc);
        } catch (ParserConfigurationException exc) {
            throw new RuntimeException("Cannot read synthetic configuration " + syntheticXML, exc);
        } catch (SAXException exc) {
            throw new RuntimeException("Cannot read synthetic configuration " + syntheticXML, exc);
        } finally {
            syntheticXMLStream.close();
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(SyntheticInitializer.class);
}
