package com.xebialabs.deployit.plugin.api.validation;

import com.google.common.collect.Sets;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.Deployed;

import java.lang.annotation.*;
import java.util.Map;
import java.util.Set;

@Retention(RetentionPolicy.RUNTIME)
@Rule(clazz = Placeholders.Validator.class, type = "placeholders")
@Target(ElementType.TYPE)
public @interface Placeholders {

	public static class Validator implements com.xebialabs.deployit.plugin.api.validation.Validator<Deployed<?, ?>> {
		@Override
		public void validate(Deployed<?, ?> value, ValidationContext context) {
			if (!value.hasProperty("placeholders")) return;

			// Deployed placeholders should always be a Map<String, String>
			Map<String, String> placeholders = value.getProperty("placeholders");
			validateAllPlaceholdersSet(placeholders, context);

			if (value.getDeployable() != null) {
				Deployable deployable = value.getDeployable();
				if (!deployable.hasProperty("placeholders")) {
					context.error("Deployed %s has placeholders but the Deployable not.", value.getId());
				} else {
					Set<String> deployablePlaceholders = deployable.getProperty("placeholders");
					validatePlaceholdersAllPresent(deployablePlaceholders, placeholders, context);
				}
			}
		}

		private void validatePlaceholdersAllPresent(Set<String> source, Map<String, String> dest, ValidationContext context) {
			Set<String> destKeys = dest.keySet();
			wrongKeys(Sets.difference(source, destKeys), context, "Deployable contains placeholder %s which is not defined in deployed");
			wrongKeys(Sets.difference(destKeys, source), context, "Deployed contains placeholder %s which was not found in deployable");
		}

		private void wrongKeys(Set<String> difference, ValidationContext context, String error) {
			for (String s : difference) {
				context.error(error, s);
			}
		}

		private void validateAllPlaceholdersSet(Map<String, String> placeholders, ValidationContext context) {
			for (String s : placeholders.keySet()) {
				if (placeholders.get(s) == null || placeholders.get(s).trim().isEmpty()) {
					context.error("Placeholders [%s] doesn't have a value.", s);
				}
			}
		}
	}
}
