package com.xebialabs.deployit.plugin.api.reflect;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.google.common.collect.Lists.newArrayList;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import com.google.common.collect.MapMaker;
import com.google.common.io.Closeables;
import com.google.common.io.Files;
import com.google.common.io.LineProcessor;

public class GlobalContext {
    private static final Logger logger = LoggerFactory.getLogger(GlobalContext.class);

    private static final Map<String, Value> context = new MapMaker().makeComputingMap(new Function<String, Value>() {
        @Override
        public Value apply(String input) {
            return new Value("", null);
        }
    });

    public static final File DEFAULTS = new File("conf", "deployit-defaults.properties");

    private GlobalContext() {
    }

    static void register(PropertyDescriptor pd, String defaultValue) {
        if (defaultValue == null && !pd.isHidden()) return;
        context.put(constructPropertyName(pd), new Value(pd.getDescription(), defaultValue));
    }

	public static void register(PropertyDescriptor pd, PropertyDescriptor inheritedFrom) {
		context.put(constructPropertyName(pd), new InheritedValue(pd.getDescription(), constructPropertyName(inheritedFrom), pd.isHidden()));
	}


    private static String constructPropertyName(PropertyDescriptor pd) {
        return pd.getDeclaringDescriptor().getType() + "." + pd.getName();
    }

    static String lookup(PropertyDescriptor pd) {
	    return lookup(constructPropertyName(pd));
    }

	private static String lookup(String key) {
		if (!context.containsKey(key)) return null;
		return context.get(key).getValue();
	}

	static void loadStoredDefaults(File configFile) {
        if (!configFile.exists()) {
            logger.info("Could not find '" + configFile + "', continuing without loading defaults");
            return;
        } else if (configFile.isDirectory()) {
            logger.error(configFile + " is a directory but should be a plain file, aborting!");
            throw new IllegalStateException("Please remove " + configFile + " and try again.");
        }
        try {
            Files.readLines(configFile, Charset.defaultCharset(), new LineProcessor<Object>() {
                @Override
                public boolean processLine(String line) throws IOException {
                    if (!line.startsWith("#") && line.contains("=")) {
                        int i = line.indexOf("=");
                        String key = line.substring(0, i);
                        Value value = context.get(key);
	                    context.put(key, new ExplicitValue(value.description, line.substring(i + 1)));
                    }
                    return true;
                }

                @Override
                public Object getResult() {
                    return null;
                }
            });
        } catch (IOException e) {
            throw new IllegalArgumentException("Could not read '" + configFile.toString() + "'", e);
        }
    }

    public static void storeDefaults() {
        storeDefaults(DEFAULTS);
    }

    static void storeDefaults(File configFile) {
        BufferedWriter bufferedWriter = null;
        try {
            if (!configFile.getParentFile().exists() || !configFile.getParentFile().isDirectory()) {
                logger.warn("Not writing {} because the directory does not exist", configFile);
                return;
            }
            bufferedWriter = Files.newWriter(configFile, Charset.defaultCharset());
            List<String> keys = newArrayList(context.keySet());
            Collections.sort(keys);
            for (String k : keys) {
                Value v  = context.get(k);
	            if (v.isShouldWrite()) {
	                if (!Strings.isNullOrEmpty(v.description)) {
	                    bufferedWriter.append("# ").append(v.description);
		                if (v instanceof InheritedValue) {
			                InheritedValue inheritedValue = (InheritedValue) v;
			                bufferedWriter.append(" (inherited from: ").append(inheritedValue.superTypeProperty).append(")");
		                }
	                    bufferedWriter.newLine();
	                }
	                bufferedWriter.append(v.isExplicit() ? "" : "#").append(k).append('=').append(v.getValue());
	                bufferedWriter.newLine();
	            }
            }
        } catch (FileNotFoundException e) {
            logger.error("Could not start writing to '" + configFile + "'", e);
        } catch (IOException e) {
            logger.error("Could not write to '" + configFile + "'", e);
        } finally {
            Closeables.closeQuietly(bufferedWriter);
        }
    }

	static void validateValues() {
		boolean valid = true;
		List<String> messages = newArrayList();
		for (String prop : context.keySet()) {
			int i = prop.lastIndexOf('.');
			String type = prop.substring(0, i);
			String propertyName = prop.substring(i + 1);

			Type tType = Type.valueOf(type);
			if (DescriptorRegistry.exists(tType)) {
				Descriptor descriptor = DescriptorRegistry.getDescriptor(tType);
				if (descriptor.isVirtual()) continue;
				PropertyDescriptor propertyDescriptor = descriptor.getPropertyDescriptor(propertyName);
				if (propertyDescriptor != null) {
					try {
						propertyDescriptor.getDefaultValue();
						if (propertyDescriptor.isRequired() && propertyDescriptor.isHidden() && isNullOrEmpty(lookup(propertyDescriptor))) {
							valid = false;
							messages.add(String.format("Cannot register empty default value for hidden required property [%s]", prop));
						}
					} catch (RuntimeException re) {
						valid = false;
						messages.add(String.format("Incorrect default registered: [%s] with value %s", prop, context.get(prop).getValue()));
					}
				}
			}
		}

		if (!valid) {
			String join = Joiner.on("\n").join(messages);
			logger.error(join);
			throw new IllegalStateException("Could not initialize the default values, please look at the log and correct the defaults file: " + DEFAULTS + "\n" + join);
		}
	}

	static class InheritedValue extends Value {
		private final String superTypeProperty;
		private final boolean shouldWrite;

		InheritedValue(String description, String superTypeProperty, boolean shouldWrite) {
			super(description, "");
			this.superTypeProperty = superTypeProperty;
			this.shouldWrite = shouldWrite;
		}

		@Override
		public String getValue() {
			return lookup(superTypeProperty);
		}

		@Override
		boolean isShouldWrite() {
			return shouldWrite;
		}
	}

	static class Value {
        Value(String description, String value) {
            this.description = description;
            this.value = value;
        }

        final String description;
        final String value;

	    boolean isExplicit() {
		    return false;
	    }

		boolean isShouldWrite() {
			return true;
		}
		
		public String getValue() {
			return value;
		}
	}

	static class ExplicitValue extends Value {
		ExplicitValue(String description, String value) {
			super(description, value);
		}

		@Override
		boolean isExplicit() {
			return true;
		}
	}
}
