/*
 * Copyright (c) 2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of Xebialabs Software and Documentation is subject to the Personal
 * License Agreement.
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * ‚ÄúDocumentation‚Äù means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the Xebialabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to  (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.was.runbook;

import java.util.List;

import com.xebialabs.deployit.Change;
import com.xebialabs.deployit.ChangePlan;
import com.xebialabs.deployit.RunBook;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.plugin.was.ci.WasCluster;
import com.xebialabs.deployit.plugin.was.ci.WasNode;
import com.xebialabs.deployit.plugin.was.step.AddSingleWasClusterMemberStep;
import com.xebialabs.deployit.plugin.was.step.CreateWasClusterMembersStep;
import com.xebialabs.deployit.plugin.was.step.CreateWasClusterStep;
import com.xebialabs.deployit.plugin.was.step.DestroyWasClusterStep;
import com.xebialabs.deployit.plugin.was.step.ModifyWasClusterMemberStep;
import com.xebialabs.deployit.plugin.was.step.RemoveWasClusterMemberStep;
import com.xebialabs.deployit.plugin.was.step.StartWasClusterMembersStep;
import com.xebialabs.deployit.plugin.was.step.StopWasClusterStep;
import com.xebialabs.deployit.plugin.was.step.SynchronizeWasNodeStep;
import com.xebialabs.deployit.reflect.ConfigurationItemReflectionUtils;

/**
 * Supports the creation, modification and deletion of a {@link com.xebialabs.deployit.plugin.was.ci.WasCluster WAS Cluster}.
 * 
 * <h4>Conditions</h4>
 * 
 * Will trigger if the change plan contains the addition, modification or deletion of a {@link WasCluster WAS Cluster} CI.
 * 
 * <h4>Actions</h4>
 * 
 * Addition:
 * 
 * <ol>
 * <li>Create cluster in the cell
 * <li>For every node in the cell create # of cluster members
 * <li>For every node in the cell synchronize
 * <li>For every node in the cell start cluster members
 * </ol>
 * 
 * Modification:
 * 
 * <ol>
 * <li>The cluster is stopped
 * <li>If the number of clusters is decreased remove excess cluster members
 * <li>Modify remaining cluster members
 * <li>For every node in the cell create additional # of cluster members if number of cluster members is increased
 * <li>For every node in the cell synchronize
 * <li>For every node in the cell start cluster members
 * </ol>
 * 
 * Deletion:
 * 
 * <ol>
 * <li>Stop the cluster
 * <li>Destroy the cluster
 * </ol>
 */
public class WasClusterRunBook extends WasRunBookBase<WasCluster> implements RunBook {

	public WasClusterRunBook() {
		super(WasCluster.class);
	}

	public void resolve(Change<WasCluster> change, ChangePlan changePlan, List<Step> steps) {
		// applies only when Cluster is not identical
		WasCluster oldCluster = change.getOldRevision();
		WasCluster newCluster = change.getNewRevision();
		boolean identical = ConfigurationItemReflectionUtils.isIdentical(oldCluster, newCluster);
		if (!identical) {

			if (change.isModification()) {
				if (!modificationWillBeHandledByWasDeploymentRunBook(change, changePlan)) {
					getStepsForModifiedCluster(steps, oldCluster, newCluster);
				}
			}

			if (change.isDeletion()) {
				WasCluster cluster = change.getOldRevision();
				steps.add(new StopWasClusterStep(cluster, false));
				steps.add(new DestroyWasClusterStep(cluster));
			}

			if (change.isAddition()) {
				WasCluster cluster = change.getNewRevision();
				steps.add(new CreateWasClusterStep(cluster));
				for (WasNode eachNode : cluster.getNodes()) {
					steps.add(new CreateWasClusterMembersStep(cluster, eachNode));
					steps.add(new SynchronizeWasNodeStep(eachNode));
					steps.add(new StartWasClusterMembersStep(cluster, eachNode));
				}
			}
		}
	}

	public void getStepsForModifiedCluster(List<Step> steps, WasCluster oldCluster, WasCluster newCluster) {
		steps.add(new StopWasClusterStep(oldCluster, false));
		for (WasNode eachNode : oldCluster.getNodes()) {
			int numberOfRemovedClones = removeClones(oldCluster, newCluster, eachNode, steps);
			for (int i = 1; i <= oldCluster.getNumberOfClusterMembers() - numberOfRemovedClones; i++) {
				steps.add(new ModifyWasClusterMemberStep(oldCluster, eachNode, i));
			}
			createAdditionalClones(oldCluster, newCluster, eachNode, steps);
			steps.add(new StartWasClusterMembersStep(newCluster, eachNode));
		}
	}

	private boolean modificationWillBeHandledByWasDeploymentRunBook(Change<WasCluster> clusterChange, ChangePlan changePlan) {
		// true when changePlan contains a modification change of a deployment that has the Cluster as its target.
		Change<Deployment> deploymentChange = findChangeForModifiedDeploymentOfModifiedCluster(clusterChange, changePlan);
		return deploymentChange != null;
	}

	@SuppressWarnings("unchecked")
	private Change<Deployment> findChangeForModifiedDeploymentOfModifiedCluster(Change<WasCluster> clusterChange, ChangePlan changePlan) {
		if (!clusterChange.isModification()) {
			return null;
		}
		for (Change<?> change : changePlan.getChanges()) {
			if (change.isModification() && change.getConfigurationItemClass() == Deployment.class) {
				WasCluster oldCluster = clusterChange.getOldRevision();
				WasCluster newCluster = clusterChange.getNewRevision();
				Deployment oldDeployment = (Deployment) change.getOldRevision();
				Deployment newDeployment = (Deployment) change.getNewRevision();
				boolean oldClusterWasTarget = oldDeployment.getTarget().getMembersOfType(WasCluster.class).contains(oldCluster);
				boolean newClusterWasTarget = newDeployment.getTarget().getMembersOfType(WasCluster.class).contains(newCluster);
				if (oldClusterWasTarget && newClusterWasTarget) {
					return (Change<Deployment>) change;
				}
			}
		}
		return null;
	}

	private int removeClones(WasCluster oldCluster, WasCluster newCluster, WasNode eachNode, List<Step> steps) {
		int diff = oldCluster.getNumberOfClusterMembers() - newCluster.getNumberOfClusterMembers();
		if (diff > 0) {
			for (int i = oldCluster.getNumberOfClusterMembers(); i > newCluster.getNumberOfClusterMembers(); i--) {
				steps.add(new RemoveWasClusterMemberStep(oldCluster, eachNode, i));
			}
			return diff;
		}
		return 0;
	}

	private void createAdditionalClones(WasCluster oldCluster, WasCluster newCluster, WasNode eachNode, List<Step> steps) {
		int diff = newCluster.getNumberOfClusterMembers() - oldCluster.getNumberOfClusterMembers();
		if (diff > 0) {
			for (int i = oldCluster.getNumberOfClusterMembers() + 1; i <= newCluster.getNumberOfClusterMembers(); i++) {
				steps.add(new AddSingleWasClusterMemberStep(newCluster, eachNode, i));
			}
		}
	}

}
