/*
 * Copyright (c) 2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to  (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.was.ci;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.core.io.ClassPathResource;

import com.xebialabs.deployit.BaseConfigurationItem;
import com.xebialabs.deployit.ConfigurationItem;
import com.xebialabs.deployit.ConfigurationItemProperty;
import com.xebialabs.deployit.StepExecutionContext;
import com.xebialabs.deployit.StepExecutionContextCallbackHandler;
import com.xebialabs.deployit.StepExecutionContextListener;
import com.xebialabs.deployit.ConfigurationItemProperty.Size;
import com.xebialabs.deployit.ci.Host;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.hostsession.CommandExecution;
import com.xebialabs.deployit.hostsession.CommandExecutionCallbackHandler;
import com.xebialabs.deployit.hostsession.HostFile;
import com.xebialabs.deployit.hostsession.HostFileUtils;
import com.xebialabs.deployit.hostsession.HostSession;
import com.xebialabs.deployit.plugin.was.step.WasStepBase;

@SuppressWarnings("serial")
@ConfigurationItem(category = "middleware", description = "A WebSphere Application Server cell. Is a member of Host on which the deployment manager runs.", expandable = true)
public class WasCell extends BaseConfigurationItem {

	/**
	 * The system property to set to disable the usage of the wsadmin daemon.
	 */
	public static final String DISABLE_DAEMON = "com.xebialabs.deployit.plugin.was.disableDaemon";

	private static final String DAEMON_STARTED = "DEPLOYIT-DAEMON-STARTED";

	private static final String DAEMON_EXIT_CODE_MARKER = "DEPLOYIT-DAEMON-EXIT-VALUE: ";

	private static final String DAEMON_END_OF_COMMAND_MARKER = "DEPLOYIT-DAEMON-END-OF-COMMAND";

	private static final String CONFIG_DIR_NAME = "config";

	private static final String PLUGIN_FILENAME = "plugin-cfg.xml";

	@ConfigurationItemProperty
	private Host host;

	@ConfigurationItemProperty(required = true, label = "WebSphere name", description = "Name of the WebSphere cell, e.g. MyCell, WASCell, Cell01")
	private String name;

	@ConfigurationItemProperty(required = false, label = "Administrative username", description = "Username which is used to login to the WebSphere Cell.")
	private String username;

	@ConfigurationItemProperty(required = false, label = "Administrative password", description = "Password which is used to login to the WebSphere Cell.", password = true)
	private String password;

	@ConfigurationItemProperty(required = true, label = "Deployment manager profile path", description = "Path of the WebSphere deployment manager profile.", size = Size.LARGE)
	private String wasHome;

	@ConfigurationItemProperty(required = true, label = "WAS version", description = "Version of WebSphere Application Server.")
	private WasVersion version;

	public String getWsadminPath() {
		String fileSep = host.getOperatingSystemFamily().getFileSeparator();
		return getWasHome() + fileSep + "bin" + fileSep + "wsadmin" + host.getOperatingSystemFamily().getScriptExtension();
	}

	public String getConfigDirPath() {
		String fileSep = host.getOperatingSystemFamily().getFileSeparator();
		return getWasHome() + fileSep + CONFIG_DIR_NAME;
	}

	public String getPluginFilePath() {
		String fileSep = host.getOperatingSystemFamily().getFileSeparator();
		return getConfigDirPath() + fileSep + "cells" + fileSep + WasCell.PLUGIN_FILENAME;
	}

	public HostSession connectToAdminHost() {
		return host.getHostSession();
	}

	public int executeWsadminJythonScript(StepExecutionContext ctx, HostSession cellHostSession, String scriptPath, String... args) {
		String disableDaemon = System.getProperty(DISABLE_DAEMON);
		if (disableDaemon == null) {
			return executeWsadminJythonScriptWithDaemon(ctx, cellHostSession, scriptPath, args);
		} else {
			return executeWsadminJythonScriptDirectly(new StepExecutionContextCallbackHandler(ctx), cellHostSession, scriptPath, args);
		}
	}

	public int executeWsadminJythonScriptWithDaemon(final StepExecutionContext ctx, HostSession cellHostSession, String scriptPath, String... args) {
		try {
			CommandExecution wsadminExecution = getWsadminDaemonCommandExecution(ctx);
			if (wsadminExecution == null) {
				logger.warn("Cannot create wsadmin daemon session, executing script directly");
				return executeWsadminJythonScriptDirectly(new StepExecutionContextCallbackHandler(ctx), cellHostSession, scriptPath, args);
			}

			StringBuffer invokePyContents = new StringBuffer();
			invokePyContents.append("import sys;\r\nsys.argv=[");
			boolean first = true;
			for (String arg : args) {
				if (!first) {
					invokePyContents.append(",");
				} else {
					first = false;
				}
				invokePyContents.append("\'");
				invokePyContents.append(arg);
				invokePyContents.append("\'");
			}
			invokePyContents.append("];\r\nexecfile(\'");
			invokePyContents.append(scriptPath);
			invokePyContents.append("\')\r\n");

			HostFile invokePyFile = cellHostSession.getTempFile(FilenameUtils.getBaseName(scriptPath), "-invoke.py");
			HostFileUtils.putStringToHostFile(invokePyContents.toString(), invokePyFile);

			String daemonLine = "execfile(\'" + invokePyFile.getPath() + "\')\r\n";
			OutputStream stdin = wsadminExecution.getStdin();
			stdin.write(daemonLine.getBytes());
			stdin.flush();

			final BufferedReader stdout = new BufferedReader(new InputStreamReader(wsadminExecution.getStdout()));
			final BufferedReader stderr = new BufferedReader(new InputStreamReader(wsadminExecution.getStderr()));
			Thread stderrThread = new Thread(new Runnable() {
				public void run() {
					try {
						for (;;) {
							String stderrLine = stderr.readLine();
							if (stderrLine == null) {
								break;
							}
							ctx.logError(stderrLine);
						}
					} catch (IOException exc) {
						exc.printStackTrace();
					}
				}
			}, "stderr printer for wsadmin daemon running on " + this);
			stderrThread.start();
			int exitCode = 0;
			for (;;) {
				String stdoutLine = stdout.readLine();
				if (stdoutLine == null) {
					throw new RuntimeIOException("Cannot execute remote command \"" + scriptPath + " " + StringUtils.join(args, ' ') + "\" on "
							+ this.getHost() + ": lost connection to the wsadmin daemon");
				}

				if (stdoutLine.startsWith(DAEMON_EXIT_CODE_MARKER)) {
					try {
						exitCode = Integer.parseInt(stdoutLine.substring(DAEMON_EXIT_CODE_MARKER.length()));
					} catch (NumberFormatException ignored) {
					}
				} else if (stdoutLine.equals(DAEMON_END_OF_COMMAND_MARKER)) {
					break;
				} else {
					ctx.logOutput(stdoutLine);
				}
			}

			return exitCode;
		} catch (IOException exc) {
			throw new RuntimeIOException("Cannot execute remote command \"" + scriptPath + " " + StringUtils.join(args, ' ') + "\" on " + this.getHost(), exc);
		}
	}

	private CommandExecution getWsadminDaemonCommandExecution(StepExecutionContext context) {
		String key = "CELL_SESSION_" + getLabel();
		WsadminDaemonExecution wrappedExecution = (WsadminDaemonExecution) context.getAttribute(key);
		if (wrappedExecution == null) {
			HostSession daemonSession = connectToAdminHost();
			HostFile uploadedDaemon = daemonSession.copyToTemporaryFile(new ClassPathResource(WasStepBase.STEP_RESOURCES_PATH + "daemon.py"));
			List<String> cmd = new ArrayList<String>();
			cmd.add(getWsadminPath());
			if (StringUtils.isNotBlank(getUsername())) {
				cmd.add("-user");
				cmd.add(getUsername());
				if (!StringUtils.isEmpty(getPassword())) {
					cmd.add("-password");
					cmd.add(getPassword());
				}
			}
			cmd.add("-lang");
			cmd.add("jython");
			cmd.add("-f");
			cmd.add(uploadedDaemon.getPath());
			String[] cmdArray = (String[]) cmd.toArray(new String[cmd.size()]);

			CommandExecution daemonExecution = daemonSession.startExecute(cmdArray);
			if (daemonExecution == null) {
				daemonSession.close();
				return null;
			}
			final BufferedReader stdout = new BufferedReader(new InputStreamReader(daemonExecution.getStdout()));
			for (;;) {
				String stdoutLine;
				try {
					stdoutLine = stdout.readLine();
					if (stdoutLine == null) {
						throw new RuntimeIOException("Cannot start wsadmin daemon: lost connection to the wsadmin daemon");
					}
					context.logOutput(stdoutLine);
					if (stdoutLine.startsWith(DAEMON_STARTED)) {
						break;
					}
				} catch (IOException exc) {
					throw new RuntimeIOException("Cannot start wsadmin daemon", exc);
				}
			}

			wrappedExecution = new WsadminDaemonExecution(daemonSession, daemonExecution);
			context.setAttribute(key, wrappedExecution);
		}
		return wrappedExecution.getCommandExecution();
	}

	public static final class WsadminDaemonExecution implements StepExecutionContextListener {

		private HostSession session;

		private CommandExecution execution;

		public WsadminDaemonExecution(HostSession session, CommandExecution execution) {
			this.session = session;
			this.execution = execution;
		}

		public CommandExecution getCommandExecution() {
			return execution;
		}

		public void contextDestroyed() {
			OutputStream stdin = execution.getStdin();
			try {
				stdin.write("QUIT\r\n".getBytes());
				stdin.flush();
				execution.waitFor();
			} catch (IOException exc) {
				logger.error("Error stopping wsadmin daemon", exc);
			} catch (RuntimeException exc) {
				logger.error("Error stopping wsadmin daemon", exc);
			}

			session.close();
		}

	}

	public int executeWsadminJythonScriptDirectly(CommandExecutionCallbackHandler handler, HostSession s, String scriptPath, String... args) {
		List<String> cmd = new ArrayList<String>();
		cmd.add(getWsadminPath());
		if (StringUtils.isNotBlank(getUsername())) {
			cmd.add("-user");
			cmd.add(getUsername());
			if (!StringUtils.isEmpty(getPassword())) {
				cmd.add("-password");
				cmd.add(getPassword());
			}
		}
		cmd.add("-lang");
		cmd.add("jython");
		cmd.add("-f");
		cmd.add(scriptPath);
		for (String arg : args) {
			cmd.add(arg);
		}
		String[] cmdArray = (String[]) cmd.toArray(new String[cmd.size()]);

		return s.execute(handler, cmdArray);
	}

	public Host getHost() {
		return host;
	}

	public void setHost(Host host) {
		this.host = host;
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public String getUsername() {
		return username;
	}

	public void setUsername(String username) {
		this.username = username;
	}

	public String getPassword() {
		return password;
	}

	public void setPassword(String password) {
		this.password = password;
	}

	public String getWasHome() {
		return wasHome;
	}

	public void setWasHome(String wasHome) {
		this.wasHome = wasHome;
	}

	public WasVersion getVersion() {
		return version;
	}

	public void setVersion(WasVersion version) {
		this.version = version;
	}

	private static Logger logger = Logger.getLogger(WasCell.class);

}
