/*
 * Copyright (c) 2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to  (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.was.runbook;

import java.util.List;

import com.xebialabs.deployit.Change;
import com.xebialabs.deployit.ChangePlan;
import com.xebialabs.deployit.RunBook;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.plugin.was.ci.WasServer;
import com.xebialabs.deployit.plugin.was.step.CreateWasServerStep;
import com.xebialabs.deployit.plugin.was.step.DestroyWasServerStep;
import com.xebialabs.deployit.plugin.was.step.ModifyWasServerStep;
import com.xebialabs.deployit.plugin.was.step.StartWasServerStep;
import com.xebialabs.deployit.plugin.was.step.StopWasServerStep;
import com.xebialabs.deployit.plugin.was.step.SynchronizeWasNodeStep;
import com.xebialabs.deployit.reflect.ConfigurationItemReflectionUtils;
import com.xebialabs.deployit.util.SingleTypeHandlingRunBook;

/**
 * Supports the creation, modification and deletion of a {@link com.xebialabs.deployit.plugin.was.ci.WasServer WAS Server}.
 * 
 * <h4>Conditions</h4>
 * 
 * Will trigger if the change plan contains the addition, modification or deletion of a {@link WasServer WAS Server} CI.
 * 
 * <h4>Actions</h4>
 * 
 * Addition:
 * 
 * <ol>
 * <li>Create server in the node
 * <li>Synchronize the node
 * <li>Start server
 * </ol>
 * 
 * Modification:
 * 
 * <ol>
 * <li>The server is stopped
 * <li>Modify server
 * <li>Synchronize the node
 * <li>Start server
 * </ol>
 * 
 * Deletion:
 * 
 * <ol>
 * <li>Stop the server
 * <li>Destroy the server
 * <li>Synchronize the node
 * </ol>
 */
public class WasServerRunBook extends SingleTypeHandlingRunBook<WasServer> implements RunBook {

	public WasServerRunBook() {
		super(WasServer.class);
	}

	public void resolve(Change<WasServer> change, ChangePlan changePlan, List<Step> steps) {
		// applies only when WasServer is not identical
		WasServer oldServer = change.getOldRevision();
		WasServer newServer = change.getNewRevision();
		boolean identical = ConfigurationItemReflectionUtils.isIdentical(oldServer, newServer);
		if (!identical) {

			if (change.isModification()) {
				if (!modificationWillBeHandledByWasDeploymentRunBook(change, changePlan)) {
					getStepsForModifiedServer(steps, oldServer, newServer);
				}
			}

			if (change.isDeletion()) {
				steps.add(new StopWasServerStep(oldServer, false));
				steps.add(new DestroyWasServerStep(oldServer));
				steps.add(new SynchronizeWasNodeStep(oldServer.getNode()));
			}

			if (change.isAddition()) {
				steps.add(new CreateWasServerStep(newServer));
				steps.add(new ModifyWasServerStep(newServer));
				steps.add(new SynchronizeWasNodeStep(newServer.getNode()));
				steps.add(new StartWasServerStep(newServer));
			}
		}
	}

	public void getStepsForModifiedServer(List<Step> steps, WasServer oldServer, WasServer newServer) {
		steps.add(new StopWasServerStep(oldServer, false));
		steps.add(new ModifyWasServerStep(newServer));
		steps.add(new StartWasServerStep(newServer));
		
	}

	private boolean modificationWillBeHandledByWasDeploymentRunBook(Change<WasServer> serverChange, ChangePlan changePlan) {
		// true when changePlan contains a modification change of a deployment that has the Server as its target.
		Change<Deployment> deploymentChange = findChangeForModifiedDeploymentOfModifiedServer(serverChange, changePlan);
		return deploymentChange != null;
	}

	@SuppressWarnings("unchecked")
	private Change<Deployment> findChangeForModifiedDeploymentOfModifiedServer(Change<WasServer> serverChange, ChangePlan changePlan) {
		if (!serverChange.isModification()) {
			return null;
		}
		for (Change<?> change : changePlan.getChanges()) {
			if (change.isModification() && change.getConfigurationItemClass() == Deployment.class) {
				WasServer oldServer = serverChange.getOldRevision();
				WasServer newServer = serverChange.getNewRevision();
				Deployment oldDeployment = (Deployment) change.getOldRevision();
				Deployment newDeployment = (Deployment) change.getNewRevision();
				boolean oldServerWasTarget = oldDeployment.getTarget().getMembersOfType(WasServer.class).contains(oldServer);
				boolean newServerWasTarget = newDeployment.getTarget().getMembersOfType(WasServer.class).contains(newServer);
				if (oldServerWasTarget && newServerWasTarget) {
					return (Change<Deployment>) change;
				}
			}
		}
		return null;
	}

}
