package com.xebialabs.deployit.plugin.was.runbook;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.xebialabs.deployit.Change;
import com.xebialabs.deployit.ChangePlan;
import com.xebialabs.deployit.RunBook;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.ci.artifact.Ear;
import com.xebialabs.deployit.ci.artifact.StaticContent;
import com.xebialabs.deployit.ci.artifact.War;
import com.xebialabs.deployit.ci.artifact.mapping.DeployableArtifactMapping;
import com.xebialabs.deployit.plugin.apache.httpd.ci.ApacheHttpdServer;
import com.xebialabs.deployit.plugin.apache.httpd.mapper.StaticContentToApacheHttpdServerMapper;
import com.xebialabs.deployit.plugin.apache.httpd.step.CreateApacheHttpdVirtualHostStep;
import com.xebialabs.deployit.plugin.apache.httpd.step.DestroyApacheHttpdVirtualHostStep;
import com.xebialabs.deployit.plugin.apache.httpd.step.RestartApacheHttpdServerStep;
import com.xebialabs.deployit.plugin.was.ci.WasCluster;
import com.xebialabs.deployit.plugin.was.ci.WasManagedApacheHttpdServer;
import com.xebialabs.deployit.plugin.was.ci.WasResource;
import com.xebialabs.deployit.plugin.was.mapper.EarToWasClusterMapper;
import com.xebialabs.deployit.plugin.was.mapper.WarToWasClusterMapper;
import com.xebialabs.deployit.plugin.was.mapper.WasDataSourceToWasClusterMapper;
import com.xebialabs.deployit.plugin.was.mapper.WasWmqQueueConnectionFactoryToWasClusterMapper;
import com.xebialabs.deployit.plugin.was.mapper.WasWmqQueueToWasClusterMapper;
import com.xebialabs.deployit.plugin.was.mapper.WasWmqTopicConnectionFactoryToWasClusterMapper;
import com.xebialabs.deployit.plugin.was.mapper.WasWmqTopicToWasClusterMapper;

/**
 * Supports the deployment, re-deployment and undeployment of a 
 * {@link com.xebialabs.deployit.ci.DeploymentPackage Application Package} to a 
 * {@link com.xebialabs.deployit.plugin.was.ci.WasCluster WAS Cluster}.
 * 
 * <h4>Conditions</h4>
 * 
 * Will trigger if the change plan contains the addition, modification or
 * deletion of a {@link Deployment Deployment} CI to an {@link com.xebialabs.deployit.ci.Environment Environment} CI 
 * which contains WAS middleware CI's.
 * 
 * <h4>Actions</h4>
 * 
 * Addition:
 * 
 * <ol>
 * <li>Deploy {@link WasResource WebSphere resources} from {@link WasCluster cluster}
 * <li>Deploy {@link Ear ear} from {@link WasCluster cluster} (Undeploy ear, destroy vhost definition, update webserver plugin config)
 * <li>Deploy {@link War war} from {@link WasCluster cluster} (Undeploy war, destroy vhost definition, update webserver plugin config)
 * <li>Create Apache Virtual Host definition on {@link WasManagedApacheHttpdServer an apache webserver}
 * <li>Deploy {@link StaticContent static content} from {@link WasManagedApacheHttpdServer an apache webserver}
 * <li>Restart {@link WasManagedApacheHttpdServer an apache webserver}
 * </ol>
 * 
 * All Modifications are handled as Deletion and Addition.
 * <p>
 * 
 * Deletion:
 * 
 * <ol>
 * <li>Undeploy {@link StaticContent static content} from {@link WasManagedApacheHttpdServer an apache webserver}
 * <li>Undeploy {@link Ear ear} from {@link WasCluster cluster} (Undeploy ear, destroy vhost definition, update webserver plugin config)
 * <li>Undeploy {@link War war} from {@link WasCluster cluster} (Undeploy war, destroy vhost definition, update webserver plugin config)
 * <li>Undeploy {@link WasResource WebSphere resources} from {@link WasCluster cluster}
 * <li>Destroy Apache Virtual Host definition on {@link WasManagedApacheHttpdServer an apache webserver}
 * <li>Restart {@link WasManagedApacheHttpdServer an apache webserver}
 * </ol>
 */
public class WasDeploymentRunBook extends WasRunBookBase<Deployment> implements RunBook {

	public WasDeploymentRunBook() {
		super(Deployment.class);
	}

	protected void resolve(Change<Deployment> change, ChangePlan changePlan, List<Step> steps) {
		EarToWasClusterMapper earMapper = new EarToWasClusterMapper(change);
		WarToWasClusterMapper warMapper = new WarToWasClusterMapper(change);
		StaticContentToApacheHttpdServerMapper staticContentMapper = new StaticContentToApacheHttpdServerMapper(change);
		WasDataSourceToWasClusterMapper dsMapper = new WasDataSourceToWasClusterMapper(change);
		WasWmqQueueToWasClusterMapper qMapper = new WasWmqQueueToWasClusterMapper(change);
		WasWmqQueueConnectionFactoryToWasClusterMapper qcfMapper = new WasWmqQueueConnectionFactoryToWasClusterMapper(change);
		WasWmqTopicToWasClusterMapper tMapper = new WasWmqTopicToWasClusterMapper(change);
		WasWmqTopicConnectionFactoryToWasClusterMapper tcfMapper = new WasWmqTopicConnectionFactoryToWasClusterMapper(change);

		Set<WasCluster> allTargetedClusters = new HashSet<WasCluster>();
		allTargetedClusters.addAll(earMapper.getAllTargets());
		allTargetedClusters.addAll(warMapper.getAllTargets());
		allTargetedClusters.addAll(dsMapper.getAllTargets());
		allTargetedClusters.addAll(qMapper.getAllTargets());
		allTargetedClusters.addAll(qcfMapper.getAllTargets());
		allTargetedClusters.addAll(tMapper.getAllTargets());
		allTargetedClusters.addAll(tcfMapper.getAllTargets());

		if (allTargetedClusters.isEmpty()) {
			return;
		}

		// deletion steps
		Map<ApacheHttpdServer, Set<String>> virtualHostsPerWebServerToDestroy = new HashMap<ApacheHttpdServer, Set<String>>();

		earMapper.setVirtualHostsPerWebServerCollector(virtualHostsPerWebServerToDestroy);
		earMapper.generateDeletionSteps(steps);

		warMapper.setVirtualHostsPerWebServerCollector(virtualHostsPerWebServerToDestroy);
		warMapper.generateDeletionSteps(steps);

		tMapper.generateDeletionSteps(steps);
		tcfMapper.generateDeletionSteps(steps);
		qMapper.generateDeletionSteps(steps);
		qcfMapper.generateDeletionSteps(steps);
		dsMapper.generateDeletionSteps(steps);
		
		if (change.isModification()) {
			WasClusterRunBook wasClusterRunBook = new WasClusterRunBook();
			Set<Change<WasCluster>> clusterChanges = findClusterChangesForDeploymentChange(change, changePlan);
			for (Change<WasCluster> eachClusterChange : clusterChanges) {
				wasClusterRunBook.getStepsForModifiedCluster(steps, eachClusterChange.getOldRevision(), eachClusterChange.getNewRevision());
			}
		}

		// addition steps
		Map<ApacheHttpdServer, Set<String>> virtualHostsPerWebServerToCreate = new HashMap<ApacheHttpdServer, Set<String>>();

		dsMapper.generateAdditionSteps(steps);
		qcfMapper.generateAdditionSteps(steps);
		qMapper.generateAdditionSteps(steps);
		tcfMapper.generateAdditionSteps(steps);
		tMapper.generateAdditionSteps(steps);

		warMapper.setVirtualHostsPerWebServerCollector(virtualHostsPerWebServerToCreate);
		warMapper.generateAdditionSteps(steps);

		earMapper.setVirtualHostsPerWebServerCollector(virtualHostsPerWebServerToCreate);
		earMapper.generateAdditionSteps(steps);

		// handle static content
		staticContentMapper.generateDeletionSteps(steps);
		staticContentMapper.generateAdditionSteps(steps);

		// handle apache
		destroyApacheVirtualHosts(virtualHostsPerWebServerToDestroy, steps);
		createApacheVirtualHosts(virtualHostsPerWebServerToCreate, steps);
		restartApacheWebServers(virtualHostsPerWebServerToDestroy, steps);
		restartApacheWebServers(virtualHostsPerWebServerToCreate, steps);
	}
	
	@SuppressWarnings("unchecked")
	private Set<Change<WasCluster>> findClusterChangesForDeploymentChange(Change<Deployment> deploymentChange, ChangePlan changePlan) {
		Set<Change<WasCluster>> clusterChanges = new HashSet<Change<WasCluster>>();
		for (Change<?> change : changePlan.getChanges()) {
			if (change.isModification() && change.getConfigurationItemClass() == WasCluster.class) {
				Change<WasCluster> clusterChange = (Change<WasCluster>) change;
				if (isClusterDeploymentTarget(clusterChange.getOldRevision(), deploymentChange.getOldRevision()) && 
						isClusterDeploymentTarget(clusterChange.getNewRevision(), deploymentChange.getNewRevision())) {
					clusterChanges.add(clusterChange);
				}
			}
		}
		return clusterChanges;
	}
	
	private boolean isClusterDeploymentTarget(WasCluster cluster, Deployment deployment) {
		List<DeployableArtifactMapping> deployableArtifactMappings = deployment.getMappingsOfType(DeployableArtifactMapping.class);
		for (DeployableArtifactMapping eachMapping : deployableArtifactMappings) {
			if (eachMapping.getTarget().equals(cluster)) {
				return true;
			}
		}
		return false;
	}

	private void createApacheVirtualHosts(Map<ApacheHttpdServer, Set<String>> virtualHostsPerWebServer, List<Step> steps) {
		for (Map.Entry<ApacheHttpdServer, Set<String>> eachVirtualHostPerWebServer : virtualHostsPerWebServer.entrySet()) {
			ApacheHttpdServer eachWebServer = eachVirtualHostPerWebServer.getKey();
			for (String eachVirtualHost : eachVirtualHostPerWebServer.getValue()) {
				steps.add(new CreateApacheHttpdVirtualHostStep(eachWebServer, eachVirtualHost, Collections.singletonList(eachWebServer)));
			}
		}
	}

	private void destroyApacheVirtualHosts(Map<ApacheHttpdServer, Set<String>> virtualHostsPerWebServer, List<Step> steps) {
		for (Map.Entry<ApacheHttpdServer, Set<String>> eachVirtualHostPerWebServer : virtualHostsPerWebServer.entrySet()) {
			ApacheHttpdServer eachWebServer = eachVirtualHostPerWebServer.getKey();
			for (String eachVirtualHost : eachVirtualHostPerWebServer.getValue()) {
				steps.add(new DestroyApacheHttpdVirtualHostStep(eachWebServer, eachVirtualHost));
			}
		}
	}

	private void restartApacheWebServers(Map<ApacheHttpdServer, Set<String>> virtualHostsPerWebServer, List<Step> steps) {
		for (ApacheHttpdServer eachWebServer : virtualHostsPerWebServer.keySet()) {
			steps.add(new RestartApacheHttpdServerStep(eachWebServer));
		}
	}

}
