/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.was.step;

import static com.google.common.base.Preconditions.checkNotNull;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Transformer;
import org.apache.commons.lang.StringUtils;

import com.google.common.collect.Lists;
import com.xebialabs.deployit.ResolutionException;
import com.xebialabs.deployit.StepExecutionContext;
import com.xebialabs.deployit.ci.Host;
import com.xebialabs.deployit.ci.artifact.Ear;
import com.xebialabs.deployit.ci.artifact.EjbJar;
import com.xebialabs.deployit.ci.artifact.NamedDeployableArtifact;
import com.xebialabs.deployit.ci.artifact.War;
import com.xebialabs.deployit.ci.mapping.EjbReference;
import com.xebialabs.deployit.ci.mapping.MdbListenerPortBinding;
import com.xebialabs.deployit.ci.mapping.ResourceReference;
import com.xebialabs.deployit.hostsession.HostFile;
import com.xebialabs.deployit.hostsession.HostFileUtils;
import com.xebialabs.deployit.hostsession.HostSession;
import com.xebialabs.deployit.plugin.was.ci.SecurityRoleUserGroupMappings;
import com.xebialabs.deployit.plugin.was.ci.WarsWebserversVirtualHostMapping;
import com.xebialabs.deployit.plugin.was.ci.WasCell;
import com.xebialabs.deployit.plugin.was.ci.WasClassLoaderMode;
import com.xebialabs.deployit.plugin.was.ci.WasClassLoaderPolicy;
import com.xebialabs.deployit.plugin.was.ci.WasManagedApacheHttpdServer;
import com.xebialabs.deployit.plugin.was.ci.WasSharedLibrary;
import com.xebialabs.deployit.plugin.was.ci.WasTarget;
import com.xebialabs.deployit.plugin.was.ci.WasWarClassLoaderMapping;
import com.xebialabs.deployit.util.ExtendedStringUtils;

/**
 * Deploys an {@link Ear}, a {@link War} or an {@link EjbJar} to a Collection of {@link WasTarget WasTargets}.
 */
@SuppressWarnings("serial")
public class WasDeployApplicationStep extends WasStepBase {
	protected static final String DEPLOY_APPLICATION_SCRIPT_RESOURCE_PATH = STEP_RESOURCES_PATH + "deploy-application.py";
	protected NamedDeployableArtifact application;
	protected String extension;
	protected Collection<? extends WasTarget> targets;
	protected Collection<WasManagedApacheHttpdServer> webservers;
	protected String virtualHostName;
	private String libraries;
	protected String contextRoot;
	protected int startingWeight = 1;
	protected List<ResourceReference> resourceEnvironmentEntryReferences;
	protected List<ResourceReference> resourceReferences;
	protected List<EjbReference> ejbReferences;
	protected List<MdbListenerPortBinding> mdbListenerPortBindings = Lists.newArrayList();
	protected List<SecurityRoleUserGroupMappings> securityRoleUserGroupMappings = Lists.newArrayList();
	protected List<WarsWebserversVirtualHostMapping> warsWebserversVirtualHostMapping = Lists.newArrayList();
	protected String earClassLoaderPolicy = "";
	protected String classLoaderMode = "";
	protected List<WasWarClassLoaderMapping> warClassLoaderMappings = Lists.newArrayList();

	public WasDeployApplicationStep(WasCell cell, Ear ear, Collection<? extends WasTarget> targets, Collection<WasManagedApacheHttpdServer> webservers,
	        String virtualHostName, final Set<WasSharedLibrary> libraries, int startingWeight, WasClassLoaderMode classLoaderMode,
	        WasClassLoaderPolicy earClassLoaderPolicy, List<WasWarClassLoaderMapping> warClassLoaderMappings,
	        List<SecurityRoleUserGroupMappings> securityRoleUserGroupMappings, List<WarsWebserversVirtualHostMapping> warsWebserversVirtualHostMapping) {
		this(cell, targets, webservers, virtualHostName, libraries);
		this.application = checkNotNull(ear);
		this.extension = ".ear";
		this.contextRoot = "";
		this.resourceEnvironmentEntryReferences = Collections.emptyList();
		this.resourceReferences = Collections.emptyList();
		this.ejbReferences = Collections.emptyList();
		if (securityRoleUserGroupMappings != null) {
			this.securityRoleUserGroupMappings = securityRoleUserGroupMappings;
		}
		if (warsWebserversVirtualHostMapping != null) {
			this.warsWebserversVirtualHostMapping = warsWebserversVirtualHostMapping;
		}
		if (earClassLoaderPolicy != null) {
			this.earClassLoaderPolicy = earClassLoaderPolicy.toString();
		}
		if (classLoaderMode != null) {
			this.classLoaderMode = classLoaderMode.toString();
		}
		if (warClassLoaderMappings != null) {
			this.warClassLoaderMappings = warClassLoaderMappings;
		}
		validate();
		initDescription();
	}

	public WasDeployApplicationStep(WasCell cell, War war, Collection<? extends WasTarget> targets, Collection<WasManagedApacheHttpdServer> webservers,
	        String virtualHostName, Set<WasSharedLibrary> libraries, String contextRoot, int startingWeight, WasClassLoaderMode warclassLoaderMode,
	        List<ResourceReference> resourceEnvironmentJndiNameMappings, List<ResourceReference> resourceReferenceJndiNameMappings,
	        List<EjbReference> ejbReferenceJndiNameMappings, List<SecurityRoleUserGroupMappings> securityRoleUserGroupMappings) {
		this(cell, targets, webservers, virtualHostName, libraries);
		this.application = checkNotNull(war);
		this.extension = ".war";
		this.resourceEnvironmentEntryReferences = resourceEnvironmentJndiNameMappings;
		this.resourceReferences = resourceReferenceJndiNameMappings;
		this.ejbReferences = ejbReferenceJndiNameMappings;
		if (securityRoleUserGroupMappings != null) {
			this.securityRoleUserGroupMappings = securityRoleUserGroupMappings;
		}
		if (warclassLoaderMode != null) {
			this.classLoaderMode = warclassLoaderMode.toString();
		}
		if (contextRoot == null) {
			throw new ResolutionException("Context root definition \"" + contextRoot + "\" is not valid. It is required for a war deployment.");
		}
		this.contextRoot = contextRoot;

		validate();
		initDescription();
	}

	public WasDeployApplicationStep(WasCell cell, EjbJar ejbJar, Collection<? extends WasTarget> targets, final Set<WasSharedLibrary> libraries,
	        int startingWeight, List<MdbListenerPortBinding> mdbListenerPortBindings) {
		this(cell, targets, null, null, libraries);
		this.application = checkNotNull(ejbJar);
		this.extension = ".jar";
		this.contextRoot = "";
		this.resourceEnvironmentEntryReferences = Collections.emptyList();
		this.resourceReferences = Collections.emptyList();
		this.ejbReferences = Collections.emptyList();
		this.mdbListenerPortBindings = mdbListenerPortBindings;

		validate();
		initDescription();
	}

	private WasDeployApplicationStep(WasCell cell, Collection<? extends WasTarget> targets, Collection<WasManagedApacheHttpdServer> webservers,
	        String virtualHostName, final Set<WasSharedLibrary> libraries) {
		super(cell);
		this.targets = checkNotNull(targets);
		if (webservers == null) {
			this.webservers = new HashSet<WasManagedApacheHttpdServer>();
		} else {
			// Make a copy of the Set because it is a EnhancedByCGLib proxy and we lose it when serializing the step.
			this.webservers = new HashSet<WasManagedApacheHttpdServer>(webservers);
		}
		this.virtualHostName = StringUtils.defaultString(virtualHostName, "default_host");
		this.libraries = convertLibrariesToString(libraries);
	}

	private void validate() {
		if (targets.isEmpty()) {
			throw new IllegalArgumentException("Collection of targets is empty");
		}

		for (WasTarget eachTarget : targets) {
			if (!eachTarget.getCell().equals(cell)) {
				throw new IllegalArgumentException(StringUtils.capitalize(eachTarget.getShortTypeDescription()) + " " + eachTarget + " is part of cell "
				        + eachTarget.getCell() + " so it cannot be targeted when deploying to cell " + cell);
			}
		}

		for (WasManagedApacheHttpdServer eachWebServer : webservers) {
			if (!eachWebServer.getNode().getCell().equals(cell)) {
				throw new IllegalArgumentException("WebServer " + eachWebServer + " is part of cell " + eachWebServer.getNode().getCell()
				        + " so it cannot be targeted when deploying to cell " + cell);
			}
		}

		for (WarsWebserversVirtualHostMapping eachMapping : warsWebserversVirtualHostMapping) {
			Set<WasManagedApacheHttpdServer> mappedWebservers = eachMapping.getWebservers();
			if (mappedWebservers != null) {
				for (WasManagedApacheHttpdServer eachWebServer : mappedWebservers) {
					if (!eachWebServer.getNode().getCell().equals(cell)) {
						throw new IllegalArgumentException("WebServer " + eachWebServer + " is part of cell " + eachWebServer.getNode().getCell()
						        + " so it cannot be targeted when deploying to cell " + cell);
					}
				}
			}
		}

		checkResourceReferences(resourceEnvironmentEntryReferences);
		checkResourceReferences(resourceReferences);
		checkEjbReferences(ejbReferences);

		for (MdbListenerPortBinding binding : mdbListenerPortBindings) {
			if (StringUtils.isBlank(binding.getEjbModuleName()) || StringUtils.isBlank(binding.getMdbName()) || StringUtils.isBlank(binding.getListenerPort())) {
				throw new IllegalArgumentException("MDB listener port binding can't contain empty value for module name, mdb name or listener port name");
			}
		}
	}

	private void checkResourceReferences(List<ResourceReference> mappings) {
		for (ResourceReference mapping : mappings) {
			String sourceJndiName = mapping.getResourceReference();
			String targetJndiName = mapping.getTargetResourceJndiName();
			if (StringUtils.isNotBlank(sourceJndiName) && StringUtils.isBlank(targetJndiName)) {
				throw new IllegalArgumentException("Reference with jndi name: " + sourceJndiName + " for artifact " + application.getName()
				        + " must be mapped to a jndi name in the target");
			}
		}
	}

	private void checkEjbReferences(List<EjbReference> mappings) {
		for (EjbReference mapping : mappings) {
			String sourceJndiName = mapping.getResourceReference();
			String classNameString = mapping.getClassName();
			String targetJndiName = mapping.getTargetResourceJndiName();
			if (StringUtils.isNotBlank(sourceJndiName) && (StringUtils.isBlank(classNameString) || StringUtils.isBlank(targetJndiName))) {
				throw new IllegalArgumentException("class value and targetJndiName must be provided for ejb reference: " + sourceJndiName);
			}
		}
	}

	private void initDescription() {
		StringBuilder desc = new StringBuilder();
		desc.append("Deploy application ");
		desc.append(application);
		desc.append(" on ");
		boolean isFirstTarget = true;
		for (WasTarget eachTarget : targets) {
			if (!isFirstTarget) {
				desc.append(" and ");
			} else {
				isFirstTarget = false;
			}
			desc.append(eachTarget.getShortTypeDescription() + " " + eachTarget);
		}
		if (webservers.size() == 1) {
			desc.append(" and webserver " + webservers.iterator().next());
		} else if (webservers.size() > 1) {
			desc.append(" and multiple webservers ");
		}
		desc.append(" in cell ");
		desc.append(cell);
		desc.append(" with virtual host ");
		desc.append(virtualHostName);
		setDescription(desc.toString());
	}

	public boolean execute(StepExecutionContext ctx) {
		HostSession rhs = cell.connectToAdminHost();
		try {
			HostFile uploadedApplicationFile = uploadApplication(rhs);

			String earFilePath = uploadedApplicationFile.getPath().replace('\\', '/');
			StringBuilder targetsString = new StringBuilder();
			for (WasTarget eachTarget : targets) {
				if (targetsString.length() > 0) {
					targetsString.append("+");
				}
				targetsString.append("WebSphere:cell=");
				targetsString.append(cell.getCellName());
				targetsString.append("," + eachTarget.getWasTargetType() + "=");
				targetsString.append(eachTarget.getName());
			}
			for (WasManagedApacheHttpdServer eachWebserver : webservers) {
				targetsString.append("+WebSphere:cell=");
				targetsString.append(cell.getCellName());
				targetsString.append(",node=");
				targetsString.append(eachWebserver.getNode().getNodeName());
				targetsString.append(",server=");
				targetsString.append(eachWebserver.getName());
			}

			int numberOfResourceEnvironmentJndiMappings = resourceEnvironmentEntryReferences.size();
			String[] resourceEnvironmentInfo = new String[numberOfResourceEnvironmentJndiMappings * 2];
			int resourceEnvironmentInfoIndex = 0;
			for (ResourceReference resourceEnvironmentJndiNameMapping : resourceEnvironmentEntryReferences) {
				resourceEnvironmentInfo[resourceEnvironmentInfoIndex++] = resourceEnvironmentJndiNameMapping.getResourceReference();
				resourceEnvironmentInfo[resourceEnvironmentInfoIndex++] = resourceEnvironmentJndiNameMapping.getTargetResourceJndiName();
			}

			int numberOfResourceReferenceMappings = resourceReferences.size();
			String[] resourceReferenceInfo = new String[numberOfResourceReferenceMappings * 2];
			int resourceReferenceInfoIndex = 0;
			for (ResourceReference resourceReferenceJndiNameMapping : resourceReferences) {
				resourceReferenceInfo[resourceReferenceInfoIndex++] = resourceReferenceJndiNameMapping.getResourceReference();
				resourceReferenceInfo[resourceReferenceInfoIndex++] = resourceReferenceJndiNameMapping.getTargetResourceJndiName();
			}

			int numberOfEjbJndiMappings = ejbReferences.size();
			String[] ejbReferenceInfo = new String[numberOfEjbJndiMappings * 3];
			int ejbReferenceInfoIndex = 0;
			for (EjbReference ejbReferenceJndiNameMapping : ejbReferences) {
				ejbReferenceInfo[ejbReferenceInfoIndex++] = ejbReferenceJndiNameMapping.getResourceReference();
				ejbReferenceInfo[ejbReferenceInfoIndex++] = ejbReferenceJndiNameMapping.getClassName();
				ejbReferenceInfo[ejbReferenceInfoIndex++] = ejbReferenceJndiNameMapping.getTargetResourceJndiName();
			}

			int numberOfMdbPortBindings = mdbListenerPortBindings.size();
			String[] mdbPortBindingsInfo = new String[numberOfMdbPortBindings * 3];
			int portBindingIndex = 0;
			for (MdbListenerPortBinding eachBinding : mdbListenerPortBindings) {
				mdbPortBindingsInfo[portBindingIndex++] = eachBinding.getEjbModuleName();
				mdbPortBindingsInfo[portBindingIndex++] = eachBinding.getMdbName();
				mdbPortBindingsInfo[portBindingIndex++] = eachBinding.getListenerPort();
			}

			int numberOfWarClassLoaderMappings = warClassLoaderMappings.size();
			String[] warClassLoaderMappingsInfo = new String[numberOfWarClassLoaderMappings * 2];
			int WarclassLoaderBindingIndex = 0;
			for (WasWarClassLoaderMapping eachBinding : warClassLoaderMappings) {
				warClassLoaderMappingsInfo[WarclassLoaderBindingIndex++] = eachBinding.getWarName();
				if (eachBinding.getClassLoaderMode() != null) {
					warClassLoaderMappingsInfo[WarclassLoaderBindingIndex++] = eachBinding.getClassLoaderMode().toString();
				} else {
					warClassLoaderMappingsInfo[WarclassLoaderBindingIndex++] = "";
				}
			}

			int numberOfSecurityRolesMappings = securityRoleUserGroupMappings.size();
			String[] securityRolesInfo = new String[numberOfSecurityRolesMappings * 3];
			int securityRoleIndex = 0;
			for (SecurityRoleUserGroupMappings eachBinding : securityRoleUserGroupMappings) {
				securityRolesInfo[securityRoleIndex++] = eachBinding.getSecurityRole();
				securityRolesInfo[securityRoleIndex++] = eachBinding.getUsers();
				securityRolesInfo[securityRoleIndex++] = eachBinding.getGroups();
			}

			int numberWarWebserverMappings = warsWebserversVirtualHostMapping.size();
			String[] webServerVHInfo = new String[numberWarWebserverMappings * 3];
			int webserverMappingIndex = 0;
			for (WarsWebserversVirtualHostMapping eachBinding : warsWebserversVirtualHostMapping) {
				webServerVHInfo[webserverMappingIndex++] = eachBinding.getWarName();
				webServerVHInfo[webserverMappingIndex++] = getWebserverTarget(eachBinding);
				webServerVHInfo[webserverMappingIndex++] = eachBinding.getVirtualHost();
			}

			String[] args = { application.getName(), earFilePath, targetsString.toString(), virtualHostName, libraries, contextRoot,
			        Integer.toString(startingWeight), classLoaderMode, earClassLoaderPolicy, Integer.toString(numberOfResourceEnvironmentJndiMappings),
			        Integer.toString(numberOfResourceReferenceMappings), Integer.toString(numberOfEjbJndiMappings), Integer.toString(numberOfMdbPortBindings),
			        Integer.toString(numberOfSecurityRolesMappings), Integer.toString(numberWarWebserverMappings),
			        Integer.toString(numberOfWarClassLoaderMappings) };

			String[] allArgs = new String[args.length + resourceEnvironmentInfo.length + resourceReferenceInfo.length + ejbReferenceInfo.length
			        + mdbPortBindingsInfo.length + securityRolesInfo.length + webServerVHInfo.length + warClassLoaderMappingsInfo.length];

			List<String> allArgsList = Lists.newArrayList();
			allArgsList.addAll(Arrays.asList(args));
			allArgsList.addAll(Arrays.asList(webServerVHInfo));
			allArgsList.addAll(Arrays.asList(resourceEnvironmentInfo));
			allArgsList.addAll(Arrays.asList(resourceReferenceInfo));
			allArgsList.addAll(Arrays.asList(ejbReferenceInfo));
			allArgsList.addAll(Arrays.asList(mdbPortBindingsInfo));
			allArgsList.addAll(Arrays.asList(securityRolesInfo));
			allArgsList.addAll(Arrays.asList(warClassLoaderMappingsInfo));
			allArgs = allArgsList.toArray(allArgs);

			int res = executeWsadminJythonScriptResource(ctx, rhs, DEPLOY_APPLICATION_SCRIPT_RESOURCE_PATH, allArgs);
			return res == 0;
		} finally {
			rhs.close();
		}
	}

	private String getWebserverTarget(WarsWebserversVirtualHostMapping eachBinding) {
		StringBuilder targetsString = new StringBuilder();
		Set<WasManagedApacheHttpdServer> webservers = eachBinding.getWebservers();
		for (WasTarget eachTarget : targets) {
			if (targetsString.length() > 0) {
				targetsString.append("+");
			}
			targetsString.append("WebSphere:cell=");
			targetsString.append(cell.getCellName());
			targetsString.append("," + eachTarget.getWasTargetType() + "=");
			targetsString.append(eachTarget.getName());
		}

		for (WasManagedApacheHttpdServer wasUnManagedApacheHttpdServer : webservers) {
			targetsString.append("+WebSphere:cell=");
			targetsString.append(cell.getCellName());
			targetsString.append(",node=");
			targetsString.append(wasUnManagedApacheHttpdServer.getNode().getNodeName());
			targetsString.append(",server=");
			targetsString.append(wasUnManagedApacheHttpdServer.getName());
		}

		return targetsString.toString();
	}

	private HostFile uploadApplication(HostSession rhs) {
		HostSession lhs = Host.getLocalHost().getHostSession();
		try {
			HostFile applicationFileToUpload = lhs.getFile(application.getLocation());
			HostFile uploadedApplicationFile = rhs.getTempFile(application.getName(), extension);
			HostFileUtils.copy(applicationFileToUpload, uploadedApplicationFile);
			return uploadedApplicationFile;
		} finally {
			lhs.close();
		}
	}

	private String convertLibrariesToString(final Set<WasSharedLibrary> libraries) {
		if (CollectionUtils.isNotEmpty(libraries)) {
			return ExtendedStringUtils.join(libraries, new Transformer() {
				public String transform(Object obj) {
					return ((WasSharedLibrary) obj).getName();
				}
			});
		}
		return "";
	}

	public NamedDeployableArtifact getApplication() {
		return application;
	}

	public String getExtension() {
		return extension;
	}

	public WasCell getCell() {
		return cell;
	}

	public Collection<? extends WasTarget> getTargets() {
		return targets;
	}

	public Collection<WasManagedApacheHttpdServer> getWebservers() {
		return webservers;
	}

	public String getVirtualHostName() {
		return virtualHostName;
	}

	public String getContextRoot() {
		return contextRoot;
	}

}
