/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.was.ci;

import java.util.Collection;
import java.util.Map;

import com.xebialabs.deployit.StepExecutionContext;
import com.xebialabs.deployit.ci.Host;
import com.xebialabs.deployit.hostsession.HostSession;

/**
 * A WAS cell is an administrative domain that holds the configuration for one or more WAS server, clusters, etc.
 */
public interface WasCell extends WasScope {

	/**
	 * Returns the name of the cell.
	 * 
	 * @return the name
	 */
	String getCellName();

	/**
	 * Sets the name of the cell (used during discovery).
	 * 
	 * @param cellName
	 *            the name of the cell.
	 */
	void setCellName(String cellName);

	/**
	 * Returns the version of WAS.
	 * 
	 * @return the version.
	 */
	WasVersion getVersion();

	/**
	 * Sets the version of WAS (used during discovery).
	 * 
	 * @param version
	 *            the version.
	 */
	void setVersion(WasVersion version);

	/**
	 * Returns the profile directory of WAS on the administrative host.
	 * 
	 * @return the profile directory
	 */
	String getWasHome();

	/**
	 * Sets the profile directory of WAS on the administrative host.
	 * 
	 * @param wasHome
	 *            the profile directory
	 */
	void setWasHome(String wasHome);

	/**
	 * Returns the administrative port.
	 * 
	 * @return the port
	 */
	int getPort();

	/**
	 * Sets the administrative port (used during discovery).
	 * 
	 * @param port
	 *            the port
	 */
	void setPort(int port);

	/**
	 * Returns the host the cell is administered on.
	 * 
	 * @return the host
	 */
	Host getCellHost();

	/**
	 * Returns the username which is used to login to the WebSphere deployment manager
	 * 
	 * @return the username
	 */
	String getUsername();

	/**
	 * Returns the password which is used to login to the WebSphere deployment manager
	 * 
	 * @return the password
	 */
	String getPassword();

	/**
	 * Returns the path of the "config" directory in the WAS installation.
	 * 
	 * @return the path of the "config" directory
	 */
	String getConfigDirPath();

	/**
	 * Returns the path to the plugin config file in the WAS installation.
	 * 
	 * @return the path to the plugin config file
	 */
	String getPluginFilePath();

	/**
	 * Returns the path to the versionInfo command in the WAS installation.
	 * 
	 * @return the path to the versionInfo command
	 */
	String getVersionInfoPath();

	/**
	 * Returns the path to the wsadmin command in the WAS installation.
	 * 
	 * @return the path to the wsadmin command
	 */
	String getWsadminPath();

	/**
	 * Returns the path to the startServer command in the WAS installation.
	 * 
	 * @return the path to the startServer command
	 */
	String getStartServerPath();

	/**
	 * Returns the path to the stopServer command in the WAS installation.
	 * 
	 * @return the path to the stopServer command
	 */
	String getStopServerPath();

	/**
	 * Returns the path to the serverStatus command in the WAS installation.
	 * 
	 * @return the path to the serverStatus command
	 */
	String getServerStatusPath();

	/**
	 * Connects to the administrative host.
	 * 
	 * @return a {@link HostSession} connected to the administrative host. The caller must close it.
	 */
	HostSession connectToAdminHost();

	/**
	 * Executes a wsadmin Jython script
	 * 
	 * @param context
	 *            the step execution context to use
	 * @param session
	 *            the host session to use
	 * @param scriptPath
	 *            the path to the script as it can be found in the classpath.
	 * @param arguments
	 *            the arguments to be passed to the script
	 * @return the exit code of the script
	 */
	int executeWsadminJythonScript(StepExecutionContext context, HostSession session, String scriptPath, String... arguments);

	/**
	 * Returns information about an object in the WAS configuration tree identified by its containmenth path.
	 * 
	 * @param ctx
	 *            the step execution context to use
	 * @param objectContainmentPath
	 *            the containmenth path of the object
	 * @return the properties of the object or <code>null</code> if it could not be found
	 */
	Map<String, String> getObjectInfo(StepExecutionContext ctx, String objectContainmentPath);

	/**
	 * Returns information about an object in the WAS configuration tree identified by its containmenth path and a sublevel name.
	 * 
	 * @param ctx
	 *            the step execution context to use
	 * @param objectContainmentPath
	 *            the containmenth path of the object
	 * @param subLevelName
	 *            the sublevel name
	 * @return the properties of the object or <code>null</code> if it could not be found
	 */
	Map<String, String> getObjectInfoWithSubLevelByName(StepExecutionContext ctx, String objectContainmentPath, String subLevelName);

	/**
	 * Returns information about an object in the WAS configuration tree identified by its containmenth path and a sublevel type.
	 * 
	 * @param ctx
	 *            the step execution context to use
	 * @param objectContainmentPath
	 *            the containmenth path of the object
	 * @param subLevelType
	 *            the sublevel type
	 * @return the properties of the object or <code>null</code> if it could not be found
	 */
	Map<String, String> getObjectInfoWithSubLevelByType(StepExecutionContext ctx, String objectContainmentPath, String subLevelType);

	/**
	 * Returns information about an object in the WAS configuration tree identified by its containmenth path and a sublevel name and type.
	 * 
	 * @param ctx
	 *            the step execution context to use
	 * @param objectContainmentPath
	 *            the containmenth path of the object
	 * @param subLevelName
	 *            the sublevel name
	 * @param subLevelType
	 *            the sublevel type
	 * @return the properties of the object or <code>null</code> if it could not be found
	 */
	Map<String, String> getObjectInfoWithSubLevelByTypeAndName(StepExecutionContext ctx, String objectContainmentPath, String subLevelType, String subLevelName);

	/**
	 * Returns the names of the properties of an object in the WAS configuration tree identified by its containmenth path and a sublevel type.
	 * 
	 * @param ctx
	 *            the step execution context to use
	 * @param objectContainmentPath
	 *            the containmenth path of the object
	 * @param subLevelType
	 *            the sublevel type
	 * @return the names of the properties of the object or <code>null</code> if it could not be found
	 */
	Collection<String> getObjectInfoWithSubLevelByTypeNamesOnly(StepExecutionContext ctx, String objectContainmentPath, String subLevelType);

	/**
	 * Returns information about the cell in the WAS configuration (which is the root object).
	 * 
	 * @param ctx
	 *            the step execution context to use
	 * @return the properties of the cell or <code>null</code> if it could not be found
	 */
	Map<String, String> getCellPropertiesInfo(StepExecutionContext ctx);

}
