/*
 * Copyright (c) 2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of Xebialabs Software and Documentation is subject to the Personal
 * License Agreement.
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * ‚ÄúDocumentation‚Äù means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the Xebialabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to  (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.wls.runbook;

import java.io.Serializable;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.xebialabs.deployit.Change;
import com.xebialabs.deployit.ChangePlan;
import com.xebialabs.deployit.RunBook;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.ci.Host;
import com.xebialabs.deployit.ci.artifact.ConfigurationFiles;
import com.xebialabs.deployit.ci.artifact.Ear;
import com.xebialabs.deployit.ci.mapping.Mapping;
import com.xebialabs.deployit.mapper.artifact.ConfigurationFilesToHostMapper;
import com.xebialabs.deployit.mapper.artifact.LibrariesToHostMapper;
import com.xebialabs.deployit.plugin.apache.httpd.mapper.StaticContentToApacheHttpdServerMapper;
import com.xebialabs.deployit.plugin.wls.ci.WlsCluster;
import com.xebialabs.deployit.plugin.wls.ci.WlsDataSource;
import com.xebialabs.deployit.plugin.wls.ci.WlsDomain;
import com.xebialabs.deployit.plugin.wls.ci.WlsJmsQueue;
import com.xebialabs.deployit.plugin.wls.ci.WlsServer;
import com.xebialabs.deployit.plugin.wls.mapper.ApacheHttpdWlsPluginConfigurationToApacheHttpdServerMapper;
import com.xebialabs.deployit.plugin.wls.mapper.EarToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.EarToWlsServerMapper;
import com.xebialabs.deployit.plugin.wls.mapper.EjbToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.EjbToWlsServerMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WarToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WarToWlsServerMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WlsDataSourceToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WlsForeignDestinationToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WlsForeignJmsConnectionFactoryToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WlsForeignServerToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WlsJmsConnectionFactoryToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WlsJmsModuleToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WlsJmsQueueToWlsServerMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WlsMaximumThreadsConstraintToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WlsQueueSetToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WlsWorkManagerToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.step.ActivatePendingChangesStep;
import com.xebialabs.deployit.plugin.wls.step.RestartServersThatRequireRestartStep;

/**
 * Supports the initial deployment, re- and undeployment of packages to a {@link WlsCluster WLS Cluster}, as well as modifications to the targeted
 * {@link WlsServer Managed Servers} required for the application.
 * 
 * <h4>Conditions</h4>
 * 
 * This runbook will trigger if the change plan contains the
 * 
 * <ul>
 * <li>addition of a new {@link Deployment} CI
 * <li>modification of an existing {@code Deployment} CI
 * <li>deletion of an existing {@code Deployment} CI
 * </ul>
 * 
 * where the <strong>deployment package contains WLS resources</strong> or if components (e.g. {@link Ear EARs} are <strong>targeted at WLS middleware
 * components.</strong>
 * <p>
 * The runbook will also handle changes to the following properties of {@link WlsServer Managed Servers} <u>to which a resource has been targeted in the current
 * change plan</u>:
 * 
 * <ul>
 * <li>{@link WlsServer#setPort(int) port}
 * <li>{@link WlsServer#setClasspath(String) classpath}
 * <li>{@link WlsServer#setBootClasspath(String) bootClasspath}
 * <li>{@link WlsServer#setInitHeapSize(int) initHeapSize}
 * <li>{@link WlsServer#setMaxHeapSize(int) maxHeapSize}
 * <li>{@link WlsServer#setArguments(String) arguments}
 * </ul>
 * 
 * Modification of <em>other</em> properties of Managed Servers, or of servers which are <em>not</em> targeted by the current deployment, are <strong>not
 * supported</strong>.
 * <h4>Update Semantics</h4>
 * 
 * If the runbook triggers because an existing {@code Deployment} has been modified, the contents of the previous deployment package are compared with the
 * contents of the new package and the following actions are performed:
 * 
 * <ul>
 * <li>if the new deployment package does <u>not</u> contain an element that was present in the previous package, this is <strong>removed</strong> from the
 * environment
 * <li>if the new deployment package contains an element that was <u>not</u> present in the previous package, it is <strong>added</strong>
 * <li>for <em>binary</em> resources ({@link Ear EARs}, {@link ConfigurationFiles} etc.) that are contained in <u>both</u> the new and previous packages, the
 * current resource is <strong>removed</strong> and the new one <strong>added</strong>
 * <li>for <em>configuration</em> resources ({@link WlsJmsQueue JMS Queues}, {@link WlsDataSource JDBC Datasources} etc.) that are contained in <u>both</u> the
 * new and previous packages, the current resource is <strong>modified</strong> if its properties in the new package do not match those in the previous package
 * </ul>
 * 
 * <h4>Actions</h4>
 * 
 * TODO: Step sequence. Note that JMS Queues are not deleted if non-empty.
 */
@SuppressWarnings("unchecked")
public class WlsDeploymentRunBook extends WlsSingleTypeHandlingRunBook<Deployment> implements RunBook {

	public WlsDeploymentRunBook() {
		super(Deployment.class);
	}

	@Override
	protected void resolve(ChangePlan changePlan, Change<Deployment> change, List<Step> steps) {
		EarToWlsClusterMapper earToClusterMapper = new EarToWlsClusterMapper(change);
		EarToWlsServerMapper earToServerMapper = new EarToWlsServerMapper(change);
		EjbToWlsClusterMapper ejbToClusterMapper = new EjbToWlsClusterMapper(change);
		EjbToWlsServerMapper ejbToServerMapper = new EjbToWlsServerMapper(change);
		WarToWlsClusterMapper warToClusterMapper = new WarToWlsClusterMapper(change);
		WarToWlsServerMapper warToServerMapper = new WarToWlsServerMapper(change);
		WlsDataSourceToWlsClusterMapper dataSourceMapper = new WlsDataSourceToWlsClusterMapper(change);
		WlsJmsModuleToWlsClusterMapper jmsModuleMapper = new WlsJmsModuleToWlsClusterMapper(change);
		WlsQueueSetToWlsClusterMapper queueSetMapper = new WlsQueueSetToWlsClusterMapper(change);
		WlsJmsQueueToWlsServerMapper queueMapper = new WlsJmsQueueToWlsServerMapper(change);
		sortQueueMappings(queueMapper);
		WlsJmsConnectionFactoryToWlsClusterMapper jmsConnectionFactoryMapper = new WlsJmsConnectionFactoryToWlsClusterMapper(change);
		WlsForeignServerToWlsClusterMapper foreignServerMapper = new WlsForeignServerToWlsClusterMapper(change);
		WlsForeignDestinationToWlsClusterMapper foreignDestinationMapper = new WlsForeignDestinationToWlsClusterMapper(change);
		WlsForeignJmsConnectionFactoryToWlsClusterMapper foreignJmsConnectionFactoryMapper = new WlsForeignJmsConnectionFactoryToWlsClusterMapper(change);
		WlsMaximumThreadsConstraintToWlsClusterMapper maxThreadsConstraintsMapper = new WlsMaximumThreadsConstraintToWlsClusterMapper(change);
		WlsWorkManagerToWlsClusterMapper workManagerMapper = new WlsWorkManagerToWlsClusterMapper(change);
		ApacheHttpdWlsPluginConfigurationToApacheHttpdServerMapper pluginConfigMapper = new ApacheHttpdWlsPluginConfigurationToApacheHttpdServerMapper(change);
		StaticContentToApacheHttpdServerMapper staticContentMapper = new StaticContentToApacheHttpdServerMapper(change);
		LibrariesToHostMapper librariesMapper = new LibrariesToHostMapper(change);
		ConfigurationFilesToHostMapper configurationFilesToHostMapper = new ConfigurationFilesToHostMapper(change);

		Set<WlsCluster> affectedClusters = new HashSet<WlsCluster>();
		affectedClusters.addAll(dataSourceMapper.getAffectedTargets());
		affectedClusters.addAll(jmsModuleMapper.getAffectedTargets());
		affectedClusters.addAll(queueSetMapper.getAffectedTargets());
		affectedClusters.addAll(jmsConnectionFactoryMapper.getAffectedTargets());
		affectedClusters.addAll(foreignServerMapper.getAffectedTargets());
		affectedClusters.addAll(foreignDestinationMapper.getAffectedTargets());
		affectedClusters.addAll(foreignJmsConnectionFactoryMapper.getAffectedTargets());
		affectedClusters.addAll(maxThreadsConstraintsMapper.getAffectedTargets());
		affectedClusters.addAll(workManagerMapper.getAffectedTargets());
		affectedClusters.addAll(earToClusterMapper.getAffectedTargets());
		affectedClusters.addAll(warToClusterMapper.getAffectedTargets());
		affectedClusters.addAll(ejbToClusterMapper.getAffectedTargets());

		Set<WlsServer> affectedServers = new HashSet<WlsServer>();
		affectedServers.addAll(queueMapper.getAffectedTargets());
		affectedServers.addAll(earToServerMapper.getAffectedTargets());
		affectedServers.addAll(warToServerMapper.getAffectedTargets());
		affectedServers.addAll(ejbToServerMapper.getAffectedTargets());

		Set<WlsCluster> allClusters = new HashSet<WlsCluster>();
		allClusters.addAll(dataSourceMapper.getAllTargets());
		allClusters.addAll(jmsModuleMapper.getAllTargets());
		allClusters.addAll(queueSetMapper.getAllTargets());
		allClusters.addAll(jmsConnectionFactoryMapper.getAllTargets());
		allClusters.addAll(foreignServerMapper.getAllTargets());
		allClusters.addAll(foreignDestinationMapper.getAllTargets());
		allClusters.addAll(foreignJmsConnectionFactoryMapper.getAllTargets());
		allClusters.addAll(maxThreadsConstraintsMapper.getAllTargets());
		allClusters.addAll(workManagerMapper.getAllTargets());
		allClusters.addAll(earToClusterMapper.getAllTargets());
		allClusters.addAll(warToClusterMapper.getAllTargets());
		allClusters.addAll(ejbToClusterMapper.getAllTargets());

		Set<WlsServer> allServers = new HashSet<WlsServer>();
		allServers.addAll(queueMapper.getAllTargets());
		allServers.addAll(earToServerMapper.getAllTargets());
		allServers.addAll(warToServerMapper.getAllTargets());
		allServers.addAll(ejbToServerMapper.getAllTargets());

		if (isWlsDeployment(allClusters, allServers)) {
			return;
		}

		Set<WlsDomain> affectedDomains = new HashSet<WlsDomain>();
		for (WlsCluster eachCluster : affectedClusters) {
			affectedDomains.add(eachCluster.getDomain());
		}

		for (WlsServer eachServer : affectedServers) {
			affectedDomains.add(eachServer.getDomain());
		}

		// undeploy apps
		int lastStepCount = steps.size();
		earToClusterMapper.generateUndeploySteps(steps);
		earToServerMapper.generateUndeploySteps(steps);
		warToClusterMapper.generateUndeploySteps(steps);
		warToServerMapper.generateUndeploySteps(steps);
		ejbToClusterMapper.generateUndeploySteps(steps);
		ejbToServerMapper.generateUndeploySteps(steps);
		if (steps.size() > lastStepCount) {
			activateChanges(steps, affectedDomains);
		}

		// destroy resources
		lastStepCount = steps.size();
		queueMapper.generateDeletionSteps(steps);
		queueSetMapper.generateDeletionSteps(steps);
		jmsModuleMapper.generateDeletionSteps(steps);
		jmsConnectionFactoryMapper.generateDeletionSteps(steps);
		foreignServerMapper.generateDeletionSteps(steps);
		foreignDestinationMapper.generateDeletionSteps(steps);
		foreignJmsConnectionFactoryMapper.generateDeletionSteps(steps);
		dataSourceMapper.generateDeletionSteps(steps);
		workManagerMapper.generateDeletionSteps(steps);
		maxThreadsConstraintsMapper.generateDeletionSteps(steps);
		if (steps.size() > lastStepCount) {
			activateChanges(steps, affectedDomains);
		}

		configurationFilesToHostMapper.generateDeletionSteps(steps);
		configurationFilesToHostMapper.generateAdditionSteps(steps);
		librariesMapper.generateDeletionSteps(steps);
		librariesMapper.generateAdditionSteps(steps);

		// Modify any servers if these were changed too, but delegate that to the WlsServerRunBook
		if (change.isModification()) {
			Set<WlsServer> oldTargetServers = new HashSet<WlsServer>();
			oldTargetServers.addAll(earToServerMapper.getOldTargets());
			oldTargetServers.addAll(warToServerMapper.getOldTargets());
			oldTargetServers.addAll(ejbToServerMapper.getOldTargets());

			Set<WlsCluster> oldTargetClusters = new HashSet<WlsCluster>();
			oldTargetClusters.addAll(earToClusterMapper.getOldTargets());
			oldTargetClusters.addAll(warToClusterMapper.getOldTargets());
			oldTargetClusters.addAll(ejbToClusterMapper.getOldTargets());
			for (WlsCluster cluster : oldTargetClusters) {
				oldTargetServers.addAll(cluster.getServers());
			}

			Set<WlsServer> newTargetServers = new HashSet<WlsServer>();
			newTargetServers.addAll(earToServerMapper.getNewTargets());
			newTargetServers.addAll(warToServerMapper.getNewTargets());
			newTargetServers.addAll(ejbToServerMapper.getNewTargets());

			Set<WlsCluster> newTargetClusters = new HashSet<WlsCluster>();
			newTargetClusters.addAll(earToClusterMapper.getNewTargets());
			newTargetClusters.addAll(warToClusterMapper.getNewTargets());
			newTargetClusters.addAll(ejbToClusterMapper.getNewTargets());
			for (WlsCluster cluster : newTargetClusters) {
				newTargetServers.addAll(cluster.getServers());
			}

			Set<Change<WlsServer>> serverChangesForDeploymentChange = findServerChangesForDeploymentChange(changePlan, change, oldTargetServers,
					newTargetServers);
			if (serverChangesForDeploymentChange != null) {
				WlsServerRunBook wlsServerRunBook = new WlsServerRunBook();
				for (Change<WlsServer> eachServerChange : serverChangesForDeploymentChange) {
					steps.addAll(wlsServerRunBook.getModificationSteps(eachServerChange));
				}
			}
		}

		// modify queues
		lastStepCount = steps.size();
		queueMapper.generateModificationSteps(steps);
		if (steps.size() > lastStepCount) {
			activateChanges(steps, affectedDomains);
		}

		// create resources
		lastStepCount = steps.size();
		jmsModuleMapper.generateAdditionSteps(steps);
		queueSetMapper.generateAdditionSteps(steps);
		queueMapper.generateAdditionSteps(steps);
		jmsConnectionFactoryMapper.generateAdditionSteps(steps);
		foreignServerMapper.generateAdditionSteps(steps);
		foreignDestinationMapper.generateAdditionSteps(steps);
		foreignJmsConnectionFactoryMapper.generateAdditionSteps(steps);
		dataSourceMapper.generateAdditionSteps(steps);
		maxThreadsConstraintsMapper.generateAdditionSteps(steps);
		workManagerMapper.generateAdditionSteps(steps);
		if (steps.size() > lastStepCount) {
			activateChanges(steps, affectedDomains);
		}

		// deploy apps
		lastStepCount = steps.size();
		earToClusterMapper.generateDeploySteps(steps);
		earToServerMapper.generateDeploySteps(steps);
		warToClusterMapper.generateDeploySteps(steps);
		warToServerMapper.generateDeploySteps(steps);
		ejbToClusterMapper.generateDeploySteps(steps);
		ejbToServerMapper.generateDeploySteps(steps);
		if (steps.size() > lastStepCount) {
			activateChanges(steps, affectedDomains);
		}

		// start apps
		earToClusterMapper.generateStartSteps(steps);
		earToServerMapper.generateStartSteps(steps);
		warToClusterMapper.generateStartSteps(steps);
		warToServerMapper.generateStartSteps(steps);
		ejbToClusterMapper.generateStartSteps(steps);
		ejbToServerMapper.generateStartSteps(steps);

		restartServersRequiringRestart(steps, affectedDomains);

		staticContentMapper.generateDeletionSteps(steps);
		staticContentMapper.generateAdditionSteps(steps);
		pluginConfigMapper.generateDeletionSteps(steps);
		pluginConfigMapper.generateAdditionSteps(steps);
	}

	private boolean isWlsDeployment(Set<WlsCluster> targetClusters, Set<WlsServer> targetServers) {
		return targetClusters.isEmpty() && targetServers.isEmpty();
	}

	private void sortQueueMappings(WlsJmsQueueToWlsServerMapper queueMapper) {
		Collections.sort(queueMapper.getAddedMappings(), new WlqJmsQueueSorter(true));
		Collections.sort(queueMapper.getDeletedMappings(), new WlqJmsQueueSorter(false));
	}

	private static final class WlqJmsQueueSorter implements Comparator<Mapping> {
		private boolean errorQueuesFirst;

		public WlqJmsQueueSorter(boolean errorQueuesFirst) {
			this.errorQueuesFirst = errorQueuesFirst;
		}

		public int compare(Mapping o1, Mapping o2) {
			WlsJmsQueue lhs = (WlsJmsQueue) o1.getSource();
			WlsJmsQueue rhs = (WlsJmsQueue) o2.getSource();
			int res;
			if (lhs.getErrorQueue() == null) {
				if (rhs.getErrorQueue() == null) {
					res = 0;
				} else {
					res = 1;
				}
			} else {
				if (rhs.getErrorQueue() == null) {
					res = -1;
				} else {
					res = 0;
				}
			}
			if (res == 0) {
				res = lhs.getName().compareTo(rhs.getName());
			}
			if (errorQueuesFirst) {
				res = -res;
			}
			return res;
		}
	}

	private void activateChanges(List<Step> steps, Set<WlsDomain> domains) {
		for (WlsDomain eachDomain : domains) {
			steps.add(new ActivatePendingChangesStep(eachDomain));
		}
	}

	private void restartServersRequiringRestart(List<Step> steps, Set<WlsDomain> domains) {
		for (WlsDomain eachDomain : domains) {
			steps.add(new RestartServersThatRequireRestartStep(eachDomain));
		}
	}

	private Set<Change<WlsServer>> findServerChangesForDeploymentChange(ChangePlan changePlan, Change<Deployment> deploymentChange,
			Set<WlsServer> oldTargetServers, Set<WlsServer> newTargetServers) {
		Set<Change<WlsServer>> serverChanges = new HashSet<Change<WlsServer>>();

		for (Change<WlsServer> serverChange : getChangesForCiType(changePlan, WlsServer.class)) {
			if (serverChange.isModification()) {
				if (oldTargetServers.contains(serverChange.getOldRevision()) && newTargetServers.contains(serverChange.getNewRevision())) {
					serverChanges.add(serverChange);
				}
			}
		}
		return serverChanges;
	}

	private Set<Change> getChangesForCiType(ChangePlan cp, Class<? extends Serializable> changeTypeClass) {
		Set<Change> changesOfType = new HashSet<Change>();
		for (Change change : cp.getChanges()) {
			if (changeTypeClass.equals(change.getConfigurationItemClass())) {
				changesOfType.add(change);
			}
		}
		return changesOfType;

	}

}