package com.xebialabs.deployit.plugin.wls.step;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.VelocityEngine;
import org.springframework.core.io.ClassPathResource;

import com.xebialabs.deployit.StepExecutionContext;
import com.xebialabs.deployit.hostsession.HostFile;
import com.xebialabs.deployit.hostsession.HostSession;
import com.xebialabs.deployit.plugin.apache.httpd.ci.ApacheHttpdServer;
import com.xebialabs.deployit.plugin.wls.ci.ApacheHttpdWlsPluginConfiguration;
import com.xebialabs.deployit.plugin.wls.ci.WlsCluster;
import com.xebialabs.deployit.plugin.wls.ci.WlsServer;

/**
 * Creates the configuration for the Apache WebLogic plugin on the basis of a
 * {@link ApacheHttpdWlsPluginConfiguration}.
 */
@SuppressWarnings("serial")
public class CreateApacheHttpdWlsPluginConfigFileStep extends ApacheWlsPluginConfigFileStep {

	public CreateApacheHttpdWlsPluginConfigFileStep(ApacheHttpdServer webServer, ApacheHttpdWlsPluginConfiguration pluginConfiguration, WlsCluster cluster,
			String virtualHost) {
		super(webServer, pluginConfiguration, cluster, virtualHost);
		setDescription("Generate Weblogic Apache Plugin configuration file on " + webServer.getHost());
	}

	public CreateApacheHttpdWlsPluginConfigFileStep(ApacheHttpdServer webServer, ApacheHttpdWlsPluginConfiguration pluginConfiguration, WlsServer server,
			String virtualHost) {
		super(webServer, pluginConfiguration, server, virtualHost);
		setDescription("Generate Weblogic Apache Plugin configuration file on " + webServer.getHost());
	}

	public boolean execute(StepExecutionContext ctx) {
		HostSession hostSession = webServer.connectToAdminHost();
		try {
			ctx.logOutput("Creating Apache Weblogic configuration file at " + wlConfigFilePath);

			HostFile apacheWlConfFile = hostSession.getFile(wlConfigFilePath);
			String apacheWLConfiguration = resolveUsingVelocity();
			if (logger.isDebugEnabled()) {
				logger.debug("resolved wl config template:" + apacheWLConfiguration);
			}
			apacheWlConfFile.put(IOUtils.toInputStream(apacheWLConfiguration), apacheWLConfiguration.length());
			return true;
		} finally {
			hostSession.close();
		}
	}

	private String resolveUsingVelocity() {
		VelocityEngine ve = new VelocityEngine();
		try {
			ve.init();
		} catch (Exception exc) {
			throw new IllegalStateException("Cannot initialize Velocity templating engine", exc);
		}
		// populate context for velocity use
		Map<String, Object> velocityContext = getVelocityContext();
		VelocityContext vc = new VelocityContext(velocityContext);

		ClassPathResource scriptResource = new ClassPathResource(DEFAULT_WL_APACHE_VHOST_TEMPLATE);
		InputStream scriptIn;
		try {
			scriptIn = scriptResource.getInputStream();
		} catch (IOException exc) {
			throw new IllegalArgumentException("Cannot read script template resource " + DEFAULT_WL_APACHE_VHOST_TEMPLATE, exc);
		}

		String evaluatedScript;
		try {
			StringWriter evaluatedTemplateWriter = new StringWriter();
			ve.evaluate(vc, evaluatedTemplateWriter, " ", new InputStreamReader(scriptIn, "UTF-8"));
			evaluatedScript = evaluatedTemplateWriter.getBuffer().toString();
		} catch (IOException exc) {
			throw new RuntimeException("Cannot evaluate script template resource " + DEFAULT_WL_APACHE_VHOST_TEMPLATE, exc);
		} finally {
			IOUtils.closeQuietly(scriptIn);
		}
		return evaluatedScript;
	}

	private Map<String, Object> getVelocityContext() {
		Map<String, Object> velocityContext = new HashMap<String, Object>();
		velocityContext.put("usingVhost", usingVirtualHost);
		if (usingVirtualHost) {
			velocityContext.put("vhostServerName", virtualHostServerName);
			velocityContext.put("vhostPort", virtualHostPort);
			velocityContext.put("documentroot", webServer.getHtdocsDirPathForVirtualHost(virtualHost));
		}
		velocityContext.put("clusterSpecification", clusterSpecification);
		if (!StringUtils.isBlank(pluginConfiguration.getErrorPage())) {
			velocityContext.put("errorPage", pluginConfiguration.getErrorPage());
		}

		List<String> mimeExpressions = new ArrayList<String>();
		if (!StringUtils.isBlank(pluginConfiguration.getMimeMatchExpressions())) {
			String[] mimeMatchExpressions = StringUtils.split(pluginConfiguration.getMimeMatchExpressions(), ",");
			mimeExpressions.addAll(Arrays.asList(mimeMatchExpressions));
		}
		velocityContext.put("matchExpressions", mimeExpressions);

		List<String> pathExpressions = new ArrayList<String>();
		if (!StringUtils.isBlank(pluginConfiguration.getPathExpressions())) {
			String[] pathExpressionsArr = StringUtils.split(pluginConfiguration.getPathExpressions(), ",");
			pathExpressions.addAll(Arrays.asList(pathExpressionsArr));
		}
		velocityContext.put("locations", pathExpressions);
		return velocityContext;
	}

	private static Logger logger = Logger.getLogger(CreateApacheHttpdWlsPluginConfigFileStep.class);

}
